/*
 * Created on Apr 17, 2008
 *
 * (c) 2006-2007 dka - edv, media, webdesign
 *
 */
package com.dkaedv.asteroids.data;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.util.ConcurrentModificationException;
import java.util.List;
import java.util.Vector;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.dkaedv.asteroids.ki.ShotTargetList;
import com.dkaedv.asteroids.ki.TargetList;
import com.dkaedv.asteroids.net.DisplayDatagram;

/**
 * Game status representation.
 *
 * Holds all information a DisplayDatagram carries as well plus some more.
 *
 * @author Daniel Kleine-Albers
 *
 */
public class GameStatus extends DisplayDatagram implements IDrawable {
    private Log log = LogFactory.getLog(GameStatus.class);

    private int collisionStatus = 1; // 1 green, 2 yellow (thrust), 3 red (warp)
    private com.dkaedv.asteroids.data.Vector hitVector = null;
    private TargetList targetList;
    private ShotTargetList shotTargetList;

    private List<Double> shotSpeeds = new Vector<Double>();

    ThreadLocal<Long> waitTimestamp = new ThreadLocal<Long>();

    public GameStatus() {
        shotSpeeds.add(490.0);
    }

    public void draw(Graphics2D g, Graphics2D textG) {

        try {
            for (Asteroid asteroid : getAsteroids()) {
                asteroid.draw(g, textG);
            }
        } catch (ConcurrentModificationException e) {
            // Ignore exception
        }

        try {
            for (Shot shot : getShots()) {
                shot.draw(g, textG);

                // only use value if shot is around for at least 50 ms
                if ( (System.currentTimeMillis() - shot.getPosition().getLastVector().timestamp) > 50
                        && shot.getType() == Shot.TYPE_SHIP) {
                    double speed = shot.getPosition().getSpeed();

                    // only add realistic values
                    if (speed > 400 && speed < 600) {
                        //log.debug(speed);
                        shotSpeeds.add(speed);

                        if (shotSpeeds.size() > 30) {
                            shotSpeeds.remove(0);
                        }
                    }
                }

            }
        } catch (ConcurrentModificationException e) {
            // Ignore exception
        }

        if (getShip() != null) {
            getShip().draw(g, textG);
        }

        if (getUfo() != null) {
            getUfo().draw(g, textG);
        }

        // Draw collision status
        Color collColor = Color.GREEN;
        if (collisionStatus == 2) {
            collColor = Color.YELLOW;
        } else if (collisionStatus == 3) {
            collColor = Color.RED;
        }

        textG.setColor(collColor);
        textG.fillRect(0, 0, 10, 10);

        if (hitVector != null) {
            g.setColor(Color.BLUE);
            g.fillOval(hitVector.x, hitVector.y, 10, 10);

            /*
            if (getShip() != null) {
                g.drawLine(getShip().getPosition().getCurVector().x, getShip().getPosition().getCurVector().y, hitVector.x, hitVector.y);
            }
            if (targetList.getFirstTarget() != null) {
                g.drawLine(targetList.getFirstTarget().getPosition().getCurVector().x, targetList.getFirstTarget().getPosition().getCurVector().y, hitVector.x, hitVector.y);
            }
            */
        }

        shotTargetList.draw(g, textG);

    }

    /**
     * Pauses the calling thread until an update is received.
     *
     * @throws InterruptedException
     */
    public void waitForUpdate() throws InterruptedException {
        // Save the receive timestamp
        synchronized (waitTimestamp) {
            waitTimestamp.set(getReceiveTimestamp());
        }

        while (waitTimestamp.get().equals(getReceiveTimestamp())) {
            //log.debug("Waiting");
            synchronized (this) {
                wait();
            }
        }
    }

    public double getAverageShotSpeed() {
        if (shotSpeeds.size() == 0) {
            return 0;
        }

        double sum = 0;
        int count = 0;

        try {
            for (Double speed : shotSpeeds) {
                sum += speed;
                count++;
            }
        } catch (ConcurrentModificationException e) {
            // Ignore
        }

        return sum / count;
    }

    public synchronized void wakeUpWaiters() {
        //log.debug("Wake up waiting threads");
        notifyAll();
    }

    /**
     * @param collisionStatus the collisionStatus to set
     */
    public void setCollisionStatus(int collisionStatus) {
        this.collisionStatus = collisionStatus;
    }

    /**
     * @param hitVector the hitVector to set
     */
    public void setHitVector(com.dkaedv.asteroids.data.Vector hitVector) {
        this.hitVector = hitVector;
    }

    /**
     * @param targetList the targetList to set
     */
    public void setTargetList(TargetList targetList) {
        this.targetList = targetList;
    }

    /**
     * @param shotTargetList the shotTargetList to set
     */
    public void setShotTargetList(ShotTargetList shotTargetList) {
        this.shotTargetList = shotTargetList;
    }
}
