/*
 * Created on Apr 17, 2008
 *
 * (c) 2006-2007 dka - edv, media, webdesign
 *
 */
package com.dkaedv.asteroids.net;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class KeysDatagram {
    private Log log = LogFactory.getLog(KeysDatagram.class);

    private int sentSerial = 0;

    private long lastSentTimestamp = 0;

    /**
     * Maximum value for sentSerial.
     *
     * If this value is reached, sentSerial is reset to 0.
     */
    private static final int MAX_SENT_SERIAL = 256;

    /**
     * Cycles count how many cycles are left for each of the keys.
     */
    private long leftCycles = 0;
    private long rightCycles = 0;
    private long fireCycles = 0;
    private long warpCycles = 0;
    private long thrustCycles = 0;


    public synchronized void packageSent() {

        // Record timestamp
        lastSentTimestamp = System.currentTimeMillis();

        // Increment sent serial
        sentSerial++;

        if (sentSerial == MAX_SENT_SERIAL)
            sentSerial = 0;

        // Decrement key counters
        if (leftCycles > 0)
            leftCycles--;

        if (rightCycles > 0)
            rightCycles--;

        if (fireCycles > 0)
            fireCycles--;

        if (warpCycles > 0)
            warpCycles--;

        if (thrustCycles > 0)
            thrustCycles--;
    }

    public int getSentSerial() {
        return sentSerial;
    }

    public synchronized void left(long time) {
        leftCycles = calcCycles(time);
        rightCycles = 0;
    }

    public synchronized void right(long time) {
        rightCycles = calcCycles(time);
        leftCycles = 0;
    }

    public synchronized void fire(long time) {
        fireCycles = calcCycles(time);

        // Fire is only done on even sent packages
        /*
        if (fireCycles == 1
                && (sentSerial % 2 != 0)) {
            fireCycles = 2;
        }
        */
    }

    public synchronized void warp(long time) {
        warpCycles = calcCycles(time);
    }

    public synchronized void thrust(long time) {
        thrustCycles = calcCycles(time);
    }

    public synchronized boolean isLeft() {
        return (leftCycles > 0);
    }

    public synchronized boolean isRight() {
        return (rightCycles > 0);
    }

    public synchronized boolean isWarp() {
        return (warpCycles > 0);
    }

    public synchronized void stopTurning() {
        rightCycles = 0;
        leftCycles = 0;
    }

    public synchronized byte[] getData() {
        byte[] data = new byte[8];

        // Copy the signature
        System.arraycopy(FrameInfo.KEYS_SIGNATURE, 0, data, 0, 6);

        // Keys
        data[6] = 0;

        if (leftCycles > 0)
            data[6] |= FrameInfo.KEY_LEFT;

        if (rightCycles > 0)
            data[6] |= FrameInfo.KEY_RIGHT;

        if (fireCycles > 0 )
            data[6] |= FrameInfo.KEY_FIRE;

        if (warpCycles > 0)
            data[6] |= FrameInfo.KEY_WARP;

        if (thrustCycles > 0)
            data[6] |= FrameInfo.KEY_THRUST;

        // Serial
        data[7] = (byte) (sentSerial & 0x00FF);

        return data;
    }

    /*
    public long getTimeToNextSend() {
        return (lastSentTimestamp + DatagramSender.SENDING_INTERVAL - System.currentTimeMillis());
    }
    */

    /**
     * Calculates the cycles from the time in ms.
     *
     * @param time
     * @return
     */
    private long calcCycles(long time) {
        if (time == 0)
            return 0;

        return 1;

        /*
        long cycles = (time / DatagramSender.IDLE_SENDING_INTERVAL);

        // We want at least one cycle
        if (cycles == 0)
            cycles = 1;

        return cycles;
        */
    }

}
