/*****************************************************************************
*  ADBotPlayerV1                                                             *
*                                                                            *
*  Copyright (C) 2008 Andreas Dittrich                                       *
*                                                                            *
*  Author:  Andreas Dittrich                                                 *
*           Am Schimmelberg 29                                               *
*           67729 Sippersfeld (Germany)                                      *
*****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
/*
#include <sys/types.h> 
#include <sys/timeb.h>
*/
#include <windows.h>

#include "inttypes.h"
#include "vector.h"

#include "../interface/ADBotPlayerV1.h"
#include "ADBotPlayerV1_private.h"

#include "gameinterface.h"
#include "vector2avi.h"
#include "tables.h"
#include "player.h"
#include "gamestate.h"

#include "botplayer.h"



BotPlayer::BotPlayer( void *hGameInterface, int *pRes ) :
	m_hGameInterface(hGameInterface),
	m_pGameState(NULL),
	m_pPlayer(NULL),
	m_hVec2avi(NULL),
	m_hLogFile(NULL),
	m_Running(1)
{
	*pRes = -1;

	if (m_hGameInterface==NULL)
		return;

	//create gamestate
	m_pGameState = (GameState_t*)malloc( sizeof(GameState_t) );
	if ( m_pGameState==NULL )
		return;

	memset( m_pGameState, 0x00, sizeof(GameState_t) );

	//create player
	m_pPlayer = Player_open();
	if ( m_pPlayer==NULL )
		return;

	//set timer resolution
	timeBeginPeriod(1);

	*pRes = 0;
}


BotPlayer::~BotPlayer()
{
	if (m_pPlayer)
	{
		Player_close(m_pPlayer);
		m_pPlayer = NULL;
	}

	if (m_pGameState)
	{
		free(m_pGameState);
		m_pGameState = NULL;
	}

	if (m_hVec2avi)
	{
		vector2avi_close(m_hVec2avi);
		m_hVec2avi = NULL;
	}

	if (m_hLogFile)
	{
		fclose(m_hLogFile);
		m_hLogFile = NULL;
	}

	timeEndPeriod(1);
}


void BotPlayer::Reset(void)
{
	//close video
	if (m_hVec2avi)
	{
		vector2avi_close( m_hVec2avi );
		m_hVec2avi = NULL;

		//rename video file
		char filenameStr[1024];

		sprintf_s( filenameStr, sizeof(filenameStr), "%06i_%s",
			m_pGameState->score,
			m_filenameAviStr );

		rename( m_filenameAviStr, filenameStr );
	}

	//close LogFile
	if (m_hLogFile)
	{
		fclose(m_hLogFile);
		m_hLogFile = NULL;

		//rename logfile
		char filenameStr[1024];

		sprintf_s( filenameStr, sizeof(filenameStr), "%06i_%s",
			m_pGameState->score,
			m_filenameLogStr );

		rename( m_filenameLogStr, filenameStr );
	}

	// generate new filenames
	{
		struct tm newtime;
		__time64_t long_time;

		_time64( &long_time ); 
		_localtime64_s( &newtime, &long_time ); 

		sprintf_s( 
			m_filenamePrefixStr, 
			sizeof(m_filenamePrefixStr), 
			"BotPlayerV1_%i-%02i-%02i_%02i%02i_%02i",
			newtime.tm_year+1900,
			newtime.tm_mon+1,
			newtime.tm_mday,
			newtime.tm_hour,
			newtime.tm_min,
			newtime.tm_sec );

		sprintf_s( m_filenameAviStr, sizeof(m_filenameAviStr), "%s.avi", m_filenamePrefixStr );
		sprintf_s( m_filenameLogStr, sizeof(m_filenameAviStr), "%s.txt", m_filenamePrefixStr );
	}

	// create new avi file 
	#ifdef RECORD_FRAME_TO_AVI
		m_hVec2avi = vector2avi_open( m_filenameAviStr );
	#endif

	//create new logfile
	#ifdef WRITE_LOGFILE
	{
		int res = fopen_s( &m_hLogFile, m_filenameLogStr, "wtS" );
		if ( (m_hLogFile) && (res!=0) )
		{
			fclose(m_hLogFile);
			m_hLogFile = NULL;
		}
	}
	#endif

	//reset game interface
	GameInterface_reset(m_hGameInterface);

	//reset gamestate
	memset( m_pGameState, 0x00, sizeof(GameState_t) );

	//reset playerstate
	Player_reset( m_pPlayer );
}


void BotPlayer::Stop(void)
{
	m_Running = 0;
}


int BotPlayer::Run(void)
{
	int res;
	int t;
	Frame_t frame;
	KeysInfo_t KeysInfoList[256];
	int KeysInfoListIndex;
	int is_GameOver;
	
StartNewGame:

	t = 0;
	KeysInfoListIndex = 0;
	memset( KeysInfoList, 0x00, sizeof(KeysInfoList) );
	is_GameOver = 0;
	Reset();

	while(1)
	{
		if (!m_Running)
			return 0;

		//send any key to notify the server, that we want to get a frame
		GameInterface_sendNewKeys( 
			m_hGameInterface,
			0,
			NULL );

		//wait for an answer
		Sleep(100);

		//synchronize to a received frame
		res = GameInterface_sync(m_hGameInterface);

		if (res==0)	//ok
			break;

		if (res<0)
			return res;

		if (res>=3)	
		{
			int N = (res==4) ? 10 : 1;

			//wait 1 or 10 seconds
			for (int n=0; (n<(N*10)) && m_Running; n++)
				Sleep( 100 );	
		}
	}

	#ifdef DO_TRANSMIT_PLAYER_NAME
	//send our name
	SendName( m_hGameInterface, BOTPLAYER_NAME );
	#endif

	//sync again
	GameInterface_sync(m_hGameInterface);

	//send first valid keys
	GameInterface_sendNewKeys( 
		m_hGameInterface,
		KeysInfoList[0].keys,
		(void*)KeysInfoListIndex );

	KeysInfoListIndex += 1;
	KeysInfoListIndex &= 0xFF;
	KeysInfoList[KeysInfoListIndex].t0 = 1;

	while(1)
	{
		//receive new frame
		while (1)
		{
			if (!m_Running)
				return 0;

			res = GameInterface_receiveNextFrame(
				m_hGameInterface,
				&frame,
				&t );

			if (res==0)	//ok
				break;

			if (res<0)
				return 0;

			if (res>=3) //busy or gameover
				goto StartNewGame;

			// no packet received (timeout) or error ==> try again
		}

		if ( KeysInfoList[((int)frame.keys_tag)&0xFF].t0 != t )
		{
			printf("frame: %d, keys delayed: %d\n", 
				t, t - KeysInfoList[((int)frame.keys_tag)&0xFF].t0 );
		}
		
		// transmit next keys 
		GameInterface_sendNewKeys( 
			m_hGameInterface,
			KeysInfoList[KeysInfoListIndex].keys,
			(void*)KeysInfoListIndex );

		KeysInfoListIndex += 1;
		KeysInfoListIndex &= 0xFF;

		// estimate the current state of the game
		UpdateGameState( 
			m_pGameState, 
			frame.pVectorRam,
			t,
			&(KeysInfoList[((int)frame.keys_tag)&0xFF]) );

		#ifdef RECEIVE_CPU_RAM
		memcpy( m_pGameState->ram, frame.pVectorRam+1026, 1024 );
		#endif

		// calculate action
		Player_update(
			m_pPlayer,
			&(KeysInfoList[KeysInfoListIndex]),
			m_pGameState,
			3,
			t );

		// write the frame (and state)
		vector2avi_writeFrame(
			m_hVec2avi,
			t,
			frame.pVectorRam,
			NUM_BYTES_VECTOR_RAM,
			(uint8_t*)m_pGameState,
			sizeof(GameState_t) );

		//log some statistical data
		if ( (m_hLogFile) && (m_pGameState->t0_game) )
		{
			int dt = m_pGameState->time - m_pGameState->t0_game;

			if (dt%60==0)
			{
				fprintf( m_hLogFile, "t: %i, level: %i, score: %i\n", 
					dt,
					m_pGameState->level,
					m_pGameState->score );
			}
		}
	}

	return 0;
}
