/*****************************************************************************
*  ADBotPlayerV1                                                             *
*                                                                            *
*  Copyright (C) 2008 Andreas Dittrich                                       *
*                                                                            *
*  Author:  Andreas Dittrich                                                 *
*           Am Schimmelberg 29                                               *
*           67729 Sippersfeld (Germany)                                      *
*****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include <windows.h>

#include "inttypes.h"
#include "vector.h"
#include "tables.h"

#include "../interface/ADBotPlayerV1.h"
#include "ADBotPlayerV1_private.h"

#include "gamestate.h"

#include "player.h"



struct tag_PlayerState_t;


typedef	struct {
	int t1;
	int t2;
	int angle_code;

	moving_object_t *pA;
	int index_A;
} target_t;


typedef struct {
	target_t targets[MAX_NUM_ASTEROIDS+1];
	int num_targets;
	int num_asteroids_and_explosions;
} target_properties_t;


typedef struct {

	tag_PlayerState_t *h;	

	void *hthread;				// system thread handle
	uint32_t dwthreadID;		// system thread ID
	void *hsignal_start;
	void *hsignal_completed;
	volatile int stop;

	target_properties_t *ptgprops;
	int cpu_usage;
	GameState_t *pGS;
	GameState_t *pGS_1;
	int time;
	int angle_code;
	int dt_latency;

} prediction_thread_t;


typedef struct tag_PlayerState_t {

	GameState_t GS_0;
	GameState_t GS_1a;
	GameState_t GS_1b;
	GameState_t GS_2;

	prediction_thread_t pred_thread_a;
	prediction_thread_t pred_thread_b;

	target_properties_t tgprops_a;
	target_properties_t tgprops_b;

	int t_hyperspace;
	int t_end_of_level;
	vector_t r_hyperspace;

	KeysInfo_t keysLog[1<<KEYSLOG_SIZE_LOG2];

	int next_shot_id;

} PlayerState_t;






static void CreateTargetProperties( 
	target_properties_t *ptgprops,
	int cpu_usage,
	GameState_t *pGS,
	GameState_t *pGS_1a,
	int time,
	int angle_code,
	int dt_latency );

static DWORD __stdcall PredictionThreadProc( prediction_thread_t * const ht);






int player_open_thread(
	PlayerState_t *h,
	prediction_thread_t *ht )
{
	DWORD dwthread = 0;
	LPTHREAD_START_ROUTINE lpfn = (LPTHREAD_START_ROUTINE)PredictionThreadProc;

	ht->h = h;

	// create events
	ht->hsignal_start = CreateEvent(NULL, FALSE, FALSE, NULL);
	if (ht->hsignal_start==NULL)
		goto Failed;

	ht->hsignal_completed = CreateEvent(NULL, FALSE, FALSE, NULL);
	if (ht->hsignal_completed==NULL)
		goto Failed;

	// create thread
	ht->hthread = CreateThread(NULL, 0, lpfn, ht, 0, &dwthread);
	if (ht->hthread==NULL)
		goto Failed;

	SetThreadPriority( ht->hthread, MY_THREAD_PRIORITY );

	ht->dwthreadID = dwthread;

	return 0;

Failed:
	if (ht->hsignal_completed)
		CloseHandle(ht->hsignal_completed);

	if (ht->hsignal_start)
		CloseHandle(ht->hsignal_start);

	ht->hthread = NULL;

	return -1;
}


void player_close_thread(
	prediction_thread_t *ht )
{
	if (ht->hthread)
	{
		ht->stop = 1;
		SetEvent(ht->hsignal_start);

		WaitForSingleObject(ht->hthread, INFINITE);

		CloseHandle(ht->hthread);
		ht->hthread = NULL;
	}

	if (ht->hsignal_start)
	{
		CloseHandle(ht->hsignal_start);
		ht->hsignal_start = NULL;
	}

	if (ht->hsignal_completed)
	{
		CloseHandle(ht->hsignal_completed);
		ht->hsignal_completed = NULL;
	}

	ht->dwthreadID =  0;
}


static DWORD __stdcall PredictionThreadProc( prediction_thread_t * const ht)
{
	while (1)
	{
		WaitForSingleObject(ht->hsignal_start, INFINITE);
		if (ht->stop)
			return 0;

		CreateTargetProperties(
			ht->ptgprops,
			ht->cpu_usage,
			ht->pGS,
			ht->pGS_1,
			ht->time,
			ht->angle_code,
			ht->dt_latency );

		SetEvent(ht->hsignal_completed);
	}

	return 0;
}



static void FillTargetList( 
	target_t *ptargets, 
	int num_targets,
	int cpu_usage,
	vector_t r_ship,
	int angle_code,
	int dt_latency,
	int time )
{
	if (num_targets<=0)
		return;

	int n_ops_max = (210*cpu_usage)/100;
	int dt_wait_max = n_ops_max / num_targets;

	if (dt_wait_max > 45)
		dt_wait_max = 45;
	else if (dt_wait_max < 1)
		dt_wait_max = 1;

	for ( int n=0; n<num_targets; n++)
	{
		ptargets[n].t1 = 1000;
		ptargets[n].t2 = 1000;
		ptargets[n].angle_code = angle_code;
	}

	for ( int w=-43; w<43; w++ )
	{
		const int ac = (angle_code + w*3)&0xFF;
		const int t_rotate = abs(w);
		const vector_t v_shot = v0_shots[ac];

		for ( int t_wait=0; t_wait<dt_wait_max; t_wait++ )
		{
			const int t1 = t_rotate + t_wait + dt_latency;
			vector_t r_shot = { r_ship.x + ((v_shot.x*5)>>1), r_ship.y + ((v_shot.y*5)>>1) };

			normalize_r( &r_shot );

			__declspec(align(32))
			struct tlist_tag {
				vector_t r;
				vector_t v;
				int state;
				int width;
				int t0;
				int reserved;
			} tlist[MAX_NUM_ASTEROIDS+1];

			for ( int n=0; n<num_targets; n++ )
			{
				vector_t rr = ptargets[n].pA->r;
				vector_t vv = ptargets[n].pA->v;

				vmuladd( &rr, vv, t1 );

				tlist[n].r     = rr;
				tlist[n].v     = vv;
				tlist[n].state = ptargets[n].pA->state;
				tlist[n].width = asteroid_width_table[ptargets[n].pA->state&0x07];
				tlist[n].t0    = ptargets[n].pA->t0 - time - t1 - 1;	// relative lifetime

				//adjust width
				switch( tlist[n].state ) 
				{
				//case 1: tlist[n].width -= 2; break;
				case 2: tlist[n].width -= 4; break;
				case 4: tlist[n].width -= 4; break;
				}

			}

			for (int t3=0; t3<67; t3++)
			{
				int n;
				for (n=0; n<num_targets; n++)
				{
					tlist_tag * ptl = &(tlist[n]);

					vector_t u = ptl->r;
					vsub( &u, r_shot );

					if ( is_hit_octagon( u, ptl->width ) ) //is_hit_shot( u, ptl->state ) )
					{
						// does it exist at the hit time?
						if ( ptl->t0 <= t3 )
							break;
					}

					vadd( &ptl->r, ptl->v );
					normalize_r( &ptl->r );
				}

				vadd( &r_shot, v_shot );
				normalize_r( &r_shot );

				if (n<num_targets)
				{
					int t2 = t1 + t3;

					if ( t2 < ptargets[n].t2 )
					{
						ptargets[n].t2 = t2;
						ptargets[n].t1 = t1;
						ptargets[n].angle_code = ac;
					}

					break;
				}
			}
		}
	}

	//_mm_empty();
}


int DoHyperspaceJump( 
	GameState_t * const pGS,
	int dt_max_jump )
{
	static const vector_t r_optimal[2] = {
		{128*8, 640*8 + 128*8},
		{896*8, 640*8 + 128*8} };

// todo check collisions
//	GameState_t GS_predict;
//	memcpy( &GS_predict, pGS, sizeof(GameState_t) );

	int do_hyperspace_jump = 0;

	if ( ( dt_max_jump>0 ) && (pGS->rgen.s!=0) )
	{
		// check current pos
		int dq_ship_min = (1<<30);
		for (int i=0; i<(sizeof(r_optimal)/sizeof(r_optimal[0])); i++)
		{
			vector_t rr = r_optimal[i];
			vsub( &rr, pGS->ship.r );
			normalize_d( &rr );

			int dq = vabsq( rr );

			if (dq<dq_ship_min)
				dq_ship_min = dq;
		}

		//check jump pos
		int dq_min = (1<<30);
		int dt_argmin = -1;	
		for (int dt=0; dt<dt_max_jump; dt++)
		{	
			vector_t r_dest;

			int is_save_jump = CheckHyperspaceJump( pGS, dt, &r_dest );
			if (!is_save_jump)
				continue;

			for (int i=0; i<(sizeof(r_optimal)/sizeof(r_optimal[0])); i++)
			{
				vector_t rr = r_optimal[i];
				vsub( &rr, r_dest );
				normalize_d( &rr );

				int dq = vabsq( rr );

				if (dq<dq_min)
				{
					dq_min = dq;
					dt_argmin = dt;
				}
			}
		}

		double d_min      = sqrt((double)dq_min);
		double d_ship_min = sqrt((double)dq_ship_min);

		if ( (dt_argmin==0) && (d_min < (d_ship_min-80*8)) )
			do_hyperspace_jump = 1;
	}

	return do_hyperspace_jump;
}


static void	PrepareCreateTargetPropsThread(
	prediction_thread_t *pht,
	target_properties_t *ptgprops,
	int cpu_usage,
	GameState_t *pGS,
	GameState_t *pGS_1,
	int time,
	int angle_code,
	int dt_latency )
{
	pht->ptgprops    = ptgprops;
	pht->cpu_usage = cpu_usage;
	pht->pGS         = pGS;
	pht->pGS_1       = pGS_1;
	pht->time        = time;
	pht->angle_code  = angle_code;
	pht->dt_latency  = dt_latency;
}


static void CreateTargetProperties( 
	target_properties_t *ptgprops,
	int cpu_usage,
	GameState_t *pGS,
	GameState_t *pGS_1,
	int time,
	int angle_code,
	int dt_latency )
{
	target_t * const targets = ptgprops->targets;

	//predict the targets in the future, until no shots are visible anymore
	{
		int dt = 0;
		while ( pGS_1->num_shots > 0 )
		{
			dt += 1;

			PredictGameState(
				pGS_1,
				time + dt_latency + dt,
				NULL );
		}

		//revert the movement
		for( int i=0; i<(MAX_NUM_ASTEROIDS+1); i++)
		{
			moving_object_t * pA = (moving_object_t*)&pGS_1->asteroids[i];
			if ( i>=MAX_NUM_ASTEROIDS )
				pA = (moving_object_t*)&pGS_1->saucer;

			if (pA->state<=0)
				continue;

			vmuladd( &pA->r, pA->v, -(dt+dt_latency-1) );
			normalize_r( &pA->r );
		}

		#ifndef RELEASE_CONTEST
		#ifndef USE_TWO_CORES
		memcpy( pGS->asteroids_predicted, pGS_1->asteroids, sizeof(pGS_1->asteroids) );
		#endif
		#endif
	}


	//fill list of targets
	int num_targets = 0;
	int num_asteroids_and_explosions = 0;

	for( int i=0; i<(MAX_NUM_ASTEROIDS+1); i++)
	{
		moving_object_t * pA = (moving_object_t*)&pGS_1->asteroids[i];
		if ( i>=MAX_NUM_ASTEROIDS )
			pA = (moving_object_t*)&pGS_1->saucer;

		if (pA->state!=0)
			num_asteroids_and_explosions++;

		if (pA->state <= 0)
			continue;

		target_t * const ptg = &(targets[num_targets]);
		num_targets++;

		ptg->pA = pA;
		ptg->index_A = i;
	}

	FillTargetList( 
		targets, 
		num_targets, 
		cpu_usage,
		pGS_1->ship.r, 
		angle_code,
		dt_latency,
		time + dt_latency );

	//sort list
	for (int n=0; n<num_targets; n++)
	{
		int t2_min = targets[n].t2;
		int nn = n;

		for (int m=n+1; m<num_targets; m++)
		{
			if ( targets[m].t2 < t2_min )
			{
				t2_min = targets[m].t2 - ( (targets[m].pA->state==1) ? 5 : 0 );
				nn = m;
			}
		}

		target_t temp = targets[n];
		targets[n] = targets[nn];
		targets[nn] = temp;
	}

	ptgprops->num_targets = num_targets;
	ptgprops->num_asteroids_and_explosions = num_asteroids_and_explosions;
}



#ifdef _DEBUG
	#define CPU_USAGE_MAX 1
#else
	#define CPU_USAGE_MAX 60
#endif


void Player_update(
	void *hnd,
	KeysInfo_t *pKeysInfo,
	GameState_t * const pGS, 
	int dt_latency,
	int time )
{
	PlayerState_t * h = (PlayerState_t*)hnd;
	if (h==NULL)
		return;

	uint32_t keys = 0;
	memset( pKeysInfo, 0x00, sizeof(KeysInfo_t) );

	if (pGS->ship.state<=0)
		goto Set_KeysInfo_And_Return;

	//new predicted states
	GameState_t * const pGS_0  = &h->GS_0;
	GameState_t * const pGS_1a = &h->GS_1a;
	GameState_t * const pGS_1b = &h->GS_1b;

	GameState_t * const pGS_2 = pGS_0;

	//predict the gamestate to compensate the latency
	//and create different scenarios
	{
		memcpy( pGS_0, pGS, sizeof(GameState_t) );

		for (int i=-(dt_latency-2); i<0; i++)
		{
			PredictGameState(
				pGS_0,
				time + i + dt_latency,
				&h->keysLog[(time+i) & ((1<<KEYSLOG_SIZE_LOG2)-1)] );
		}

		memcpy( pGS_1b, pGS_0, sizeof(GameState_t) );

		KeysInfo_t keysinfo_fire = {};
		keysinfo_fire.keys = KEY_FIRE;

		PredictGameState(
			pGS_1b,
			time + dt_latency,
			&keysinfo_fire );

		PredictGameState(
			pGS_0,
			time + dt_latency,
			NULL );

		memcpy( pGS_1a, pGS_0, sizeof(GameState_t) );
	}

	//get gun state
	int is_gun_ready        = pGS_0->ship.is_gun_ready;
	int num_shots_available = pGS_0->ship.num_shots_available;
	if (!(pGS_0->ship.flags & IS_TRACKED))
		num_shots_available = 1;

	//get angle
	const int angle_code = pGS_0->ship.angle_code;	
	const vector_t v0 = v0_shots[angle_code&0xFF];

	//check hyperspace state
	if (time <= h->t_hyperspace)
		goto Set_KeysInfo_And_Return;


	//check collision shot with ship
	int dt_collision = 1000;
	for (int j=0; j<pGS_0->num_shots; j++)
	{
		const int width_correction = -3;
		int width = MAX( 0, asteroid_width_table[1] - width_correction );

		shot_t * const pC = &(pGS_0->shots[j]);

		vector_t r = pGS_0->ship.r;
		vsub( &r, pC->r );
		normalize_d( &r );
		
		for (int dt=0; dt<10; dt++)
		{
			vsub( &r, pC->v );

			if ( is_hit_octagon(r, width) )
			{
				if (dt<dt_collision)
					dt_collision = dt;

				break;
			}
		}
	}


	//check collision of asteroid or saucer with ship
	int num_asteroids_and_saucer = 0;
	for( int i=0; i<(MAX_NUM_ASTEROIDS+1); i++)
	{
		moving_object_t * pA = (moving_object_t*)&pGS_2->asteroids[i];
		if ( i>=MAX_NUM_ASTEROIDS )
			pA = (moving_object_t*)&pGS_2->saucer;

		if (pA->state <= 0)
			continue;

		num_asteroids_and_saucer++;

		//check collision of asteroid or saucer with ship
		{
			const int width_correction = -3;

			int width = MAX( 0, ship_width_table[pA->state&0x07] - width_correction );

			vector_t r = pA->r;
			vsub( &r, pGS_2->ship.r );
			normalize_d( &r );		

			for (int dt=0; dt<10; dt++)
			{
				vadd( &r, pA->v );

				if ( is_hit_octagon(r, width) )
				{
					if (dt<dt_collision)
						dt_collision = dt;

					break;
				}
			}
		}
	}


	//get sorted list of targets

#ifndef USE_TWO_CORES 

	CreateTargetProperties(
		&h->tgprops_a,
		CPU_USAGE_MAX/2,
		pGS,
		pGS_1a,
		time,
		angle_code,
		dt_latency );

	CreateTargetProperties(
		&h->tgprops_b,
		CPU_USAGE_MAX/2,
		pGS,
		pGS_1b,
		time,
		angle_code,
		dt_latency );

#else

	PrepareCreateTargetPropsThread(
		&h->pred_thread_a,
		&h->tgprops_a,
		CPU_USAGE_MAX,
		pGS,
		pGS_1a,
		time,
		angle_code,
		dt_latency );

	SetEvent(h->pred_thread_a.hsignal_start);

	PrepareCreateTargetPropsThread(
		&h->pred_thread_b,
		&h->tgprops_b,
		CPU_USAGE_MAX,
		pGS,
		pGS_1b,
		time,
		angle_code,
		dt_latency );

	SetEvent(h->pred_thread_b.hsignal_start);

	void * wait_signals[2] = {
		h->pred_thread_a.hsignal_completed,
		h->pred_thread_b.hsignal_completed };

	WaitForMultipleObjects(	2, wait_signals, TRUE, INFINITE);

#endif


	target_t *targets = h->tgprops_a.targets;
	int num_targets   = h->tgprops_a.num_targets;
	int num_asteroids_and_explosions = h->tgprops_a.num_asteroids_and_explosions;


	//select hit targets
	target_t * ptg_fire   = NULL;
	target_t * ptg_rotate = NULL;

	if (num_targets>0)
		ptg_rotate = &(targets[0]);

	for (int n=0; n<num_targets; n++)
	{
		moving_object_t * const pA = targets[n].pA;
		target_t * const ptg = &(targets[n]);

		//shoot now?
		if ( (angle_code == ptg->angle_code) &&
			 (is_gun_ready) &&
			 (dt_latency == targets[n].t1) )
		{
			if ( ( n==0 ) ||
				 (pA->flags&IS_SAUCER) || 
				 (ptg->t2 < 9) )
			{
				ptg_fire = ptg;
				break;
			}

			if ( pA->state == 1 )
			{
				ptg_fire = ptg;
				break;
			}

			if ( pA->state == 4 )
			{
				ptg_fire = ptg;
				break;
			}
		}
	}

	if (ptg_fire)
	{
		targets       = h->tgprops_b.targets;
		num_targets   = h->tgprops_b.num_targets;
		num_asteroids_and_explosions = h->tgprops_b.num_asteroids_and_explosions;

		if (num_targets>0)
			ptg_rotate = &(targets[0]);
	}

	if (num_targets==0)
	{
		int nx = ( pGS_1a->ship.r.x / (1024*8/2) )&1;
		int ny = ( (pGS_1a->ship.r.y - 128*8) / (768*8/2) )&1;
		int angle_code_new = 0;

		switch (nx*2+ny) 
		{
		case 0: angle_code_new = 153; break;
		case 1: angle_code_new = 103; break;
		case 2: angle_code_new = 217; break;
		case 3: angle_code_new =  25; break; 
		}

		targets[0].angle_code = ( angle_code + ((angle_code_new-angle_code)/3)*3 )&0xFF;
		ptg_rotate = &(targets[0]);
	}

	//end of level?
	if ( ((time - h->t_end_of_level)>250) && 
		 ((time - pGS->t0_game) > 200 ) &&
		 (num_asteroids_and_saucer==0) )
	{
		h->t_end_of_level = time;
	}

	//change position?
	int do_hyperspace_jump = 0;
	if ( ((time - h->t_end_of_level) < 30) &&
 		 ((time - h->t_hyperspace) > 50)  &&
		 ((time - pGS->t0_game) > 200 )       )
	{
		do_hyperspace_jump = DoHyperspaceJump( pGS_0, 28 );
	}

	//hyperspace?
	if (  ( (dt_collision <= 15) || do_hyperspace_jump ) && 
		  (time > h->t_hyperspace)  )
	{
		if (dt_collision > 1)
		{
			int second_chance = 0;

			for (int n=1; n<dt_collision; n++)
			{
				randgen_t rgen_temp1 = pGS_0->rgen;

				PredictGameState(
					pGS_0,
					time + dt_latency + n,
					NULL );

				randgen_t rgen_temp2 = pGS_0->rgen;

				pGS_0->rgen = rgen_temp1;

				CycleRandomGenerator(
					pGS_0,
					time,
					1 );

				second_chance = IsSaveHyperspaceJump( pGS_0, &h->r_hyperspace );
				if ( second_chance )
					break;

				pGS_0->rgen = rgen_temp2;
			}

			if (!second_chance)
				dt_collision = 1;
		}
	
		if ( (do_hyperspace_jump) || 
			 (dt_collision<=1) )
		{
			keys |= KEY_HYPERSPACE;
			h->t_hyperspace = time + dt_latency;

			#ifndef RELEASE_CONTEST
			pGS->r_destination = h->r_hyperspace;
			#endif

			goto Set_KeysInfo_And_Return;
		}
	}

	//fire?
	if ( ptg_fire )
	{	
		moving_object_t * const pX = ptg_fire->pA;
		int t_hit = ptg_fire->t2 - dt_latency;

		pKeysInfo->id_target = pX->id;
		pKeysInfo->t1        = time + dt_latency;
		pKeysInfo->t2        = time + dt_latency + t_hit;
		pKeysInfo->id_shot   = h->next_shot_id++;

		keys |= KEY_FIRE;
	}

	//rotate?
	if ( ptg_rotate )
	{
		int angle_delta = (int8_t)(uint8_t)(ptg_rotate->angle_code - angle_code);
		
		if ( angle_delta > 0 )
			keys |= KEY_LEFT;
		else if ( angle_delta < 0 )
			keys |= KEY_RIGHT;
	}


Set_KeysInfo_And_Return:

	pKeysInfo->t0   = time + dt_latency - 1;
	pKeysInfo->keys = keys;

	h->keysLog[time & ((1<<KEYSLOG_SIZE_LOG2)-1)] = *pKeysInfo;
}


void * Player_open()
{
	PlayerState_t * h = 
		(PlayerState_t*)malloc( sizeof(PlayerState_t) );

	if (h==NULL)
		return NULL;

	memset( h, 0x00, sizeof(PlayerState_t) );

	int res = 0;
	
	res = player_open_thread( h, &h->pred_thread_a );
	if (res<0)
		goto Failed;

	res = player_open_thread( h, &h->pred_thread_b );
	if (res<0)
		goto Failed;

//Success
	return h;

Failed:
	Player_close( h );
	return NULL;
}


void Player_close( void *hnd )
{
	PlayerState_t * h = (PlayerState_t*)hnd;
	if (h==NULL)
		return;

	player_close_thread( &h->pred_thread_a );
	player_close_thread( &h->pred_thread_b );

	free(h);
}


void Player_reset( void *hnd )
{
	PlayerState_t * h = (PlayerState_t*)hnd;
	if (h==NULL)
		return;

	h->t_hyperspace = 0;
	memset( h->keysLog, 0x00, sizeof(h->keysLog) );
	h->next_shot_id = 0;
}
