package haui.asteroid.control;

import java.io.IOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;

/**
 * Steuerschleife Datenempfang vom Emulator, Interpretation des
 * Bildschirminhalts, Berechnung von Tastendrücken und Senden des Steuerpakets.
 * 
 * @author Harald Bögeholz / c't
 * @author Bernhard Haumacher
 */
public class Player {
	DatagramSocket socket;
	
	byte[] rxBuffer = new byte[FramePacket.PACKET_SIZE];
	byte[] txBuffer = new byte[KeysPacket.PACKET_SIZE];
	
	DatagramPacket rxPacket = new DatagramPacket(rxBuffer, FramePacket.PACKET_SIZE);
	DatagramPacket txPacket = new DatagramPacket(txBuffer, KeysPacket.PACKET_SIZE);

	int[] vectorRamWords = new int[FramePacket.VECTOR_RAM_SIZE / 2];
	

	public Player(DatagramSocket socket, InetAddress serverIp) {
		this.socket = socket;
		
		this.txPacket.setAddress(serverIp);
		this.txPacket.setPort(1979);
	}

	void run() throws IOException
	{
		FramePacket framePacket = new FramePacket();
		KeysPacket keyPacket = new KeysPacket();
		GameStatus game = new GameStatus();
		byte prevFrameNumber = 0;
		int t = 0;

		for (;;)
		{
			// Zeit
			++t;         
			// jedes gesendete Päckchen erhält eine individuelle Nummer zur Latenzmessung
			++keyPacket.ping; 
			sendPacket(keyPacket);
			receivePacket(framePacket);

			if (framePacket.frameNumber != ++prevFrameNumber || framePacket.ping != keyPacket.ping)
			{
				System.out.println("Latenz " + (keyPacket.ping - framePacket.ping) + ". " + (framePacket.frameNumber - prevFrameNumber) + "Frames verloren.");
				prevFrameNumber = framePacket.frameNumber;
			}

			interpretScreen(framePacket, game);

			keyPacket.clear();   // alle Tasten loslassen
			int minDist = 0x7fffffff;
			int minDx = 0;
			int minDy = 0;
			if (game.shipPresent)
			{
				for (int i=0; i<game.asteroidCount; ++i)
				{   // nächstgelegenen Asteroiden suchen
					int dx = game.asteroids[i].x - game.shipX;
					while (dx < -512) dx += 1024; // dx normalisieren auf -512 ... 511
					while (dx > 511) dx -= 1024;
					int dy = game.asteroids[i].y - game.shipY;
					while (dy < -384) dy += 768;  // dy normalisieren auf -384 ... 383
					while (dy > 383) dy -= 768;
					int dist = dx*dx+dy*dy;  // Quadrat des Abstands zu diesem Asteroiden
					switch (game.asteroids[i].scaleFactor)
					{	// Abstand um den ungefähren Radius des Asteroiden korrigieren
						case 0:  // großer Asteroid
							dist -= 40*40;
							break;
						case 15: // mittlerer Asteroid
							dist -= 20*20;
							break;
						case 14: // kleiner Asteroid
							dist -= 8*8;
							break;
					}
					if (dist < minDist)
					{
						minDist = dist;
						minDx = dx;
						minDy = dy;
					}
				}
				if (game.ufoPresent)
				{
					int dx = game.ufoX - game.shipX;
					while (dx < -512) dx += 1024;
					while (dx > 511) dx -= 1024;
					int dy = game.ufoY - game.shipY;
					while (dy < -384) dy += 768;
					while (dy > 383) dy -= 768;
					int dist = dx*dx+dy*dy;
					switch (game.ufoSize)
					{	// Abstand um den ungefähren Radius des UFOs korrigieren
					case 15: // großes UFO
						dist -= 20*12;
						break;
					case 14: // kleines UFO
						dist -= 10*6;
						break;
					}
					if (dist < minDist)
					{
						minDist = dist;
						minDx = dx;
						minDy = dy;
					}
				}

				// Schiff in Richtung auf das nächstgelegene Objekt drehen
				// mathematisch wird hier das Kreuzprodukt aus den Vektoren 
				// shipDx/y/0 und minDx/y/0 berechnet
				if (game.shipDx * minDy - game.shipDy * minDx > 0)
					keyPacket.left(true);
				else
					keyPacket.right(true);

				if (minDist < 27*27)  // Flucht, wenn Kollision unausweichlich
					keyPacket.hyperspace(true);

				if (minDist > 400*400) // beschleunigen, wenn nichts in der Nähe
					keyPacket.thrust(true);

				if (t % 2 == 0)  // Feuerknopf drücken, so schnell es geht
					keyPacket.fire(true);
			}
		}
	}

	void interpretScreen(FramePacket packet, GameStatus game)
	{
		for (int r = 0, w = 0; w < 512; r += 2, w++) {
			vectorRamWords[w] = 
				(asInt(packet.vectorRam[r])) | (asInt(packet.vectorRam[r + 1]) << 8);
		}
		
		int dx, dy, sf, vx = 0, vy = 0, vz, vs = 0;
		int v1x = 0;
		int v1y = 0;
		int shipdetect = 0;

		game.clear();
		if (asInt(packet.vectorRam[1]) != 0xe0 && asInt(packet.vectorRam[1]) != 0xe2) {
			// sollte nicht vorkommen; erster Befehl ist immer ein JMPL
			return;
		}

		int pc = 1;
		for (;;)
		{
			int op = vectorRamWords[pc] >> 12;
			switch (op)
			{
			case 0xa: // LABS
				vy = vectorRamWords[pc] & 0x3ff;
				vx = vectorRamWords[pc+1] & 0x3ff;
				vs = vectorRamWords[pc+1] >> 12;
				break;
			case 0xb: // HALT
				return;
			case 0xc: // JSRL
				switch (vectorRamWords[pc] & 0xfff)
				{
				case 0x8f3:
					game.asteroids[game.asteroidCount++].set(vx, vy, 1, vs);
					break;
				case 0x8ff:
					game.asteroids[game.asteroidCount++].set(vx, vy, 2, vs);
					break;
				case 0x90d:
					game.asteroids[game.asteroidCount++].set(vx, vy, 3, vs);
					break;
				case 0x91a:
					game.asteroids[game.asteroidCount++].set(vx, vy, 4, vs);
					break;
				case 0x929:
					game.ufoPresent = true;
					game.ufoX = vx;
					game.ufoY = vy;
					game.ufoSize = vs;
					break;
				}  
				break;
			case 0xd: // RTSL
				return;
			case 0xe: // JMPL
				/*
				pc = vectorRamWords[pc] & 0xfff;
				break;
				*/
				return;
			case 0xf: // SVEC
				/*
				dy = vectorRamWords[pc] & 0x300;
				if ((vectorRamWords[pc] & 0x400) != 0)
					dy = -dy;
				dx = (vectorRamWords[pc] & 3) << 8;
				if ((vectorRamWords[pc] & 4) != 0)
					dx = -dx;
				sf = (((vectorRamWords[pc] & 8) >> 2) | ((vectorRamWords[pc] & 0x800) >> 11)) + 2;
				vz = (vectorRamWords[pc] & 0xf0) >> 4;
				*/
				break;
			default:
				dy = vectorRamWords[pc] & 0x3ff;
				if ((vectorRamWords[pc] & 0x400) != 0)
					dy = -dy;
				dx = vectorRamWords[pc+1] & 0x3ff;
				if ((vectorRamWords[pc+1] & 0x400) != 0)
					dx = -dx;
				sf = op;
				vz = vectorRamWords[pc+1] >> 12;
				if (dx == 0 && dy == 0 && vz == 15)
					game.shots[game.shotCount++].set(vx, vy);
				if (op == 6 && vz == 12 && dx != 0 && dy != 0)
				{
					switch (shipdetect)
					{
					case 0:
						v1x = dx;
						v1y = dy;
						++shipdetect;
						break;
					case 1:
						game.shipPresent = true;
						game.shipX = vx;
						game.shipY = vy;
						game.shipDx = v1x - dx;
						game.shipDy = v1y - dy;
						++shipdetect;
						break;
					}
				}
				else if (shipdetect == 1)
					shipdetect = 0;

				break;
			}
			if (op <= 0xa)
				++pc;
			if (op != 0xe) // JMPL
				++pc;
		}   

	}

	/**
	 * Convert given byte interpreted as unsigned into an <code>int</code> value.
	 */
	private int asInt(byte byteValue) {
		return ((int) byteValue) & 0xFF;
	}

	void receivePacket(FramePacket packet) throws IOException
	{
		socket.receive(rxPacket);
		
		int bytesReceived = rxPacket.getLength();
		if (bytesReceived != rxPacket.getData().length)
		{
			System.err.println("Error while receiving packet, " + bytesReceived + " bytes received.");
			System.exit(1);
		}
		
		packet.copyFrom(rxBuffer);
	}

	void sendPacket(KeysPacket packet) throws IOException
	{
		int size = packet.copyTo(txBuffer);
		txPacket.setLength(size);
		
		socket.send(txPacket);
	}
}
