unit CubeWorldU;  // 22-JAN-00 as (Arne Schpers)
{ D3DRM: Flug durch ein 3D-Feld aus Wrfeln, mit
  Reflektion an den Grenzen des Spielfelds

  "T" = vorwrts, "G" = Stop, "V" = Rckwrts
  "D" = Blickrichtung = Bewegungsrichtung
  Pfeiltasten fr Links/Rechts, Aufwrts/Abwrts

  Unterschiede zu D3DRMini: Tastatur-Schnittstelle, BuildScene
}

{$DEFINE DIRECTX7}
interface
uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls,
  Forms, Dialogs, ExtCtrls, StdCtrls, DXTimer, // schneller Timer
  MMSystem,
{$IFDEF DIRECTX7}  // DX7-Header, seit 17-JAN-00
  DirectDraw, Direct3D, Direct3DRM;
{$ELSE}  // DX6-Header
  DDraw,D3D,D3DTypes,D3DCaps,D3DRM,D3DRmDef,D3DRMObj,D3DRMWin;
{$ENDIF}

type
  TCWorldForm = class(TForm)
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormPaint(Sender: TObject);
    procedure FormKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure FormKeyUp(Sender: TObject; var Key: Word;
      Shift: TShiftState);
  private  // 50 Fortschreibungen pro Sekunde
    DXTimer: TDXTimer;
    procedure OnDXTimer(Sender: TObject);
  private  // Direct3DRM-Schnittstellen
    D3dInitialized: Boolean;
    D3DDevGUID: TGUID;  // GUID des (HAL-)Treibers
    Device: IDirect3DRMDevice3;
    D3DRMInterface: IDirect3DRM3;
    View: IDirect3DRMViewPort2;
    Scene, Camera: IDirect3DRMFrame3;
    Speed: TD3DValue;
  protected
    function InitD3DInterfaces: Boolean;
    function BuildScene:Boolean;
    function RenderScene:Boolean;
    procedure OnCameraMove(Sender: TObject);
  end;

var CWorldForm: TCWorldForm;

const CUBEDIMXYSIZE = 4;   // X und Y: 0..5, 0..5
      CUBEDIMZSIZE = CUBEDIMXYSIZE*10; // Z: 0..40

implementation
{$R *.DFM}

function CheckRes(Res: HResult; ErrMsg: String): Boolean;
begin  // Makro, sozusagen
  Result := SUCCEEDED(Res);
  if not Result then ShowMessage('Fehler bei: '+ErrMsg);
end;

// D3D-Treiber abzhlen, Clipper und D3DRM-Schnittstellen anlegen
function TCWorldForm.InitD3DInterfaces: Boolean;
var D3DRMTemp: IDirect3DRM;  // Urversion
    DDrawClipper: IDirectDrawClipper;

  // Rckruf beim Abzhlen der D3D-Gerte bzw. Treiber; bricht ab,
  // sobald der erste Hardware-Treiber gemeldet wird
  function EnumDevicesCallback(const lpGuid: TGUID;
     lpDeviceDescription, lpDeviceName: LPSTR;
     const lpD3DHWDeviceDesc, lpD3DHELDeviceDesc: TD3DDeviceDesc;
        lpUserArg: Pointer) : HResult; stdcall;
  begin
    if Assigned(@lpGUID) then PGUID(lpUserArg)^ := lpGuid;
    if lpD3DHWDeviceDesc.dcmColorModel = D3DCOLOR_RGB
      then Result := DDENUMRET_CANCEL  // OK, ende
      else Result := DDENUMRET_OK;  // weitermachen
  end;

  // DirectDraw-Schnittstelle anlegen, dort nach IDirect3D fragen,
  // IDirect3D zum Abzhlen der Gerte verwenden, danach beide
  // Schnittstellen (automatisch) wieder freigeben
  function Enum3DDevices(var D3DDevGuid: TGuid): Boolean;
  var DDrawObject: IDirectDraw; D3DObject: IDirect3D;
  begin
    Result := SUCCEEDED(DirectDrawCreate(nil, DDrawObject, nil))
     and SUCCEEDED(DDrawObject.QueryInterface(IID_IDirect3D,
       D3DObject)) and SUCCEEDED(D3DObject.EnumDevices
        (@EnumDevicesCallback, @D3DDevGUID));
  end;

begin
  Result := False;

  // DirectD3DRM-Schnittstelle (Urversion), Abfrage nach RM3
  if not CheckRes(Direct3DRMCreate(D3DRMTemp),
    'Direct3DRM-Schnittstelle anlegen')
   or not CheckRes(D3DRMTemp.QueryInterface(IID_IDirect3DRM3,
     D3DRMInterface),' QueryInterface nach RM3') then Exit;

  // DirectDrawClipper-Objekt anlegen, mit dem Fenster verbinden
  if not CheckRes(DirectDrawCreateClipper(0, DDrawClipper, nil),
    'DirectDraw-Clipper anlegen')
   or not CheckRes(DDrawClipper.SetHWnd(0, Handle),
    'DirectDraw-Clipper mit Fenster verbinden') then Exit;

  // GUID des ersten HAL-Treibers abfragen. Das ginge auch
  // ohne die vorherige Initialisierung von D3DRM
  if not Enum3DDevices(D3DDevGUID) then Exit;

  // D3DRM-Gert ber den DirectDrawClipper anlegen. NIL anstelle
  // des GUIDs wre "HEL ramp" anstelle von "HAL"
  if not CheckRes(D3DRMInterface.CreateDeviceFromClipper(
    DDrawClipper, @D3DDevGUID, ClientWidth, ClientHeight,
    Device), 'CreateDeviceFromClipper') then Exit;

  // Master- (= Hintergrund) und Kamera-Frame anlegen
  if not CheckRes(D3DRMInterface.CreateFrame(nil, Scene),
    'CreateFrame fr den Hintergrund')
   or not CheckRes(D3DRMInterface.CreateFrame(Scene, Camera),
    'CreateFrame fr die Kamera') then Exit;

   // Viewport anlegen, hier fr das gesamte Fenster
   // (GetWidth/GetHeight = ClientWidth/ClientHeight)
   Result := CheckRes(D3DRMInterface.CreateViewport(Device,
    Camera, 0, 0, Device.GetWidth, Device.GetHeight, View),
     'CreateViewport');
end;

procedure TCWorldForm.FormCreate(Sender: TObject);
begin
  ClientWidth := 400; ClientHeight := 300;
  D3DInitialized := InitD3DInterfaces;
  if D3DInitialized and BuildScene then
  begin
    DXTimer := TDXTimer.Create(Self);
    DXTimer.Interval := 20; DXTimer.OnTimer := OnDXTimer;
  end
    else PostQuitMessage(0);
end;

procedure TCWorldForm.FormDestroy(Sender: TObject);
begin  // Die Reihenfolge spielt mal wieder eine Rolle...
  D3DInitialized:=False;
  View := nil; Camera := nil;
  Scene :=nil; Device := nil;
  D3DRMInterface := nil;
end;

// Sorgt dafr, dass D3DRM bei WM_PAINT den gesamten Zeichen-
// puffer neu malt - und nicht nur die Bereiche, die bei
// normaler Fortschreibung aktualisiert werden mten
procedure TCWorldForm.FormPaint(Sender: TObject);
var WinDev: IDirect3DRMWinDevice;
begin
  // WinDevice-Schnittstelle von RMDevice3
  // bernimmt die Reaktion auf WM_PAINT
  if SUCCEEDED(Device.QueryInterface(
    IID_IDirect3DRMWinDevice,WinDev))
   then WinDev.HandlePaint(Canvas.Handle);
end;


procedure OnCameraMoveHelper(lpD3RMFrame: IDirect3DRMFrame3;
   lpArg: Pointer; delta: TD3DValue); cdecl;
begin
  TCWorldForm(lpArg).OnCameraMove(nil);
end;

function TCWorldForm.BuildScene: Boolean;
var
  MeshBuilder: IDirect3DRMMeshBuilder3;
  LightA: IDirect3DRMLight;
  x,y,z: Integer;
  CubeFrame: IDirect3DRMFrame3;
  Material: IDirect3DRMMaterial2;

  procedure AddLight(PosX, PosY, R,G,B: TD3DValue);
  var Frame: IDirect3DRMFrame3; Light: IDirect3DRMLight;
  begin   // R, G und W-Licht an der Mitte der Z-Achse
    with D3DRMInterface do
      if Succeeded(CreateFrame(Scene, Frame)) then
      begin
        Frame.SetPosition(Scene, PosX, PosY, CUBEDIMZSIZE / 2);
        CreateLightRGB(D3DRMLIGHT_PARALLELPOINT, R, G, B, Light);
        Frame.AddLight(Light);
      end;
  end;

begin
  Result := False;
  // Render-Qualitt (in der Praxis wird da Gouraud draus)
  if not CheckRes(Device.SetQuality(D3DRMRENDER_PHONG),
    'SetQuality') then Exit;
  // Meshbuilder anlegen und Utah-Teetopf aus einer .X-Datei laden
  if not CheckRes(D3DRMInterface.CreateMeshBuilder(MeshBuilder),
    'CreateMeshBuilder')
   or not CheckRes(MeshBuilder.Load(PChar('cube.x'),nil,
     D3DRMLOAD_FROMFILE, nil, nil), 'Load CUBE.X') then Exit;
  // Das Gitternetz mit einem Material versehen (5.0 = metallisch,
  // je hher, desto plastikartiger wird es), Farbe auf hellgrn
  if not CheckRes(D3DRMInterface.CreateMaterial(35.0,Material),
    'CreateMaterial (fr den Teetopf)')
   or not CheckRes(MeshBuilder.SetMaterial(Material),
     'SetMaterial (Cube)')
   or not CheckRes(MeshBuilder.SetColorRGB(0.9, 0.9, 0.9),
     'SetColorRGB (Cube-Material)') then Exit;

  // die Wrfel sind verhltnismig klein
  MeshBuilder.Scale(0.125, 0.125, 0.125);

  // Denkbare Mglichkeiten zum Einsetzen einiger hundert Wrfel
  // 1: Mesh n mal per Clone kopieren, ein gemeinsamer Frame
  // 2: n Frames anlegen, immer wieder denselben Mesh einsetzen
  for x := 0 to CUBEDIMXYSIZE do
    for y := 0 to CUBEDIMXYSIZE do
      for z := 0 to CUBEDIMZSIZE-1 do
        if Random(10) > 6 then  // 30 Prozent besetzen
          // Mglichkeit 2: n Frames, 1 Objekt
          if SUCCEEDED(D3DRMInterface.CreateFrame(Scene,
            CubeFrame)) then
           begin
             CubeFrame.AddVisual(MeshBuilder);
             // X und Y: -2..+2, Z: 0..39
             CubeFrame.SetPosition(Scene,
             2*(x-CUBEDIMXYSIZE div 2),y-CUBEDIMXYSIZE div 2,z);
        end;

  // Kameraposition setzen. Objekte (= Frames) mit denselben X/Y-
  // Koordinaten wie die Kamera liegen sozusagen vor der Nase
  // des Benutzers, der Z-Wert bestimmt die Entfernung.
  // Mitten ins Geschehen...
  Camera.SetPosition(Scene, 0, 0, CUBEDIMZSIZE / 2);
  // Callback vor jeder Bewegung der Kamera: Kollisionsprfung
  Camera.AddMoveCallback(OnCameraMoveHelper, Self,
      D3DRMCALLBACK_PREORDER);

  AddLight(-CUBEDIMXYSIZE*10,0, 1.0, 0,0);   // links, rot
  AddLight(CUBEDIMXYSIZE*10, 0, 0, 1.0, 0);  // rechts, grn
  AddLight(0, CUBEDIMXYSIZE*10, 1.0, 1.0, 1.0); // oben, wei

  // Umgebungslicht, hier ohne eigenen Frame
  D3DRMInterface.CreateLightRGB(D3DRMLIGHT_AMBIENT,
    0.1,0.1,0.1, LightA);
  Scene.AddLight(LightA);

  Result:=True;  // wunnebar...

  // Begrenzung des Sichtfelds = massive Beschleunigung des
  // Renderings. Ohne Fogging erscheinen Wrfel an der Sichtbar-
  // keitsgrenze schlagartig im Bild
  View.Setback(8);    // erhht die Framerate um den Faktor 4(!)
  // Hintergrundfarbe Schwarz, "Nebelfarbe" ebenfalls
  Scene.SetSceneBackground(0); Scene.SetSceneFogColor(0);
  Scene.SetSceneFogEnable(True);  // Fogging an
  // ab Entfernung 3 einnebeln, ab 8 nur noch "Nebelfarbe"
  Scene.SetSceneFogParams(3,8,1);

  // View.SetFront(1); // das wre die vordere Sichtbarkeitsgrenze
end;


function TCWorldForm.RenderScene: Boolean;
begin
  // Objekt(e) um 1 Frame bewegen. Scene.Move(1.0) ginge auch
  Result := CheckRes(D3DRMInterface.Tick(1.0),'Tick')
   // Viewport lschen
  and CheckRes(View.Clear(D3DRMCLEAR_ALL),'View.Clear')
  // Szene in den Viewport zeichnen
  and CheckRes(View.Render(Scene),'View.Render')
  // und Fenster aktualisieren
  and CheckRes(Device.Update,'Device.Update (RenderScene)');
end;

var FrameTimes: Single; // Framecounter einfachster Art
    LastTick: Cardinal;

procedure TCWorldForm.OnDXTimer(Sender: TObject);
var Dir, Up, Pos, Vel: TD3DVector; NextTick: Cardinal;
begin
  NextTick := timeGetTime;
  if LastTick = 0 then LastTick := NextTick-20;  // "50 Frames"
  FrameTimes := FrameTimes - FrameTimes / 20 + NextTick-LastTick;
  LastTick := NextTick;

  if D3DInitialized then
  begin  // reine Informationsfunktion
    Camera.GetOrientation(Scene, Dir, Up);
    Camera.GetPosition(Scene, Pos);
    Camera.GetVelocity(Scene, Vel, False);
    Caption := Format('F: %2.1f Dir (%.2f %.2f %.2f) '+
      'Up (%.2f %.2f %.2f) Pos (%2.2f %2.2f %2.2f) ' +
      'Vel (%.2f %.2f %.2f)',
    [(1000*20)/FrameTimes, Dir.x, Dir.y,Dir.z,
      Up.x,Up.y,Up.z,
    Pos.x, Pos.y, Pos.z, Vel.x, Vel.y, Vel.z]);

    if not RenderScene then
    begin
      PostQuitMessage(0);
      DXTimer.Enabled := False;
    end;
  end;
end;

// Reflektiert den Vektor V an N (N normalisiert),
// vereinfachte Version aus Mike Lischkes OpenGL-Units
// HINWEIS: D3DRMVectorReflect produziert Unsinn!
function VectorReflect(V,N: TD3DVector): TD3DVector;
var Dot: TD3DValue; // Single;
begin
  D3DRMVectorNormalize(N);
  Dot := D3DRMVectorDotProduct(V,N);
  Result.x := V.x-2*Dot*N.x;
  Result.y := V.y-2*Dot*N.y;
  Result.z := V.z-2*Dot*N.z;
end;

procedure TCWorldForm.OnCameraMove(Sender: TObject);
var Pos, NewPos, Dir, NewDir: TD3DVector;

  function D3DVector(x,y,z: TD3DValue): TD3DVector;
  begin
    Result.x := x; Result.y := y; Result.z := z;
  end;

  // Vergleich und Begrenzung
  function MaxPos(var Value: TD3DValue; Max: TD3DValue): Boolean;
  begin
    if Max < 0 then Result := Value < Max
      else Result := Value > Max;
    if Result then Value := Max;
  end;

begin
  // Positionsprfung und Reflektion an den Grenzen des Spielfelds
  // Blickrichtung (GetOrientation) ungleich Bewegungsvektor,
  // ndert sich bei Kollisionen nicht
  Camera.GetPosition(Scene, Pos);
  Camera.GetVelocity(Scene, Dir, False);

  NewPos := Pos; NewDir := Dir;

  if MaxPos(NewPos.x, -CUBEDIMXYSIZE / 4)
    then NewDir := VectorReflect(Dir, D3DVector(1,0,0));
  if MaxPos(NewPos.x, CUBEDIMXYSIZE / 4)
    then NewDir := VectorReflect(Dir, D3DVector(-1,0,0));

  if MaxPos(NewPos.y, -CUBEDIMXYSIZE / 4)
    then NewDir := VectorReflect(Dir, D3DVector(0,1,0));
  if MaxPos(NewPos.y, CUBEDIMXYSIZE / 4)
    then NewDir := VectorReflect(Dir, D3DVector(0,-1,0));

  if MaxPos(NewPos.z, -0.01)
    then NewDir := VectorReflect(Dir, D3DVector(0,0,-1));
  if MaxPos(NewPos.z, CUBEDIMZSIZE * 3 / 4)
    then NewDir := VectorReflect(Dir, D3DVector(0,0,1));


  if (NewDir.x <> Dir.x) or (NewDir.y <> Dir.y) or
    (NewDir.z <> Dir.z) then
    with NewDir do
    begin  // Aufprall halbiert die Geschwindigkeit
      Camera.SetVelocity(Scene, x / 2, y / 2, z / 2, False);
      if (NewPos.x <> Pos.x) or (NewPos.y <> Pos.y) or
        (NewPos.z <> Pos.z) then  // Position auerhalb!
           with NewPos do Camera.SetPosition(Scene, x, y, z);
    end;
  // Hier knnte man jetzt eine Trefferprfung fr die
  // einzelnen Wrfel nach demselben Muster einbauen
end;


procedure TCWorldForm.FormKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
var Direction, Up, Right: TD3DVector;

  // Beschleunigung in oder gegen Blickrichtung; Direction
  // mu vorher mit GetOrientation besetzt worden sein
  procedure SetCameraSpeed(Delta: TD3DValue);
  var Velocity: TD3DVector;
  begin
    Camera.GetVelocity(Scene, Velocity, False);
    D3DRMVectorNormalize(Direction);
    with Velocity do
    begin
      x := x + Direction.x * Delta;
      y := y + Direction.y * Delta;
      z := z + Direction.z * Delta;
      Camera.SetVelocity(Scene, x,y,z, False);
    end;
  end;

begin
  // Blickrichtung und Up-Vektor
  Camera.GetOrientation(Scene, Direction, Up);
  // Achse quer zur Blickrichtung
  D3DRMVectorCrossProduct(Right, Up, Direction);

  case  Key of
    VK_F1:
      ShowMessage('CubeWorld - simple interactive D3DRM Demo '+
         'as (Arne Schpers) JAN-2000'#13#10+
         'T = forward, G = stop, V = backward; '+
         'arrow keys to change direction');
    VK_ESCAPE:
      begin
        Speed := 0;
        Camera.SetRotation(Scene,0,0,0,0);
        Camera.SetPosition(Scene,0,0,CUBEDIMZSIZE / 2);
        Camera.SetOrientation(Scene,0,0,1,0,1,0);
        Camera.SetVelocity(Scene, 0,0,0, False);
      end;
    Ord('T'):  // Vorwrts-Beschleunigung
      SetCameraSpeed(0.01);
    Ord('G'):  // Stop
        Camera.SetVelocity(Scene, 0,0,0, False);
    Ord('V'):  // rckwrts-Beschleunigung
      SetCameraSpeed(-0.01);

    VK_RIGHT:  // Rotation um Up (senkrecht zu Direction)
        with Up do Camera.SetRotation(Scene, x,y,z,0.01);
    VK_LEFT:  // dito
        with Up do Camera.SetRotation(Scene, x,y,z,-0.01);
    VK_UP:  // Rotation um die Achse quer zur Blickrichtung
      with Right do Camera.SetRotation(Scene, x,y,z,0.01);
    VK_DOWN: // dito
      with Right do Camera.SetRotation(Scene, x,y,z,-0.01);
  end;
end;

procedure TCWorldForm.FormKeyUp(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  case Key of
    VK_RIGHT, VK_LEFT,
    VK_UP,VK_DOWN:
      Camera.SetRotation(Scene,0,0,0,0);  // Stop
  end;
end;

end.
