/*lfs.c: 64-Bit-Dateizugriff unter Linux */
/* Bei Auskommentieren der beiden #defines entsteht ein 32-Bit-Binary */
/* (lfs32), bei Drinlassen ein 64-Bit-Binary (lfs64) */

#define _LARGEFILE64_SOURCE
#define _FILE_OFFSET_BITS 64

#include <stdio.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>

#define BYTES_TO_WRITE 1   /* after lseek() */
#define BLOCK_SIZE 4096    /* for write() */

off_t file_size;           /* for sparse file via lseek() */
off_t blocks_to_write;     /* for "real" file via write() */
char path[256];


void usage()
{
  printf("usage: lfs -d path -s n[k|M] -w n[k|M]\n");
  printf("       -d: path where to create the files\n");
  printf("       -s: size of sparse file\n");
  printf("       -w: size of write file\n");
  exit(0);
}   /* usage() */


void params(int argc, char *argv[])
{
  int i;
  char *c;
  off_t bytes;

  path[0] = '\0';
  file_size=0;
  blocks_to_write=0;

  for (i=1; i<argc; i++)
  {
    c = argv[i];
    if ('-' == *c)
    {
      c++;
      if ('d' == *c)
      {
        i++;
        if (i<argc) strcpy(path, argv[i]); else usage();
      }
      else if ('s' == *c)
      {
        i++; 
        if (i<argc) c = argv[i]; else usage();
        file_size = atoi(c);
        while (*c != '\0') c++; c--;
        if (('k' == *c) || ('K' == *c)) file_size*=1024;
        else if (('m' == *c) || ('M' == *c)) file_size*=1048576;
      }
      else if ('w' == *c)
      {
        i++;
        if (i<argc) c = argv[i];
        else usage();
        blocks_to_write = atoi(c);
        while (*c != '\0') c++; c--;
        if (('k' == *c) || ('K' == *c)) blocks_to_write*=1024;
        else if (('m' == *c) || ('M' == *c)) blocks_to_write*=1048576;
        bytes = blocks_to_write;
        blocks_to_write /= BLOCK_SIZE;
        if (bytes % BLOCK_SIZE) blocks_to_write++;
      }
      else usage();
    }   /* if ('-' == *c) */
    else usage();
  }   /* for i */
  if ('\0' == path[0]) usage();
  printf("writing files to %s\n", path);
  printf("sparse file: %1lld + %1d Bytes (%3.1f MBytes)\n",
         file_size, BYTES_TO_WRITE, file_size / (1024.0 * 1024.0));
  printf("'real' file: %1lld Bytes (%1lld blocks of %1d Bytes)\n\n",
         blocks_to_write*BLOCK_SIZE, blocks_to_write, BLOCK_SIZE);
}   /* params() */


int create_sparse_file(char *path, off_t file_size)
{ /* 0: error, 1: ok */
  char fname[256];
  int f;
  ssize_t err;
  struct stat st;
  
  sprintf(fname, "%s%s", path, "/sparse");
  f = open(fname, O_RDWR|O_CREAT|O_TRUNC, 511);
  if (f<0)
  {
    perror("open(sparse) failed");
    return(0);
  }
  if (lseek(f, file_size, SEEK_SET) != file_size)
  {
    perror("lseek(sparse) failed");
    close(f); return(0);
  }
  err = write(f, fname, BYTES_TO_WRITE);
  if (-1 == err)
  {
    perror("write(sparse) failed");
    close(f); return(0);
  }
  if (BYTES_TO_WRITE != err)
  {
    printf("write(sparse) failed: only %1d of %1d Bytes written\n",
           (int) err, BYTES_TO_WRITE);
    close(f); return(0);
  }
  fstat(f, &st); close(f);
  printf("%s created as a sparse file of %1lld + %1d Bytes\n",
         fname, file_size, BYTES_TO_WRITE);
  printf("stat(sparse) says: %1lld Bytes\n", st.st_size);
  return(1);
}   /* create_sparse_file() */


int write_file(char *path, off_t blocks_to_write)
{ /* 0: error, 1: ok */
  char fname[256];
  int f;
  char buf[BLOCK_SIZE];
  off_t i;
  ssize_t err;
  struct stat st;

  for (i=0; i<BLOCK_SIZE; i++) buf[i] = (char) i;
  sprintf(fname, "%s%s", path, "/written");
  f = open(fname, O_RDWR|O_CREAT|O_TRUNC, 511);
  if (f<0)
  {
    perror("open(written) failed");
    return(0);
  }
  for (i=0; i<blocks_to_write; i++)
  {
    err = write(f, buf, BLOCK_SIZE);
    if (-1 == err)
    {
      sprintf(fname, "write(written) failed on block # %1lld\n", i);
      perror(fname);
      close(f); return(0);
    }
    if (BLOCK_SIZE != err)
    {
      printf("write(written) failed on block # %1lld\n", i);
      close(f); return(0);
    }
  }   /* for i */
  fstat(f, &st); close(f);
  printf("%s created with %1lld blocks of %1d Bytes\n",
         fname, blocks_to_write, BLOCK_SIZE);
  printf("stat(written) says: %1lld Bytes\n", st.st_size);
  return(1);
}   /* write_file() */


int main(int argc, char *argv[])
{
  struct stat st;
  printf("sizeof(off_t): %1d -- ", sizeof(off_t));
  printf("system uses %1d bit file offsets\n", sizeof(off_t)*8);
  printf("sizeof(stat->st_size): %1d\n\n", sizeof(st.st_size));
  params(argc, argv);
  if (file_size>0) create_sparse_file(path, file_size);
  if (blocks_to_write>0) write_file(path, blocks_to_write);
  exit(0);
}   /* main() */

