//Titel:        Mail-Client
//Version:      0.99
//Copyright:    Copyright (c) 2000
//Autor:        Christoph Borowski, cb@ordix.de
//Firma:        ORDIX AG, www.ordix.de, Westernmauer 12-16, 33098 Paderborn
//Beschreibung: Beispielanwendung, die die JavaMail-API einsetzt.

package mail.model;

import java.util.*;
import javax.mail.*;

/**
  * Die Klasse MailReceiver implementiert die Funktionalitt
  * zum Empfangen, Auslesen und Lschen von e-Mails.
  */
public class MailReceiver extends Observable {
	private Properties props;
	private Session session;
	private Store store;
	private Folder folder;
	private Message[] messages;

	/**
	  * Default-Konstruktor
	  */
	public MailReceiver() {
		// Referenz auf Properties-Objekt holen
		props = System.getProperties();

		// Referenz auf ein Session-Objekt holen
		session = Session.getDefaultInstance(props, null);
	}

	/**
	  * Holt alle Mails von dem Mailserver ab.
	  */
	public void readMessages() {
		try {
			// falls ein Ordner vom letzten Zugriff geffnet ist ...
			if (folder != null && folder.isOpen()) {
				// ... dann wird der Ordner und die Verbindung zum
				// Mailkonto geschlossen
				folder.close(false);
				store.close();
			}

			// Mailserver-URL zusammenstellen, die das Format
			// pop3://user:password@host besitzt
			String pop3URL = "pop3://"+MailOptions.get().getLogin()+
								":"+MailOptions.get().getPassword()+
								"@"+MailOptions.get().getMailServerIn();

			store = session.getStore(new URLName(pop3URL));
			store.connect();

			// andere Mglichkeit sich zum Mailkonto zu verbinden:
			// store.connect(host, port, user, password);

	    // Defaultordner auswhlen
			folder = store.getDefaultFolder();
			if (folder == null) {
				System.out.println("Kein Defaultorder vorhanden.");
				return;
			}

			// Ordner INBOX auswhlen
			folder = folder.getFolder("INBOX");
			if (folder == null) {
				System.out.println("Kein Ordner 'Inbox' vorhanden.");
				return;
			}

			// Ordner INBOX zum Lesen und Schreiben ffnen
			folder.open(Folder.READ_WRITE);

			// Anzahl der im Ordner INBOX gespeicherten Mails ermitteln
			int totalMessages = folder.getMessageCount();

			if (totalMessages == 0) {
				System.out.println("Keine Mails im Ordner INBOX");
				folder.close(false);
				store.close();
        messages = null;
  			setChanged();
	  		notifyObservers();
				return;
			}

			// Mails als Leichtgewichtsobjekte holen
			messages = folder.getMessages();

			// Fetchprofil fr die wichtigsten Headerwerte erstellen
			FetchProfile fp = new FetchProfile();
			fp.add(FetchProfile.Item.ENVELOPE);
			fp.add(FetchProfile.Item.CONTENT_INFO);
			fp.add(FetchProfile.Item.FLAGS);
			fp.add("X-mailer");

			// die im Fetchprofil angegebenen Headerwerte fr die
			// im Ordner INBOX gespeicherten Mails abrufen
			folder.fetch(messages, fp);

			// alle Observer informieren, da ein aktuallisierter
			// Mailbestand vorliegt
			setChanged();
			notifyObservers();
   	} catch (Exception e) {
			System.out.println("Fehler in Methode readMessages():" +
                                                e.getMessage());
			e.printStackTrace();
		}
	}

	/**
	  * Liefert alle E-Mails als Message-Array zurck.
	  */
	public Message[] getMessages() {
		return messages;
	}

	/**
	  * Liefert den Mailtext einer E-Mail zurck.
	  */
	public String getMessageText(int index) {
		try {
      String contentType = messages[index].getContentType();
			if (contentType.substring(0, 11).equals("text/plain;"))
				return (String)messages[index].getContent();
			else
				if (messages[index].getContent() instanceof Multipart) {
				  	StringBuffer mailText = new StringBuffer();

					Multipart mp = (Multipart)messages[index].getContent();
					BodyPart bp;
					for (int i=0; i<mp.getCount(); i++) {
						bp = mp.getBodyPart(i);
            contentType = bp.getContentType();
						if (contentType.substring(0,11).equals("text/plain;"))
							mailText.append((String)bp.getContent());
					}
					return mailText.toString();
				}
   		} catch (Exception e) {
			System.out.println("Fehler in Methode getMessageText(...):" +
                                                    e.getMessage());
			e.printStackTrace();
		}
		return "Inhaltstyp wird nicht untersttzt ...";
	}

	/**
	  * Lscht eine E-Mail.
	  */
	public void deleteMessage(int index) {
		try {
			// zu lschende Mail markieren und ...
			messages[index].setFlag(Flags.Flag.DELETED, true);
			// ber Methode folder.close(true) lschen - folder.expunge()
			// wird von der POP3-Implementierung nicht untersttzt.
			folder.close(true);
			// aktuallisierten Mailbestand vom Mailserver lesen
			readMessages();
		} catch (Exception e) {
			System.out.println("Fehler in Methode deleteMessage(...):" +
                                                    e.getMessage());
			e.printStackTrace();
		}
	}
}