unit D3DXDemoU;  // 06-AUG-2000 as (Arne Schpers)
// D3DX with Delphi, based on the "tentacle" demo from
// the DirectX 7 SDK.
// Requires: DirectX 7 Headers, D3DX.PAS, D3DXAS[D].DLL

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls,
  Forms, Dialogs, Direct3D, DirectDraw, D3DX, DXTimer, Registry,
  StdCtrls, MMSystem, // timeGetTime
  DirectInput, DXInputAS;  // DirectInput, Delphi-Objekte dazu;

type // Textures
  TTexIndices = (TEX_GROUND, TEX_BOX,
    TEX_FLARE0, TEX_FLARE1, TEX_FLARE2, TEX_FLARE3);

const
  TexIndexMax = TEX_FLARE3;
  g_szTexName: Array[TTexIndices] of String =
    ('ground1.bmp', 'dx_logo.bmp',
    'flare0.bmp', 'flare1.bmp', 'flare2.bmp', 'flare3.bmp');

// Delphi 3 compatible lookalike for D3DXFrameTimer
// uses the system performance counter (if available)
type
  TD3DXFrameTimer = class(TComponent)
  private
    class function LI2Comp(LI: TLargeInteger): Comp;
  protected
    FLastTick, FTicksPerSec: Comp;
    FTicksPerFrame: Single;
  public
    function GetTicks: Comp;
    function GetFramesPerSec: Single;
    function GetSecsPerFrame: Single;
  public
    constructor Create(AOwner: TComponent); override;
    procedure Start(_FramesPerSec: Single);
    procedure Frame;
    property TicksPerSec: Comp read FTicksPerSec;
    property TicksPerFrame: Single read FTicksPerFrame;
  public  // aliases to simplify SDK sample ports
    property GetTicksPerSec: Comp read FTicksPerSec;
    property GetTicksPerFrame: Single read FTicksPerFrame;
  end;

// ------------------------------------------
// (C) TLensFlare
// Copyright (c) 1999 Microsoft Corporation. All rights reserved.

type
  TFlareData = record
    Texture: IDirectDrawSurface7;
    Opacity, Radius, Position: Float;
  end;
  PFlareData = ^TFlareData;

  TLensFlare = class(TObject)
  protected
    FDistance, FSourceRadius: Float;
    m_pSourceTexture: IDirectDrawSurface7;
    FlareData: TList;
    m_colorLight: TD3DXCOLOR;
    m_vecLight: TD3DXVECTOR4;
    procedure CleanupData;
  public
    constructor Create;
    destructor Destroy; override;
    function Initialize(uFlares: UINT; Distance: Float): HResult;
    procedure SetLightPosition(Vec: TD3DXVector4);
    procedure SetLightColor(Color: TD3DXColor);
    procedure SetSource(fRadius: Float;
       const pTexture: IDirectDrawSurface7);
    procedure SetFlare(uFlare: UINT; fRadius, fOpacity,
       fPosition: Float; const pTexture: IDirectDrawSurface7);
    function Draw(matPos: TD3DXMatrix;
       const D3DDevice: IDirect3DDevice7): HResult;
  end;

// ------------------------------------------
type
  TD3DXDemoForm = class(TForm)
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure FormPaint(Sender: TObject);
    procedure FormKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
  private
    FGroundWidth, FGroundHeight, FGroundTiles: Float;
    m_dwColor: DWord;
    m_uVertices, m_uIndices: UINT;
    m_pwIndices: PWord;
    m_pvbVertices: IDirect3DVertexBuffer7;
    function CreateGroundSurface: HResult;
    function CreateGroundMesh: HResult;
  private
    g_matIdentity: TD3DXMatrix;
    g_matPosition, g_matView, g_matProjection: TD3DXMatrix;
    g_D3DDeviceDesc: TD3DDEVICEDESC7;
    g_pD3DX: ID3DXContext;
    g_pDD: IDirectDraw7;
    g_pD3D: IDirect3D7;
    g_pD3DDevice: IDIRECT3DDEVICE7;
  private
    g_bDrawLensFlare, g_bCapsLensFlare,
    g_bReady, g_bActive, g_bFullScreen: Boolean;
    g_dwHwLevel, g_dwNumBackBuffers: DWord;
    g_pLensFlare: TLensFlare;
    Torus, Box: ID3DXSimpleShape;
    g_colorLight: TD3DXColor;
    g_vecLight: TD3DXVECTOR4;
    // Background
    g_colorClear: TD3DXCOLOR;
    // Ground plane
    g_bDrawGround: Boolean;
    g_planeGround: TD3DXPLANE;  // unused
    g_colorGround: TD3DXCOLOR;
    // speed & scaling
    g_fSpeed, g_fAngularSpeed: Float;
    g_vecVelocity, g_vecAngularVelocity: TD3DXVector3;
    g_ppTex: Array[TTexIndices] of IDirectDrawSurface7;
    // UI
    g_bWireframe, g_bDrawTorii, g_bDrawBoxes: Boolean;
    g_bOnIdle: Boolean;
    g_bUnlimitedUniverse: Boolean;
    FramesPerSec: Single; LastTick: Comp;
    g_bDrawFramerate, g_bFogging: Boolean;
    g_bDrawHelp: Boolean;
  protected
    DXTimer: TDXTimer;
    FrameTimer: TD3DXFrameTimer;
    function HandleWindowedModeChanges: HResult;
    procedure OnDXTimer(Sender: TObject);
    procedure OnIdle(Sender: TObject; var Done: Boolean);
  public
    procedure ActivateApp(Sender: TObject);
    procedure DeactivateApp(Sender: TObject);
    function CreateScene: HResult;
    function ReleaseScene: HResult;
    function CreateContext: HResult;
    function RestoreContext: HResult;
    function ReleaseContext: HResult;
    function Shutdown: HResult;
    function Draw: HResult;
  end;

var
  D3DXDemoForm: TD3DXDemoForm;

implementation
{$R *.DFM}

{ TD3DXFrameTimer }
constructor TD3DXFrameTimer.Create(AOwner: TComponent);
var Res: TLargeInteger;
begin
  inherited;
  if QueryPerformanceFrequency(Res)
    then FTicksPerSec := LI2Comp(Res)
    else FTicksPerSec := 1000;
end;

class function TD3DXFrameTimer.LI2Comp(LI: TLargeInteger): Comp;
begin
{$IFDEF VER130}  // Delphi 5
  Result := LI;
{$ELSE}  // Delphi 3 and 4
  asm fild qword ptr [LI]
      fistp qword ptr @Result;
  end;
{$ENDIF}
end;

procedure TD3DXFrameTimer.Frame;
var ThisFrameTick, TickDelta: Comp;
begin
  ThisFrameTick := GetTicks;
  TickDelta := ThisFrameTick - FLastTick;
  FLastTick := ThisFrameTick;
  if TickDelta < 0 then Exit;  // crossed 0
  FTicksPerFrame := FTicksPerFrame * 0.9 + TickDelta * 0.1;
end;

function TD3DXFrameTimer.GetTicks: Comp;
var Res: TLargeInteger;
begin
  if QueryPerformanceCounter(Res) then
  begin
   while LI2Comp(Res) = 0 do
     QueryPerformanceCounter(Res);
   Result := LI2Comp(Res);
  end
    else Result := timeGetTime;
end;

function TD3DXFrameTimer.GetFramesPerSec: Single;
begin
  if GetTicksPerFrame = 0 then Result := 0
   else Result := TicksPerSec / GetTicksPerFrame;
end;

function TD3DXFrameTimer.GetSecsPerFrame: Single;
begin
  Result := TicksPerFrame / TicksPerSec;
end;

procedure TD3DXFrameTimer.Start(_FramesPerSec: Single);
begin
  FLastTick := GetTicks;
  FTicksPerFrame := TicksPerSec / _FramesPerSec;
end;


procedure TD3DXDemoForm.FormCreate(Sender: TObject);
var Res: HResult;
begin
  g_bWireframe := False; g_bDrawFrameRate := True;
  g_bDrawTorii := True; g_bDrawBoxes := True;
  g_bDrawLensFlare := True; g_bReady := False;
  g_bActive := True; g_bFullScreen := False;
  g_dwHwLevel := D3DX_DEFAULT;
  g_dwNumBackBuffers := 2;

  // Light
  g_colorLight := D3DXCOLOR(1.0, 0.95, 0.8, 1.0);
  g_vecLight := D3DXVECTOR4(0.0, -0.5, 1.5, 0.0);
  // Background
  g_colorClear := D3DXCOLOR (0.4, 0.5, 0.8, 1.0);

  // Ground plane
  g_bDrawGround := TRUE;

  g_planeGround := D3DXPLANE(0.0, 1.0, 0.0, 0.0);
  g_colorGround := D3DXCOLOR(0.17, 0.35, 0.7, 0.75);

  g_fSpeed := 5.0; g_fAngularSpeed := 1.0;  // const

  g_vecVelocity := D3DXVECTOR3(0.0, 0.0, 0.0);
  g_vecAngularVelocity := D3DXVECTOR3(0.0, 0.0, 0.0);

  D3DXMatrixIdentity(g_matIdentity);

  Res := CreateScene;
  if SUCCEEDED(Res) then Res := CreateContext;
  if FAILED(Res) then ShowMessage(D3DXGetErrorMsg(Res));

  DXTimer := TDXTimer.Create(Self);
  with DXTimer do
  begin
    Interval := 10; OnTimer := OnDXTimer;
    Enabled := g_bReady;
  end;
  FrameTimer := TD3DXFrameTimer.Create(Self);
  FrameTimer.Start(100);

  Application.OnActivate := ActivateApp;
  Application.OnDeactivate := DeactivateApp;
  Application.OnIdle := OnIdle;

end;

// If GroundDivs <> FGroundTiles then CreateGroundMesh will
// adjust texture coords accordingly
const GroundDivs = 8;

function TD3DXDemoForm.CreateGroundMesh: HResult;
type TWordArray = Array[0..5] of Word;
var uX, uZ, uVertex: UINT;
    pwIndex: ^TWordArray;
begin
  FGroundWidth  := 256; FGroundHeight := 256;
  FGroundTiles   := 10;
  m_dwColor := D3DRGBA(0.17, 0.35, 0.7, 0.75);


  // Calculate number of vertices and indices
  m_uVertices := (GroundDivs + 1) * (GroundDivs + 1);
  m_uIndices  := (GroundDivs * GroundDivs) * 6;

  // Create indices
  GetMem(m_pwIndices, m_uIndices * SizeOf(Word));

  // Fill in indicies
  pwIndex := Pointer(m_pwIndices);

  for uZ := 0 to GroundDivs-1 do
    for uX := 0 to GroundDivs-1 do
    begin
      uVertex := uX + uZ * (GroundDivs + 1);

      pwIndex[0] := uVertex + 0;
      pwIndex[1] := uVertex + 1;
      pwIndex[2] := uVertex + (GroundDivs + 1);
      pwIndex[3] := uVertex + (GroundDivs + 1);
      pwIndex[4] := uVertex + 1;
      pwIndex[5] := uVertex + (GroundDivs + 2);

      Inc(pwIndex);
    end;
  Result := S_OK;
end;

type
  TGroundVertex = record
    m_vecPos: TD3DXVector3;
    m_dwDiffuse: TD3DColor;
    m_vecTex: TD3DXVector2;
  end;
  TGroundVertexArray = Array[0..999] of TGroundVertex;

function TD3DXDemoForm.CreateGroundSurface: HResult;
var vbdesc: TD3DVertexBufferDesc;
    pVertices: ^TGroundVertexArray;
    uX, uZ: UINT;
    fXInc, fZInc, fSInc, fTInc, fZ, fT, fX, fS: TD3DValue;
    DummySize: DWord;
begin
  // Create vertices
  vbdesc.dwSize := sizeof(vbdesc);
  vbdesc.dwCaps := 0;
  vbdesc.dwFVF := D3DFVF_XYZ or D3DFVF_DIFFUSE or D3DFVF_TEX1 or D3DFVF_TEXCOORDSIZE2(0);
  vbdesc.dwNumVertices := m_uVertices;

  if (g_D3DDeviceDesc.dwDevCaps and D3DDEVCAPS_HWTRANSFORMANDLIGHT = 0) or
   EqualGUID(g_D3DDeviceDesc.deviceGUID, IID_IDirect3DRefDevice)
      then vbdesc.dwCaps := vbDesc.dwCaps or D3DVBCAPS_SYSTEMMEMORY;

  Result := g_pD3D.CreateVertexBuffer(vbdesc, m_pvbVertices, 0);
  if FAILED(Result) then Exit;

  if SUCCEEDED(m_pvbVertices.Lock(DDLOCK_WAIT or DDLOCK_WRITEONLY, Pointer(pVertices), DummySize)) then
  begin
    fXInc := FGroundWidth / GroundDivs; fZInc := FGroundHeight / GroundDivs;

    fSInc := FGroundTiles / GroundDivs; fTInc := FGroundTiles / GroundDivs;

    fZ := FGroundHeight * -0.5; fT := 0.0;

    for uZ := 0 to GroundDivs do
    begin
      fX := FGroundWidth * -0.5; fS := 0.0;
      for uX := 0 to GroundDivs do
        with pVertices[uX + uZ * (GroundDivs + 1)] do
        begin
          m_vecPos := D3DXVector3(fX,0,fZ);
          m_dwDiffuse := m_dwColor;
          m_vecTex := D3DXVector2(fS, fT);
          fX := fX + fXInc; fS := fS + fSInc;
        end;
      fZ := fZ + fZInc; fT := fT + fTInc;
    end;

    m_pvbVertices.Unlock;
    m_pvbVertices.Optimize(g_pD3DDevice, 0);
  end;
end;

// = OneTimeSceneInit
function TD3DXDemoForm.CreateScene: HResult;
begin
  // Create ground
  CreateGroundMesh;
  // Create LensFlare
  g_pLensFlare := TLensFlare.Create;
  Result := g_pLensFlare.Initialize(6, 95.0);
  if Failed(Result) then Exit;

  g_pLensFlare.SetLightColor(g_colorLight);
  g_pLensFlare.SetLightPosition(g_vecLight);

  // Misc stuff
  D3DXMatrixTranslation(g_matView, 0.0, 0.0, -20.0);
  D3DXMatrixTranslation(g_matPosition, 0.0, 0.0, 20.0);
  D3DXMatrixPerspectiveFov(g_matProjection, D3DXToRadian(60.0), 3.0 / 4.0, 0.1, 1000.0);

  Result := S_OK;
end;


function TD3DXDemoForm.CreateContext: HResult;
var fFogStart, fFogEnd: Float; d3dLight: TD3DLight7; d3dMaterial: TD3DMaterial7;
    x: TTexIndices;
    sf: TD3DX_SURFACEFORMAT;
    g_szPath: String;  // D3DXMediaPath

  function GetDXSDKMediaPath: String;
  var Reg: TRegistry;
  begin
    Reg := TRegistry.Create;
    with Reg do
    begin
      RootKey := HKEY_LOCAL_MACHINE;
      if OpenKey('Software\Microsoft\DirectX',False) then
      begin
        Result := ReadString('DXSDK Samples Path');
        if Result <> '' then Result := Result + '\D3DX\Media\';
      end;
      Free;
    end;
  end;

begin
  // Initialize D3DX
  Result := D3DXInitialize;
  if Failed(Result) then Exit;

  if g_bFullScreen then
    Result := D3DXCreateContextEx(g_dwHwLevel, D3DX_CONTEXT_FULLSCREEN,  Handle, 0, D3DX_DEFAULT, 0,
          D3DX_DEFAULT, 0, g_dwNumBackBuffers, 640, 480, D3DX_DEFAULT, g_pD3DX)
  else
  begin
    g_dwNumBackBuffers := 1;
    Result := D3DXCreateContextEx(g_dwHwLevel, 0, Handle, 0, D3DX_DEFAULT, 0,
          D3DX_DEFAULT, 0, g_dwNumBackBuffers, D3DX_DEFAULT, D3DX_DEFAULT, D3DX_DEFAULT, g_pD3DX);
    if Failed(Result) then
    begin
      ShowMessage('D3DXCreateContext failed, trying HWLEVEL_2D "Acceleration"');
      Result := D3DXCreateContext(
      // D3DX handle, flags, Window handle, colorbits,
      D3DX_HWLEVEL_2D, 0, Handle,  D3DX_DEFAULT, // colorbits
      D3DX_DEFAULT, g_pD3DX); // numdepthbits, D3DX interface
    end;
  end;

  if Failed(Result) then Exit;

  // das tte es genauso
  // g_pD3D := IDirect3D7(g_pD3DX.GetD3D);

  if not (D3DXDDFromContext(g_pD3DX, g_pDD) and
          D3DXD3DDeviceFromContext(g_pD3DX, g_pD3DDevice) and
          D3DXD3DFromContext(g_pD3DX, g_pD3D) ) then
  begin
    Result := E_FAIL; Exit;
  end;

  //
  // Grok device caps
  //

  FillChar(g_D3DDeviceDesc, SizeOf(TD3DDeviceDesc7), 0);
  g_pD3DDevice.GetCaps(g_D3DDeviceDesc);

  with g_D3DDeviceDesc do
    g_bCapsLensFlare  := (dpcTriCaps.dwSrcBlendCaps and D3DPBLENDCAPS_ONE <> 0) and
                         (dpcTriCaps.dwDestBlendCaps and D3DPBLENDCAPS_INVSRCCOLOR <> 0) and
                         (dpcTriCaps.dwShadeCaps and D3DPSHADECAPS_ALPHAFLATBLEND <> 0);

  //
  // Setup render states
  //

  g_pD3DX.SetClearColor(D3DXColorToDWord(g_colorClear));
  with g_pD3DDevice do
  begin
    SetTransform(D3DTRANSFORMSTATE_VIEW, TD3DMatrix(g_matView));
    SetTransform(D3DTRANSFORMSTATE_WORLD, TD3DMatrix(g_matIdentity));
    SetTransform(D3DTRANSFORMSTATE_PROJECTION, TD3DMatrix(g_matProjection));

    LightEnable(0, True);
    SetRenderState(D3DRENDERSTATE_AMBIENT, $ffffffff);

    fFogEnd := Sqrt(20*20+100*100); fFogStart := fFogEnd * 0.9;
    SetRenderState(D3DRENDERSTATE_FOGCOLOR, D3DXColorToDWord(g_colorClear));

    SetRenderState(D3DRENDERSTATE_FOGTABLEMODE, Ord(D3DFOG_LINEAR));

    SetRenderState(D3DRENDERSTATE_FOGSTART, PDWord(@fFogStart)^);
    SetRenderState(D3DRENDERSTATE_FOGEND, PDWord(@fFogEnd)^);
    SetRenderState(D3DRENDERSTATE_FOGENABLE, Ord(True));
    SetRenderState(D3DRENDERSTATE_RANGEFOGENABLE, Ord(True));
    SetTextureStageState(0, D3DTSS_MINFILTER, Ord(D3DTFN_LINEAR));
    SetTextureStageState(0, D3DTSS_MAGFILTER, Ord(D3DTFG_LINEAR));
    SetTextureStageState(0, D3DTSS_MIPFILTER, Ord(D3DTFP_POINT));
    SetTextureStageState(0, D3DTSS_COLOROP, Ord(D3DTOP_MODULATE));
    SetTextureStageState(0, D3DTSS_ALPHAOP, Ord(D3DTOP_MODULATE));
  end;

  //
  // Create light
  FillChar(d3dLight, Sizeof(d3dLight), 0);
  with d3dLight do
  begin
    dltType := D3DLIGHT_DIRECTIONAL;
    dcvDiffuse := TD3DColorValue(g_colorLight);
    dcvSpecular := TD3DColorValue(D3DXColor(1,1,1,0));
    dvDirection.x := g_vecLight.x;
    dvDirection.y := g_vecLight.y;
    dvDirection.z := g_vecLight.z;
  end;
  g_pD3DDevice.SetLight(0, d3dLight);


  // Create material
  FillChar(d3dMaterial, SizeOf(d3dMaterial), 0);
  with d3dMaterial do
  begin
    dcvDiffuse := TD3DColorValue(D3DXColor(0.75,0.75,0.75,0.25));
    dcvAmbient.r := g_colorLight.r;
    dcvAmbient.g := g_colorLight.g;
    dcvAmbient.b := g_colorLight.b;
    dcvSpecular := TD3DColorValue(D3DXColor(0.5,0.5,0.5,0.5));
    dvPower := 10.0;
  end;

  g_pD3DDevice.SetMaterial(d3dMaterial);

  // Simple Shapes
  D3DXCreateTorus(g_pD3DDevice, 0.8, 1.5, 16, 32, Cardinal(D3DX_DEFAULT), Torus);
  D3DXCreateBox(g_pD3DDevice, 3,3,3,1, Box);

  // Get media path from registry
  g_szPath := GetDXSDKMediaPath;
  // Create textures
  for x := TEX_GROUND to TexIndexMax do
  begin
   sf := D3DX_SF_UNKNOWN;
   if FAILED(D3DXCreateTextureFromFile(g_pD3DDevice, nil, nil, nil, sf, nil,
          g_ppTex[x], nil, PChar(g_szTexName[x]), D3DX_FT_LINEAR))
     and FAILED(D3DXCreateTextureFromFile(g_pD3DDevice, nil, nil, nil, sf, nil,
          g_ppTex[x], nil, PChar(g_szPath+g_szTexName[x]), D3DX_FT_LINEAR))

     then g_ppTex[x] := nil;
  end;

  // Create surfaces
  Result := CreateGroundSurface;
  if Failed(Result) then Exit;

  with g_pLensFlare do
  begin
    SetSource(17.0, g_ppTex[TEX_FLARE0]);
    // (uFlare: UINT; fRadius, fOpacity, fPosition: Float
    SetFlare(0,  8.00, 0.06,  1.00, g_ppTex[TEX_FLARE1]);
    SetFlare(1, 12.00, 0.04,  0.8, g_ppTex[TEX_FLARE2]);
    SetFlare(2,  4.00, 0.10,  0.50, g_ppTex[TEX_FLARE2]);
    SetFlare(3,  8.00, 0.08, -0.30, g_ppTex[TEX_FLARE2]);
    SetFlare(4, 12.00, 0.04, -0.60, g_ppTex[TEX_FLARE3]);
    SetFlare(5, 30.00, 0.04, -1.00, g_ppTex[TEX_FLARE1]);
  end;
  g_bReady := TRUE;
  Result := S_OK;
end;

procedure TD3DXDemoForm.OnIdle(Sender: TObject; var Done: Boolean);
begin
  if not g_bActive then Done := True
    else Done := not g_bOnIdle;
  if not Done and g_bReady then Draw;
end;

procedure TD3DXDemoForm.OnDXTimer(Sender: TObject);
begin
  Draw;
  DXTimer.Enabled := g_bReady;
end;

function TD3DXDemoForm.Draw: HResult;
var Data: TDIKeyboardState;  // Tastaturabfrage
    vecT, vecR: TD3DXVector3;
    matT, matR: TD3DXMatrix;
    qR: TD3DXQuaternion;
    fTimeKey: Float; x: Integer;
    DrawMode: TD3DPrimitiveType;
    matGround: TD3DXMatrix; GroundModulo: Double;

  function KeyDown(Index: Integer): Boolean;
  begin
    Result := Data[Index] and $80 <> 0;
  end;

  function ModF(const Arg, ModBy: Double): Double;
  begin
    Result := Trunc(Arg / ModBy) * ModBy;
  end;

begin
  Result := S_OK;
  if not g_bReady then Exit;
  if g_bActive and SUCCEEDED(DIKeyboard.Acquire) and SUCCEEDED(
    DIKeyboard.DIObject.GetDeviceState(SizeOf(Data),@Data)) then
  begin // Process keyboard input
    vecT := D3DXVECTOR3(0.0, 0.0, 0.0);
    vecR := D3DXVECTOR3(0.0, 0.0, 0.0);

    if KeyDown(DIK_A) or KeyDown(DIK_NUMPAD1) or KeyDown(DIK_LEFT)
       then vecT.x := vecT.x - 1.0; // Slide Left
    if KeyDown(DIK_D) or KeyDown(DIK_NUMPAD3) or KeyDown(DIK_RIGHT)
      then vecT.x := vecT.x + 1.0; // Slide Right
    if KeyDown(DIK_DOWN) then vecT.y := vecT.y - 1.0; // Slide Down
    if KeyDown(DIK_UP) then vecT.y := vecT.y + 1.0; // Slide Up
    if KeyDown(DIK_W) then vecT.z := vecT.z - 2.0; // Move Forward
    if KeyDown(DIK_S) then vecT.z := vecT.z + 2.0; // Move Backward

    if KeyDown(DIK_NUMPAD8) then vecR.x := vecR.x - 1.0; // Pitch Down
    if KeyDown(DIK_NUMPAD2) then vecR.x := vecR.x + 1.0; // Pitch Up
    if KeyDown(DIK_E) or KeyDown(DIK_NUMPAD6)
      then vecR.y := vecR.y - 1.0; // Turn Right
    if KeyDown(DIK_Q) or KeyDown(DIK_NUMPAD4)
      then vecR.y := vecR.y + 1.0; // Turn Left
    if KeyDown(DIK_NUMPAD9) then vecR.z := vecR.z - 2.0; // Roll CW
    if KeyDown(DIK_NUMPAD7) then vecR.z := vecR.z + 2.0; // Roll CCW

// C++: g_vecVelocity = g_vecVelocity * 0.9 + vecT * 0.1;
    g_vecVelocity := D3DXVec3Add(
      D3DXVec3Scale(g_vecVelocity, g_vecVelocity, 0.9)^,
      D3DXVec3Scale(vecT, vecT, 0.1)^);

// C++: g_vecAngularVelocity = g_vecAngularVelocity * 0.9 + vecR * 0.1;
    g_vecAngularVelocity := D3DXVec3Add(
      D3DXVec3Scale(g_vecAngularVelocity, g_vecAngularVelocity, 0.9)^,
      D3DXVec3Scale(vecR, vecR, 0.1)^);
    //
    // Update position and view matricies
    //

// C++: vecT = g_vecVelocity * g_pFrameTimer->GetSecsPerFrame() * g_fSpeed;
// C++: vecR = g_vecAngularVelocity * g_pFrameTimer->GetSecsPerFrame() * g_fAngularSpeed;
    D3DXVec3Scale(vecT, g_vecVelocity, FrameTimer.GetSecsPerFrame * g_fSpeed);
    D3DXVec3Scale(vecR, g_vecAngularVelocity, FrameTimer.GetSecsPerFrame * g_fAngularSpeed);

    D3DXMatrixTranslation(matT, vecT.x, vecT.y, vecT.z);
    D3DXMatrixMultiply(g_matPosition, matT, g_matPosition);

    D3DXQuaternionRotationYawPitchRoll(qR, vecR.y, vecR.x, vecR.z);
    D3DXMatrixRotationQuaternion(matR, qR);
    D3DXMatrixMultiply(g_matPosition, matR, g_matPosition);

    if g_bUnlimitedUniverse then
    begin
      with g_matPosition do
      begin
        if m31 < 1.0 then m31 := 1.0;  // ground
        if m31 > 20 then m31 := 20;  // ceiling
      end;
    end else
    begin
      with g_matPosition do
      begin
        if m30 > 100 then m30 := 100;
        if m30 < -100 then m30 := -100;

        if m31 < 1.0 then m31 := 1.0;  // ground
        if m31 > 20 then m31 := 20;  // ceiling

        if m32 < -100 then m32 := -100;
        if m32 > 100 then m32 := 100;
      end;
    end;
    D3DXMatrixInverse(g_matView, nil, g_matPosition);
  end;

  Result := g_pD3DDevice.BeginScene;
  if SUCCEEDED(Result) then
  begin
    g_pD3DX.Clear(D3DCLEAR_TARGET or D3DCLEAR_ZBUFFER);
    // Draw ground
    g_pD3DDevice.SetTransform(D3DTRANSFORMSTATE_VIEW, TD3DMatrix(g_matView));
    if g_bUnlimitedUniverse then
    begin  // ground repeats every 25.6 units
      D3DXMatrixIdentity(matGround);
      GroundModulo := FGroundWidth / FGroundTiles;
      matGround.m30 := Trunc(g_matPosition.m30 / GroundModulo) * GroundModulo;
      matGround.m32 := Trunc(g_matPosition.m32 / GroundModulo) * GroundModulo;
      g_pD3DDevice.SetTransform(D3DTRANSFORMSTATE_WORLD, TD3DMatrix(matGround));
    end else
    begin  // draw ground statically at (0,0,0)
      g_pD3DDevice.SetTransform(D3DTRANSFORMSTATE_WORLD, TD3DMatrix(g_matIdentity));
    end;
    if g_bDrawGround then
    begin
      with g_pD3DDevice do
      begin
        SetRenderState(D3DRENDERSTATE_CULLMODE, Ord(D3DCULL_CCW));
        SetRenderState(D3DRENDERSTATE_LIGHTING, Ord(False));
        if g_bFogging then SetRenderState(D3DRENDERSTATE_FOGENABLE, Ord(True));

        if g_bWireFrame then
        begin
          DrawMode := D3DPT_LINELIST;
          SetTexture(0, nil);
        end else
        begin
          DrawMode := D3DPT_TRIANGLELIST;
          if g_bUnlimitedUniverse then SetTexture(0, g_ppTex[TEX_BOX])
           else SetTexture(0, g_ppTex[TEX_GROUND]);
        end;

       //g_pGround->Draw();
      DrawIndexedPrimitiveVB(DrawMode, // D3DPT_TRIANGLELIST,
        m_pvbVertices, 0, m_uVertices,
        m_pwIndices^, m_uIndices, D3DDP_WAIT);

      if g_bUnlimitedUniverse or g_bWireFrame
        then SetTexture(0, g_ppTex[TEX_GROUND]);

      SetRenderState(D3DRENDERSTATE_LIGHTING, Ord(True));
      end;
    end;

    with g_pD3DDevice do
    begin
      fTimeKey := (timeGetTime mod 10000) / 10000;
      // sonst sieht's sehr hohl aus
      SetRenderState(D3DRENDERSTATE_CULLMODE, Ord(D3DCULL_CW));

      D3DXMatrixRotationX(matR,3*Pi * fTimeKey);
      if g_bDrawTorii then
      begin
        if not g_bDrawGround then SetTexture(0, g_ppTex[TEX_GROUND]);
        matR.m31 := 4.0;
        for x := -2 to 2 do
        begin
          matR.m30 := x*3;
          SetTransform(D3DTRANSFORMSTATE_WORLD, TD3DMatrix(matR));
          Torus.Draw;
        end;

        D3DXMatrixRotationZ(matR, 5*Pi* fTimeKey);
        matR.m31 := 4.5; matR.m32 := -3;
        SetTransform(D3DTRANSFORMSTATE_WORLD, TD3DMatrix(matR));
        Torus.Draw;
      end;

      if g_bDrawBoxes then
      begin
        SetTexture(0, g_ppTex[TEX_BOX]);
//      D3DXMatrixRotationY(matR, Pi* fTimeKey);
//      D3DXMatrixIdentity(matR);
        D3DXMatrixRotationX(matR, Pi);  // sonst stehen die Dinger auf dem Kopf

        matR.m31 := 2;  // Y
        for x := 0 to 3 do
        begin
          if Odd(x) then matR.m30 := -10 else matR.m30 := 10;
          if x < 2 then matR.m32 := -10 else matR.m32 := 10;
          SetTransform(D3DTRANSFORMSTATE_WORLD, TD3DMatrix(matR));
          Box.Draw;
        end;
        SetTransform(D3DTRANSFORMSTATE_WORLD, TD3DMatrix(g_matIdentity));
        SetRenderState(D3DRENDERSTATE_CULLMODE, Ord(D3DCULL_CCW));
      end;
    end;

    // Draw lens flare
    if g_bCapsLensFlare and g_bDrawLensFlare then
      with g_pD3DDevice do
      begin
        SetTransform(D3DTRANSFORMSTATE_WORLD, TD3DMatrix(g_matIdentity));
        SetRenderState(D3DRENDERSTATE_CULLMODE, Ord(D3DCULL_CCW));

        SetRenderState(D3DRENDERSTATE_FOGENABLE, Ord(False));
        SetRenderState(D3DRENDERSTATE_LIGHTING, Ord(False));
        SetRenderState(D3DRENDERSTATE_ALPHABLENDENABLE, Ord(True));
        SetRenderState(D3DRENDERSTATE_VERTEXBLEND, Ord(D3DVBLEND_DISABLE));
        SetRenderState(D3DRENDERSTATE_ZWRITEENABLE, Ord(False));
        SetRenderState(D3DRENDERSTATE_SRCBLEND, Ord(D3DBLEND_ONE));
        SetRenderState(D3DRENDERSTATE_DESTBLEND, Ord(D3DBLEND_INVSRCCOLOR));

        g_pLensFlare.Draw(g_matPosition, g_pD3DDevice);

        SetRenderState(D3DRENDERSTATE_ALPHABLENDENABLE, Ord(False));
        SetRenderState(D3DRENDERSTATE_LIGHTING, Ord(True));
        SetRenderState(D3DRENDERSTATE_ZWRITEENABLE, Ord(True));
        if g_bFogging then SetRenderState(D3DRENDERSTATE_FOGENABLE, Ord(True));
      end;

        // Draw framerate
      if g_bDrawFramerate then
      begin
        if LastTick < FrameTimer.GetTicks then
        begin  // update 3 times a second
          with FrameTimer do LastTick := GetTicks + GetTicksPerSec / 3;
          FramesPerSec := FrameTimer.GetFramesPerSec;
        end;
        if g_bUnlimitedUniverse then
          with g_matPosition do g_pD3DX.DrawDebugText(0.0, 0.0, $ffffff00,
            PChar(Format(' %.3d fps'#13#10+
              ' Pos (X: %3.2f, Y: %3.2f, Z: %3.2f)'#13#10+
              ' Gnd (X: %3.2f, Z: %3.2f)', [Trunc(FramesPerSec),
            m30, m31, m32, matGround.m30, matGround.m32])))
        else
          g_pD3DX.DrawDebugText(0.01, 0.0, $ff7fcf7f,
            PChar(Format('%d fps', [Trunc(FramesPerSec)])));

        if not g_bDrawHelp then
         g_pD3DX.DrawDebugText(0.89, 0.0, $ff7fcf7f, 'F1=help');
      end;

      // Draw help
      if g_bDrawHelp then
      begin
          g_pD3DX.DrawDebugText(0.01, 0.52, $ff7fcf7f,
              'Keyboard controls:'#13#10+
              '  Move'#13#10+
              '  Slide'#13#10+
              '  Turn'#13#10+
              '  Pitch'#13#10+
              '  Roll'#13#10+
              '  Exit'#13#10+
              ''#13#10+
              'Other controls:'#13#10+
              '  WireFrame'#13#10+
              '  Framerate'#13#10+
              '  LensFlare'#13#10+
              '  Torii, Boxes'#13#10+
              '  Fogging'#13#10+
              ''#13#10+
              '  OnIdle loop'#13#10+
              '  Unlimited');
          g_pD3DX.DrawDebugText(0.18, 0.52, $ff7fcf7f,
              ''#13#10+
              'W,S'#13#10+
              'Arrow Keys'#13#10+
              'Numpad 4,6'#13#10+
              'Numpad 2,8'#13#10+
              'Numpad 7,9'#13#10+
              'Escape'#13#10+
              ''#13#10+
              ''#13#10+
              'F2'#13#10+
              'R'#13#10+
              'L'#13#10+
              'T, B'#13#10+
              'F'#13#10+
              ''#13#10+
              'I'#13#10+
              'U');
      end;  // if g_bDrawHelp

    g_pD3DDevice.EndScene;
  end;  // if Succeeded(BeginScene)


  // Update frame
  Result := g_pD3DX.UpdateFrame(0);

  if (Result = DDERR_SURFACELOST) or (Result = DDERR_SURFACEBUSY) then
    Result := HandleWindowedModeChanges;

  if g_bActive then FrameTimer.Frame;
end;


function TD3DXDemoForm.ReleaseContext: HResult;
var x: TTexIndices;
begin
  g_bReady := FALSE;
  Torus := nil; Box := nil;

  for x := TEX_GROUND to TexIndexMax do
     g_ppTex[x] := nil;

   m_pvbVertices := nil;
   g_pD3DDevice := nil;
   g_pD3D := nil;
   g_pDD := nil;
   g_pD3DX := nil;
   Result := D3DXUninitialize;

   if Failed(Result) then ShowMessage('Release: '+D3DXGetErrorMsg(Result));
end;

function TD3DXDemoForm.ReleaseScene: HResult;
begin
  if m_pwIndices <> nil then FreeMem(m_pwIndices);
  m_pwIndices := nil;
  g_pLensFlare.Free; g_pLensFlare := nil;
  Result := S_OK;
end;

function TD3DXDemoForm.RestoreContext: HResult;
begin
  Result := g_pD3DX.RestoreSurfaces;
  if Failed(Result) then Exit;
  Result := CreateGroundSurface;   // recreate
end;

function TD3DXDemoForm.Shutdown: HResult;
begin
  ReleaseContext;
  ReleaseScene;
  Result := S_OK;
end;



procedure TD3DXDemoForm.FormDestroy(Sender: TObject);
begin
  Shutdown;
end;

function TD3DXDemoForm.HandleWindowedModeChanges: HResult;
begin
  Result := g_pDD.TestCooperativeLevel;

  if SUCCEEDED(Result) then
  begin
        // This means that mode changes had taken place, surfaces
        // were lost but still we are in the original mode, so we
        // simply restore all surfaces and keep going.
        Result := g_pDD.RestoreAllSurfaces;
        if FAILED(Result) then Exit;
  end else if Result = DDERR_WRONGMODE then
  begin
        // This means that the desktop mode has changed
        // we can destroy and recreate everything back again.
        Result := ReleaseContext; if FAILED(Result) then Exit;
        Result := CreateContext; if FAILED(Result) then Exit;
  end else if Result = DDERR_EXCLUSIVEMODEALREADYSET then
  begin
        // This means that some app took exclusive mode access
        // we need to sit in a loop till we get back to the right mode.
        repeat
           Sleep(500);
           Result := g_pDD.TestCooperativeLevel;
        until Result <> DDERR_EXCLUSIVEMODEALREADYSET;
        if SUCCEEDED(Result) then
        begin
            // This means that the exclusive mode app relinquished its
            // control and we are back to the safe mode, so simply restore
            Result := g_pDD.RestoreAllSurfaces;
            if FAILED(Result) then Exit;
        end else if Result = DDERR_WRONGMODE then
        begin
            // This means that the exclusive mode app relinquished its
            // control BUT we are back to some strange mode, so destroy
            // and recreate.
          Result := ReleaseContext; if FAILED(Result) then Exit;
          Result := CreateContext; if FAILED(Result) then Exit;
        end;
  end;
end;

procedure TD3DXDemoForm.FormResize(Sender: TObject);
begin
  if g_bReady then
  begin
    if FAILED(g_pD3DX.Resize(ClientWidth,ClientHeight)) then
    begin
      g_bReady := False;
      PostQuitMessage(0);
    end;
//    D3DXMatrixPerspectiveFov(g_matProjection, D3DXToRadian(60.0), 3.0 / 4.0, 0.1, 1000.0);
//    g_pD3DDevice.SetTransform(D3DTRANSFORMSTATE_PROJECTION, TD3DMatrix(g_matProjection));
  end;
end;


procedure TD3DXDemoForm.FormPaint(Sender: TObject);
begin
  if g_bReady then Draw;
end;

procedure TD3DXDemoForm.ActivateApp(Sender: TObject);
begin
  g_bActive := True;
end;

procedure TD3DXDemoForm.DeactivateApp(Sender: TObject);
begin
  if g_bActive then
  begin
    g_bActive := False;
    if Assigned(g_pDD) then g_pDD.FlipToGDISurface;
    RedrawWindow(Handle, nil, 0, RDW_FRAME);
  end;
end;

procedure TD3DXDemoForm.FormKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  case Key of
    VK_ESCAPE:
      PostQuitMessage(0);
    VK_F1:
      g_bDrawHelp :=  not g_bDrawHelp;
    VK_F2:
      g_bWireframe := not g_bWireframe;
    Ord('R'):
      g_bDrawFramerate := not g_bDrawFramerate;
    Ord('F'):
      g_bFogging := not g_bFogging;
    Ord('G'):
       g_bDrawGround := not g_bDrawGround;
    Ord('B'):
       g_bDrawBoxes := not g_bDrawBoxes;
    Ord('T'):
       g_bDrawTorii := not g_bDrawTorii;
    Ord('L'):
       g_bDrawLensFlare := not g_bDrawLensFlare;
    Ord('I'):
       g_bOnIdle := not g_bOnIdle;
    Ord('U'):
      g_bUnlimitedUniverse := not g_bUnlimitedUniverse;
  end;
end;


constructor TLensFlare.Create;
begin
  inherited;
  FlareData := TList.Create;
  m_vecLight := D3DXVECTOR4(0.0, -1.0, 0.0, 0.0);
end;

procedure TLensFlare.CleanupData;
var x: Integer; P: PFlareData;
begin
  for x := 0 to FlareData.Count-1 do
  begin
    P := FlareData[x];
    P^.Texture := nil;
    Dispose(P);
  end;
  FlareData.Clear;
end;

destructor TLensFlare.Destroy;
begin
  m_pSourceTexture := nil;
  CleanupData;
  FlareData.Free;
  inherited;
end;

function TLensFlare.Initialize(uFlares: UINT; Distance: Float): HResult;
var x: Integer; P: PFlareData;
begin
  if uFlares <> Cardinal(FlareData.Count) then
  begin
    CleanupData;
    for x := 0 to uFlares-1 do
    begin
      New(P); FillChar(P^,Sizeof(TFlareData),0);
      FlareData.Add(P);
    end;
  end;
  FDistance := Distance;
  Result := S_OK;
end;

procedure TLensFlare.SetLightPosition(Vec: TD3DXVector4);
begin
  D3DXVec4Normalize(m_vecLight, Vec);
end;

procedure TLensFlare.SetLightColor(Color: TD3DXColor);
begin
  m_colorLight := color;
end;


procedure TLensFlare.SetSource(fRadius: Float; const pTexture: IDirectDrawSurface7);
begin
  FSourceRadius := fRadius;
  m_pSourceTexture := pTexture;
end;

procedure TLensFlare.SetFlare(uFlare: UINT; fRadius, fOpacity, fPosition: Float;
       const pTexture: IDirectDrawSurface7);
begin
  with PFlareData(FlareData[uFlare])^ do
  begin
    Radius := fRadius; Opacity := 2*fOpacity; Position := fPosition;
    Texture := pTexture;
  end;
end;

// Manually transform a vertex from local 3D space to 2D screen coordinates.
function TransformVertex(var Vertex: TD3DTLVertex; const D3DDevice: IDirect3DDevice7): HResult;
var vp: TD3DViewport7;
    matWorld, matView, matProj, matSet: TD3DXMatrix;
    fX, fY, fZ, fXp, fYp, fZp, fWp, fWpInv: Float;
begin
  // Get the width and height of the viewport. This is needed to scale the
  // transformed vertices to fit the render window.
  D3DDevice.GetViewport(vp);

  // Get the current matrix set. This is needed for the transformation.
  with D3DDevice do
  begin
    GetTransform(D3DTRANSFORMSTATE_WORLD, TD3DMatrix(matWorld));
    GetTransform(D3DTRANSFORMSTATE_VIEW, TD3DMatrix(matView));
    GetTransform(D3DTRANSFORMSTATE_PROJECTION, TD3DMatrix(matProj));
  end;
  D3DXMatrixMultiply(matSet, matWorld, matView);
  D3DXMatrixMultiply(matSet, matSet, matProj);

  // Get the untransformed vertex position
  with Vertex do
  begin
    fX := sx; fy := sy; fZ := sz;
  end;

  // Transform it through the current matrix set
  with matSet do
  begin
    fXp := m00*fX + m10*fY + m20*fZ + m30;
    fYp := m01*fX + m11*fY + m21*fZ + m31;
    fZp := m02*fX + m12*fY + m22*fZ + m32;
    fWp := m03*fX + m13*fY + m23*fZ + m33;
  end;

   fWpInv := 1.0 / fWp;

  // Finally, scale the vertices to screen coords. This step first
  // "flattens" the coordinates from 3D space to 2D device coordinates,
  // by dividing each coordinate by the wp value. Then, the x- and
  // y-components are transformed from device coords to screen coords.
  // Note 1: device coords range from -1 to +1 in the viewport.
  // Note 2: the sz-coordinate will be used in the z-buffer.
  with Vertex do
  begin
    sx  := (1.0 + (fXp * fWpInv)) * (vp.dwWidth / 2);
    sy  := (1.0 - (fYp * fWpInv)) * (vp.dwHeight / 2);
    sz  := fZp * fWpInv;
    rhw := fWp;

    if (sx < 0.0) or (sx > vp.dwWidth)
      or (sy < 0.0) or (sy > vp.dwHeight)
    then Result := E_FAIL
    else Result := S_OK;
  end;
end;


function TLensFlare.Draw(matPos: TD3DXMatrix; const D3DDevice: IDirect3DDevice7): HResult;
var pSurface, pZBuffer: IDirectDrawSurface7;
    ddsc: TDDSCAPS2;
    bFlares: BOOL; dwLight: Cardinal;
    vecPos, vecDir, vecCenter,
    vecLight, vec, vecAxis, vecDx, vecDy,
    vecX, vecY, vecSx, vecSy: TD3DXVector3;
    fDot: Float;
    D3DTL: TD3DTLVertex;
    ddsd: TDDSURFACEDESC2; cb: UINT;
    pb, pbLim: PByte;

    x: Integer;
    Vertexes: Array[0..5] of TD3DLVertex;

    dwFlareColor: Cardinal; FlareColor: TD3DXColor;
    dwZB: DWord;
begin
  Result := S_OK;
  bFlares := True;
  dwLight := D3DXColorToDWord(m_colorLight);

  //
  // Compute center and axis of flares
  //
  // Calculate our position and direction
  vecPos := D3DXVECTOR3(matPos.m30, matPos.m31, matPos.m32);
  vecDir := D3DXVECTOR3(0.0, 0.0, -1.0);

  D3DXVec3TransformNormal(vecDir, vecDir, matPos);
  D3DXVec3Normalize(vecDir, vecDir);

  // Figure out of light (or flare) might be visible
  vecLight := D3DXVECTOR3(-m_vecLight.x, -m_vecLight.y, -m_vecLight.z);
  D3DXVec3Normalize(vecLight, vecLight);

  fDot := D3DXVec3Dot(vecLight, vecDir);

  if fDot <= 0.00001 then Exit;

  // Calculate the point directly in front of us, on the far clip plane
//  vecCenter := vecDir * m_fDistance + vecPos;
  D3DXVec3Scale(vecCenter, vecDir, FDistance);
  vecCenter := D3DXVec3Add(vecCenter, vecPos);

  // Calculate position of light on the far clip plane
//  vecLight = vecLight * (m_fDistance / fDot) + vecPos;
  D3DXVec3Scale(vecLight, vecLight, (FDistance / fDot));
  vecLight := D3DXVec3Add(vecLight, vecPos);

  // Compute axis which goes from light through the center of the screen
//  vecAxis = vecLight - vecCenter;
  vecAxis := D3DXVec3Subtract(vecLight, vecCenter);

  D3DXVec3Normalize(vecDx, vecAxis);
  D3DXVec3Cross(vecDy, vecDx, vecDir);

  //
  // Figure out if light is behind something else
  //
  Result := D3DDevice.GetRenderTarget(pSurface);
  if Succeeded(Result) then
  begin
    FillChar(ddsc, SizeOf(ddsc), 0);
    ddsc.dwCaps := DDSCAPS_ZBUFFER;
    Result := pSurface.GetAttachedSurface(ddsc, pzBuffer);
    if Succeeded(Result) then
    begin
      d3dTL.sx := vecLight.x;
      d3dTL.sy := vecLight.y;
      d3dTL.sz := vecLight.z;

      Result := TransformVertex(D3DTL, D3DDevice);
      if Succeeded(Result) then
      begin
        ddsd.dwSize := sizeof(ddsd);
        pSurface.GetSurfaceDesc(ddsd);
        cb := ddsd.ddpfPixelFormat.dwZBufferBitDepth shr 3;

        Result := pZBuffer.Lock(nil, ddsd, DDLOCK_READONLY or DDLOCK_WAIT, 0);
        if Succeeded(Result) then
        begin
          pb := PByte(Cardinal(ddsd.lpSurface) + Trunc(d3dTL.sy) * ddsd.lPitch + Trunc(d3dTL.sx) * cb);
          pbLim := PByte(Cardinal(pb) + cb);

          while Cardinal(pb) < Cardinal(pbLim) do
            if pb^ <> $FF then
            begin
              bFlares := False; Break;
            end
              else Inc(pb);

          pzBuffer.Unlock(nil);
        end;
      end;
      pzBuffer := nil;
    end;
    pSurface := nil;
  end;
  //
  // Draw light source
  //

  // Figure out screen X and Y axes in model coordinates
  vecX := D3DXVECTOR3(1.0, 0.0, 0.0);
  D3DXVec3TransformNormal(vecX, vecX, matPos);
  D3DXVec3Normalize(vecX, vecX);
  D3DXVec3Cross(vecY, vecX, vecDir);

  D3DXVec3Scale(vecSx, vecX, FSourceRadius);
  D3DXVec3Scale(vecSy, vecY, FSourceRadius);

  FillChar(Vertexes, Sizeof(Vertexes), 0);
  for x := 0 to 5 do Vertexes[x].color := dwLight;

  Vertexes[0].x := vecLight.x + vecSx.x - vecSy.x;
  Vertexes[0].y := vecLight.y + vecSx.y - vecSy.y;
  Vertexes[0].z := vecLight.z + vecSx.z - vecSy.z;
  Vertexes[0].tu := 1.0; Vertexes[0].tv := 1.0;

  Vertexes[1].x := vecLight.x - vecSx.x - vecSy.x;
  Vertexes[1].y := vecLight.y - vecSx.y - vecSy.y;
  Vertexes[1].z := vecLight.z - vecSx.z - vecSy.z;
  Vertexes[1].tv := 1.0;

  Vertexes[2].x := vecLight.x - vecSx.x + vecSy.x;
  Vertexes[2].y := vecLight.y - vecSx.y + vecSy.y;
  Vertexes[2].z := vecLight.z - vecSx.z + vecSy.z;

  Vertexes[3].x := vecLight.x + vecSx.x - vecSy.x;
  Vertexes[3].y := vecLight.y + vecSx.y - vecSy.y;
  Vertexes[3].z := vecLight.z + vecSx.z - vecSy.z;
  Vertexes[3].tu := 1.0; Vertexes[3].tv := 1.0;

  Vertexes[4].x := vecLight.x - vecSx.x + vecSy.x;
  Vertexes[4].y := vecLight.y - vecSx.y + vecSy.y;
  Vertexes[4].z := vecLight.z - vecSx.z + vecSy.z;

  Vertexes[5].x := vecLight.x + vecSx.x + vecSy.x;
  Vertexes[5].y := vecLight.y + vecSx.y + vecSy.y;
  Vertexes[5].z := vecLight.z + vecSx.z + vecSy.z;
  Vertexes[5].tu := 1.0;

  D3DDevice.SetTexture(0, m_pSourceTexture);
  D3DDevice.DrawPrimitive(D3DPT_TRIANGLELIST, D3DFVF_LVERTEX, Vertexes, 6, D3DDP_WAIT);

  //
  // Disable Z-buffer and draw flares
  //
  if not bFlares or (FlareData.Count = 0) then Exit;

  D3DDevice.GetRenderState(D3DRENDERSTATE_ZENABLE, dwZB);
  D3DDevice.SetRenderState(D3DRENDERSTATE_ZENABLE, Ord(D3DZB_FALSE));

//    Texture: IDirectDrawSurface7;
//    Opacity, Radius, Position: Float;

  for x := 0 to FlareData.Count-1 do
  begin
    with PFlareData(FlareData[x])^ do
    begin
      vec := D3DXVec3Add(D3DXVec3Scale(vec, vecAxis, Position)^, vecCenter);
      D3DXVec3Scale(vecSx, vecDx, Radius);
      D3DXVec3Scale(vecSY, vecDY, Radius);

      dwFlareColor := D3DXColorToDWord(D3DXColorScale(
         FlareColor, m_colorLight, Opacity)^);

      D3DDevice.SetTexture(0, Texture);
    end;

    Vertexes[0].x := vec.x + vecSx.x - vecSy.x;
    Vertexes[0].y := vec.y + vecSx.y - vecSy.y;
    Vertexes[0].z := vec.z + vecSx.z - vecSy.z;
    Vertexes[0].color := dwFlareColor;

    Vertexes[1].x := vec.x - vecSx.x - vecSy.x;
    Vertexes[1].y := vec.y - vecSx.y - vecSy.y;
    Vertexes[1].z := vec.z - vecSx.z - vecSy.z;
    Vertexes[1].color := dwFlareColor;

    Vertexes[2].x := vec.x - vecSx.x + vecSy.x;
    Vertexes[2].y := vec.y - vecSx.y + vecSy.y;
    Vertexes[2].z := vec.z - vecSx.z + vecSy.z;
    Vertexes[2].color := dwFlareColor;

    Vertexes[3].x := vec.x + vecSx.x - vecSy.x;
    Vertexes[3].y := vec.y + vecSx.y - vecSy.y;
    Vertexes[3].z := vec.z + vecSx.z - vecSy.z;
    Vertexes[3].color := dwFlareColor;

    Vertexes[4].x := vec.x - vecSx.x + vecSy.x;
    Vertexes[4].y := vec.y - vecSx.y + vecSy.y;
    Vertexes[4].z := vec.z - vecSx.z + vecSy.z;
    Vertexes[4].color := dwFlareColor;

    Vertexes[5].x := vec.x + vecSx.x + vecSy.x;
    Vertexes[5].y := vec.y + vecSx.y + vecSy.y;
    Vertexes[5].z := vec.z + vecSx.z + vecSy.z;
    Vertexes[5].color := dwFlareColor;


    D3DDevice.DrawPrimitive(D3DPT_TRIANGLELIST, D3DFVF_LVERTEX, Vertexes, 6, D3DDP_WAIT);
  end;

  D3DDevice.SetRenderState(D3DRENDERSTATE_ZENABLE, dwZB);
end;

end.


