/* LW_PALM.c:  common function for PALM programming
 *
 * Arndt Kritzner
 * kritzner@logicway.de
 */

#include <PalmOS.h>
#include "LW_PALM.h"

UInt8 LW_NumError = LW_NO_ERROR;

UInt8 LW_Error( UInt8 e )
{
	UInt8 eOld;
	eOld = LW_NumError;
	LW_NumError = e;
	return( eOld );
}

Int8 LW_DoubleToStr( char* StrVal, const UInt8 maxLen, const double FVal )
{
	double val = FVal;
	Int8 sign = 0;
	Char *String, *strPtr, *roundPtr;
	UInt8 strPos = 0, roundPos;
	Int8 strShift = 1;
	Int8 err = 0;
	Int8 decimalPoint = 0, behinddp;
	double Digit;
	Char CDigit, RDigit, WDigit;
	MemHandle h;
	UInt8 IsZero = 0;
	UInt8 ValidDigits = 0;
	UInt8 hangOver = 1;
	UInt8 noTail = 0;
	h = MemHandleNew( maxLen+1 );
	String = (Char *) MemHandleLock( h );
	StrPrintF( String, "");
	strPtr=String;
	if ( val < 0.0 ) {
		val = -val;
		sign = 1;
		*strPtr='-';
		strPtr++;
		strPos++;
	};
	while ( val >= 10.0 ) {
		val = val / 10.0;
		strShift++;
	};
	while (( ! IsZero ) && (strPos<maxLen) && (! err)) {
		if (( ! strShift ) && ( ! decimalPoint )) {
			*strPtr='.';
			strPos++;
			strPtr++;
			decimalPoint = 1;
		} else {
			for ( Digit=9.0, CDigit='9'; Digit > val; Digit-=1.0, CDigit-- );
			if ( ValidDigits || ( CDigit > '0' ) )
				ValidDigits++;
			if ( ValidDigits > MAX_VALID_DIGITS ) {
				if ( CDigit > '5' ) {
					roundPtr = strPtr;
					roundPos = strPos;
					behinddp = decimalPoint;
					do {
						roundPtr--;
						roundPos--;
						if ( *roundPtr == '9' ) {
							*roundPtr = '0';
							if ( behinddp ) {
								strPtr--;
								strPos--;
								noTail = 1;
							};
						} else
							if ( *roundPtr == '.' ) {
								behinddp = 0;
								strPtr--;
								strPos--;
								noTail = 1;
							} else
								if ( *roundPtr != '-' ) {
									CDigit = *roundPtr;
									CDigit++;
									*roundPtr = CDigit;
									hangOver = 0;
								};
					} while ( hangOver && ( roundPos > 0 ) );
					if ( hangOver ) {
						while ( roundPos < strPos ) {
							if ( hangOver ) {
								if (*roundPtr != '-') {
									WDigit = *roundPtr;
									*roundPtr = '1';
									hangOver = 0;
								};
							} else {
								RDigit = *roundPtr;
								*roundPtr = WDigit;
								WDigit = RDigit;
							};
							roundPtr++;
							roundPos++;
						};
						if ( ! hangOver ) {
							strPtr++;
							strPos++;
						};
					};
				} else {
					roundPtr = strPtr;
					roundPos = strPos;
					behinddp = decimalPoint;
					hangOver = 1;
					do {
						roundPtr--;
						roundPos--;
						if ( *roundPtr == '0' ) {
							if ( behinddp ) {
								strPtr--;
								strPos--;
								noTail = 1;
							};
						} else
							if ( *roundPtr == '.' ) {
								behinddp = 0;
								strPtr--;
								strPos--;
								noTail = 1;
								hangOver = 0;
							} else
								hangOver = 0;
					} while ( hangOver && ( roundPos > 0 ) );
				};
				CDigit = '0';
				val = 0.0;
			} else {
				val = val - Digit;
				val = val * 10.0;
			};
			if ( ! noTail ) {
				*strPtr = CDigit; strPos++; strPtr++;
			};
			if ( strShift ) strShift--;
		};
		if ( ( val == 0.0 ) && ( strShift == 0 ) )
			IsZero = 1;
	};
	*strPtr='\0';
	if ( err ) {
		LW_Error( LW_ERR_DToStr );
		StrPrintF(StrVal,"Error");
	} else
		StrPrintF(StrVal,"%s",String);
	MemHandleUnlock( h );
	MemHandleFree( h );
	return( sign );
}

Int8 LW_StrToDouble( char* StrVal, double* FVal )
{
	double val = 0.0;
	double Factor = 1.0;
	Char *strPtr;
	Int8 err = 0;
	double Digit;
	Char CDigit;
	strPtr = StrVal;
	while ( (*strPtr != '\0') && (! err) ) {
		if (*strPtr == '.') {
			Factor = 0.1;
			strPtr++;
		} else {
			for ( Digit=0.0, CDigit = '0'; (CDigit != *strPtr) && (CDigit != '9'); Digit+=1, CDigit++ );
			if (CDigit == *strPtr) {
				if ( Factor == 1.0 )
					val = val * 10.0 + Digit;
				else {
					val = val + Factor * Digit;
					Factor = Factor * 0.1;
				};
				strPtr++;
			} else
				err = 1;
		};
	};
	if ( ! err )
		*FVal = val;
	else
		LW_Error( LW_ERR_StrToD );
	return ( err );
}

Int16 LW_DoubleToInt( const double FVal )
{
	double val = FVal;
	Int8 strShift = 0;
	double Digit;
	Int8 IDigit;
	Int8 sign = 0;
	Int16 IVal = 0;
	if ( val < 0.0 ) {
		val = -val;
		sign = 1;
	};
	val += 0.5;
	while ( val >= 10.0 ) {
		val = val / 10.0;
		strShift++;
	};
	while ( strShift >= 0 ) {
		for ( Digit=9.0, IDigit=9; Digit > val; Digit-=1.0, IDigit-- );
		val = val - Digit;
		val = val * 10.0;
		IVal *= 10;
		IVal += IDigit;
		strShift--;
	};
	if ( sign ) IVal = -IVal;
	return( IVal );
}

double LW_IntToDouble( Int16 IVal )
{
	double val = 0.0;
	Int16 v = IVal, v_neu;
	Int8 Digit;
	Int8 sign = 0;
	double Faktor = 1.0;
	double DDigit;
	if ( v < 0 ) {
		v = -v;
		sign = 1; 
	};
	while ( v > 0 ) {
		v_neu = v / 10;
		for ( Digit = 9, DDigit=9.0; 10 * v_neu + Digit > v; Digit--, DDigit -= 1.0 );
		val = val + Faktor * DDigit;
		Faktor *= 10.0;
		v = v_neu;
	};
	if ( sign )
		val = -val;
	return ( val );
}

double LW_abs( const double x )
{
	if ( x < 0.0 )
		return( -x );
	else
		return( x );
}

double LW_sqrt( const double x )
{
	double x_ = 0.0;
	double prev_x_;
	double diff = 0.0;
	UInt16 Loop = 0;
	if ( x > 0.0 ) {
		x_ = x;
		do {
			prev_x_ = x_;
			x_ = ( x_ + (x/x_)) / 2.0;
			diff = LW_abs( prev_x_ - x_ );
			Loop++;
		} while ( ( diff > LW_ITERATION_ACCURACY ) && ( Loop < LW_ITERATION_LOOPS ) );
		return(x_);
	} else {
		if ( x < 0.0 )
			LW_Error( LW_ERR_SQRT );
		return(x);
	};
}

double LW_sin( const double x )
{
	double x_ = x;
	double x__;
	double prev_x_;
	double fac;
	double facbase;
	double power;
	double sign;
	double diff;
	UInt16 Loop = 0;
	while ( x_ > LW_PI ) {
		x_ -= 2.0 * LW_PI;
	};
	while ( x_ < -LW_PI ) {
		x_ += 2.0 * LW_PI;
	};
	x__ = x_;
	fac = 1.0;
	sign = 1.0;
	facbase = 1.0;
	power = x_;
	do {
		prev_x_ = x__;
		fac *= (facbase+1.0) * (facbase+2.0);
		facbase += 2.0;
		sign = -sign;
		power = power*x_*x_;
		x__ += sign*power/fac;
		diff = LW_abs( prev_x_-x__ );
		Loop++;
	} while ( ( diff > LW_ITERATION_ACCURACY ) && ( Loop < LW_ITERATION_LOOPS ) );
	return(x__);
}

double LW_cos( const double x )
{
	return( LW_sin(x+LW_PI/2.0) );
}

double LW_arcsin( const double x )
{
	double x_ = x;
	double li, ri, mi;
	UInt16 Loop = 0;
	if ( x_ >= 1.0 ) {
		x_ = LW_PI / 2.0;
		if ( x_ > 1.0 )
			LW_Error( LW_ERR_ArcSin );
	} else
		if ( x_ <= ( -1 ) ) {
			x_ = ( -LW_PI / 2.0 );
			if ( x_ < -1.0 )
				LW_Error( LW_ERR_ArcSin );
		} else {
			li = - ( LW_PI / 2.0 );
			ri = LW_PI / 2.0;
			do {
				mi = ( li + ri ) / 2.0;
				if ( LW_sin( mi ) > x )
					ri = mi;
				else
					li = mi;
				Loop++;
			} while ( (( ri - li ) > LW_ITERATION_ACCURACY) && ( Loop < LW_ITERATION_LOOPS ) );
			x_ = mi;
		};
	return( x_ );
}

double LW_arccos( const double x )
{
	return( - (LW_arcsin( x ) - LW_PI / 2.0 ) );
}
