// to compile with GNU C++:
// g++ -Wall -O3 benchmark.cpp

/*
Title: ctbench
Description: a small benchmark for C#, Java, and C++
Copyright (c) 2002
Company: c't magazine, www.heise.de/ct
Authors: Jrn Loviscach, Ulrich Breymann
Version: 1.0
*/

#include <limits.h>
#if (SHRT_MAX != 32767) || (INT_MAX != 2147483647)
#error Required: short is two bytes long, int is four bytes long.
#endif

// workarounds to adapt Microsoft (TM) Visual C++ (R) to ISO C++
// or use compiler option /Za
#ifdef _MSC_VER
// see Knowledge Base Q167748
#define for if(0);else for
// see Knowledge Base Q167733
#include <new>
#include <new.h>
int my_new_handler(size_t) {
	throw std::bad_alloc();
	return 0; // never reached
}
#endif



// ******** integrated header to facilitate distribution *******

#include <iostream>
#include <sys/timeb.h>
#include <string>
#include <vector>
#include <list>
#include <algorithm>



// simple stop timer with formatted output
class Stopwatch {
	timeb startTime, stopTime;
	long getMillis();
	public:
	void start();
	void stop();
	void print(std::string name, int num);
};

// unsigned fixed point type of "arbitrary" precision
class LongNumber {
	int length;
	short* array; // using 15 bits in lack of an unsigned type in Java
	void addAndCarry(int i, int s);
	void subAndCarry(int i, int s);
	public:
	LongNumber(int len, short x = 0);
	LongNumber(const LongNumber& a);
	LongNumber& operator= (const LongNumber& a);
	virtual ~LongNumber() { delete[] array; }
	// subtraction: a has to be smaller than the objects itself: unsigned op!
	void sub(const LongNumber& a);
	void half();
	void addProd(const LongNumber& a, const LongNumber& b);
	bool almostEqual(const LongNumber& a) const;
	short get(int i) const { return array[i]; }
	int getLength() const { return length; }
	void swap(LongNumber& a);
};

// calculate sqrt(2) up to "arbitrary" precision
class SqrtTwo {
	static const int size;
	static LongNumber sqrtTwoValue;
	public:
	// calculate sqrt(2) and return number of iterations needed
	static int init();
	// return copy of internal LongNumber object
	static LongNumber getValue();
	private:
	// iteration step for calculating sqrt(2): r = n*(3-n*n/2)/2
	static void iterate(const LongNumber& n, LongNumber& r);
};

// pseudo-random number generator taking values from a LongNumber object
class RandomNumberCollection {
	mutable int num;
	LongNumber val;
	public:
	// initialize by a LongNumber
	RandomNumberCollection (const LongNumber& ln);
	// get the next short as random number
	short next(short max) const;
};

// 3D vector with arithmetics and a pre-defined chaos vector field
class Vector3D {
	public:
	double x, y, z;
	Vector3D(double x_, double y_, double z_) : x(x_), y(y_), z(z_) {}
	Vector3D operator+(const Vector3D& v) const;

	// vector field of the Lorenz differential equation
	static Vector3D lorenz(const Vector3D& v) {
		const double a = 10.0;
		const double r = 28.0;
		const double b = 8.0/3.0;
		return Vector3D(a*(v.y-v.x), r*v.x-v.y-v.x*v.z, v.x*v.y-b*v.z);
	}
};

Vector3D operator*(double c, const Vector3D& v);

const LongNumber benchSqrtTwo(); // call and bench the initializer of the square root class
void benchVector(const RandomNumberCollection& r); // search in a vector-like container
void benchList(const RandomNumberCollection& r); // search in a list-like container
void benchBlur(const RandomNumberCollection& r); // Gaussian blur of a random grayscale picture
void benchRungeKutta(); // solve the Lorenz differential equation by Runge-Kutta integration
void benchMemorySimple(const RandomNumberCollection& r); // allocate and free memory, no crisscross pointers involved



// ******** implementation *******

using namespace std;

int main() {
	#ifdef _MSC_VER
	_set_new_handler(my_new_handler); // for MSVC, see above

	#endif

	cout << "Please wait ..." << endl;
	// use the bits of sqrt(2) as pseudo-random numbers
	RandomNumberCollection r = RandomNumberCollection(benchSqrtTwo());
	benchVector(r);
	benchList(r);
	benchBlur(r);
	benchRungeKutta();
	benchMemorySimple(r);

	return 0;
}

const LongNumber benchSqrtTwo() {
	Stopwatch s;
	s.start();
	int i = SqrtTwo::init();
	s.stop();
	s.print("Square Root", i);
	return SqrtTwo::getValue();
}

void benchVector(const RandomNumberCollection& r) {
	const short count = 20000;
	const short size = 10000;
	vector<int> c;
	for(int i=0; i < size; ++i) {
		c.push_back(i);
	}
	int seeked;
	vector<int>::iterator pos;
	Stopwatch s;
	s.start();
	for(int j = 0; j < count; ++j) {
		seeked =r.next(size);
		pos = find(c.begin(), c.end(), seeked);
	}
	s.stop();
	s.print("Vector", (count*size)/2); // (count*size)/2 is an estimate
}

void benchList(const RandomNumberCollection& r) {
	const short count = 5000;
	const short size = 10000;
	list<int> c;
	for(int i=0; i < size; ++i) {
		c.push_back(i);
	}
	int seeked;
	list<int>::iterator pos;
	Stopwatch s;
	s.start();
	for(int j = 0; j < count; ++j) {
		seeked = r.next(size);
		pos = find(c.begin(), c.end(), seeked);
	}
	s.stop();
	s.print("List", (count*size)/2); // (count*size)/2 is an estimate
}

void benchBlur(const RandomNumberCollection& r) {
	const short size = 1000;
	const short num	= 5;
	Stopwatch s;
	// size is constant here, so one could write unsigned char (*arr1)[size] = new ...,
	// but in real-world apps, size won't be constant.
	unsigned char* arr1 = new unsigned char[size*size];
	unsigned char* arr2 = new unsigned char[size*size];

	for(int i = 0; i < size; ++i) {
		for(int j = 0; j < size; ++j) {
			arr1[i*size+j] = (unsigned char) r.next(128); // Java has no unsigned byte type, so use only 0...127
		}
	}
	s.start();
	for(int n = 0; n < num; n++) { // num rounds of blurring
		for (int i = 3; i < size-3; ++i) { // vertical blur arr1 -> arr2
			for(int j = 0; j < size; ++j) {
				arr2[i*size+j] = (arr1[(i-3)*size+j] + arr1[(i+3)*size+j]
					+ 6*(arr1[(i-2)*size+j]+arr1[(i+2)*size+j])
					+ 15*(arr1[(i-1)*size+j]+arr1[(i+1)*size+j])
					+ 20*arr1[i*size+j] + 32)>>6;
			}
		}

		for(int j = 3; j < size-3; ++j) { // horizontal blur arr1 -> arr2
			for(int i = 0; i < size; ++i) {
				arr1[i*size+j] = (arr2[i*size+j-3] + arr2[i*size+j+3]
					+ 6*(arr2[i*size+j-2]+arr2[i*size+j+2])
					+ 15*(arr2[i*size+j-1]+arr2[i*size+j+1])
					+ 20*arr2[i*size+j] + 32)>>6;
			}
		}
	}
	s.stop();
	s.print("Blur", num*(size-6)*(size-6));

	delete[] arr2;
	delete[] arr1;
}

void benchRungeKutta() {
	Stopwatch s;
	const double dt = 1e-4;
	double t = 0.0;
	Vector3D v(2.0, 2.0, 1.0);

	s.start();
	while(t < 50.0 + 0.5*dt){ // 0.5*dt to compensate for possible roundoff errors
		Vector3D dv1 = dt*Vector3D::lorenz(v);
		t += 0.5*dt;
		Vector3D dv2 = dt*Vector3D::lorenz(v + 0.5*dv1);
		Vector3D dv3 = dt*Vector3D::lorenz(v + 0.5*dv2);
		t += 0.5*dt;
		Vector3D dv4 = dt*Vector3D::lorenz(v + dv3);
		v = v + (1.0/6.0)*dv1 + (1.0/3.0)*dv2 + (1.0/3.0)*dv3 + (1.0/6.0)*dv4;
	}
	s.stop();
	s.print("Runge-Kutta", (int)(50.0/dt + 0.5));
}

void benchMemorySimple(const RandomNumberCollection& r) {
	Stopwatch s;
	const short maxSize = 2000;
	const short minSize = 1;
	const short num = 2000;
	int** arr = new int*[num];
	const short count = 20000;

	for(int i = 0; i < num; ++i) {
		arr[i] = new int[minSize + r.next(maxSize - minSize + 1)];
	}

	s.start();
	for(int i = 0; i < count; ++i) {
		int n = r.next(num);
		delete[] arr[n];
		int size = minSize + r.next(maxSize - minSize + 1);
		arr[n] = new int[size];
		memset(arr[n], 0, size*sizeof(int));
		// We use memset because Java and C# deliver only initialized memory.
		// Of course, this means putting the brake on
		// since initialization by zero is not often needed.

		// for analogy to Java and C# versions:
		arr[n][size-1] = 42;
	}
	s.stop();
	s.print("Memory", count);
	for(int i = 0; i < num; ++i) {
		delete[] arr[i];
	}
	delete[] arr;
}

long Stopwatch::getMillis() {
	return (long)(1000*(stopTime.time - startTime.time)
		+ stopTime.millitm - startTime.millitm);
}

void Stopwatch::start() { ftime(&startTime); }

void Stopwatch::stop() { ftime(&stopTime); }
		
void Stopwatch::print(std::string name, int num) {
	string tUnit = "ns";
	double tPerIteration = (1e9/1000.0/num)*getMillis();
	if (tPerIteration > 1e8) {
		tPerIteration /= 1e6;
		tUnit = "ms";
	} else if (tPerIteration > 1e5) {
		tPerIteration /= 1e3;
		tUnit = "s";
	}

	cout << name << ": "
		<< (int)(0.5 + (1000.0/1000.0)*getMillis())
		<< " ms for " << num << " iterations; "
		<< (int)(0.5 + tPerIteration)
		<< " " << tUnit << " per iteration"
		<< endl;
}

const int SqrtTwo::size = 10000;
LongNumber SqrtTwo::sqrtTwoValue = LongNumber(size);

// calculate sqrt(2) and return number of iterations needed
int SqrtTwo::init() {
	int i = 1;
	LongNumber n(size, 1);
	iterate(n, sqrtTwoValue);
	while(! n.almostEqual(sqrtTwoValue) ) {
		n.swap(sqrtTwoValue);
		iterate(n, sqrtTwoValue);
		++i;
	}
	return i;
}

// return copy of internal LongNumber object
LongNumber SqrtTwo::getValue() {
	return LongNumber(sqrtTwoValue);
}

// iteration step for calculating sqrt(2): r = n*(3-n*n/2)/2
inline void SqrtTwo::iterate(const LongNumber& n, LongNumber& r) {
	LongNumber x(size, 3);
	LongNumber y(size);
	y.addProd(n, n);
	y.half();
	x.sub(y);
	x.half();
	LongNumber z(size);
	z.addProd(n, x);
	z.swap(r);
}

// initialize by a LongNumber
RandomNumberCollection::RandomNumberCollection(const LongNumber& ln)
: num(0), val(ln) // use a copy so that later changes to ln don't matter
{}

// get the next short as random number
inline short RandomNumberCollection::next(short max) const {
	++num;
	if(num >= val.getLength()) {
		num = 0;
	}
	return (short)((max*(int)val.get(num)) >> 15);
}

LongNumber::LongNumber(int len, short x)
: length(len), array(new short[length])
{
	memset(array, 0, length*sizeof(short));
	array[0] = x;
}

LongNumber::LongNumber(const LongNumber& a)
: length(a.length), array(new short[length])
{
	memcpy(array, a.array, length*sizeof(short));
}

LongNumber& LongNumber::operator=(const LongNumber& a) {
	if(this != &a) { // needn't copy into itself
		short* temp = new short[a.length];
		memcpy(temp, a.array, a.length*sizeof(short));
		delete[] array; // no 'new' exception has been thrown, so it's ok to destroy old data
		length = a.length;
		array = temp;
	}
	return *this;
}

inline void LongNumber::addAndCarry(int i, int s) {
	int s1 = array[i] + s; // array may contain 0x7FFF, s may contain 0x3FFFF001
	if(s1 <= 0x7FFF) { // only 15 bits because Java doesn't do unsigned types
		array[i] = (short) s1;
	} else {
		int twoShort = array[i-1]*0x8000 + s1;
		if(twoShort >= 0x8000*0x8000) { // need to carry over more than 2*15 bits
			twoShort -= 0x8000*0x8000; // enough in any case, because s1 is smaller than this number
			int j = i-2;
			while(array[j] == 0x7FFF) {
				array[j] = 0;
				--j;
			}
			++array[j];
		}
		array[i] = (twoShort&0x7FFF);
		array[i-1] = (twoShort>>15);
	}
}

inline void LongNumber::subAndCarry(int i, int s) { // note: s may be filled up to 31 bits
	if(array[i] >= s) {
		array[i] = (short)(array[i]-s);
	} else {
		int twoShort = array[i-1]*0x8000 + array[i] - s;
		if(twoShort <= 0) { // need to carry over more than 2*15 bits
			twoShort += 0x8000*0x8000; // enough in any case, because s is smaller than this number
			int j = i-2;
			while(array[j] == 0) {
				array[j] = 0x7FFF;
				--j;
			}
			--array[j];
		}
		array[i] = (short)(twoShort&0x7FFF);
		array[i-1] = (short)(twoShort>>15);
	}
}

inline void LongNumber::sub(const LongNumber& a) {
	int len = length;
	if(a.length < len) {
		len = a.length;
	}
	for(int i = len-1; i >= 0; --i) {
		subAndCarry(i, a.array[i]);
	}
}

inline void LongNumber::half() {
	array[length-1]>>=1;
	for(int i = length-2; i >= 0; --i) {
		if( (array[i] & 0x1) == 1 ) {
			array[i+1] += 0x4000; // this bit has been cleared in the step before
		}
	array[i]>>=1;
	}
}

void LongNumber::addProd(const LongNumber& a, const LongNumber& b) {
	int len = length;
	if(a.length < len) {
		len = a.length;
	}
	if(b.length < len) {
		len = b.length;
	}
	for(int i = len-1; i >= 0; --i) {
		for(int j = len-1-i; j >= 0; --j) {
			addAndCarry(i+j, a.array[i] * b.array[j]);
		}
	}
}

bool LongNumber::almostEqual(const LongNumber& a) const {
	int len = length;
	if(a.length < len) {
		len = a.length;
	}
	int carry = 0;
	for(int i = 0; i < len-2; ++i) { // skip the last two
		if(carry == 0) {
			if(array[i] != a.array[i]) {
				carry = array[i]-a.array[i];
				if( (carry != 1) && (carry != -1) ) return false;
			}
		} else if(carry > 0) { // carry == +1
			if(array[i] != 0) return false;
			if(a.array[i] != 0x7FFF) return false;
		} else { // carry == -1
			if(a.array[i] != 0) return false;
			if(array[i] != 0x7FFF) return false;
		}
	}
	return true;
}

inline void LongNumber::swap(LongNumber& a){
	short* temp = array;
	array = a.array;
	a.array = temp;
	int itemp = length;
	length = a.length;
	a.length = itemp;
}

inline Vector3D Vector3D::operator+(const Vector3D& v) const {
	return Vector3D(x+v.x, y+v.y, z+v.z);
}

inline Vector3D operator*(double c, const Vector3D& v) {
	return Vector3D(c*v.x, c*v.y, c*v.z);
}
