// to compile: csc /o /w:4 benchmark.cs

using System;

namespace ct
{
	namespace bench
	{
		/// <summary>
		/// Title: ctbench
		/// Description: a small benchmark for C#, Java, and C++
		/// Copyright: Copyright (c) 2002
		/// Company: c't magazine, www.heise.de/ct
		/// Authors: Jrn Loviscach, Ulrich Breymann 
		/// Version 1.0
		/// </summary>
		class Benchmark
		{
			static void Main()
			{
				Console.WriteLine("Please wait ...");
				// use the bits of sqrt(2) as pseudo-random numbers
				RandomNumberCollection r = new RandomNumberCollection(BenchSqrtTwo());
				GC.Collect();
				BenchVector(r);
				GC.Collect();
				BenchList(r);
				GC.Collect();
				BenchBlur(r);
				GC.Collect();
				BenchRungeKutta();
				GC.Collect();
				BenchMemorySimple(r);
			}

			/// <summary>
			/// call and bench the initializer of the square root class
			/// </summary>
			private static LongNumber BenchSqrtTwo() 
			{
				Stopwatch s = new Stopwatch();
				s.Start();
				int i = SqrtTwo.Init();
				s.Stop();
				s.Print("Square Root", i);
				return SqrtTwo.GetValue();
			}

			/// <summary>
			///  search in a vector-like container
			/// </summary>
			private static void BenchVector(RandomNumberCollection r) 
			{
				const int count = 20000;
				const int size = 10000;
				System.Collections.ArrayList c = new System.Collections.ArrayList();
				for(int i=0; i < size; i++)
					c.Add(i);
				int seeked;
				int pos;
				Stopwatch s = new Stopwatch();
				s.Start();
				for(int j = 0; j < count; j++) 
				{
					seeked = r.Next(size);
					pos = c.IndexOf(seeked);
				}
				s.Stop();
				s.Print("Vector", (count*size)/2); // (count*size)/2 is an estimate
			}

			/// <summary>
			/// search in a list-like container
			/// </summary>
			private static void BenchList(RandomNumberCollection r) 
			{
				const int count = 5000;
				const int size = 10000;
				System.Collections.Queue c = new System.Collections.Queue();
				for(int i = 0; i < size; i++)
					c.Enqueue(i);
				int seeked;
				Stopwatch s = new Stopwatch();
				s.Start();
				for(int j = 0; j < count; j++) 
				{
					seeked = r.Next(size);
					foreach(int x in c)
					{
						if(x == seeked) break;
					}
				}
				s.Stop();
				s.Print("List", (count*size)/2); // (count*size)/2 is an estimate
			}

			/// <summary>
			/// Gaussian blur of a random grayscale picture
			/// </summary>
			private static void BenchBlur(RandomNumberCollection r) 
			{
				const int size = 1000;
				const int num  = 5;
				Stopwatch s = new Stopwatch();
				byte[,] arr1 = new byte[size, size];
				byte[,] arr2 = new byte[size, size];

				for(int i = 0; i < size; i++)
					for(int j = 0; j < size; j++)
						arr1[i, j] = (byte) r.Next((short)128); // Java has no unsigned byte type, so use only 0...127
				s.Start();
				for(int n = 0; n < num; n++) 
				{ // num rounds of blurring
					for(int i = 3; i < size-3; i++) // vertical blur arr1 -> arr2
						for(int j = 0; j < size; j++)
							arr2[i, j] = (byte)((arr1[i-3, j] + arr1[i+3, j]
								+ 6*(arr1[i-2, j]+arr1[i+2, j])
								+ 15*(arr1[i-1, j]+arr1[i+1, j])
								+ 20*arr1[i, j] + 32)>>6);

					for(int j = 3; j < size-3; j++) // horizontal blur arr1 -> arr2
						for(int i = 0; i < size; i++)
							arr1[i, j] = (byte)((arr2[i, j-3] + arr2[i, j+3]
								+ 6*(arr2[i, j-2]+arr2[i, j+2])
								+ 15*(arr2[i, j-1]+arr2[i, j+1])
								+ 20*arr2[i, j] + 32)>>6);
				}
				s.Stop();
				s.Print("Blur", num*(size-6)*(size-6));
			}

			/// <summary>
			/// solve the Lorenz differential equation by Runge-Kutta integration
			/// </summary>
			private static void BenchRungeKutta() 
			{
				Stopwatch s = new Stopwatch();
				const double dt = 1e-4;
				double t = 0.0;
				Vector3D v = new Vector3D(2.0, 2.0, 1.0);

				s.Start();
				while(t < 50.0 + 0.5*dt)
				{ // 0.5*dt to compensate for possible roundoff errors
					Vector3D dv1 = dt*Vector3D.Lorenz(v);
					t += 0.5*dt;
					Vector3D dv2 = dt*Vector3D.Lorenz(v + 0.5*dv1);
					Vector3D dv3 = dt*Vector3D.Lorenz(v + 0.5*dv2);
					t += 0.5*dt;
					Vector3D dv4 = dt*Vector3D.Lorenz(v + dv3);
					v = v + (1.0/6.0)*dv1 + (1.0/3.0)*dv2 + (1.0/3.0)*dv3 + (1.0/6.0)*dv4;
				}
				s.Stop();
				s.Print("Runge-Kutta", (int)(50.0/dt + 0.5));
			}

			/// <summary>
			/// allocate and free memory, no crisscross pointers involved
			/// </summary>
			private static void BenchMemorySimple(RandomNumberCollection r)  
			{
				Stopwatch s = new Stopwatch();
				const int maxSize = 2000;
				const int minSize = 1;
				const int num = 2000;
				int[][] arr = new int[num][]; // 32 bit
				const int count = 20000;

				for(int i = 0; i < num; i++) 
				{
					arr[i] = new int[minSize + r.Next((short)(maxSize - minSize + 1))];
				}

				s.Start();
				for(int i = 0; i < count; i++) 
				{
					int n = r.Next(num);
					int size = minSize + r.Next((short)(maxSize - minSize + 1));
					arr[n] = new int[size];
					arr[n][size-1] = 42; // only to make sure that memory is being used
				}
				s.Stop();
				s.Print("Memory", count);
			}

			/// <summary>
			/// simple stop timer with formatted output
			/// </summary>
			private class Stopwatch 
			{
				DateTime startTime, stopTime;
				public void Start() { startTime = DateTime.Now; }
				public void Stop() { stopTime = DateTime.Now; }
				public void Print(String name, int num) 
				{
					string tUnit = "ns";
					double tPerIteration = (1e9/1000.0/num)*(stopTime-startTime).TotalMilliseconds;
					if (tPerIteration > 1e8)
					{
						tPerIteration /= 1e6;
						tUnit = "ms";
					}
					else if (tPerIteration > 1e5) 
					{
						tPerIteration /= 1e3;
						tUnit = "s";
					}

					Console.WriteLine(
						name + ": "
						+ (int)(0.5 + (1000.0/1000.0)*(stopTime-startTime).TotalMilliseconds)
						+ " ms for " + num + " iterations; "
						+ (int)(0.5 + tPerIteration)
						+ " " + tUnit + " per iteration"
						);
				}
			}

			/// <summary>
			/// calculate sqrt(2) up to "arbitrary" precision
			/// </summary>
			private class SqrtTwo 
			{
				private const int size = 10000;
				private static LongNumber sqrtTwoValue = new LongNumber(size);
				/// <summary>
				/// calculate sqrt(2)
				/// </summary>
				/// <returns>number of iterations needed</returns>
				public static int Init() 
				{
					int i = 1;
					LongNumber n = new LongNumber(size,(short)1);
					sqrtTwoValue = Iterate(n);
					while(! n.AlmostEqual(sqrtTwoValue) ) 
					{
						n = sqrtTwoValue;
						sqrtTwoValue = Iterate(n);
						i++;
					}
					return i;
				}

				///<returns>copy of internal LongNumber object</returns>
				public static LongNumber GetValue() {
					return new LongNumber(sqrtTwoValue);
				}

				/// <summary>
				/// iteration step for calculating sqrt(2): n = n*(3-n*n/2)/2
				/// </summary>
				private static LongNumber Iterate(LongNumber n) 
				{
					LongNumber x = new LongNumber(size, (short)3);
					LongNumber y = new LongNumber(size);
					y.AddProd(n, n);
					y.Half();
					x.Sub(y);
					x.Half();
					LongNumber z = new LongNumber(size);
					z.AddProd(n, x);
					return z;
				}
			}

			/// <summary>
			/// pseudo-random number generator taking values from a LongNumber object
			/// </summary>
			private class RandomNumberCollection {
				private LongNumber val;
				private int num = 0;
				/// <summary>
				/// initialize by a LongNumber
				/// </summary>
				public RandomNumberCollection (LongNumber ln) {
					val = new LongNumber(ln); // use a copy so that later changes to ln don't matter
				}
				/// <summary>
				/// get the next short as random number
				/// </summary>
				public short Next(short max) {
					num++;
					if(num >= val.GetLength()) {
						num = 0;
					}
					return (short)((max*(int)val.Get(num)) >> 15);
				}
			}

			/// <summary>
			/// unsigned fixed point type of "arbitrary" precision
			/// </summary>
			private class LongNumber 
			{
				private int length;
				private short[] array; // using 15 bits in lack of an unsigned type in Java
				public LongNumber(int len) 
				{
					length = len;
					array = new short[length];
				}
				public LongNumber(int len, short x) 
				{
					length = len;
					array = new short[length];
					array[0] = x;
				}
				public LongNumber(LongNumber ln) {
					length = ln.length;
					array = new short[length];
					for (int i=0; i<length; i++) {
						array[i] = ln.array[i];
					}
				}
				private void AddAndCarry(int i, int s) 
				{
					int s1 = array[i] + s; // array may contain 0x7FFF, s may contain 0x3FFFF001
					if(s1 <= 0x7FFF) 
					{ // only 15 bits because Java doesn't do unsigned types
						array[i] = (short) s1;
					} 
					else 
					{
						int twoShort = array[i-1]*0x8000 + s1;
						if(twoShort >= 0x8000*0x8000) 
						{ // need to carry over more than 2*15 bits
							twoShort -= 0x8000*0x8000; // enough in any case, because s1 is smaller than this number
							int j = i-2;
							while(array[j] == 0x7FFF) 
							{ // exception on j = -1 means overflow
								array[j] = 0;
								j--;
							}
							array[j]++;
						}
						array[i] = (short)(twoShort&0x7FFF);
						array[i-1] = (short)(twoShort>>15);
					}
				}
				private void SubAndCarry(int i, int s) 
				{ // note: s may be filled up to 31 bit
					if(array[i] >= s) 
					{
						array[i] = (short)(array[i]-s);
					} 
					else 
					{
						int twoShort = array[i-1]*0x8000 + array[i] - s;
						if(twoShort <= 0) 
						{ // need to carry over more than 2*15 bits
							twoShort += 0x8000*0x8000; // enough in any case, because s is smaller than this number
							int j = i-2;
							while(array[j] == 0) 
							{ // exception on j = -1 means overflow
								array[j] = 0x7FFF;
								j--;
							}
							array[j]--;
						}
						array[i] = (short)(twoShort&0x7FFF);
						array[i-1] = (short)(twoShort>>15);
					}
				}
				/// <summary>
				/// subtraction: a has to be smaller than the objects itself: unsigned op!
				/// </summary>
				public void Sub(LongNumber a) 
				{
					int len = length;
					if(a.length < len) 
					{
						len = a.length;
					}
					for(int i = len-1; i >= 0; i--) 
					{
						SubAndCarry(i, a.array[i]);
					}
				}
				public void Half() 
				{
					array[length-1]>>=1;
					for(int i = length-2; i >= 0; i--) 
					{
						if( (array[i] & 0x1) == 1 ) 
						{
							array[i+1] += 0x4000; // this bit has been cleared in the step before
						}
						array[i]>>=1;
					}
				}
				public void AddProd(LongNumber a, LongNumber b) 
				{
					int len = length;
					if(a.length < len) 
					{
						len = a.length;
					}
					if(b.length < len) 
					{
						len = b.length;
					}
					for(int i = len-1; i >= 0; i--) 
					{
						for(int j = len-1-i; j >= 0; j--) 
						{
							AddAndCarry(i+j, a.array[i] * b.array[j]);
						}
					}
				}
				public bool AlmostEqual(LongNumber a) 
				{
					int len = length;
					if(a.length < len) 
					{
						len = a.length;
					}
					int carry = 0;
					for(int i = 0; i < len-2; i++) // skip the last two
					{
						if(carry == 0)
						{
							if(array[i] != a.array[i]) 
							{
								carry = array[i]-a.array[i];
								if( (carry != 1) && (carry != -1) ) return false;
							}
						}
						else if(carry > 0) // carry == +1
						{
							if(array[i] != 0) return false;
							if(a.array[i] != 0x7FFF) return false;
						}
						else // carry == -1
						{
							if(a.array[i] != 0) return false;
							if(array[i] != 0x7FFF) return false;
						}
					}
					return true;
				}
				public short Get(int i) 
				{
					return array[i];
				}
				public int GetLength() {
					return length;
				}
			}

			/// <summary>
			/// 3D vector with arithmetics and a pre-defined chaos vector field
			/// </summary>
			private struct Vector3D 
			{
				public double x, y, z;

				public Vector3D(double x_, double y_, double z_) 
				{
					x = x_; y = y_; z = z_;
				}

				public static Vector3D operator* (double c, Vector3D v) 
				{
					return new Vector3D(c*v.x, c*v.y, c*v.z);
				}

				public static Vector3D operator+ (Vector3D v, Vector3D u) 
				{
					return new Vector3D(v.x+u.x, v.y+u.y, v.z+u.z);
				}

				/// <summary>
				/// vector field of the Lorenz differential equation
				/// </summary>
				public static Vector3D Lorenz(Vector3D v) 
				{
					const double a = 10.0;
					const double r = 28.0;
					const double b = 8.0/3.0;
					return new Vector3D(a*(v.y-v.x), r*v.x-v.y-v.x*v.z, v.x*v.y-b*v.z);
				}
			}
		}
	}
}