/* $Id: cache-benchmark.c 56 2005-05-04 06:40:14Z olau $ */

#include <ctype.h>
#include <getopt.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <malloc.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <assert.h>
#include <math.h>

#include <openssl/aes.h>
#include <openssl/ssl.h>

#include "benchmark.h"
#include "../core/padlock.h"
#include "../timer/timer.h"

#ifdef WIN32

#include <windows.h>
#include "../padlock-sdk/windows/padlock.h"

#else /* WIN32 */

#ifdef __GNUC__
#include <unistd.h>
#endif

#include "../padlock-sdk/linux/include/padlock.h"

#endif /* WIN32 */

static const char *appname = "padlock";
static const char *appdescription = "VIA PadLock(tm) Cache Benchmark Tool";
static const char *appversion = "$LastChangedRevision: 56 $";
static const char *appauthor = "Oliver Lau <ola@ct.heise.de>";

size_t (*cipher)(const void *ibuf,
                 const void *obuf,
                 size_t cnt,
                 const AES_KEY *key,
                 controlword_t *ctl,
                 const void *iv);
static int repetitions = DEFAULT_REPETITIONS;
static int iterations = DEFAULT_ITERATIONS;
static int keysize = DEFAULT_KEYSIZE;
static int blocksize = DEFAULT_BLOCKSIZE;
static int mode = DEFAULT_MODE;
static char *modestring = NULL;
static char *logfilename = NULL;
static char *inputfilename = NULL;
static bool decrypt = TRUE;
static byte *ibuf;
static byte *obuf;
static byte *tbuf;
static byte *key;
static AES_KEY keyschedule;
static byte *iv;
static controlword_t *ctl;

/* Ein pseudo-zuflliger Vorgabeschlssel */
static byte default_key[DEFAULT_KEYSIZE] = {
  0x71, 0x63, 0xa1, 0x5b,
  0xbf, 0x08, 0x3b, 0x2b,
  0xb4, 0x38, 0xc1, 0x9b,
  0x9a, 0x21, 0x72, 0x80,
  0x45, 0x59, 0x60, 0x33,
  0xf6, 0x2d, 0x74, 0x88,
  0x7f, 0x36, 0xb3, 0x04,
  0xa7, 0x2d, 0xbf, 0x4a,
};

/* Ein pseudo-zuflliger Initialisierungsvektor */
static byte default_iv[DEFAULT_IVSIZE] = {
  0x99, 0xcc, 0xdf, 0x02,
  0x66, 0xe9, 0xb9, 0x44,
  0x68, 0xe7, 0xbf, 0x25,
  0x15, 0xfd, 0xef, 0xaf,
};

static struct option long_options[] = {
  { PARAM_BLOCKSIZE,   required_argument, 0, SELECT_BLOCKSIZE   },
  { PARAM_KEY,         required_argument, 0, SELECT_KEY         },
  { PARAM_KEYSIZE,     required_argument, 0, SELECT_KEYSIZE     },
  { PARAM_IV,          required_argument, 0, SELECT_IV          },
  { PARAM_REPETITIONS, required_argument, 0, SELECT_REPETITIONS },
  { PARAM_ITERATIONS,  required_argument, 0, SELECT_ITERATIONS  },
  { PARAM_MODE,        required_argument, 0, SELECT_MODE        },
  { PARAM_LOG,         required_argument, 0, SELECT_LOG         },
  { 0,                 0,                 0, 0                  }
};


/* Eine ASCII-kodierte Hexadezimalziffer
in einen Byte-Wert wandeln. */
byte hex2char(char hex) {
  if (hex >= 'a' && hex <= 'f') {
    return hex - 'a' + 10;
  }
  else if (hex >= '0' && hex <= '9') {
    return hex - '0';
  }
  return 0;
}


/* Eine ASCII-kodierte Hexadezimalzahl
in ein Byte-Array konvertieren */
void hex2byte_array(char *hex, byte *arr) {
  int i, j;
  int len = (int) strlen(hex);
  for (i = 0, j = 0; i < len; i+=2, j++) {
    arr[j] = (hex2char(hex[i]) << 4) +
      hex2char(hex[i+1]);
  }
}


/* Einen null-terminierten String in Kleinbuchstaben wandeln.
Diese Funktion arbeitet auf einer Kopie des bergebenen Strings. */
char *strtolower(const char *s) {
  char *d = (char *) malloc(strlen(s));
  register int i;
  for (i = 0; s[i] != 0; i++)
    d[i] = tolower(s[i]);
  return d;
}


/* usage() */
void usage(void) {
  printf("\nAufruf: %s <Datei> [<Optionen>]\n"
    "\nOptionen:"
    "\n  --blocksize <size> Groesse einer Verschluesselungseinheit"
    "\n                     (size muss durch %d teilbar sein)"
    "\n  --mode <mode>      Betriebsart: EBC, CBC, CFB, OFB"
    "\n  --repetitions <i>  i Durchlaeufe der Iterationen"
    "\n  --iterations <i>   i-mal das Ver- und Entschluesseln wiederholen"
    "\n  --key <key>        Schluessel key in hexadezimaler Darstellung"
    "\n  --keysize <size>   Schluessellaenge in Bits fuer Standardschluessel"
    "\n  --iv <iv>          Initialisierungsvektor iv"
    "\n                     in hexadezimaler Darstellung"
    "\n  --log Datei        Ergebnisse im CSV-Format in Datei schreiben"
    "\n  -h | -?            diese Hilfe anzeigen"
    "\n  -e                 nur verschluesseln, aber nicht entschluesseln"
    "\n  -s                 per Software verschlusseln, nicht per Hardware"
    "\n"
    "\n",
    appname, AES_BLOCK_SIZE);
}

/* Los gehts ... */
int main(int argc, char *argv[]) {
  struct stat fattr;
  FILE *ifp = NULL;
  FILE *lfp = NULL;
  size_t inputsize;
  LONGLONG time_encr = -1;
  double thruput_encr;
  TIMER stopwatch;
  int i, j, k;
  EVP_CIPHER_CTX ctx;
  const EVP_CIPHER *(*openssl_cipher)(void) = EVP_aes_256_cbc;
  int outlen, tmplen;
  bool software = FALSE;
  size_t buflen;
  size_t bytesread;
  size_t bytesstored;
  size_t rc;
  int option_index;
  int c;

  printf("%s (%s)\n"
    "Copyright (c) 2005 %s\n"
    "Alle Rechte vorbehalten.\n",
    appdescription, appversion, appauthor);

#ifdef DEBUG
  printf("cache line size = %u bytes\n", get_cache_line_size());
#endif

  /* set default key and initialization vector */
#ifdef WIN32
  key = (byte *) VirtualAlloc(NULL, MAX_KEY_LENGTH, MEM_COMMIT, 0);
  iv = (byte *) VirtualAlloc(NULL, IV_LENGTH, MEM_COMMIT, 0);
  ctl = (controlword_t *) VirtualAlloc(NULL, sizeof(controlword_t), MEM_COMMIT, 0);
#else
  key = (byte *) memalign(PADLOCK_BLOCK_BOUNDARY, MAX_KEY_LENGTH);
  iv = (byte *) memalign(PADLOCK_BLOCK_BOUNDARY, IV_LENGTH);
  memcpy(key, default_key, DEFAULT_KEYSIZE);
  memcpy(iv, default_iv, DEFAULT_IVSIZE);
  /* reserve memory for ACE control word */
  ctl = (controlword_t *) memalign(PADLOCK_BLOCK_BOUNDARY, sizeof(controlword_t));
#endif /* WIN32 */


  for (;;) {
    option_index = 0;
    c = getopt_long(argc, argv, "h?es", long_options, &option_index);
    if (c == -1)
      break;
    switch (c) {
    case 'h':
      /* fall-through */
    case '?':
      usage();
      return 0;
    case 'e':
      decrypt = FALSE;
      break;
    case 's':
      software = TRUE;
      break;
    case SELECT_LOG:
      if (optarg) {
        logfilename = optarg;
      }
      break;
    case SELECT_BLOCKSIZE:
      if (optarg) {
        blocksize = atoi(optarg);
        if (blocksize < AES_BLOCK_SIZE) {
          usage();
          return 1;
        }
      }
      break;
    case SELECT_MODE:
      if (optarg) {
        modestring = strtolower(optarg);
        if (strcmp(modestring, MODESTRING_ECB) == 0) {
          mode = MODE_ECB;
          cipher = padlock_xcrypt_ecb;
          openssl_cipher = EVP_aes_256_ecb;
        }
        else if (strcmp(modestring, MODESTRING_CBC) == 0) {
          mode = MODE_CBC;
          cipher = padlock_xcrypt_cbc;
          openssl_cipher = EVP_aes_256_cbc;
        }
        else if (strcmp(modestring, MODESTRING_CFB) == 0) {
          mode = MODE_CFB;
          cipher = padlock_xcrypt_cfb;
          openssl_cipher = EVP_aes_256_cfb;
        }
        else if (strcmp(modestring, MODESTRING_OFB) == 0) {
          mode = MODE_OFB;
          cipher = padlock_xcrypt_ofb;
          openssl_cipher = EVP_aes_256_ofb;
        }
        else {
          usage();
          return 1;
        }
      }
      break;
    case SELECT_REPETITIONS:
      if (optarg) {
        repetitions = atoi(optarg);
      }
      break;
    case SELECT_ITERATIONS:
      if (optarg) {
        iterations = atoi(optarg);
      }
      break;
    case SELECT_KEYSIZE:
      if (optarg) {
        keysize = atoi(optarg) / 8;
        if (keysize < MIN_KEY_LENGTH || keysize > MAX_KEY_LENGTH) {
          usage();
          return 1;
        }
      }
      break;
    case SELECT_KEY:
      if (optarg) {
        hex2byte_array(optarg, key);
        keysize = (int) strlen(optarg) / 2;
        switch(keysize) {
		case 128/8:
		  ctl->ksize  = KEY_LENGTH_128BIT;
		  ctl->rounds = ROUNDS_128BIT_KEY;
		  break;
		case 192/8:
		  ctl->ksize  = KEY_LENGTH_192BIT;
		  ctl->rounds = ROUNDS_192BIT_KEY;
		  break;
		case 256/8:
		  ctl->ksize  = KEY_LENGTH_256BIT;
		  ctl->rounds = ROUNDS_256BIT_KEY;
		  break;
		default:
		  fprintf(stderr, "Fehler: ungueltige Schluessellaenge. Der Schluessel muss eine Laenge von 128, 192 oder 256 Bits haben\n");
		  return 1;
        }
      }
      else {
        usage();
        return 1;
      }
      break;
    case SELECT_IV:
      if (optarg) {
        if (strlen(optarg) >= (MAX_KEY_LENGTH * 2)) {
          usage();
          return 1;
        }
        else {
          hex2byte_array(optarg, iv);
        }
      }
      else {
        usage();
        return 1;
      }
      break;
    default:
      usage();
      return 1;
    }
  }

  if (optind >= argc) {
    usage();
    return 1;
  }
  inputfilename = argv[optind++];

  if (stat(inputfilename, &fattr) != 0) {
    perror("Dateizugriff fehlgeschlagen");
    return 1;
  }
  inputsize = (fattr.st_size + 16) & 0xfffffff0UL; /* align to 16 byte boundary */

  if (!has_cpuid()) {
    fprintf(stderr, "Fehler: CPUID-Instruktion nicht verfuegbar\n");
    return 1;
  }
  if ((padlock_status() & PADLOCK_EXISTS) == 0) {
    fprintf(stderr, "Fehler: PadLock(tm) nicht verfuegbar\n");
    return 1;
  }
  if ((padlock_status() & PADLOCK_ENABLED) == 0) {
    fprintf(stderr, "Fehler: PadLock(tm) verfuegbar, aber nicht eingeschaltet\n");
    return 1;
  }
  buflen = (inputsize + blocksize + AES_BLOCK_SIZE) & 0xfffffff0;
#ifdef WIN32
  if ((ibuf = (char *) VirtualAlloc(NULL, buflen, MEM_COMMIT, 0)) == 0) {
#else
  if ((ibuf = (char *) memalign(PADLOCK_BLOCK_BOUNDARY, buflen)) == 0) {
#endif /* WIN32 */
    perror("nicht genuegend Speicher vorhanden");
    return 1;
  }
  memset(ibuf, 0, buflen);
#ifdef WIN32
  if ((obuf = (char *) VirtualAlloc(NULL, buflen, MEM_COMMIT, 0)) == 0) {
#else
  if ((obuf = (char *) memalign(PADLOCK_BLOCK_BOUNDARY, buflen)) == 0) {
#endif /* WIN32 */
    perror("nicht genuegend Speicher vorhanden");
    return 1;
  }
  memset(obuf, 0, buflen);
#ifdef WIN32
  if ((tbuf = (char *) VirtualAlloc(NULL, blocksize, MEM_COMMIT, 0)) == 0) {
#else
  if ((tbuf = (char *) memalign(PADLOCK_BLOCK_BOUNDARY, blocksize)) == 0) {
#endif /* WIN32 */
    perror("nicht genuegend Speicher vorhanden");
    return 1;
  }
  memset(tbuf, 0, blocksize);

  if (logfilename) {
    lfp = fopen(logfilename, "a+");
    if (!lfp) {
      perror("Kann Logdatei nicht zum Schreiben oeffnen");
      return 1;
    }
  }
  ifp = fopen(inputfilename, "rb");
  if (!ifp) {
    perror("Kann Eingabedatei nicht oeffnen");
    return 1;
  }
  bytesread = 0;
  printf("Lesen der Datei '%s' .. ", inputfilename);
  fflush(stdout);
  while (!feof(ifp)) {
    rc = fread(ibuf, 1, inputsize, ifp);
    bytesread += rc;
  }
  fclose(ifp);
  printf("%ld (%ld) Byte.\n", (long) bytesread, (long) inputsize); 

  OpenSSL_add_all_algorithms();
  EVP_CIPHER_CTX_init(&ctx);

  /***********/
  /* encrypt */
  /***********/
  printf("\nVerschluesseln .. ");
  fflush(stdout);
  ctl->algo   = ALGORITHM_AES;
  ctl->keygen = KEY_LOADED_FROM_MEMORY;
  ctl->interm = NO_EXAMINATION;
  ctl->encdec = ENCRYPT;
  ctl->res0   = 0;
  ctl->res[0] = 0;
  ctl->res[1] = 0;
  ctl->res[2] = 0;
  padlock_reload_key();
  thruput_encr = 1.0;
  for (i = 0; i < repetitions; i++) {
    bytesstored = 0;
    START();
    AES_set_encrypt_key(key, keysize * 8, &keyschedule);
    for (j = 0; j < iterations; j++) {
      for (k = 0; k < (int) inputsize; k += blocksize) {
        if (!software) {
          cipher(ibuf, obuf, blocksize / AES_BLOCK_SIZE, &keyschedule, ctl, iv);
        }
        else {
          EVP_EncryptInit_ex(&ctx, openssl_cipher(), NULL, key, iv);
          EVP_CIPHER_CTX_set_key_length(&ctx, keysize);
          if (!EVP_EncryptUpdate(&ctx, obuf, &outlen, ibuf, blocksize))
            printf("Fehler in EVP_EncryptUpdate()");
          if (!EVP_EncryptFinal_ex(&ctx, obuf + outlen, &tmplen))
            printf("Fehler in EVP_EncryptFinal_ex()");
        }
      }
    }
    STOP(time_encr);
    bytesstored = iterations * inputsize;
    thruput_encr *= (double) bytesstored / (double) time_encr;
    printf("\n%8lu KByte / %7lld ms = %6.1lf MByte/s",
      (unsigned long) bytesstored / 1024,
      time_encr,
      1000.0 * (double) bytesstored / (double) time_encr / 1024 / 1024);
    fflush(stdout);
  }
  thruput_encr = pow(thruput_encr, 1.0 / (double) repetitions);
  if (lfp) {
    fprintf(lfp,
      "%s;%s;%d;%d;%.1lf\n",
      "enc",
      modestring,
      keysize * 8,
      blocksize,
      1000.0 * thruput_encr / 1024 / 1024);
    fclose(lfp);
  }

  printf("\n");
  return 0;
}
