/*
 * Copyright 1998-2004 VIA Technologies, Inc. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * VIA, AND/OR ITS SUPPLIERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

#include "padlock.h"
#include <stdio.h>
#include <stdlib.h>
#include <memory.h>
#include <string.h>


///////////////////////////////////////////////////////////////////////////////////
//    Rijndael AES Cryptography Testing Vestors from 
//    NIST Recommendation for Block Cipher Modes of Operation : Methods and Techniques
//    Appendix F: Example Vectors for Modes of Operation of the AES
//    http://csrc.nist.gov/publications/nistpubs/800-38a/sp800-38a.pdf 
//
//    By  Danial Miao
//    2004-06-18

/***************** ecb mode aes key plaintxt ciphertxt test vector *****************/

static unsigned char 
ecb_aes128_key[16] =    { 
						  0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 
					      0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f 
					    };

static unsigned char 
ecb_aes192_key[24] =    {
						  0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
					      0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
					      0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17 
						 };

static unsigned char 
ecb_aes256_key[32] =    { 
						  0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
  					      0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
					      0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
					      0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f 
						 };

static unsigned char 
ecb_aes_plain[64] =     { 
						  0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
			  		      0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff, 	
					      0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
					      0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff,
					      0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
					      0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff, 	
					      0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
						  0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff 
					    };

static unsigned char 
ecb_aes128_cipher[64] = { 
						  0x69, 0xc4, 0xe0, 0xd8, 0x6a, 0x7b, 0x04, 0x30,
						  0xd8, 0xcd, 0xb7, 0x80, 0x70, 0xb4, 0xc5, 0x5a,
						  0x69, 0xc4, 0xe0, 0xd8, 0x6a, 0x7b, 0x04, 0x30,
						  0xd8, 0xcd, 0xb7, 0x80, 0x70, 0xb4, 0xc5, 0x5a,
						  0x69, 0xc4, 0xe0, 0xd8, 0x6a, 0x7b, 0x04, 0x30,
						  0xd8, 0xcd, 0xb7, 0x80, 0x70, 0xb4, 0xc5, 0x5a,
						  0x69, 0xc4, 0xe0, 0xd8, 0x6a, 0x7b, 0x04, 0x30,
						  0xd8, 0xcd, 0xb7, 0x80, 0x70, 0xb4, 0xc5, 0x5a 
					    };

static unsigned char 
ecb_aes192_cipher[64] = { 
						  0xdd, 0xa9, 0x7c, 0xa4, 0x86, 0x4c, 0xdf, 0xe0,
						  0x6e, 0xaf, 0x70, 0xa0, 0xec, 0x0d, 0x71, 0x91,  
						  0xdd, 0xa9, 0x7c, 0xa4, 0x86, 0x4c, 0xdf, 0xe0,
						  0x6e, 0xaf, 0x70, 0xa0, 0xec, 0x0d, 0x71, 0x91,
						  0xdd, 0xa9, 0x7c, 0xa4, 0x86, 0x4c, 0xdf, 0xe0,
						  0x6e, 0xaf, 0x70, 0xa0, 0xec, 0x0d, 0x71, 0x91,  
						  0xdd, 0xa9, 0x7c, 0xa4, 0x86, 0x4c, 0xdf, 0xe0,
						  0x6e, 0xaf, 0x70, 0xa0, 0xec, 0x0d, 0x71, 0x91 
						};

static unsigned char 
ecb_aes256_cipher[64] = { 
						  0x8e, 0xa2, 0xb7, 0xca, 0x51, 0x67, 0x45, 0xbf,
						  0xea, 0xfc, 0x49, 0x90, 0x4b, 0x49, 0x60, 0x89,  
						  0x8e, 0xa2, 0xb7, 0xca, 0x51, 0x67, 0x45, 0xbf,
						  0xea, 0xfc, 0x49, 0x90, 0x4b, 0x49, 0x60, 0x89,
						  0x8e, 0xa2, 0xb7, 0xca, 0x51, 0x67, 0x45, 0xbf,
						  0xea, 0xfc, 0x49, 0x90, 0x4b, 0x49, 0x60, 0x89,  
						  0x8e, 0xa2, 0xb7, 0xca, 0x51, 0x67, 0x45, 0xbf,
						  0xea, 0xfc, 0x49, 0x90, 0x4b, 0x49, 0x60, 0x89 
					    };

/***************** cbc mode aes key plaintxt ciphertxt test vector *****************/
int cbc_aes128_key[4] =		{ 
								 0x16157e2b, 0xa6d2ae28, 0x8815f7ab, 0x3c4fcf09
							};

int cbc_aes192_key[6] =		{
								 0xf7b0738e, 0x52640eda, 0x2bf310c8, 0xe5799080, 
								 0xd2eaf862, 0x7b6b2c52
							};

int cbc_aes256_key[8] =		{
								 0x10eb3d60, 0xbe71ca15, 0xf0ae732b, 0x81777d85,
								 0x072c351f, 0xd708613b, 0xa310982d, 0xf4df1409
							};

int cbc_aes_iv[4] =			{
								 0x03020100, 0x07060504, 0x0b0a0908, 0x0f0e0d0c
							};
	
int cbc_aes_plain[16] =		{
								 0xe2bec16b, 0x969f402e, 0x117e3de9, 0x2a179373, 
 								 0x578a2dae, 0x9cac031e, 0xac6fb79e, 0x518eaf45, 
			  					 0x461cc830, 0x11e45ca3, 0x19c1fbe5, 0xef520a1a, 
               					 0x45249ff6, 0x179b4fdf, 0x7b412bad, 0x10376ce6
							};

int cbc_aes128_cipher[16] = {
								 0xacab4976, 0x46b21981, 0x9b8ee9ce, 0x7d19e912, 
							   	 0x9bcb8650, 0xee197250, 0x3a11db95, 0xb2787691, 
								 0xb8d6be73, 0x3b74c1e3, 0x9ee61671, 0x16952222, 
							     0xa1caf13f, 0x09ac1f68, 0x30ca0e12, 0xa7e18675
							};

int cbc_aes192_cipher[16] = {	
								 0xb21d024f, 0x3d63bc43, 0x3a187871, 0xe871a09f, 
						   	     0xa9add9b4, 0xf4ed7dad, 0x7638e7e5, 0x5a14693f, 
							     0x20241b57, 0xe07afb12, 0xacbaa97f, 0xe002f13d, 
							     0x79e2b008, 0x81885988, 0xe6a920d9, 0xcd15564f
							};

int cbc_aes256_cipher[16]=  {
								 0x044c8cf5, 0xbaf1e5d6, 0xfbab9e77, 0xd6fb7b5f, 
						   		 0x964efc9c, 0x8d80db7e, 0x7b779f67, 0x7d2c70c6, 
								 0x6933f239, 0xcfbad9a9, 0x63e230a5, 0x61142304, 
								 0xe205ebb2, 0xfce99bc3, 0x07196cda, 0x1b9d6a8c
							};

/***************** 128bits data block cfb mode aes key plaintxt ciphertxt test vector *****************/
//      Because VIA ACE hardware can only support 16 bytes data cryptography
//      in Padlock SDK vesion 1.0.0, only 128bits cfb is supported
//      while 1bit cfb and 8bits cfb are not supported by Padlock SDK version 1.0.0

int cfb128_aes128_key[4] =    {
								 0x16157e2b, 0xa6d2ae28, 0x8815f7ab, 0x3c4fcf09
							  };

int cfb128_aes192_key[6] =    {
								 0xf7b0738e, 0x52640eda, 0x2bf310c8, 0xe5799080,
							     0xd2eaf862, 0x7b6b2c52
							  };

int cfb128_aes256_key[8] =    {
								 0x10eb3d60, 0xbe71ca15, 0xf0ae732b, 0x81777d85, 
							     0x072c351f, 0xd708613b, 0xa310982d, 0xf4df1409
							  };

int cfb128_aes_iv[4] =        {
								 0x03020100, 0x07060504, 0x0b0a0908, 0x0f0e0d0c
							  };
	
int cfb128_aes_plain[16] =    {
								 0xe2bec16b, 0x969f402e, 0x117e3de9, 0x2a179373, 
						   	     0x578a2dae, 0x9cac031e, 0xac6fb79e, 0x518eaf45, 
			  					 0x461cc830, 0x11e45ca3, 0x19c1fbe5, 0xef520a1a, 
               					 0x45249ff6, 0x179b4fdf, 0x7b412bad, 0x10376ce6
							  };

int cfb128_aes128_cipher[16]= {	
								 0x2ed93f3b, 0x20ad2db7, 0xf8493433, 0x4afb3ce8, 
					   	         0x3745a6c8, 0x3fa9b3a0, 0xadcde3cd, 0x8be51c9f, 
								 0x671f7526, 0x40b1cba3, 0xf18c80b1, 0xdff4a487, 
							     0x35054bc0, 0x0e1c5d7c, 0x6fc6c4ea, 0xe6f2f79f
							  };

int cfb128_aes192_cipher[16]= {  
								 0x6f0dc8cd, 0xab8cf1dd, 0x0959c234, 0x74419ac9, 
					   	         0x7f7fce67, 0x21361781, 0x702b1a96, 0x7a3d1d17, 
						         0x1d8a1e2e, 0xb1889bd5, 0xed0fe6c8, 0xc9c4fa1e, 
							     0x9c9f5fc0, 0xa04f83a9, 0xba8fae42, 0xff094b58
							  };

int cfb128_aes256_cipher[16]= {	 
								 0xbf847edc, 0x4b1679da, 0x8684cd7e, 0x60385d98, 
					   	         0x14edff39, 0xc8b1283b, 0x633c1132, 0x7b40e531, 
							     0x241310df, 0x924be515, 0xa8d03ea1, 0xf9e27a26, 
							     0x7485a375, 0xf8ceb91a, 0x3d623120, 0x71e4b155
							  };

/*****************  ofb mode aes key plaintxt ciphertxt test vector *****************/
//      Because VIA ACE hardware can only support 16 bytes data cryptography
//      in Padlock SDK vesion 1.0.0, only 128bits ofb is supported

int ofb_aes128_key[4] =		{
								 0x16157e2b, 0xa6d2ae28, 0x8815f7ab, 0x3c4fcf09
							};

int ofb_aes192_key[6] =		{
								 0xf7b0738e, 0x52640eda, 0x2bf310c8, 0xe5799080,
						 		 0xd2eaf862, 0x7b6b2c52
							};

int ofb_aes256_key[8] =		{
								 0x10eb3d60, 0xbe71ca15, 0xf0ae732b, 0x81777d85, 
								 0x072c351f, 0xd708613b, 0xa310982d, 0xf4df1409
							};

int ofb_aes_iv[4] =			{
								 0x03020100, 0x07060504, 0x0b0a0908, 0x0f0e0d0c
							};
	
int ofb_aes_plain[16] =		{
								 0xe2bec16b, 0x969f402e, 0x117e3de9, 0x2a179373, 
							     0x578a2dae, 0x9cac031e, 0xac6fb79e, 0x518eaf45, 
					  			 0x461cc830, 0x11e45ca3, 0x19c1fbe5, 0xef520a1a,
	                			 0x45249ff6, 0x179b4fdf, 0x7b412bad, 0x10376ce6
							};

int ofb_aes128_cipher[16]=  {
								 0x2ed93f3b, 0x20ad2db7, 0xf8493433, 0x4afb3ce8, 
						   	     0x8d508977, 0x038f9116, 0xda523cf5, 0x25d84ec5, 
								 0x1e054097, 0xf6ec5f9c, 0xa8f74443, 0xcced6022, 
								 0x28654c30, 0x78c759f6, 0xd910a566, 0x5eaed6c1
							};

int ofb_aes192_cipher[16]=  {
								 0x6f0dc8cd, 0xab8cf1dd, 0x0959c234, 0x74419ac9, 
						   	     0x8d8bc2fc, 0x7c83634c, 0x0017e809, 0x010410c1, 
							     0xea9a9a8d, 0x6f59f6c0, 0x4d6d9c55, 0xf2a559af, 
								 0x08209f6d, 0x3e6cca57, 0x4b52ac9c, 0x2ac9acd9
							};

int ofb_aes256_cipher[16]=  {
								0xbf847edc, 0x4b1679da, 0x8684cd7e, 0x60385d98, 
					   			0x67dceb4f, 0x3a0bd240, 0xd86a8fc8, 0x8db04f2a, 
								0xa047ab71, 0xed6ee886, 0x5b1c9df3, 0x08c497ba, 
								0x1d142601, 0xe87bf367, 0x8b5a8f53, 0x84e440e7
							};

static unsigned char scratch[64] = {0,};

static void 
dump_rand(void *buffer, int len)
{
	int i;
	int *buf = (int *)buffer;
	for(i = 0;i < (int)(len/(sizeof(int))); i++)
	{
		if(i%8)
			printf("0x%08x,",buf[i]);
		else
			printf("\n");
	}
	printf("\n");
}

static void 
print_ace_error(AES_RESULT res)
{
	switch (res)
	{
	case AES_ADDRESS_NOT_ALIGNED:   printf("Addresses Not Aligned Error for fast Cryptography!\n");
									break;
	case AES_NOT_BLOCKED:			printf("Invalid Data Size Error! Should be n*16 bytes!\n");
									break;
	case AES_KEY_NOT_SUPPORTED:	    printf("Invalid Key Length Error! Should be 16/24/32 bytes!\n");
									break;
	case AES_MODE_NOT_SUPPORTED:	printf("Invalid Cipher Mode Error! Should be ecb/cbc/cfb/ofb!\n");
									break;
	}
}

static void 
print_star()
{
	int i;

	printf("\n");
	for(i = 0;i < 50; i++)
	{
		printf("*");
	}
	printf("\n");

}

// test VIA Padlock SDK ACE plain aes API
static int
test_padlock_aes(ACE_AES_MODE mode, KEY_LENGTH key_len, int nbytes)
{
	struct ace_aes_context *ctx;
	char cipher_mode[10]= {0,};
	char cipher_key[10] = {0,};
	
	unsigned char *p_key;
	unsigned char *p_plain;
	unsigned char *p_cipher;
	unsigned char temp_iv[16]={0,};
	unsigned char orig_iv[16]={0,};
	
	AES_RESULT res;

	switch (mode)
	{
	case ACE_AES_ECB:		memcpy(cipher_mode,"ECB",3);
							p_plain = (unsigned char *)ecb_aes_plain;
							switch(key_len)
							{
							case KEY_128BITS:		memcpy(cipher_key,"128",3);
													p_key = (unsigned char *)ecb_aes128_key;
													p_cipher = (unsigned char *)ecb_aes128_cipher;
													break;
							case KEY_192BITS:		memcpy(cipher_key,"192",3);
													p_key = (unsigned char *)ecb_aes192_key;
													p_cipher = (unsigned char *)ecb_aes192_cipher;
													break;
							case KEY_256BITS:		memcpy(cipher_key,"256",3);
													p_key = (unsigned char *)ecb_aes256_key;
													p_cipher = (unsigned char *)ecb_aes256_cipher;
													break;
							default:				printf("Invalid Key Length Error!\n");
													return -1;
							}
							break;
	case ACE_AES_CBC:		memcpy(cipher_mode,"CBC",3);
							p_plain = (unsigned char *)cbc_aes_plain;
							memcpy(orig_iv,(unsigned char *)cbc_aes_iv,16);
							switch(key_len)
							{
							case KEY_128BITS:		memcpy(cipher_key,"128",3);
													p_key = (unsigned char *)cbc_aes128_key;
													p_cipher = (unsigned char *)cbc_aes128_cipher;
													break;
							case KEY_192BITS:		memcpy(cipher_key,"192",3);
													p_key = (unsigned char *)cbc_aes192_key;
													p_cipher = (unsigned char *)cbc_aes192_cipher;
													break;
							case KEY_256BITS:		memcpy(cipher_key,"256",3);
													p_key = (unsigned char *)cbc_aes256_key;
													p_cipher = (unsigned char *)cbc_aes256_cipher;
													break;
							default:				printf("Invalid Key Length Error!\n");
													return -1;
							}
							break;
	case ACE_AES_CFB128:	memcpy(cipher_mode,"CFB128",6);
							p_plain = (unsigned char *)cfb128_aes_plain;
							memcpy(orig_iv,(unsigned char *)cfb128_aes_iv,16);
							switch(key_len)
							{
							case KEY_128BITS:		memcpy(cipher_key,"128",3);
													p_key = (unsigned char *)cfb128_aes128_key;
													p_cipher = (unsigned char *)cfb128_aes128_cipher;
													break;
							case KEY_192BITS:		memcpy(cipher_key,"192",3);
													p_key = (unsigned char *)cfb128_aes192_key;
													p_cipher = (unsigned char *)cfb128_aes192_cipher;
													break;
							case KEY_256BITS:		memcpy(cipher_key,"256",3);
													p_key = (unsigned char *)cfb128_aes256_key;
													p_cipher = (unsigned char *)cfb128_aes256_cipher;
													break;
							default:				printf("Invalid Key Length Error!\n");
													return -1;
							}
							break;
	case ACE_AES_OFB128:	memcpy(cipher_mode,"OFB128",6);
							p_plain = (unsigned char *)ofb_aes_plain;
							memcpy(orig_iv,(unsigned char *)ofb_aes_iv,16);
							switch(key_len)
							{
							case KEY_128BITS:		memcpy(cipher_key,"128",3);
													p_key = (unsigned char *)ofb_aes128_key;
													p_cipher = (unsigned char *)ofb_aes128_cipher;
													break;
							case KEY_192BITS:		memcpy(cipher_key,"192",3);
													p_key = (unsigned char *)ofb_aes192_key;
													p_cipher = (unsigned char *)ofb_aes192_cipher;
													break;
							case KEY_256BITS:		memcpy(cipher_key,"256",3);
													p_key = (unsigned char *)ofb_aes256_key;
													p_cipher = (unsigned char *)ofb_aes256_cipher;
													break;
							default:				printf("Invalid Key Length Error!\n");
													return -1;
							}
							break;
	default :				printf("Invalid Cipher Mode Error!\n");
							return -1;
	}
	
	print_star();
	printf("%s mode plain AES %sbits key cipher test:\n",cipher_mode,cipher_key);
	print_star();

	// create the ace aes cipher context 
	ctx = padlock_aes_begin();

	// set cipher key
	res = padlock_aes_setkey( ctx, p_key, key_len);
	if(res != AES_SUCCEEDED)
	{
		print_ace_error(res);
		return -1;
	}
	
	// because the iv will be updated by the plain aes encryption and decryption API
	// here we reserve it in the orig_iv
	memcpy(temp_iv, orig_iv, 16);

	// set cipher mode and iv
	res = padlock_aes_setmodeiv( ctx, mode, temp_iv);
	if(res != AES_SUCCEEDED)
	{
		print_ace_error(res);
		return -1;
	}
		
	// call ACE plain aes encryption API
	res = padlock_aes_encrypt( ctx, p_plain, scratch, nbytes);
	if(res != AES_SUCCEEDED)
	{
		print_ace_error(res);
		return -1;
	}
	
	// see whether the iv has been updated by ACE plain aes encryption API
	if(memcmp (temp_iv, orig_iv, 16))
	{
		printf("\niv has been changed after encryption!\n");
	}
	else
	{
		printf("\niv hasn't been changed after encryption!\n");
	}
	
	// verify the result of encryption
	if (memcmp (scratch, p_cipher, nbytes))
	{
        printf("%s mode plain AES-%s encryption test failed.\n",cipher_mode,cipher_key);
	}
    else
	{
		printf("%s mode plain AES-%s encryption test ok\n",cipher_mode,cipher_key);
	}

	print_star();
	
	/////////////////////////////////////////////////////////////////////
	
	// because the temp_iv has been updated 
	// we have to restore orig_iv to it here
	memcpy(temp_iv, orig_iv, 16);

	// reset iv for decrytion
	res = padlock_aes_setmodeiv( ctx, mode, temp_iv);
	if(res != AES_SUCCEEDED)
	{
		print_ace_error(res);
		return -1;
	}

	// call ACE plain aes decryption API
	res = padlock_aes_decrypt( ctx, p_cipher, scratch, nbytes);
	if(res != AES_SUCCEEDED)
	{
		print_ace_error(res);
		return -1;
	}
	
	// see whether the iv has been updated by ACE plain aes decryption API
	if(memcmp (temp_iv, orig_iv, 16))
	{
		printf("\niv has been changed after decryption!\n");
	}
	else
	{
		printf("\niv hasn't been changed after decryption!\n");
	}
	
	// verify the result of decryption
	if (memcmp (scratch, p_plain, nbytes))
	{
        printf("%s mode plain AES-%s decryption test failed.\n",cipher_mode,cipher_key);
    }
	else 
	{
		printf("%s mode plain AES-%s decryption test ok\n",cipher_mode,cipher_key);
	}

	// destroy the ace aes cipher context
	padlock_aes_close( ctx);
    
	print_star();

	return 0;
}

// test VIA Padlock SDK ACE aligned aes API
static int
test_padlock_aligned_aes(ACE_AES_MODE mode, KEY_LENGTH key_len, int nbytes)
{
	struct aligned_memory_context *buf_aligned_mctx;
	
	char cipher_mode[10]= {0,};
	char cipher_key[10] = {0,};
	
	unsigned char *p_key;
	unsigned char *p_plain;
	unsigned char *p_cipher;
	unsigned char temp_iv[16]={0,};
	unsigned char orig_iv[16]={0,};
	
	AES_RESULT res;

	switch (mode)
	{
	case ACE_AES_ECB:		memcpy(cipher_mode,"ECB",3);
							p_plain = (unsigned char *)ecb_aes_plain;
							switch(key_len)
							{
							case KEY_128BITS:		memcpy(cipher_key,"128",3);
													p_key = (unsigned char *)ecb_aes128_key;
													p_cipher = (unsigned char *)ecb_aes128_cipher;
													break;
							case KEY_192BITS:		memcpy(cipher_key,"192",3);
													p_key = (unsigned char *)ecb_aes192_key;
													p_cipher = (unsigned char *)ecb_aes192_cipher;
													break;
							case KEY_256BITS:		memcpy(cipher_key,"256",3);
													p_key = (unsigned char *)ecb_aes256_key;
													p_cipher = (unsigned char *)ecb_aes256_cipher;
													break;
							default:				printf("Invalid Key Length Error!\n");
													return -1;
							}
							break;
	case ACE_AES_CBC:		memcpy(cipher_mode,"CBC",3);
							p_plain = (unsigned char *)cbc_aes_plain;
							memcpy(orig_iv,(unsigned char *)cbc_aes_iv,16);
							switch(key_len)
							{
							case KEY_128BITS:		memcpy(cipher_key,"128",3);
													p_key = (unsigned char *)cbc_aes128_key;
													p_cipher = (unsigned char *)cbc_aes128_cipher;
													break;
							case KEY_192BITS:		memcpy(cipher_key,"192",3);
													p_key = (unsigned char *)cbc_aes192_key;
													p_cipher = (unsigned char *)cbc_aes192_cipher;
													break;
							case KEY_256BITS:		memcpy(cipher_key,"256",3);
													p_key = (unsigned char *)cbc_aes256_key;
													p_cipher = (unsigned char *)cbc_aes256_cipher;
													break;
							default:				printf("Invalid Key Length Error!\n");
													return -1;
							}
							break;
	case ACE_AES_CFB128:	memcpy(cipher_mode,"CFB128",6);
							p_plain = (unsigned char *)cfb128_aes_plain;
							memcpy(orig_iv,(unsigned char *)cfb128_aes_iv,16);
							switch(key_len)
							{
							case KEY_128BITS:		memcpy(cipher_key,"128",3);
													p_key = (unsigned char *)cfb128_aes128_key;
													p_cipher = (unsigned char *)cfb128_aes128_cipher;
													break;
							case KEY_192BITS:		memcpy(cipher_key,"192",3);
													p_key = (unsigned char *)cfb128_aes192_key;
													p_cipher = (unsigned char *)cfb128_aes192_cipher;
													break;
							case KEY_256BITS:		memcpy(cipher_key,"256",3);
													p_key = (unsigned char *)cfb128_aes256_key;
													p_cipher = (unsigned char *)cfb128_aes256_cipher;
													break;
							default:				printf("Invalid Key Length Error!\n");
													return -1;
							}
							break;
	case ACE_AES_OFB128:	memcpy(cipher_mode,"OFB128",6);
							p_plain = (unsigned char *)ofb_aes_plain;
							memcpy(orig_iv,(unsigned char *)ofb_aes_iv,16);
							switch(key_len)
							{
							case KEY_128BITS:		memcpy(cipher_key,"128",3);
													p_key = (unsigned char *)ofb_aes128_key;
													p_cipher = (unsigned char *)ofb_aes128_cipher;
													break;
							case KEY_192BITS:		memcpy(cipher_key,"192",3);
													p_key = (unsigned char *)ofb_aes192_key;
													p_cipher = (unsigned char *)ofb_aes192_cipher;
													break;
							case KEY_256BITS:		memcpy(cipher_key,"256",3);
													p_key = (unsigned char *)ofb_aes256_key;
													p_cipher = (unsigned char *)ofb_aes256_cipher;
													break;
							default:				printf("Invalid Key Length Error!\n");
													return -1;
							}
							break;
	default :				printf("Invalid Cipher Mode Error!\n");
							return -1;
	}
	
	print_star();
	printf("%s mode aligned AES %sbits key cipher test:\n",cipher_mode,cipher_key);
	print_star();

	// create address aligned buffer context
	buf_aligned_mctx = padlock_aligned_malloc(nbytes);
	
	// copy data to be encrypted to aligned buffer
	res = padlock_aligned_memcpy_to(buf_aligned_mctx, p_plain, nbytes);
	if(res != AES_SUCCEEDED)
	{
		printf("aligned memory copy error!\n");
		return -1;
	}
	
	// because the temp_iv will be updated 
	// we have to reserve it to orig_iv and copy it to temp_iv here
	memcpy(temp_iv, orig_iv, 16);

	// call ACE aligned aes encryption API 
	res = padlock_aes_aligned_encrypt(p_key, key_len, mode, buf_aligned_mctx, temp_iv);
	if(res != AES_SUCCEEDED)
	{
		print_ace_error(res);
		return -1;
	}
	
	// see whether the iv has been updated by ACE aligned aes encryption API
	if(memcmp (temp_iv, orig_iv, 16))
	{
		printf("\niv has been changed after encryption!\n");
	}
	else
	{
		printf("\niv hasn't been changed after encryption!\n");
	}

	// copy the encrypted result from aligned buffer to scratch
	res = padlock_aligned_memcpy_from(buf_aligned_mctx, scratch, nbytes);
	if(res != AES_SUCCEEDED)
	{
		printf("aligned memory copy error!\n");
		return -1;
	}
	
	// verify the result of encryption
	if (memcmp (scratch, p_cipher, nbytes))
	{
        printf("%s mode aligned AES-%s encryption test failed.\n",cipher_mode,cipher_key);
    }
	else 
    {
		printf("%s mode aligned AES-%s encryption test ok\n",cipher_mode,cipher_key);
	}

	print_star();
	/////////////////////////////////////////////////////////////////////
	
	// copy data to be decrypted to aligned buffer
	res = padlock_aligned_memcpy_to(buf_aligned_mctx, p_cipher, nbytes);
	if(res != AES_SUCCEEDED)
	{
		printf("aligned memory copy error!\n");
		return -1;
	}
	
	// because the temp_iv has been updated 
	// we have to restore orig_iv to it here
	memcpy(temp_iv, orig_iv, 16);
		
	// call ACE aligned aes decryption API
	res = padlock_aes_aligned_decrypt(p_key, key_len, mode, buf_aligned_mctx, temp_iv);
	if(res != AES_SUCCEEDED)
	{
		print_ace_error(res);
		return -1;
	}
	
	// see whether the iv has been updated by ACE aligned aes decryption API
	if(memcmp (temp_iv, orig_iv, 16))
	{
		printf("\niv has been changed after decryption!\n");
	}
	else
	{
		printf("\niv hasn't been changed after decryption!\n");
	}
	
	// copy the decrypted result from aligned buffer to scratch
	res = padlock_aligned_memcpy_from(buf_aligned_mctx, scratch, nbytes);
	if(res != AES_SUCCEEDED)
	{
		printf("aligned memory copy error!\n");
		return -1;
	}
	
	// verify the result of decryption
	if (memcmp (scratch, p_plain, nbytes))
	{
        printf("%s mode aligned AES-%s decryption test failed.\n",cipher_mode,cipher_key);
    }
	else 
    {
		printf("%s mode aligned AES-%s decryption test ok\n",cipher_mode,cipher_key);
	}

	// destroy the aligned buffer context
	padlock_aligned_mfree( buf_aligned_mctx);
    
	print_star();

	return 0;
}

//  test VIA Padlock SDK ACE fast aes API
static int
test_padlock_fast_aes(ACE_AES_MODE mode, KEY_LENGTH key_len, int nbytes)
{
	unsigned char *p_temp_buf;
	unsigned char *p_aligned_buf;
	
	char cipher_mode[10]= {0,};
	char cipher_key[10] = {0,};
	
	unsigned char *p_key;
	unsigned char *p_plain;
	unsigned char *p_cipher;
	unsigned char temp_iv[16]={0,};
	unsigned char orig_iv[16]={0,};
	
	AES_RESULT res;

	switch (mode)
	{
	case ACE_AES_ECB:		memcpy(cipher_mode,"ECB",3);
							p_plain = (unsigned char *)ecb_aes_plain;
							switch(key_len)
							{
							case KEY_128BITS:		memcpy(cipher_key,"128",3);
													p_key = (unsigned char *)ecb_aes128_key;
													p_cipher = (unsigned char *)ecb_aes128_cipher;
													break;
							case KEY_192BITS:		memcpy(cipher_key,"192",3);
													p_key = (unsigned char *)ecb_aes192_key;
													p_cipher = (unsigned char *)ecb_aes192_cipher;
													break;
							case KEY_256BITS:		memcpy(cipher_key,"256",3);
													p_key = (unsigned char *)ecb_aes256_key;
													p_cipher = (unsigned char *)ecb_aes256_cipher;
													break;
							default:				printf("Invalid Key Length Error!\n");
													return -1;
							}
							break;
	case ACE_AES_CBC:		memcpy(cipher_mode,"CBC",3);
							p_plain = (unsigned char *)cbc_aes_plain;
							memcpy(orig_iv,(unsigned char *)cbc_aes_iv,16);
							switch(key_len)
							{
							case KEY_128BITS:		memcpy(cipher_key,"128",3);
													p_key = (unsigned char *)cbc_aes128_key;
													p_cipher = (unsigned char *)cbc_aes128_cipher;
													break;
							case KEY_192BITS:		memcpy(cipher_key,"192",3);
													p_key = (unsigned char *)cbc_aes192_key;
													p_cipher = (unsigned char *)cbc_aes192_cipher;
													break;
							case KEY_256BITS:		memcpy(cipher_key,"256",3);
													p_key = (unsigned char *)cbc_aes256_key;
													p_cipher = (unsigned char *)cbc_aes256_cipher;
													break;
							default:				printf("Invalid Key Length Error!\n");
													return -1;
							}
							break;
	case ACE_AES_CFB128:	memcpy(cipher_mode,"CFB128",6);
							p_plain = (unsigned char *)cfb128_aes_plain;
							memcpy(orig_iv,(unsigned char *)cfb128_aes_iv,16);
							switch(key_len)
							{
							case KEY_128BITS:		memcpy(cipher_key,"128",3);
													p_key = (unsigned char *)cfb128_aes128_key;
													p_cipher = (unsigned char *)cfb128_aes128_cipher;
													break;
							case KEY_192BITS:		memcpy(cipher_key,"192",3);
													p_key = (unsigned char *)cfb128_aes192_key;
													p_cipher = (unsigned char *)cfb128_aes192_cipher;
													break;
							case KEY_256BITS:		memcpy(cipher_key,"256",3);
													p_key = (unsigned char *)cfb128_aes256_key;
													p_cipher = (unsigned char *)cfb128_aes256_cipher;
													break;
							default:				printf("Invalid Key Length Error!\n");
													return -1;
							}
							break;
	case ACE_AES_OFB128:	memcpy(cipher_mode,"OFB128",6);
							p_plain = (unsigned char *)ofb_aes_plain;
							memcpy(orig_iv,(unsigned char *)ofb_aes_iv,16);
							switch(key_len)
							{
							case KEY_128BITS:		memcpy(cipher_key,"128",3);
													p_key = (unsigned char *)ofb_aes128_key;
													p_cipher = (unsigned char *)ofb_aes128_cipher;
													break;
							case KEY_192BITS:		memcpy(cipher_key,"192",3);
													p_key = (unsigned char *)ofb_aes192_key;
													p_cipher = (unsigned char *)ofb_aes192_cipher;
													break;
							case KEY_256BITS:		memcpy(cipher_key,"256",3);
													p_key = (unsigned char *)ofb_aes256_key;
													p_cipher = (unsigned char *)ofb_aes256_cipher;
													break;
							default:				printf("Invalid Key Length Error!\n");
													return -1;
							}
							break;
	default :				printf("Invalid Cipher Mode Error!\n");
							return -1;
	}
	
	print_star();
	printf("%s mode fast AES %sbits key cipher test:\n",cipher_mode,cipher_key);
	print_star();

	// malloc temporary buffer for later use
	p_temp_buf = (unsigned char *)malloc((nbytes + 16));
	if(p_temp_buf == NULL)
	{
		printf("Fail to malloc %d bytes memory!\n",(nbytes + 16));
		return -1;
	}
	
	// get address aligned buffer from temporary buffer
	p_aligned_buf = (unsigned char *)((((unsigned long)p_temp_buf) + 15 )&(~15UL));
	
	// copy the data to be encrypted to aligned buffer
	memcpy(p_aligned_buf, p_plain,nbytes);
	
	// because the temp_iv will be updated 
	// we have to reserve it to orig_iv and copy it to temp_iv here
	memcpy(temp_iv, orig_iv, 16);
	
	// call ACE fast aes encryption API 
	res = padlock_aes_fast_encrypt(p_key, key_len, mode, p_aligned_buf, p_aligned_buf, nbytes, temp_iv);
	if(res != AES_SUCCEEDED)
	{
		print_ace_error(res);
		return -1;
	}
	
	// copy the encryption result to scratch
	memcpy(scratch, p_aligned_buf, nbytes);
	
	// see whether the iv has been updated by ACE fast aes encryption API
	if(memcmp (temp_iv, orig_iv, 16))
	{
		printf("\niv has been changed after encryption!\n");
	}
	else
	{
		printf("\niv hasn't been changed after encryption!\n");
	}

	// verify the result of encryption
	if (memcmp (scratch, p_cipher, nbytes))
	{
        printf("%s mode fast AES-%s encryption test failed.\n",cipher_mode,cipher_key);
    }
	else 
    {
		printf("%s mode fast AES-%s encryption test ok\n",cipher_mode,cipher_key);
	}

	print_star();
	/////////////////////////////////////////////////////////////////////
	
	// copy the data to be decrypted to aligned buffer
	memcpy(p_aligned_buf, p_cipher, nbytes);
	
	// because the temp_iv has been updated 
	// we have to restore orig_iv to it here
	memcpy(temp_iv, orig_iv, 16);

	// call ACE fast aes decryption API 
	res = padlock_aes_fast_decrypt(p_key, key_len, mode, p_aligned_buf, p_aligned_buf, nbytes, temp_iv);
	if(res != AES_SUCCEEDED)
	{
		print_ace_error(res);
		return -1;
	}

	// copy the decryption result to scratch
	memcpy(scratch, p_aligned_buf, nbytes);

	// see whether the iv has been updated by ACE fast aes decryption API
	if(memcmp (temp_iv, orig_iv, 16))
	{
		printf("\niv has been changed after decryption!\n");
	}
	else
	{
		printf("\niv hasn't been changed after decryption!\n");
	}

	// verify the result of decryption
	if (memcmp (scratch, p_plain, nbytes))
	{
        printf("%s mode fast AES-%s decryption test failed.\n",cipher_mode,cipher_key);
    }
	else 
    {
		printf("%s mode fast AES-%s decryption test ok\n",cipher_mode,cipher_key);
	}

	// free temporary buffer
	free( p_temp_buf);
    
	print_star();

	return 0;
}

int main(int argc, char* argv[])
{
    int rng_available;
	unsigned char random_num1[1024] = {0,};
	unsigned char random_num2[1024] = {0,};

	int ace_available;
	ACE_AES_MODE mode;
	KEY_LENGTH key_len;
	int nbytes = 64;

	///////////////////////////////////////////////////////////////////////////

	print_star();
	printf("Begin to test Padlock SDK RNG API functions:\n");
	print_star();


	// Padlock SDK RNG API
	rng_available = padlock_rng_available();
	if(rng_available)
	{
		printf("VIA RNG hardware is available!\n");
	}
	else
	{
		printf("VIA RNG hardware isn't available!\n");
		return -1;
	}
		
	print_star();
	padlock_rng_rand( random_num1, 1024);
	printf("VIA RNG generates random number 1 :\n");
	dump_rand(random_num1, 1024);	
	print_star();

	print_star();
	padlock_rng_rand( random_num2, 1024);
	printf("VIA RNG generates random number 2 :\n");
	dump_rand(random_num2, 1024);	
	print_star();

	if((strcmp(random_num1, random_num2)))
	{
		printf("\nVIA RNG random number generation function test ok!\n");
	}
	else
	{
		printf("\nVIA RNG random number generation function test failed!\n");
	}
	
	print_star();

	////////////////////////////////////////////////////////////////////////////////////

	print_star();
	printf("Begin to test Padlock SDK ACE AES API functions:\n");
	print_star();

	// Padlock SDK ACE AES API
	ace_available = padlock_ace_available();
	if(ace_available)
	{
		printf("VIA ACE hardware is available!\n");
	}
	else
	{
		printf("VIA ACE hardware isn't available!\n");
		return -1;
	}
	print_star();

	mode = ACE_AES_ECB;
	key_len = KEY_128BITS;
	test_padlock_aes(mode,key_len,nbytes);

	mode = ACE_AES_ECB;
	key_len = KEY_192BITS;
	test_padlock_aes(mode,key_len,nbytes);
	
	mode = ACE_AES_ECB;
	key_len = KEY_256BITS;
	test_padlock_aes(mode,key_len,nbytes);

	mode = ACE_AES_CBC;
	key_len = KEY_128BITS;
	test_padlock_aes(mode,key_len,nbytes);
	
	mode = ACE_AES_CBC;
	key_len = KEY_192BITS;
	test_padlock_aes(mode,key_len,nbytes);

	mode = ACE_AES_CBC;
	key_len = KEY_256BITS;
	test_padlock_aes(mode,key_len,nbytes);

	mode = ACE_AES_CFB128;
	key_len = KEY_128BITS;
	test_padlock_aes(mode,key_len,nbytes);

	mode = ACE_AES_CFB128;
	key_len = KEY_192BITS;
	test_padlock_aes(mode,key_len,nbytes);

	mode = ACE_AES_CFB128;
	key_len = KEY_256BITS;
	test_padlock_aes(mode,key_len,nbytes);

	mode = ACE_AES_OFB128;
	key_len = KEY_128BITS;
	test_padlock_aes(mode,key_len,nbytes);

	mode = ACE_AES_OFB128;
	key_len = KEY_192BITS;
	test_padlock_aes(mode,key_len,nbytes);

	mode = ACE_AES_OFB128;
	key_len = KEY_256BITS;
	test_padlock_aes(mode,key_len,nbytes);

	//////////////////////////////////////////////////////////////////

	print_star();
	printf("Begin to test Padlock SDK aligned aes API functions:\n");

	mode = ACE_AES_ECB;
	key_len = KEY_128BITS;
	test_padlock_aligned_aes(mode,key_len,nbytes);

	mode = ACE_AES_ECB;
	key_len = KEY_192BITS;
	test_padlock_aligned_aes(mode,key_len,nbytes);
	
	mode = ACE_AES_ECB;
	key_len = KEY_256BITS;
	test_padlock_aligned_aes(mode,key_len,nbytes);

	mode = ACE_AES_CBC;
	key_len = KEY_128BITS;
	test_padlock_aligned_aes(mode,key_len,nbytes);
	
	mode = ACE_AES_CBC;
	key_len = KEY_192BITS;
	test_padlock_aligned_aes(mode,key_len,nbytes);

	mode = ACE_AES_CBC;
	key_len = KEY_256BITS;
	test_padlock_aligned_aes(mode,key_len,nbytes);

	mode = ACE_AES_CFB128;
	key_len = KEY_128BITS;
	test_padlock_aligned_aes(mode,key_len,nbytes);

	mode = ACE_AES_CFB128;
	key_len = KEY_192BITS;
	test_padlock_aligned_aes(mode,key_len,nbytes);

	mode = ACE_AES_CFB128;
	key_len = KEY_256BITS;
	test_padlock_aligned_aes(mode,key_len,nbytes);

	mode = ACE_AES_OFB128;
	key_len = KEY_128BITS;
	test_padlock_aligned_aes(mode,key_len,nbytes);

	mode = ACE_AES_OFB128;
	key_len = KEY_192BITS;
	test_padlock_aligned_aes(mode,key_len,nbytes);

	mode = ACE_AES_OFB128;
	key_len = KEY_256BITS;
	test_padlock_aligned_aes(mode,key_len,nbytes);

	///////////////////////////////////////////////////////////////////////////////////////

	print_star();
	printf("Begin to test Padlock SDK fast aes API functions:\n");

	mode = ACE_AES_ECB;
	key_len = KEY_128BITS;
	test_padlock_fast_aes(mode,key_len,nbytes);

	mode = ACE_AES_ECB;
	key_len = KEY_192BITS;
	test_padlock_fast_aes(mode,key_len,nbytes);
	
	mode = ACE_AES_ECB;
	key_len = KEY_256BITS;
	test_padlock_fast_aes(mode,key_len,nbytes);

	mode = ACE_AES_CBC;
	key_len = KEY_128BITS;
	test_padlock_fast_aes(mode,key_len,nbytes);
	
	mode = ACE_AES_CBC;
	key_len = KEY_192BITS;
	test_padlock_fast_aes(mode,key_len,nbytes);

	mode = ACE_AES_CBC;
	key_len = KEY_256BITS;
	test_padlock_fast_aes(mode,key_len,nbytes);

	mode = ACE_AES_CFB128;
	key_len = KEY_128BITS;
	test_padlock_fast_aes(mode,key_len,nbytes);

	mode = ACE_AES_CFB128;
	key_len = KEY_192BITS;
	test_padlock_fast_aes(mode,key_len,nbytes);

	mode = ACE_AES_CFB128;
	key_len = KEY_256BITS;
	test_padlock_fast_aes(mode,key_len,nbytes);

	mode = ACE_AES_OFB128;
	key_len = KEY_128BITS;
	test_padlock_fast_aes(mode,key_len,nbytes);

	mode = ACE_AES_OFB128;
	key_len = KEY_192BITS;
	test_padlock_fast_aes(mode,key_len,nbytes);
	
	mode = ACE_AES_OFB128;
	key_len = KEY_256BITS;
	test_padlock_fast_aes(mode,key_len,nbytes);
	
	return 0;
}
