/* c't-Netz-Schalter Demo Source
 * Datei: event.c
 * Info: Routinen fr die Zeitschaltuhr
 * Autor: Benjamin Benz (bbe@heise.de)
 * Datum: 29.09.05
*/
#include "../mcu.h"

#include "event.h"
#include <avr/eeprom.h>
#include "rtc.h"
#include "../schalt.h"
#include "uart.h"


#ifdef EVENT_AVAILABLE

#define DONTCARE	255
#define MAXEVENT 	20

#ifdef RTC_FULL_AVAILABLE
	// Ein erstes Event, um 00:00:30; Tag, Monat, Jahr, Wochentag egal
	#define GETTIMEEVENT {{10, 0,0,  DONTCARE, DONTCARE,DONTCARE,DONTCARE,{0,EVENT_SPECIAL,EVENT_GETTIME}}}
#else
#define GETTIMEEVENT {{10, 0,0,  DONTCARE,{EVENT_SPECIAL,0,EVENT_GETTIME}}}
#endif	

// Wieviele Events stehen im EEPROM
char eventsInEEPROM  __attribute__ ((section (".eeprom")))=1;
char events=0;

// Adresse im EEPROM ab der wir Events ablegen, initialisieren mit 1x gettime 
event_t eventEEPROM[MAXEVENT] __attribute__ ((section (".eeprom"))) = GETTIMEEVENT;
char eventPtr =0;

event_t event;	// Ein Event im RAM um damit zu arbeiten

/* sichert das Arbeitsevent event an die Stelle des eventPtr im EEPROM*/
void event_ToEEPROM(unsigned char eventPtr){
	if ((eventPtr <= events) & (eventPtr <MAXEVENT)){
		eeprom_write_block(&event,&eventEEPROM[eventPtr],sizeof(event_t));
		
		if (eventPtr == events){ //neues Event geschrieben
			events++;	  // event-Zhler hoch	
			eeprom_write_byte(&eventsInEEPROM,events); //sichern
		}
	}
}

/* ldt das Arbeitsevent event von der Stelle des eventPtr des EEPROMs*/
void event_FromEEPROM(unsigned char eventPtr){
	if (eventPtr < events)
		eeprom_read_block(&event,&eventEEPROM[eventPtr],sizeof(event_t));
}

void event_init(void){
	events=eeprom_read_byte(&eventsInEEPROM);
	if (events > MAXEVENT){
		events=0;
	}
}

#ifdef RTC_FULL_AVAILABLE	 
	#define day_matches \
		 (((event.day    == rtc_day)    | (event.day    == DONTCARE)) & \
		 ((event.month  == rtc_month)  | (event.month  == DONTCARE)) & \
		 ((event.year   == rtc_year)   | (event.year   == DONTCARE)))
#else
	#define day_matches (1)
#endif


#define event_matches \
	(((event.second == rtc_second) | (event.second == DONTCARE)) & \
	 ((event.minute == rtc_minute) | (event.minute == DONTCARE)) & \
	 ((event.hour   == rtc_hour)   | (event.hour   == DONTCARE)) & \
	 ((event.weekday & (1 <<(rtc_weekday-1))) >0)  & day_matches)

void event_cron(){
	int i;
	for (i=0; i<events; i++){
		event_FromEEPROM(i);
		if (event_matches){
			if (event.action.special == EVENT_SPECIAL){
				#ifdef RTC_HTTPTIME_AVAILABLE
					if (event.action.data== EVENT_GETTIME){
							rtc_get_httptime();
					}
				#endif
			}else
				Relais_set_masked(event.action.data, event.action.channel);
		}
	}
}

#ifdef UART_AVAILABLE	

/* Transmit one event via uart */ 
void event_transmit(char ev){
	char i;
	char * ptr = (char *) &event;

	event_FromEEPROM(ev);	
	for (i=0; i<sizeof(event_t); i++)
		uart_send(*ptr++);		
}

/* Receive one event from uart */ 
char event_receive(char ev){
	char i;
	char * ptr = (char *) &event;
	
	uart_timeout=0;
	for (i=0; i<sizeof(event_t); i++)
		*ptr++=uart_read();

	if (uart_timeout==0){
		event_ToEEPROM(ev);
		return 1;
	} else 
		return 0;
}
#endif
/* Delete Event ev from EEPROM */
void event_delete(char ev){
	if (ev < events){
		while (ev < (events-1)){	// Beim letzten mssen 
			event_FromEEPROM(ev+1);	//wir nicht mehr schieben
			event_ToEEPROM(ev++);
		}
		events--;
	}
}

/*
void event_test(void){	
	event.channel=00;		// Maske fr die betroffenen Kanle
	event.data=01;		// Werte fr die betroffenen Kanle
	
	event.second=02;		// Zeitpunkt des Ereignisses
	event.minute=3;		// wird ein Feld nicht benutzt, steht es 
	event.hour=4;		// auf DONTCARE
	event.day=5;
	event.month=6;
	event.year=0x0708;	// Achtung das Jahr 255 bedeutet DONTCARE
	
	
	event.minute=DONTCARE;
	event.hour=DONTCARE;
	event.day=DONTCARE;
	event.month=DONTCARE;
	event.year=DONTCARE;

	event.channel= 01;
	event.data=00;
	
	event.second=10;
	
	event_ToEEPROM(0);

	event.channel= 01;
	event.data=00;	
	event.second=20;

	event_ToEEPROM(1);
}
*/
#endif
