/* c't Mikrocontroller-im-LAN Demo Source
 * Datei: uart.c 
 * Info: Routinen zur seriellen Kommunikation
 * Autor: Benjamin Benz (bbe@heise.de)
 * Datum: 24.06.04
*/

#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/signal.h>
#include "../mcu.h"
#include "uart.h"
#include "led.h"

#ifdef UART_AVAILABLE

#define UART_RX_BUFFER_SIZE 16

#define UART_RX_BUFFER_MASK ( UART_RX_BUFFER_SIZE - 1 )
#if ( UART_RX_BUFFER_SIZE & UART_RX_BUFFER_MASK )
	#error RX buffer size is not a power of 2
#endif

#define UART_TIMEOUT	20000	// Timeout. Wartet UART_TIMEOUT CPU-Takte


static unsigned char UART_RxBuf[UART_RX_BUFFER_SIZE];
static volatile unsigned char UART_RxHead;
static volatile unsigned char UART_RxTail;

char uart_dest_akt =DEST_XPORT; // Hier sichern wir den aktuellen Zustand
char uart_dest_old =DEST_XPORT; // Hier sichern wir den letzten Zustand

char uart_timeout;	// 0, wenn uart_read/uart_send erfolgreich
			// 1, wenn timeout erreicht

void uart_init(void){
	DDRB|= 0x07;  // Multiplexleitungen seriell PB0-2 als Ausgang	
	PORTB|= 0x03; // UART verbinden mit COM
	
	
	UBRRH= 0x0;  // UART auf 9600 baud
	UBRRL= 0x5F; //  UBRR= (fquarz/ (16* BAUD) ) -1
	
	UCSRC=0x86; // 8 bit 1 Stop No Parity

	//Transmit&Receive Enable + RX Int
	UCSRB= ((1<<RXEN) | (1<<TXEN)|(1<<RXCIE)); 

	/* Flush receive buffer */

	UART_RxTail = 0;
	UART_RxHead = 0;
}

/* Interrupt Handler for UART RECV */
SIGNAL (SIG_UART_RECV){
	unsigned char data;
	unsigned char tmphead;
	
	data = UDR;                 /* Read the received data */

	/* Calculate buffer index */
	tmphead = ( UART_RxHead + 1 ) & UART_RX_BUFFER_MASK;
	UART_RxHead = tmphead;      /* Store new index */

	if ( tmphead == UART_RxTail ){
		/* ERROR! Receive buffer overflow */
	}
	
	UART_RxBuf[tmphead] = data; /* Store received data in buffer */
}

char uart_data_available(void){
	if (UART_RxHead == UART_RxTail) 
		return 0;
	else return 1;
}

/* Liefert ein Zeichen aus dem uart-buffer
 * Kehrt nach UART_TIMEOUT auf jedenfall zurck
 * liefert in uart_timeout zurck, ob das lesen erfolgreich war
 */
char uart_read(void){	
	int tmptail=0;
	//uart_timeout=0;
	while ( (UART_RxHead == UART_RxTail) & (tmptail < UART_TIMEOUT) ){
		tmptail++;
	}
	
	if (tmptail == UART_TIMEOUT){
		uart_timeout=1;
		return 0;
	}
	
	tmptail = ( UART_RxTail + 1 ) & UART_RX_BUFFER_MASK;/* Calculate buffer index */
	
	UART_RxTail = tmptail;                /* Store new index */
	
	return UART_RxBuf[tmptail];           /* Return data */
}

void uart_send(char data){ // Achtung ist noch blockierend!!!!
	while (((UCSRA >> UDRE) & 1) ==0){}	// warten bis UART sendebereit
	UDR= data;
}

void uart_dest(char dest){
	if (dest == DEST_OLD)	// Zurck zum letzten Zustand
		dest=uart_dest_old;
	
	switch (dest) {
		case DEST_XPORT: 
			PORTB &= 0xFC;	// PB0=0; PB1=0  ==> Z=Z0 Y=Y0 
			PORTB |= 0x04; 	// PB2=1 ==> (X=X1)	
			break;
		case DEST_COM: 
			PORTB |=  0x03;	// PB0=1; PB1=1  ==> Z=Z1 Y=Y1 
			PORTB &=  0xFB;	// PB2=0 ==> (X=X0) (eigentlich don't care)
			break;
		case DEST_PASSTHROUGH:	
			//PORTB |=  0x03; //  
			PORTB 	&=  0xFC; // PB0=1; PB1=1  ==> Z=Z0 Y=Y0 
			break;
	}
	uart_dest_old=uart_dest_akt;
	uart_dest_akt=dest;	// Zustand sichern
}
#endif
