/* c't Mikrocontroller-im-LAN Demo Source
 * Datei: adc.c 
 * Info: Routinen zum Einlesen der Analogeingnge
 * Autor: Benjamin Benz (bbe@heise.de)
 * Datum: 24.06.04
*/

#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/signal.h>
#include "../mcu.h"

#include "adc.h"
#include "uart.h"
#include "tools.h"
#include "mcu.h"

#ifdef FILTER_AVAILABLE
	#include "adc_filter.h"
#endif

#ifdef ADC_AVAILABLE

volatile int adc_buffer[ADC_BUFFER];	// Ringpuffer
volatile unsigned char adc_ptr=0;	// Zeiger auf den Ringpuffer
volatile char adc_channel=0;	// Welchen kanal soll der Interrupt samplen?
				// Das hchste Bit legt fest, ob der 50Hz-Filter
				// verwendet wird: 0= Filter aus, 1= Filter an?
				
// Initialisert den AD-Umsetzer. Fr jeden Kanal, den man nutzen mchte, 
// muss das entsprechende Bit in channel gesetzt sein:
// Bit0 = Kanal 0 usw.
void adc_init(char channel){
	DDRA &= ~ channel;	// Pin als input
	PORTA &= ~ channel;	// Alle Pullups aus.
	
	// interne Referenzspannung, rechts Ausrichtung
	ADMUX=_BV(REFS1) | _BV(REFS0);

	SFIOR|= _BV(ADTS0)|_BV(ADTS2);	// ADC Trigger Source= Timer1 Compare B
	
	adc_channel=0;		// Wir fangen mit Kanal 0 an
	adc_ptr=0;		// Zeiger in den Ringpuffer
	
	ADMUX = (ADMUX & 0xE0) | adc_channel;	// Und jetzt Kanal whlen

	
	ADCSRA= _BV(ADPS2) | _BV(ADPS1)|	// prescale faktor= 128 ADC luft
		_BV(ADPS0) |			// mit 14,7456MHz/ 128 = 115,2kHz 
		_BV(ADEN)|			// ADC an
		_BV(ADSC)|			// Beginne mit der Konvertierung
		_BV(ADIE)|			// ADC-Interrupt an
		_BV(ADATE);			// Auto trigger on
	
	sei();
}

/* Interrupt, ausgelst, wenn der AD-Wandler fertig ist 
 * speichert einen Messwert. (vorher wendet sie evtl. noch eine Filterfkt an)
 */
SIGNAL (SIG_ADC){
	int adc=ADCW;			// ADC einlesen
	
	#ifdef FILTER_AVAILABLE
		int filtered=filter(adc);    // Der Filter muss weiterlaufen
	#endif

	if (adc_ptr < ADC_BUFFER){
		#ifdef FILTER_AVAILABLE
			if ((adc_channel & 0x80) != 0) // mit 50 Hz Bandpass
				adc_buffer[adc_ptr++]=filtered;
			else // ohne 50Hz Bandpass
		#endif
				adc_buffer[adc_ptr++]=adc;
	}
}


//Liest einen analogen Kanal aus blockierend!!!
//Der Parameter ist die Nummer des Pins (0x00 fr PA0; 0x01 fr PA1, ..)
//Lscht den ganzen Ringpuffer, wenn channel != adc_channel
int adc_read(char channel){
	if (adc_channel != channel) {
		adc_select_channel(channel);
		while (adc_ptr==0){}
	}
	return adc_buffer[adc_ptr];
}

/* Wechselt einen ADU-kanal. Dafr muessen auch die Puffer zurckgesetzt werden
 * Das oberste Bit legt fest, ob der 50 Hz Filter mitlaufen soll, oder nicht
 * es wird aber erst in der adc_isr() ausgewertet!!!!
*/
void adc_select_channel(char channel){
	adc_channel=channel;  	
	adc_ptr=0;
	ADMUX = (ADMUX & (_BV(REFS0)|_BV(REFS1)|_BV(ADLAR)) )  | channel;	// Und jetzt Kanal whlen
}

#ifdef UART_AVAILABLE	

/* bertrage den Inhalt des ADC-Puffers per uart */
void adc_transmit(void){
	unsigned char i;
	char hex[6];

	while (adc_ptr != ADC_BUFFER) {
		asm ("nop\n");
	}	//warten bis Puffer voll
	
	ADCSRA &= ~_BV(ADEN);	// Alle AD-Wandler aus, damit Puffer bleibt
	
	for (i=0; i<64; i++){		
		int_to_str(adc_buffer[i],hex);
		uart_send(hex[0]); uart_send(hex[1]);
		uart_send(hex[2]); uart_send(hex[3]);
		uart_send(hex[4]); uart_send(hex[5]);
		uart_send('\n');
	}
	ADCSRA |= _BV(ADEN);			// Und weiter gehts
}
#endif
#endif
