using System;
using System.Drawing;
using System.Windows.Forms;
using Microsoft.DirectX;
using Microsoft.DirectX.Direct3D;
using DirectInput = Microsoft.DirectX.DirectInput; // wegen der Doppelung der Klasse Device
using DirectSound = Microsoft.DirectX.DirectSound; // wegen der Doppelung der Klasse Device
using System.Runtime.InteropServices;
using System.Reflection;
using System.IO;
using System.Diagnostics;

namespace Plonk
{
	public class Game : Form
	{
		private const float width = 0.1f;
		private const float depth = 5.0f;
		private const float paddleHeight = 0.5f;
		private const float ballRadius = 0.1f; // sollte mit .x-Datei bereinstimmen
		private const float epsilon = 0.01f;
		private const float shadowRange = 0.2f;
		private readonly Vector3 rotationAxis = new Vector3(0.4f, 0.7f, -0.2f);
		private const float rotationSpeed = (float)(2.0*Math.PI/3.0); // Radian pro Sekunde fr Kugeldrehung

		private bool useHighResTimer = true;
		private long timerFreq = 1000;
		private long oldTimerCount = 0;
		private DateTime oldTime = DateTime.Now;

		private float fieldOfView = 0.01f; // darf nicht 0 werden
		private Vector2 camera = new Vector2();

		private Device device;
		private Vector2 ballPosition = new Vector2(-0.5f, 0.0f);
		private Vector2 ballVelocity = new Vector2(-0.5f, 0.3f); // Lngeneinheiten pro Sekunde
		private Vector2 paddlePosition = new Vector2(1.0f, 0.0f);
		private Vector2 oldPaddlePosition = new Vector2();
		private Vector2 paddleVelocity = new Vector2();
		private float rotationAngle = 0.0f;
		private Mesh wallHorizontal;
		private Mesh wallVertical;
		private Mesh paddle;
		private Mesh ball;
		private Material wallMaterial;
		private Material paddleMaterial;
		private Material ballMaterial;
		private Texture ballTexture;

		private DirectInput.Device input;
		private bool useFFJoystick;
		private DirectInput.Effect forceEffect;
		private DirectInput.EffectObject forceEffectObject;
		private bool isAcquired = false;
		private bool hasToAcquire = false;

		private DirectSound.Device audio;
		private DirectSound.SecondaryBuffer wallSound;
		private DirectSound.SecondaryBuffer paddleSound;

		public static void Main()
		{
			try
			{
				new Game().Run();
			}
			catch(Exception e)
			{
				MessageBox.Show(e.Message, "Fehler", MessageBoxButtons.OK, MessageBoxIcon.Error);
			}
		}
 
		[DllImport("kernel32.dll")]
		extern static short QueryPerformanceCounter(ref long x);
		[DllImport("kernel32.dll")]
		extern static short QueryPerformanceFrequency(ref long x);

		public Game()
		{
			Text = "Plonk (Esc: Ende, b: neuer Ball, Cursor/Blttern fr Perspektive)";
			Size = new Size(640, 480);
			Location = new Point(40, 40);
			SetStyle(ControlStyles.AllPaintingInWmPaint | ControlStyles.UserPaint, true);

			useHighResTimer = (0 != QueryPerformanceFrequency(ref timerFreq));
			if(useHighResTimer)
			{
				QueryPerformanceCounter(ref oldTimerCount);
				Trace.WriteLine("Using high-resolution timer at " + timerFreq + " Hz.");
			}
			else
			{
				Trace.WriteLine("Using standard time function.");
			}

			Assembly assembly = Assembly.GetAssembly(typeof(Game));
			Stream stream;

			PresentParameters pp = new PresentParameters();
			pp.AutoDepthStencilFormat = DepthFormat.D24X8;
			pp.EnableAutoDepthStencil = true;
			pp.SwapEffect = SwapEffect.Discard;
			pp.Windowed = true;
			bool canDoHardware = Manager.CheckDeviceType(0, DeviceType.Hardware, Format.X8R8G8B8, Format.A8R8G8B8, true)
				&& Manager.GetDeviceCaps(0, DeviceType.Hardware).DeviceCaps.SupportsHardwareTransformAndLight;
			if(canDoHardware)
			{
				device = new Device(0, DeviceType.Hardware, this, CreateFlags.HardwareVertexProcessing, pp);
			}
			else
			{
				bool canDoReference = Manager.CheckDeviceType(0, DeviceType.Reference, Format.X8R8G8B8, Format.A8R8G8B8, true);
				if(canDoReference)
				{
					MessageBox.Show(this, "Keine passende Hardwareuntersttzung gefunden,\ndeshalb auf den langsamen Reference Rasterizer geschaltet.",
						"Warnung", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
					device = new Device(0, DeviceType.Reference, this, CreateFlags.HardwareVertexProcessing, pp);
				}
				else
				{
					throw new ApplicationException("Programmabbruch:\nKeine passende Grafikuntersttzung gefunden,\nnicht einmal den Reference Rasterizer.");
				}
			}

			wallHorizontal = Mesh.Box(device, 2.0f, width, depth);
			wallVertical = Mesh.Box(device, width, 2.0f, depth);
			paddle = Mesh.Box(device, width, paddleHeight, depth);
			stream = assembly.GetManifestResourceStream("Plonk.ball.x");
			ball = Mesh.FromStream(stream, MeshFlags.Managed, device);
			stream.Close();
			stream = null;

			wallMaterial = new Material();
			wallMaterial.Ambient = Color.White;
			wallMaterial.Diffuse = Color.White;

			paddleMaterial = new Material();
			paddleMaterial.Ambient = Color.Green;
			paddleMaterial.Diffuse = Color.Green;

			ballMaterial = new Material();
			ballMaterial.Ambient = Color.White;
			ballMaterial.Diffuse = Color.White;
			ballMaterial.Specular = Color.White;
			ballMaterial.SpecularSharpness = 20.0f;
			stream = assembly.GetManifestResourceStream("Plonk.ball.jpg");
			ballTexture = TextureLoader.FromStream(device, stream);
			stream.Close();
			stream = null;

			device.DeviceReset += new EventHandler(reset);
			reset(null, EventArgs.Empty);

			audio = new DirectSound.Device();
			audio.SetCooperativeLevel(this, DirectSound.CooperativeLevel.Priority);
			stream = assembly.GetManifestResourceStream("Plonk.wall.wav");
			wallSound = new DirectSound.SecondaryBuffer(stream, audio);
			stream.Close();
			stream = null;
			stream = assembly.GetManifestResourceStream("Plonk.paddle.wav");
			paddleSound = new DirectSound.SecondaryBuffer(stream, audio);
			stream.Close();
			stream = null;

			// Haben wir einen hinreichend ausgestatteten Force-Feedback-Joystick?
			DirectInput.DeviceList dl = DirectInput.Manager.GetDevices(DirectInput.DeviceType.Joystick,
				DirectInput.EnumDevicesFlags.AttachedOnly | DirectInput.EnumDevicesFlags.ForceFeeback);
			if(dl.Count > 0)
			{
				dl.GetEnumerator().MoveNext(); // nimm den ersten Joystick
				DirectInput.DeviceInstance di = (DirectInput.DeviceInstance)dl.Current;
				input = new DirectInput.Device(di.InstanceGuid);
				input.SetDataFormat(DirectInput.DeviceDataFormat.Joystick);
				input.Properties.AutoCenter = false;
				input.SetCooperativeLevel(this, DirectInput.CooperativeLevelFlags.Foreground | DirectInput.CooperativeLevelFlags.Exclusive);

				DirectInput.EffectList el = input.GetEffects(DirectInput.EffectType.ConstantForce);
				if(el.Count > 0)
				{
					useFFJoystick = true;

					el.GetEnumerator().MoveNext(); // nimm den ersten Constant-Force-Effekt
					DirectInput.EffectInformation ei = (DirectInput.EffectInformation)el.Current;

					int axis1Offset = 0;
					int axis2Offset = 0;
					bool foundAxis1 = false;
					bool foundAxis2 = false;
					foreach (DirectInput.DeviceObjectInstance oi in input.GetObjects(DirectInput.DeviceObjectTypeFlags.All))
					{
						// die Force-Feedback-Achsen merken
						if ( (oi.ObjectId & (int)DirectInput.DeviceObjectTypeFlags.Axis) != 0
								&& (oi.Flags & (int)DirectInput.ObjectInstanceFlags.Actuator) != 0)
						{
							if (!foundAxis1)
							{
								axis1Offset = oi.Offset;
								foundAxis1 = true;
							}
							else if(!foundAxis2)
							{
								axis2Offset = oi.Offset;
								foundAxis2 = true;
							}
						}

						// den Bereich der xy-Koordinaten einstellen
						if( (oi.ObjectId & (int)DirectInput.DeviceObjectTypeFlags.Axis) != 0
							&& (oi.Flags & (int)DirectInput.ObjectInstanceFlags.Position) != 0)
						{
							input.Properties.SetRange(DirectInput.ParameterHow.ById, oi.ObjectId,
								new DirectInput.InputRange(-10000, 10000));
						}
					}

					forceEffect = new DirectInput.Effect();
					forceEffect.SetAxes(new int[]{axis1Offset, axis2Offset});
					forceEffect.SetDirection(new int[2]);
					forceEffect.EffectType = (DirectInput.EffectType)ei.EffectType;
					forceEffect.Constant.Magnitude = 10000;
					forceEffect.Duration = 100000; // Mikrosekunden
					forceEffect.SamplePeriod = 0;
					forceEffect.Gain = 10000;
					forceEffect.Flags = DirectInput.EffectFlags.ObjectOffsets | DirectInput.EffectFlags.Polar;
					forceEffect.TriggerButton = (int)DirectInput.Button.NoTrigger;
					forceEffect.TriggerRepeatInterval = (int)DirectInput.DI.Infinite;
					forceEffect.ConditionStruct = new DirectInput.Condition[2];
					forceEffect.UsesEnvelope = false;

					forceEffectObject = new DirectInput.EffectObject(ei.EffectGuid, forceEffect, input);
				}
			}

			if(useFFJoystick)
			{
				Trace.WriteLine("Using the joystick for control.");
			}
			else
			{
				Trace.WriteLine("Using the mouse for control.");
				if(input != null)
				{
					input.Dispose();
				}
				input = new DirectInput.Device(DirectInput.SystemGuid.Mouse);
				input.SetCooperativeLevel(this, DirectInput.CooperativeLevelFlags.Foreground | DirectInput.CooperativeLevelFlags.Exclusive);
				input.SetDataFormat(DirectInput.DeviceDataFormat.Mouse);
			}
		}

		public void Run()
		{
			Application.Idle += new EventHandler(idle);
			Application.Run(this);
		}

		[StructLayout(LayoutKind.Sequential)]
		private struct Message
		{
			public IntPtr hWnd;
			public int msg;
			public IntPtr wParam;
			public IntPtr lParam;
			public uint time;
			public System.Drawing.Point p;
		}

		[System.Security.SuppressUnmanagedCodeSecurity]
		[DllImport("User32.dll", CharSet=CharSet.Auto)]
		private static extern bool PeekMessage(out Message msg, IntPtr hWnd, uint messageFilterMin, uint messageFilterMax, uint flags);

		private void idle(object sender, EventArgs e)
		{
			Message msg;
			while (!PeekMessage(out msg, IntPtr.Zero, 0, 0, 0))
			{
				animate();
				render();
			}
		}

		protected override void OnPaint(PaintEventArgs e) // hier nur zum Neuzeichnen whrend Fenstergrennderung
		{
			render();
		}

		private void reset(object o, System.EventArgs e)
		{
			device.Lights[0].Type = LightType.Directional;
			device.Lights[0].Direction = new Vector3(0.7f, -2.0f, 1.0f);
			device.Lights[0].Ambient = Color.Black;
			device.Lights[0].Diffuse = Color.White;
			device.Lights[0].Specular = Color.White;
			device.Lights[0].Update();
			device.Lights[0].Enabled = true;

			device.Lights[1].Type = LightType.Directional;
			device.Lights[1].Direction = new Vector3(-0.9f, -1.0f, 2.0f);
			device.Lights[1].Ambient = Color.Black;
			device.Lights[1].Diffuse = Color.FromArgb(20, 40, 0);
			device.Lights[1].Specular = Color.Black;
			device.Lights[1].Update();
			device.Lights[1].Enabled = true;

			device.RenderState.Lighting = true;
			device.RenderState.SpecularEnable = true;
			device.RenderState.Ambient = Color.DarkSlateBlue;
		}

		private void animate()
		{
			double timeDifference;
			if(useHighResTimer)
			{
				long timerCount = oldTimerCount;
				QueryPerformanceCounter(ref timerCount);
				timeDifference = (timerCount-oldTimerCount)/(double)timerFreq;
				oldTimerCount = timerCount;
			}
			else
			{
				DateTime t = DateTime.Now;
				timeDifference = (t-oldTime).TotalSeconds;
				oldTime = t;
			}

			rotationAngle += rotationSpeed * (float)timeDifference;

			if(hasToAcquire)
			{
				try
				{
					input.Acquire();
					isAcquired = true;
					hasToAcquire = false;
				}
				catch(DirectInput.OtherApplicationHasPriorityException) // dann eben beim nchsten Durchgang ...
				{}
			}

			if(isAcquired)
			{
				input.Poll();
				if(useFFJoystick)
				{
					DirectInput.JoystickState s = input.CurrentJoystickState;
					paddlePosition = 0.0001f * new Vector2(s.X, -s.Y);
				}
				else
				{
					DirectInput.MouseState s = input.CurrentMouseState;
					paddlePosition += 0.005f * new Vector2(s.X, -s.Y);
				}
			}

			paddlePosition.X = Math.Max(0.0f, Math.Min(1.5f, paddlePosition.X));
			paddlePosition.Y = Math.Max(-1.0f + paddleHeight/2.0f, Math.Min(1.0f - paddleHeight/2.0f, paddlePosition.Y));
			paddleVelocity = 1.0f/(0.001f + (float) timeDifference) * (paddlePosition - oldPaddlePosition);
			oldPaddlePosition = paddlePosition;

			ballPosition += ballVelocity *(float)timeDifference;
			ballPosition.X = Math.Max(Math.Min(ballPosition.X, 10000.0f), -10000.0f); // Bereich begrenzen
			ballPosition.Y = Math.Max(Math.Min(ballPosition.Y, 10000.0f), -10000.0f);
		
			if(ballPosition.X < -1.0f + ballRadius)
			{
				ballVelocity.X *= -1.0f;
				ballPosition.X = -1.0f + ballRadius + epsilon;
				wallSound.SetCurrentPosition(0);
				wallSound.Play(0, DirectSound.BufferPlayFlags.Default);
			}
			else if(ballPosition.X > paddlePosition.X - ballRadius
				&& ballPosition.X < paddlePosition.X + width + ballRadius
				&& Math.Abs(ballPosition.Y - paddlePosition.Y) < paddleHeight/2.0f)
				// Kollisionen werden nur auf der Vorderseite des Schlgers richtig behandelt
			{
				ballVelocity.X = 0.7f * paddleVelocity.X - ballVelocity.X; // etwas Impulsbertrag
				ballVelocity.X = Math.Max(Math.Min(ballVelocity.X, 1000.0f), -1000.0f); // Bereich begrenzen 
				// Ball aus dem Schlger herausbewegen und noch etwas geschwindigkeitsabh. Vorsprung geben
				ballPosition.X = paddlePosition.X - ballRadius - epsilon
					+ 2.0f*paddleVelocity.X*(float)timeDifference;
				paddleSound.SetCurrentPosition(0);
				paddleSound.Play(0, DirectSound.BufferPlayFlags.Default);
				if(useFFJoystick && isAcquired)
				{
					Vector2 hit = ballVelocity - paddleVelocity;
					double angle = 180.0/Math.PI*Math.Atan2(hit.X, -hit.Y);
					if(angle < 0.0) // Atan2 liefert auch negative Winkel
					{
						angle += 360.0;
					}
					forceEffect.SetDirection(new int[]{(int)(100.0*angle), 0});
					forceEffect.Gain = (int)Math.Min(5000.0*hit.Length(), Int32.MaxValue);
					forceEffectObject.SetParameters(forceEffect,
						DirectInput.EffectParameterFlags.Direction | DirectInput.EffectParameterFlags.Gain);
					forceEffectObject.Start(1);
				}
			}

			if(ballPosition.Y > 1.0f - ballRadius && ballPosition.X < 1.0f + ballRadius)
			{
				ballPosition.Y = 1.0f - ballRadius - epsilon;
				ballVelocity.Y *= -1.0f;
				wallSound.SetCurrentPosition(0);
				wallSound.Play(0, DirectSound.BufferPlayFlags.Default);
			}
			else if(ballPosition.Y < -1.0f + ballRadius && ballPosition.X < 1.0f + ballRadius)
			{
				ballPosition.Y = -1.0f + ballRadius + epsilon;
				ballVelocity.Y *= -1.0f;
				wallSound.SetCurrentPosition(0);
				wallSound.Play(0, DirectSound.BufferPlayFlags.Default);
			}
		}

		private void render()
		{
			float cameraDistance = (1.0f + width)/(float)Math.Tan(fieldOfView/2.0);
			device.Transform.View = Matrix.LookAtLH(new Vector3(camera.X, camera.Y, -cameraDistance - depth/2.0f),
				new Vector3(), new Vector3(0.0f, 1.0f, 0.0f));
			device.Transform.Projection = Matrix.PerspectiveFovLH(fieldOfView,
				ClientRectangle.Width/(0.001f+ClientRectangle.Height),
				0.5f*cameraDistance, 1.1f*(cameraDistance + depth + camera.Length()));  
			
			device.Clear(ClearFlags.Target | ClearFlags.ZBuffer, Color.Black, 1.0f, 0);
			device.BeginScene();

			device.Material = wallMaterial;
			device.Transform.World = Matrix.Translation(0.0f, 1.0f + width/2.0f, 0.0f);
			wallHorizontal.DrawSubset(0);
			device.Transform.World = Matrix.Translation(0.0f, -1.0f - width/2.0f, 0.0f);
			wallHorizontal.DrawSubset(0);
			device.Transform.World = Matrix.Translation(-1.0f - width/2.0f, 0.0f, 0.0f);
			wallVertical.DrawSubset(0);

			device.Material = paddleMaterial;
			device.Transform.World = Matrix.Translation(paddlePosition.X + width/2.0f, paddlePosition.Y, 0.0f);
			paddle.DrawSubset(0);

			device.SetTexture(0, ballTexture);
			device.Material = ballMaterial;
			device.Transform.World = Matrix.RotationAxis(rotationAxis, rotationAngle)
				* Matrix.Translation(ballPosition.X, ballPosition.Y, 0.0f);
			ball.DrawSubset(0);
			device.SetTexture(0, null);

			drawShadows();

			device.EndScene();
			device.Present();
		}

		private void drawSingleShadow(double distance, Vector4 direction, Plane plane)
		{
			int attenuation = (int)(0.5 + 255.0*Math.Min(1.0, Math.Max(0.0, distance/shadowRange)));
			device.RenderState.BlendFactor = Color.FromArgb(attenuation, attenuation, attenuation);
			Matrix shadowMatrix = new Matrix();
			shadowMatrix.Shadow(direction, plane);
			device.Transform.World = Matrix.Translation(ballPosition.X, ballPosition.Y, 0.0f) * shadowMatrix;
			ball.DrawSubset(0);
		}

		private void drawShadows()
		{
			device.RenderState.AlphaBlendEnable = true;
			device.RenderState.BlendOperation = BlendOperation.Add;
			device.RenderState.SourceBlend = Blend.Zero;
			device.RenderState.DestinationBlend = Blend.BlendFactor;
			
			// auf den horizontalen Wnden
			if(ballPosition.X < 1.0f - ballRadius) // kein freischwebender Schatten jenseits der Wnde
			{
				drawSingleShadow(1.0 + ballPosition.Y,
					new Vector4(0.0f, -1.0f, 0.0f, 0.0f), new Plane(0.0f, -1.0f, 0.0f, -1.0f+0.01f));

				drawSingleShadow(1.0 - ballPosition.Y,
					new Vector4(0.0f, -1.0f, 0.0f, 0.0f), new Plane(0.0f, -1.0f, 0.0f, 1.0f-0.01f));
			}

			// auf der vertikalen Wand
			drawSingleShadow(1.0 + ballPosition.X,
				new Vector4(-1.0f, 0.0f, 0.0f, 0.0f), new Plane(-1.0f, 0.0f, 0.0f, -1.0f+0.01f));

			// auf dem Schlger
			if(ballPosition.X < paddlePosition.X
				&& Math.Abs(ballPosition.Y - paddlePosition.Y) < paddleHeight/2.0f)
			{
				drawSingleShadow(paddlePosition.X - ballPosition.X,
					new Vector4(-1.0f, 0.0f, 0.0f, 0.0f), new Plane(-1.0f, 0.0f, 0.0f, paddlePosition.X-0.01f));
			}

			device.RenderState.AlphaBlendEnable = false;
		}
	
		protected override void OnKeyDown(KeyEventArgs e)
		{
			if(!e.Handled)
			{
				e.Handled = true;
				switch(e.KeyData)
				{
					case Keys.Escape:
						Application.Exit();
						break;
					case Keys.B:
						ballPosition = new Vector2(-0.5f, 0.0f);
						ballVelocity = new Vector2(-0.5f, 0.3f);
						break;
					case Keys.PageUp:
						fieldOfView *= 1.1f;
						if(fieldOfView > (float)Math.PI/2.0f)
						{
							fieldOfView = (float)Math.PI/2.0f;
						}
						break;
					case Keys.PageDown:
						fieldOfView /= 1.1f;
						if(fieldOfView < 0.01f)
						{
							fieldOfView = 0.01f;
						}
						break;
					case Keys.Up:
						camera.Y += 0.1f;
						break;
					case Keys.Down:
						camera.Y -= 0.1f;
						break;
					case Keys.Left:
						camera.X -= 0.1f;
						break;
					case Keys.Right:
						camera.X += 0.1f;
						break;
					default:
						e.Handled = false;
						break;
				}
			}
			base.OnKeyDown(e);
		}
	
		protected override void OnActivated(EventArgs e)
		{
			base.OnActivated(e);
			hasToAcquire = true; // nicht sofort akquiriert, weil sonst ggf. das Loslassen der Maustaste verloren geht
		}
	
		protected override void OnDeactivate(EventArgs e)
		{
			base.OnDeactivate(e);
			hasToAcquire = false;
			isAcquired = false;
			input.Unacquire();
		}
	}
}
