/* $Id: CFileNTFS.cpp 776 2006-04-27 10:38:18Z olau $ */

// Copyright (c) 2006 Caspar Gorvin, http://www.gorvin.de/
// Copyright (c) 2006 Heise Zeitschriften Verlag, http://www.heise.de/
// All rights reserved.

#include "stdafx.h"
#include "CFileNTFS.h"

//******************************************************************
// Implementierung der Klasse CFileNTFS zum Lesen von Streams
//******************************************************************

CFileNTFS::CFileNTFS(void)
{
	m_lnOffsetToNextStreamID.QuadPart = 0;
}

CFileNTFS::~CFileNTFS(void)
{
}

HRESULT CFileNTFS::CreateFile(LPCWSTR lpFileName,
                              DWORD dwDesiredAccess,					
                              DWORD dwShareMode,
                              DWORD dwCreationDisposition,
                              DWORD dwFlagsAndAttributes)
{
	// Stream-Variablen initialisieren
	m_lpReadContext = NULL;
	m_lnOffsetToNextStreamID.QuadPart = 0;

	// Datei oder Verzeichnis ffnen
	m_hFile = ::CreateFile(
		lpFileName,					// Dateiname
		dwDesiredAccess,			// gewnschte Zugriffsart
		dwShareMode,				// geteilter Zugriff erwnscht?
		NULL,						// Dateihandle nicht vererbbar
		dwCreationDisposition,		// vorhanden, neu, lschen...
		dwFlagsAndAttributes,		// z. B. Backup-Zugriff...
		NULL);						// ohne Template-File ffnen

	if (m_hFile == INVALID_HANDLE_VALUE) 
	{
		// IO-Fehler, wahrscheinlichste Ursachen: 
		// * Zugriff verweigert mangels Zugriffsrechten
		// * Datei wird von anderem Programm benutzt
		// * Lock auf Datei
		// * Datentrger wurde entfernt
		// * Datentrger ist defekt
		return E_FAIL;
	}

	// Dateinamen fr Testausgaben merken
	StringCchCopy(m_wszFileName, MAX_PATH, lpFileName);

  return S_OK;
}

/// Stream-ID-Struktur lesen
HRESULT CFileNTFS::ReadNextStreamInfo(CFileNTFS::FILE_STREAM_ID* pStreamID)
{
	DWORD dwNumberOfBytesRead;
	LARGE_INTEGER lnBytesToSeek;
	DWORD dwLowBytesSeeked;
	DWORD dwHighBytesSeeked;

	// Setzen des File-Pointers auf den Anfang der nchsten Stream-ID
	if(m_lnOffsetToNextStreamID.QuadPart > 0) 
	{
		// Die Stream-IDs sind an 32-Bit-Grenzen ausgerichtet.
		// Zwischen dem Ende der Nutzdaten des laufende Streams
		// und dem Anfang der nchsten Stream-ID kann also eine 
		// Lcke sein. Der SeekOffset mu gleich oder grer
		// als der Abstand zur nchsten Stream-ID sein, BackupSeek
		// hlt dann genau am Anfang der nchsten Stream-ID an.
		lnBytesToSeek.QuadPart = m_lnOffsetToNextStreamID.QuadPart;		
		BackupSeek(m_hFile, 
			lnBytesToSeek.LowPart, 
			lnBytesToSeek.HighPart,
			&dwLowBytesSeeked, 
			&dwHighBytesSeeked,
			&m_lpReadContext);
	}

	// Lesen der Stream-ID-Struktur
	if (! BackupRead(						
		m_hFile,                      // File Handle
		(LPBYTE)pStreamID,            // Datenziel
		sizeof (WIN32_STREAM_ID) - 4, // Lnge der Stream-ID
		&dwNumberOfBytesRead,         // 
		FALSE,                        // Abort Flag
		FALSE,                        // Security Flag
		&m_lpReadContext))            // Adresse Read-Context
		return E_FAIL;

	if (!dwNumberOfBytesRead)         // Dateiende erreicht
		return S_FALSE;						

	if (pStreamID->dwStreamNameSize > MAX_PATH + 36) // Formatfehler
		return E_FAIL;

	if (pStreamID->dwStreamNameSize)   // Hat der Stream einen Namen?
	{
		// Lesen des Stream-Namens
		if (!BackupRead(							
			m_hFile,                     // File Handle
			(LPBYTE)&(pStreamID->cStreamName[0]),	// Datenziel
			pStreamID->dwStreamNameSize, // Lnge
			&dwNumberOfBytesRead,        // 
			FALSE,                       // Abort Flag
			FALSE,                       // Security Flag (ACL restore flag)
			&m_lpReadContext))
			return E_FAIL;

		if (dwNumberOfBytesRead != pStreamID->dwStreamNameSize)	
			return E_FAIL;			// Fehler beim Lesen des Namens

		// Null-Byte an den Namen anhngen
		pStreamID->cStreamName[pStreamID->dwStreamNameSize / 
			sizeof (WCHAR)] = (WCHAR) 0; 

#ifdef _DEBUG
		const unsigned int MAX_TEXT = 1000;
		WCHAR wszMsg [MAX_TEXT];
		StringCchPrintf(wszMsg, MAX_TEXT, 
			L"Aufruf der Funktion: %s\n\n%s \"%s\"\n%s \"%s\"",
			_T(__FUNCTION__),
			L"Dateiname:", m_wszFileName,
			L"Streamname und Typ:", pStreamID->cStreamName);
		MessageBox(NULL, wszMsg, L"Projekt ctShellExtension", 
			MB_OK | MB_ICONINFORMATION);
#endif
	}

	// Abstand zur nchsten Stream-ID in der Datei berechnen,
	// dabei 32-Bit-Ausrichtung der Stream-IDs bercksichtigen
	m_lnOffsetToNextStreamID.QuadPart = pStreamID->Size.QuadPart;
	if (pStreamID->Size.LowPart & 0x03)
		m_lnOffsetToNextStreamID.QuadPart += 
		4 - pStreamID->Size.LowPart & 0x03; 

	return S_OK;
}

/// Datei schlieen
HRESULT CFileNTFS::CloseFile()
{
	// Read-Context freigeben
	if (m_lpReadContext)
		BackupRead(				
		m_hFile,			// File Handle
		NULL,				// nicht benutzt
		0,					// nicht benutzt
		NULL,				// nicht benutzt
		TRUE,				// Abort Flag: Context freigeben
		FALSE,				// nicht benutzt
		&m_lpReadContext);	// Adresse des Read-Context-Pointers

	CloseHandle(m_hFile);
	return S_OK;
}
