using System;

namespace Sudoku
{
    public partial class SudokuField
    {
        private static Random r = new Random();
        private SudokuField UniquePuzzle;
        private int DeleteIterations, MinDigitCount, UniquePuzzleDigitCount;

        /// <summary>
        /// Generates a new Sudoku puzzle with a unique solution.
        /// </summary>
        /// <param name="MinNumberOfDigits">Requested number of given digits</param>
        public SudokuField(int MinNumberOfDigits)
        {
            Digits = new int[nn, nn];
            Lists = new SudokuList[nn, nn];
            SudokuStatus Status;
            do
            {
                Clear();
                int c = 0;
                while (c != nn * nn / 4)          // nn == 9 fills 20 cells with digits
                {
                    int i = r.Next(nn), j = r.Next(nn), Digit = r.Next(nn) + 1;
                    if (Digits[i, j] == 0 && TestRules(i, j, Digit))
                    {
                        Digits[i, j] = Digit;
                        c++;
                    }
                }
                Status = GetStatus(10 * nn * nn); // until it is solvable (unique or not)
            }
            while (Status != SudokuStatus.OneSolution && Status != SudokuStatus.MoreSolutions ||
                   AddDigits() != SudokuStatus.OneSolution);  // make it unique

            SudokuField ShortestUniquePuzzle = new SudokuField(UniquePuzzle);
            int Tries = 15;
            while (true)
            {
                Copy(UniquePuzzle);
                if (DeleteDigits(MinNumberOfDigits) != SudokuStatus.OneSolution)
                    break;
                if (UniquePuzzle.DigitCount < ShortestUniquePuzzle.DigitCount)
                    ShortestUniquePuzzle.Copy(UniquePuzzle);
                if (ShortestUniquePuzzle.DigitCount <= MinNumberOfDigits || --Tries < 0)
                    break;                       // Sudoku small enough or timeout

                Copy(ShortestUniquePuzzle);
                int i, j;
                do
                {
                    i = r.Next(nn);
                    j = r.Next(nn);
                }
                while (Digits[i, j] == 0);
                Digits[i, j] = 0;                // delete one Digit and make it unique again
                if (AddDigits() != SudokuStatus.OneSolution)
                    break;
            }
            Copy(ShortestUniquePuzzle);
            ShortestUniquePuzzle.Solve();        // returns SudokuStatus.OneSolution always
            while (DigitCount < MinNumberOfDigits)
            {
                int i, j;
                do
                {
                    i = r.Next(nn);
                    j = r.Next(nn);
                }
                while (Digits[i, j] != 0);
                Digits[i, j] = ShortestUniquePuzzle.Digits[i, j];
            }
            InitLists();
        }

        /// <summary>
        /// Adds digits to a SudokuStatus.MoreSolutions Sudoku until it is 
        /// SudokuStatus.OneSolution (unique).
        /// </summary>
        /// <returns>OneSolution, Timeout (result is in UniquePuzzle)</returns>
        private SudokuStatus AddDigits()
        {
            SudokuStatus Status = GetStatus(10 * nn * nn);
            if (Status == SudokuStatus.OneSolution)
            {
                UniquePuzzle = new SudokuField(this);
                return Status;
            }
            if (Status == SudokuStatus.MoreSolutions)
            {
                int i = r.Next(nn), j = r.Next(nn), i0 = i, j0 = j;
                do
                {
                    if (Solution.Digits[i, j] != SecondSolution.Digits[i, j])
                    {
                        Digits[i, j] = Solution.Digits[i, j];
                        Status = AddDigits();
                        if (Status == SudokuStatus.OneSolution ||
                            Status == SudokuStatus.Timeout)
                            return Status;
                        Digits[i, j] = 0;
                    }
                    i = (i + (j + 1) / nn - j / nn) % nn;
                    j = (j + 1) % nn;
                } while (i != i0 || j != j0);
            }
            return Status;
        }

        /// <summary>
        /// Deletes digits from a SudokuStatus.OneSolution sudoku as long as it
        /// stays unique and MinNumberOfDigits is not reached. 
        /// </summary>
        /// <param name="MinNumberOfDigits">Requested number of given digits</param>
        /// <returns>OneSolution, Timeout (result is in UniquePuzzle)</returns>
        private SudokuStatus DeleteDigits(int MinNumberOfDigits)
        {
            UniquePuzzle = null;
            DeleteIterations = 3 * nn * nn;
            MinDigitCount = MinNumberOfDigits;
            DeleteDigits(DigitCount, 0, 0);
            return UniquePuzzle != null ? SudokuStatus.OneSolution : SudokuStatus.Timeout;
        }

        private SudokuStatus DeleteDigits(int DigitCount, int i, int j)
        {
            if (--DeleteIterations <= 0)
                return SudokuStatus.Timeout;
            SudokuStatus Status = GetStatus(2 * nn * nn);  // shorter Timeout to save time
            if (Status == SudokuStatus.Timeout)            // treat Timeout like MoreSolutions
                Status = SudokuStatus.MoreSolutions;

            if (Status == SudokuStatus.OneSolution)
            {
                if (UniquePuzzle == null || DigitCount < UniquePuzzleDigitCount)
                {
                    UniquePuzzle = new SudokuField(this);    // take this as best result
                    UniquePuzzleDigitCount = UniquePuzzle.DigitCount;
                    if (UniquePuzzleDigitCount <= MinDigitCount)
                        return SudokuStatus.Timeout;
                }
                do
                {
                    if (Digits[i, j] != 0)
                    {
                        int Digit = Digits[i, j];
                        Digits[i, j] = 0;
                        if (DeleteDigits(DigitCount - 1, i, j) == SudokuStatus.Timeout)
                            return SudokuStatus.Timeout;
                        Digits[i, j] = Digit;
                    }
                    i = (i + (j + 1) / nn - j / nn) % nn;
                    j = (j + 1) % nn;
                } while (i != 0 || j != 0);
            }
            return Status;
        }
    }
}
