/* $Id: bzr.c 1332 2006-11-30 14:47:26Z olau $ 
 *
 * Copyright (c) 2006 Oliver Lau <ola@ctmagazin.de>
 * Copyright (c) 2006 Heise Zeitschriften Verlag
 * Alle Rechte vorbehalten. All rights reserved.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <mpi.h>
#include <gd.h>

#include "globaldefs.h"
#include "helper.h"


#define ROOT                 (0)

// Tag fuer Uebertragung des Halos zu den Nachbarn
#define TAG_NEIGHBORS      0x23

// Tag fuer Uebertragung der Teilwelten zu den Prozessen und wieder zurueck
#define TAG_DIST_GATHER    0x42

// Anzahl der zu berechnenden Generationen (Vorgabe)
#define DEFAULT_T          (100)

// Anzahl der Generationen, nach deren Berechnung ein PNG erzeugt werden soll
#define DEFAULT_TICK         (1)

// Breite und Hoehe der Welt
#define WIDTH             (1280/2)
#define HEIGHT             (960/2)

// maximaler Wert einer Zelle
#define MAX_CELL_VALUE     (255)

// Einfluss erregter Zellen in der Nachbarschaft
// auf den Zustand einer Zelle 
#define DEFAULT_K1           (3)

// Einfluss aktiver Zellen in der Nachbarschaft
// auf den Zustand einer Zelle 
#define DEFAULT_K2           (3)

// Ausbreitungsgeschwindigkeit der Erregungswelle
#define DEFAULT_G           (41)

// Konstanten fuer Moore'sche Nachbarschaft
const int num_neighbors = 8;
enum { N = 0, S, E, W, NW, SE, NE, SW };

#define MAX(a, b) ((a) > (b)? (a) : (b))
#define MIN(a, b) ((a) < (b)? (a) : (b))


typedef struct _direction_t {
    int send_rank;   // ID des Prozesses, an den der aktuelle Daten sendet
    int send_offset; // Offset ins Array sec0 fuer die gesendeten Daten
    int recv_rank;   // ID des Prozesses, von dem der aktuellen Daten empfaengt
    int recv_offset; // Offset ins Array sec0 fuer die empfangenen Daten
    MPI_Datatype type;
} direction_t;


/** Eine Iteration der BZR durchfuehren. (Vgl. A.K. Dewdney, Wellen
 * aus der Computer-Retorte, Spektrum der Wissenschaft, Sonderheft 8:
 * Computer-Kurzweil III, 1989)
 */
void BZR_iterate(int *cell, int *cell_new,
                 int width, int height,
                 int k1, int k2, int g, int n) {
    for (int y = 0; y < height; ++y) {
        for (int x = 0; x < width; ++x) {
            int c = cell[x + y * (width+2)];
            int S = c;
            int excited = 0;
            int active = 0;
            for (int yy = -1; yy <= 1; ++yy) {
                for (int xx = -1; xx <= 1; ++xx) {
                    int v = cell[(x + xx) + (y + yy) * (width+2)];
                    S += v;
                    if (v == n)
                        ++active;
                    else if (v != 0)
                        ++excited;
                }
            }

            int c_new;
            if (c == 0)
                c_new = excited / k1 + active / k2;
            else if (c == n)
                c_new = 0;
            else 
                c_new = S / (excited + active + 1) + g;

            if (c_new > n)
                c_new = n;

            cell_new[x + y * (width+2)] = c_new;
        }
    }
}


// hier gehts los ...
int main(int argc, char *argv[]) {
    int size; // Anzahl der Jobs
    int myrank; // Job-ID
    int width; // Breite des Weltausschnitts
    int height; // Hoehe des Weltausschnitts
    int dims[2] = { 0, 0 }; // Anzahl Spalten/Zeilen je Dimension
    int periods[2] = { 1, 1 }; // Welt ist in beiden Dimensionen periodisch
    int cart_coords[2], coords[2];
    int *sec0, *sec0_o, *sec1, *sec1_o;
    int *tmpworld, *world = NULL;
    direction_t direction[8];
    MPI_Status stat;
    MPI_Request req;
    MPI_Comm comm;

    // Konstanten fuer Belousov-Zhabotinsky-Reaktion (BZR)
    int k1 = DEFAULT_K1;
    int k2 = DEFAULT_K2;
    int g  = DEFAULT_G;
    int n  = 255;
  
    // Ausmasse der Welt
    int globalwidth = WIDTH;
    int globalheight = HEIGHT;

    int colors[MAX_CELL_VALUE+1];
    FILE *pngout;
    char pngname[100];
    int pngsize;
    BYTE *pngbuf = 0;
    gdImagePtr im = 0;
    
    int tmax = DEFAULT_T;
    if (argc > 1)
        tmax = atoi(argv[1]);

    int ttick = DEFAULT_TICK;
    if (argc > 2)
        ttick = atoi(argv[2]);

    // MPI-Umgebung initialisieren
    MPI_Init(&argc, &argv);

    // Anzahl der Prozesse im Kommunikator ermitteln
    MPI_Comm_size(MPI_COMM_WORLD, &size);

    // Anzahl der Abschnitte berechnen, in die die Welt in
    // horizontaler und vertikaler Richtung zerlegt werden soll
    MPI_Dims_create(size, 2, dims);

    // Einen neuen Kommunikator mit der 2D-Welt verknuepfen
    MPI_Cart_create(MPI_COMM_WORLD, 2, dims, periods, 1, &comm);

    // eigene ID innerhalb der zweidimensionalen Welt bestimmen
    MPI_Comm_rank(comm, &myrank);

    // Breite und Hoehe der Welt muessen durch die Anzahl 
    // der Abschnitte in jeder Dimension teilbar sein, damit
    // die Abschnitte nahtlos aneinander liegen
    globalwidth  -= globalwidth  % dims[0];
    globalheight -= globalheight % dims[1];

    // Die Breite eines Abschnitts ergibt sich aus der Breite 
    // bzw. Hoehe der Welt geteilt durch die Anzahl der Abschnitte
    // in der jeweiligen Dimension
    width  = globalwidth  / dims[0];
    height = globalheight / dims[1];

    // Horizontales Halo
    MPI_Datatype row_type;
    MPI_Type_vector(1, width, width, MPI_INT, &row_type);
    MPI_Type_commit(&row_type);

    // Vertikales Halo
    MPI_Datatype column_type;
    MPI_Type_vector(height, 1, width+2, MPI_INT, &column_type);
    MPI_Type_commit(&column_type);

    // Rechteckiger Ausschnitt aus der Welt
    MPI_Datatype submatrix_type;
    MPI_Type_vector(height, width, globalwidth, MPI_INT, &submatrix_type);
    MPI_Type_commit(&submatrix_type);

    // Rechteckiger Ausschnitt der Welt
    MPI_Datatype section_type;
    MPI_Type_vector(height, width, width+2, MPI_INT, &section_type);
    MPI_Type_commit(&section_type);

    // Speicher fuer eigenen Ausschnitt inklusive Halo belegen
    sec0_o = (int *) safe_malloc((width+2)*(height+2)*sizeof(*sec0));
    sec0 = sec0_o + 1 + (width + 2);
    sec1_o = (int *) safe_malloc((width+2)*(height+2)*sizeof(*sec1));
    sec1 = sec1_o + 1 + (width + 2);

    if (myrank == ROOT) {
        printf("Erzeugen der 2D-Welt mit %d Abschnitten: dims[]=(%d, %d)\n",
               size, dims[0], dims[1]);
        world = safe_malloc(globalwidth * globalheight * sizeof(*world));
        // Welt initialisieren
        srand((unsigned int) time(0));
        for (int y = 0; y < globalheight; ++y)
            for (int x = 0; x < globalwidth; ++x)
                world[x + y * globalwidth] = rand() % (MAX_CELL_VALUE+1);

        im = gdImageCreate(globalwidth, globalheight);
        for (int i = 0; i < MAX_CELL_VALUE+1; ++i) 
            colors[i] = gdImageColorAllocate(im, i, i, 255 - i);
        printf("Berechnen von %d Iterationen ..\n", tmax);
    }

#ifdef DEBUG
    printf("Job %d arbeitet an Block [ %d, %d ]\n",
           myrank, cart_coords[0], cart_coords[1]);
#endif

    MPI_Cart_shift(comm, 0, +1, &direction[W].send_rank, &direction[W].recv_rank);
    direction[W].type = column_type;
    direction[W].send_offset = 0;
    direction[W].recv_offset = width;

    MPI_Cart_shift(comm, 0, -1, &direction[E].send_rank, &direction[E].recv_rank);
    direction[E].type = column_type;
    direction[E].send_offset = width - 1;
    direction[E].recv_offset = -1;

    MPI_Cart_shift(comm, 1, +1, &direction[N].send_rank, &direction[N].recv_rank);
    direction[N].type = row_type;
    direction[N].send_offset = 0;
    direction[N].recv_offset = (width + 2) * height;

    MPI_Cart_shift(comm, 1, -1, &direction[S].send_rank, &direction[S].recv_rank);
    direction[S].type = row_type;
    direction[S].send_offset = (width + 2) * (height - 1);
    direction[S].recv_offset = -(width + 2);

    // Die kartesischen Koordinaten des zugeteilten Abschnitts ermitteln
    MPI_Cart_coords(comm, myrank, 2, cart_coords);

    coords[0] = cart_coords[0] - 1;
    coords[1] = cart_coords[1] - 1;
    MPI_Cart_rank(comm, coords, &direction[NW].send_rank);
    MPI_Cart_rank(comm, coords, &direction[SE].recv_rank);
    direction[NW].type = MPI_INT;
    direction[NW].send_offset = 0;
    direction[NW].recv_offset = (width + 2) * height + width;

    coords[0] = cart_coords[0] + 1;
    coords[1] = cart_coords[1] - 1;
    MPI_Cart_rank(comm, coords, &direction[NE].send_rank);
    MPI_Cart_rank(comm, coords, &direction[SW].recv_rank);
    direction[NE].type = MPI_INT;
    direction[NE].send_offset = width - 1;
    direction[NE].recv_offset = (width + 2) * height - 1;

    coords[0] = cart_coords[0] - 1;
    coords[1] = cart_coords[1] + 1;
    MPI_Cart_rank(comm, coords, &direction[SW].send_rank);
    MPI_Cart_rank(comm, coords, &direction[NE].recv_rank);
    direction[SW].type = MPI_INT;
    direction[SW].send_offset = (width + 2) * (height - 1);
    direction[SW].recv_offset = -2;

    coords[0] = cart_coords[0] + 1;
    coords[1] = cart_coords[1] + 1;
    MPI_Cart_rank(comm, coords, &direction[SE].send_rank);
    MPI_Cart_rank(comm, coords, &direction[NW].recv_rank);
    direction[SE].type = MPI_INT;
    direction[SE].send_offset = (width + 2) * (height - 1) + width - 1;
    direction[SE].recv_offset = -1 - (width + 2);

    // Welt abschnittsweise an Knoten verteilen
    MPI_Irecv(sec0, 1, section_type, ROOT, TAG_DIST_GATHER, comm, &req);
    if (myrank == ROOT) {
        int xy[2], rank;
        for (xy[1] = 0; xy[1] < dims[1]; ++xy[1]) {
            for (xy[0] = 0; xy[0] < dims[0]; ++xy[0]) {
                MPI_Cart_rank(comm, xy, &rank);
                MPI_Send(world + xy[0] * width + xy[1] * height * globalwidth,
                         1, submatrix_type, rank, TAG_DIST_GATHER,
                         comm);
            }
        }
    }
    // auf Ende von MPI_Irecv() warten
    MPI_Wait(&req, &stat);

    int t = 0;
    double t_sum = 0.0;
    double t_max = -1e38;
    double t_min = 1e38;
    do {
        if (myrank == ROOT) {
            printf("\r%7d ", t);
            fflush(stdout);
            
            sprintf(pngname, "bzr-%06d.png", t);
            pngout = fopen(pngname, "wb+");
            if (pngout == 0)
                MPI_Abort(MPI_COMM_WORLD, EXIT_FAILURE);
            
            for (int x = 0; x < globalwidth; ++x)
                for (int y = 0; y < globalheight; ++y)
                    gdImageSetPixel(im, x, y,
                                    colors[world[x + y * globalwidth]]);
            
            pngbuf = (BYTE *) gdImagePngPtr(im, &pngsize);
            fwrite(pngbuf, sizeof(BYTE), pngsize, pngout);
            fclose(pngout);
            gdFree((void *) pngbuf);
        }

        double t0 = MPI_Wtime();
        for (int dt = 0; dt < ttick; ++dt) {
            for (int i = 0; i < num_neighbors; ++i) {
                MPI_Sendrecv(sec0 + direction[i].send_offset, 1, direction[i].type,
                             direction[i].send_rank, TAG_NEIGHBORS,
                             sec0 + direction[i].recv_offset, 1, direction[i].type,
                             direction[i].recv_rank, TAG_NEIGHBORS,
                             comm, &stat);
            }
            BZR_iterate(sec0, sec1, width, height, k1, k2, g, n);
            
            // Matrizen tauschen
            tmpworld = sec0;
            sec0 = sec1;
            sec1 = tmpworld;
        }
        t += ttick;

        // Berechnung fertig, Ergebnis an Hauptprozess schicken
        MPI_Isend(sec0, 1, section_type, ROOT, TAG_DIST_GATHER, comm, &req);
        
        // Der Hauptprozess sammelt die Ergebnisse ein ...
        if (myrank == ROOT) {
            int xy[2], rank;
            for (xy[1] = 0; xy[1] < dims[1]; ++xy[1]) {
                for (xy[0] = 0; xy[0] < dims[0]; ++xy[0]) {
                    MPI_Cart_rank(comm, xy, &rank);
                    MPI_Recv(world + xy[0] * width + xy[1] * globalwidth * height,
                             1, submatrix_type, rank, TAG_DIST_GATHER,
                             comm, &stat);
                }
            }
        }

        // Warten, bis Hauptprozess die Daten empfangen hat
        MPI_Wait(&req, &stat);

        t0 = MPI_Wtime() - t0;
        t_max = MAX(t_max, t0);
        t_min = MIN(t_min, t0);
        t_sum += t0;
        if (myrank == ROOT)
            printf("  %10.3lf ms", 1000 * t0);
    }
    while (t < tmax);

    if (myrank == ROOT) {
        gdImageDestroy(im);
        free(world);
        printf("\nmin./mittl./max. Zeit je %d Iteration%s:"
               " %.3lf / %.3lf / %.3lf ms",
               ttick, (ttick == 1)? "" : "en",
               1000 * t_min,
               1000 * t_sum / tmax * ttick,
               1000 * t_max);
        printf("\nFertig.\n");
    }

    /* Aufraeumen */
    free(sec0_o);
    free(sec1_o);
    MPI_Type_free(&submatrix_type);
    MPI_Type_free(&section_type);
    MPI_Type_free(&column_type);
    MPI_Type_free(&row_type);
    MPI_Comm_free(&comm);
    MPI_Finalize();
    
    return EXIT_SUCCESS;
}
