// ThreadTest.cpp : Defines the entry point for the console application.
//

#include "stdafx.h"


#pragma region ThreadClassTest
class ConsumerThread : public Thread
{
public:
    ConsumerThread(HANDLE hEvent) : hEvent(hEvent) {}
    ~ConsumerThread() {}

private:
    HANDLE hEvent;

protected:
    THREADRESULT run(void);
};



THREADRESULT ConsumerThread::run(void)
{
    printf("  consumer is waiting .. (thread 0x%x)\n", id());
    WaitForSingleObject(hEvent, INFINITE);
    return 0;
}


class ProducerThread : public Thread
{
public:
    ProducerThread(HANDLE hEvent) : hEvent(hEvent) {}
    ~ProducerThread() {}

private:
    HANDLE hEvent;

protected:
    THREADRESULT run(void);
};


THREADRESULT ProducerThread::run(void)
{
    printf("  producer awakens consumer .. (thread 0x%x)\n", id());
    SetEvent(hEvent);
    return 0;
}

void TestThreadClass()
{
    printf("Testing class Thread ..\n");
    HANDLE hEvent = CreateEvent(NULL, FALSE, FALSE, _T("TestEvent"));

    ConsumerThread *consumerThread = new ConsumerThread(hEvent);
    consumerThread->setName("ConsumerThread");
    consumerThread->start();

    ProducerThread *producerThread = new ProducerThread(hEvent);
    producerThread->setName("ProducerThread");
    producerThread->start();

    producerThread->wait();
    consumerThread->wait();

    delete producerThread;
    delete consumerThread;

    CloseHandle(hEvent);
}


#pragma endregion


struct TILE
{
    HBITMAP hBitmap;
    BITMAP bitmap;
    BITMAPINFO bi;
    double reC;
    double imC;
    double scale;
};


#pragma region ThreadPoolClassTest

class DemoRenderJob : public Job
{
public:
    DemoRenderJob(TILE *tile, bool *restart, int id)
        : m_tile(tile), m_restart(restart), m_iterations(0), m_id(id)
    { }
    int run(void);
    TILE *m_tile;
    bool *m_restart;
    __int64 m_iterations;
    int m_id;

    __int64 iterations() { return m_iterations; }

private:
    MandelbrotSet mandel;
};


int DemoRenderJob::run(void)
{
    printf("RenderJob::run() thread 0x%04x, job %2d, re(c) = %.13lf, im(c) = %.13lf\n",
        GetCurrentThreadId(), m_id, m_tile->reC, m_tile->imC);
    m_iterations = mandel.render(m_tile->reC, m_tile->imC, m_tile->scale, &m_tile->bitmap, m_restart);
    printf("RenderJob::run() thread 0x%04x, job %2d, %I64d iterations\n",
        GetCurrentThreadId(), m_id, m_iterations);
    return 0;
}

void TestThreadPoolClass()
{
    printf("Testing class ThreadPool ..\n");

    bool restart = false;
    const int NumJobs = 16;
    ThreadPool *pool = new ThreadPool(NUMTHREADS);

    //double reC = 0.36380648408020388;
    //double imC = 0.66475706843006455;
    //double scaleFactor = 1.87743e-011;

    int width = 1280;
    int height = 1024;
    const double MinZoom = 1e-3;
    const double MaxZoom = 1e-15;
    const double ZoomInFactor  = 0.94387431268169349664191315666753; // 1/(2^(1/12))
    __declspec(align(16)) double reC = -0.66;
    __declspec(align(16)) double imC =  0.0;
    __declspec(align(16)) double scale;

    TILE *tile = new TILE[NumJobs];
    int tileHeight = height / NumJobs;
    DWORD bpp;
    for (int i = 0; i < NumJobs; ++i)
    {
        tile[i].reC = reC;
        tile[i].hBitmap = CreateBitmap(width, tileHeight, 1, 32, NULL);
        GetObject(tile[i].hBitmap, sizeof(BITMAP), (LPVOID) &tile[i].bitmap);
        bpp = (int) (tile[i].bitmap.bmPlanes * tile[i].bitmap.bmBitsPixel) / 8;
        tile[i].bitmap.bmBits = VirtualAlloc(NULL, bpp * width * tileHeight, MEM_COMMIT, PAGE_READWRITE);
        tile[i].bi.bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
        tile[i].bi.bmiHeader.biWidth = width;
        tile[i].bi.bmiHeader.biHeight = tileHeight;
        tile[i].bi.bmiHeader.biPlanes = tile[i].bitmap.bmPlanes;
        tile[i].bi.bmiHeader.biBitCount = tile[i].bitmap.bmBitsPixel;
        tile[i].bi.bmiHeader.biCompression = BI_RGB;
    }

    for (scale = MinZoom; scale > MaxZoom; scale *= ZoomInFactor)
    {
        for (int i = 0; i < NumJobs; ++i)
        {
            printf("Enqueueing job %2d ..\n", i);
            tile[i].imC = imC + scale * height / 2 - scale * tileHeight / 2 * (i * 2 + 1);
            tile[i].scale = scale;
            DemoRenderJob *renderJob = new DemoRenderJob(&tile[i], &restart, i);
            pool->enqueue(renderJob);
        }

        printf("Executing jobs ..\n");
        pool->run();

        printf("Waiting for jobs to be executed ..\n");
        pool->waitForAll();

        // save bitmap to file ...
    }

    printf("Ready.\n");
    pool->stop();

    for (int i = 0; i < NumJobs; ++i)
    {
        VirtualFree(tile[i].bitmap.bmBits, 0, MEM_RELEASE);
    }
    delete [] tile;
    delete pool;
}

#pragma endregion 


int _tmain(int argc, _TCHAR* argv[])
{
    SetThreadName(-1, "_tmain");
    Stopwatch t;
    t.start();
    TestThreadPoolClass();
    printf("time: %g secs\n", t.elapsed());
    return 0;
}
