// $Id: Threads.cpp 277 2007-04-30 10:31:18Z olau $

#include "stdafx.h"

double sumSpeed = 0.0;
int speedSamples = 0;


// signal worker threads to continue
void SignalWorkerThreads()
{
    for (int i = 0; i < numThreads; ++i)
    {
        if (!SetEvent(param[i].hTriggerEvent))
            fprintf(stderr, "[%s, line %d] SetEvent() failed in function SignalWorkerThreads()\n", __FILE__, __LINE__);
    }
}


void RestartWorkerThreads()
{
    for (int i = 0; i < numThreads; ++i)
    {
        if (param[i].active)
        {
            param[i].restart = true;
        }
        else
        {
            if (!SetEvent(param[i].hTriggerEvent))
                fprintf(stderr, "[%s, line %d] SetEvent() failed in function RestartWorkerThreads()\n", __FILE__, __LINE__);
        }
    }
}


void WaitForWorkerThreads()
{
    WaitForMultipleObjects(numThreads, hReadyEvent, TRUE, INFINITE);
    EnterCriticalSection(&critSecZooming);
    int currentlyZooming = zooming;
    LeaveCriticalSection(&critSecZooming);
    if (!zooming)
        RedrawWindow(hMainWindow, NULL, NULL, RDW_INVALIDATE);
}


static void PrintStatus(double speed)
{
    ++speedSamples;
    sumSpeed += speed;
    double avgSpeed = sumSpeed / speedSamples;
    printf("** %.1lfM iters/s (avg. %.1lfM iters/s)\n"
        "  scale=%g, re(c)=%.17lf, im(c)=%.17lf\n",
        1e-6 * speed, 1e-6 * avgSpeed,
        scaleFactor, centerR, centerI);
}


static double Recalculate()
{
    SignalWorkerThreads();
    WaitForWorkerThreads();
    double speed = CalcItersPerSecond();
    PrintStatus(speed);
    return speed;
}


static double ZoomIn(int multiple)
{
    scaleFactor *= pow(AutomaticZoomInFactor, multiple);
    return Recalculate();
}


static double ZoomOut(int multiple)
{
    scaleFactor *= pow(AutomaticZoomOutFactor, multiple);
    return Recalculate();
}


// this thread controls the automatic zoom in/out
THREADID WINAPI AutomaticZoomThread(LPVOID)
{
    SetThreadName(-1, "AutomaticZoomThread");
    bool forceAbort;
    for (;;)
    {
        while (scaleFactor > MaxZoom)
        {
            EnterCriticalSection(&critSecAbort);
            forceAbort = doAbort;
            LeaveCriticalSection(&critSecAbort);
            if (forceAbort)
            {
                _endthreadex(0);
                return 0;
            }
            ZoomIn(1);
        }
        while (scaleFactor < MinZoom)
        {
            EnterCriticalSection(&critSecAbort);
            forceAbort = doAbort;
            LeaveCriticalSection(&critSecAbort);
            if (forceAbort)
            {
                _endthreadex(0);
                return 0;
            }
            ZoomOut(1);
        }
    }
    _endthreadex(0);
    return 0;
}


THREADID WINAPI TriggerThread(LPVOID params)
{
    TriggerParam *p = (TriggerParam *) params;
    SetThreadName(-1, "TriggerThread");
    bool forceAbort;
    for (;;)
    {
        EnterCriticalSection(&critSecAbort);
        forceAbort = doAbort;
        LeaveCriticalSection(&critSecAbort);
        if (forceAbort)
        {
            _endthreadex(0);
            return 0;
        }

        // wait for new work
        DWORD rc = WaitForSingleObject(hTriggerEvent, INFINITE);
        if (rc != WAIT_OBJECT_0)
            fprintf(stderr, "TriggerThread() line %d: waiting on event failed\n", __LINE__);

        switch (p->action)
        {
        case TriggerPan:
            Recalculate();
            break;
        case TriggerZoom:
            Recalculate();
            zoomDelta = 1.0;
            zooming = NotZooming;
            break;
        default:
            break;
        }
    }
    _endthreadex(0);
    return 0;
}


// this thread renders a tile of the Mandelbrot set
THREADID WINAPI RenderThread(void* params)
{
    MandelbrotSetParam *p = (MandelbrotSetParam *) params;
#ifdef _DEBUG
    const int StrBufMaxLen = 100;
    char strbuf[StrBufMaxLen];
    sprintf_s(strbuf, StrBufMaxLen, "RenderThread %d", p->id);
    SetThreadName(-1, strbuf);
    SetThreadNameViaTIB((LPCTSTR)strbuf);
#endif //_DEBUG
    Stopwatch t(Stopwatch::START_MANUALLY);
    for (;;)
    {
        EnterCriticalSection(&critSecAbort);
        bool doQuit = doAbort;
        LeaveCriticalSection(&critSecAbort);
        if (doQuit)
            break;
        p->active = true;
        MandelbrotSet mandel;
        double imc = centerI + scaleFactor * p->totalHeight / 2 - scaleFactor * p->height / 2 * (p->id * 2 + 1);
        t.start();
        p->restart = false;
        logBuf.addEntry("RenderThread: vor MandelbrotSet::render()");
        p->iterations = mandel.render(centerR, imc, scaleFactor, &p->bitmap, &p->restart);
        logBuf.addEntry("RenderThread: nach MandelbrotSet::render()");
        p->deltaT = t.elapsed();
        p->ready = true;
        // signal that work is done
        if (!SetEvent(hReadyEvent[p->id]))
            fprintf(stderr, "RenderThread() line %d: SetEvent() failed\n", __LINE__);
        p->active = false;
        // wait for new work
        DWORD rc = WaitForSingleObject(p->hTriggerEvent, INFINITE);
        if (rc != WAIT_OBJECT_0)
            fprintf(stderr, "RenderThread() line %d: waiting on event failed\n", __LINE__);
        p->restart = false;
    }
    return 0;
}
