/*
 * addr.c
 * ------
 *
 * "Kernel" functions for address comparison.
 *
 * Copyright (c):
 * 2007:  Joerg MICHAEL, Adalbert-Stifter-Str. 11, 30655 Hannover, Germany
 *
 * SCCS: @(#) addr.c  1.0  2008-11-30
 *
 * This file is subject to the GNU Lesser General Public License (LGPL)
 * (formerly known as GNU Library General Public Licence)
 * as published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this file; if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Actually, the LGPL is __less__ restrictive than the better known GNU General
 * Public License (GPL). See the GNU Library General Public License or the file
 * LIB_GPLA.TXT for more details and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * There is one important restriction: If you modify this program in any way
 * (e.g. modify the underlying logic or translate this program into another
 * programming language), you must also release the changes under the terms
 * of the LGPL.
 * That means you have to give out the source code to your changes,
 * and a very good way to do so is mailing them to the address given below.
 * I think this is the best way to promote further development and use
 * of this software.
 *
 * If you have any remarks, feel free to e-mail to:
 *     ct@ct.heise.de
 *
 * The author's email address is:
 *    astro.joerg@googlemail.com
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "addr_ext.h"



/****  char set = iso8859-1  ****/
#define MAX_FIELDS_ADDR       30

#define GENDER_IS_DIFFERENT    0
#define GENDER_MAY_BE_EQUAL    5
#define GENDER_IS_EQUAL       10

#define POSTF_1             "Postfach "
#define POSTF_2             "POSTFACH "
#define POSTF_3             "POSTF."
#define POSTF_4             "PF "
#define POSTF_5             "PF."
#define POSTF_FORMATTED     "Postf."

#define DORFSTR_1           "Dorfstr."
#define DORFSTR_2           "DORFSTR."
#define DORFSTR_3           "Nr."
#define DORFSTR_4           "NR."

#define STRASSE_1           "Strae "
#define STRASSE_2           "Strasse "
#define STRASSE_FORMATTED   "Str."


/****  first postal address (pointers to variables)  ****/
#define a_gender           (a_pointer_array [IS_GENDER])
#define a_first_name       (a_pointer_array [IS_FIRST_NAME])
#define a_first_name_ph    (a_pointer_array [IS_FIRST_NAME_PHONET])
#define a_fam_name         (a_pointer_array [IS_FAM_NAME])
#define a_fam_name_ph      (a_pointer_array [IS_FAM_NAME_PHONET])
#define a_c_o_name         (a_pointer_array [IS_C_O_NAME])
#define a_street           (a_pointer_array [IS_STREET])
#define a_street_ph        (a_pointer_array [IS_STREET_PHONET])
#define a_zip_code         (a_pointer_array [IS_ZIP_CODE])
#define a_city             (a_pointer_array [IS_CITY])
#define a_country          (a_pointer_array [IS_COUNTRY])
#define a_full_birthday    (a_pointer_array [IS_FULL_BIRTHDAY])
#define a_date_format      (a_pointer_array [IS_DATE_FORMAT])
#define a_b_day            (a_pointer_array [IS_BIRTH_DAY])
#define a_b_month          (a_pointer_array [IS_BIRTH_MONTH])
#define a_b_year           (a_pointer_array [IS_BIRTH_YEAR])
#define a_phone_number     (a_pointer_array [IS_PHONE_NUMBER])
#define a_mobile_number    (a_pointer_array [IS_MOBILE_NUMBER])
#define a_cust_number      (a_pointer_array [IS_CUST_NUMBER])
#define a_email_addr       (a_pointer_array [IS_EMAIL_ADDR])
#define a_iban_code        (a_pointer_array [IS_IBAN_CODE])
#define a_bank_account     (a_pointer_array [IS_BANK_ACCOUNT])


/****  second postal address (pointers to variables)  ****/
#define a2_gender          (a2_pointer_array [IS_GENDER])
#define a2_first_name      (a2_pointer_array [IS_FIRST_NAME])
#define a2_first_name_ph   (a2_pointer_array [IS_FIRST_NAME_PHONET])
#define a2_fam_name        (a2_pointer_array [IS_FAM_NAME])
#define a2_fam_name_ph     (a2_pointer_array [IS_FAM_NAME_PHONET])
#define a2_c_o_name        (a2_pointer_array [IS_C_O_NAME])
#define a2_street          (a2_pointer_array [IS_STREET])
#define a2_street_ph       (a2_pointer_array [IS_STREET_PHONET])
#define a2_zip_code        (a2_pointer_array [IS_ZIP_CODE])
#define a2_city            (a2_pointer_array [IS_CITY])
#define a2_country         (a2_pointer_array [IS_COUNTRY])
#define a2_full_birthday   (a2_pointer_array [IS_FULL_BIRTHDAY])
#define a2_date_format     (a2_pointer_array [IS_DATE_FORMAT])
#define a2_b_day           (a2_pointer_array [IS_BIRTH_DAY])
#define a2_b_month         (a2_pointer_array [IS_BIRTH_MONTH])
#define a2_b_year          (a2_pointer_array [IS_BIRTH_YEAR])
#define a2_phone_number    (a2_pointer_array [IS_PHONE_NUMBER])
#define a2_mobile_number   (a2_pointer_array [IS_MOBILE_NUMBER])
#define a2_email_addr      (a2_pointer_array [IS_EMAIL_ADDR])
#define a2_cust_number     (a2_pointer_array [IS_CUST_NUMBER])
#define a2_iban_code       (a2_pointer_array [IS_IBAN_CODE])
#define a2_bank_account    (a2_pointer_array [IS_BANK_ACCOUNT])



struct WEIGHTS_COUNTRY
  {
    char *country;
    int w_first_name;
    int w_fam_name;
    int w_street;
    int w_zip_code;
    int w_city;
  };

static struct WEIGHTS_COUNTRY WEIGHT_LIST[] = WEIGHTS_ADDR_COMPARE;


static char *jun[] =
     {
       "jr",  "jr.",  "Jr.",
       "jun", "jun.", "Jun.", "JUN.",
       "sen", "sen.", "Sen.", "SEN.",
       NULL
     };


/****  first postal address (internal variables)  ****/
static char *a_pointer_array [MAX_FIELDS_ADDR+1];
static char a_temp_array [MAX_FIELDS_ADDR] [LENGTH_INTERNAL_VAR+1];
static char a_temp_gender [LENGTH_INTERNAL_VAR+1];
static char a_temp_birth [LENGTH_INTERNAL_VAR+1];

/****  second postal address (internal variables)  ****/
static char *a2_pointer_array [MAX_FIELDS_ADDR+1];
static char a2_temp_array [MAX_FIELDS_ADDR] [LENGTH_INTERNAL_VAR+1];
static char a2_temp_gender [LENGTH_INTERNAL_VAR+1];
static char a2_temp_birth [LENGTH_INTERNAL_VAR+1];




/************************************************************/
/****  private (static) functions  **************************/
/************************************************************/

static int count_names (char *name)
/****  count number of "full" names in "name"  ****/
{
 int i,k,n;
 n = 0;
 k = 0;
 i = 0;

 if (name == NULL)
   {
    name = "";
   }

 for (i=0; name[i] != '\0'; i++)
   {
    if (IS_LETTER (name[i]))
      {
       k++;
      }
    else
      {
       if (k >= 3
       && (name[i] == ' '  ||  name[i] == '-'  ||  name[i] == '\0'))
         {
          n++;
         }
       k = 0;
      }
   }

 return (n);
}




static void format_number (char *text, int run_mode)
{
  char *s;
  char wildcard_any_string = MATCHES_ANY_STRING;
  char wildcard_any_char = MATCHES_ANY_CHAR;

  if (run_mode & DB_WILDCARDS_FOR_LIKE)
    {
     wildcard_any_string = DB_LIKE_ANY_STRING;
     wildcard_any_char = DB_LIKE_ANY_CHAR;
    }

  s = text;
  while (*text != '\0')
    {
     if (IS_XLETTER (*text)
     ||  *text == wildcard_any_string  ||  *text == wildcard_any_char)
       {
        *s = *text;
        s++;
       }
     text++;     
    }
  *s = '\0';
}




static void get_internal_addr
        (struct MAIL_ADDR x_address[], char *x_pointer_array[],
        char x_temp_array [MAX_FIELDS_ADDR] [LENGTH_INTERNAL_VAR+1],
        char *temp_birth, int run_mode)
/****  map internal variables on an actual mail address  ****/
{
  int  i,k;
  char *s;

  /****  reset all internal variables  ****/
  for (i=0; i< MAX_FIELDS_ADDR; i++)
    {
     x_pointer_array[i] = "";
    }

  x_pointer_array [IS_FIRST_NAME_PHONET] = NULL;
  x_pointer_array [IS_FAM_NAME_PHONET] = NULL;
  x_pointer_array [IS_STREET_PHONET]   = NULL;
  x_pointer_array [IS_FULL_BIRTHDAY]   = NULL;

  if (x_address != NULL)
    {
     /****  read external mail address  ****/
     for (i=0; x_address[i].info > 0; i++)
       {
        k = x_address[i].info;
        s = x_temp_array[i];

        if (x_address[i].text != NULL  &&  k <= MAX_FIELDS_ADDR)
          {
           if (k == IS_ZIP_CODE
           ||  k == IS_PHONE_NUMBER
           ||  k == IS_MOBILE_NUMBER
           ||  k == IS_IBAN_CODE
           ||  k == IS_BANK_ACCOUNT)
             {
              StrNCpy (s, x_address[i].text, LENGTH_INTERNAL_VAR+1);
              x_pointer_array[k] = s;
              format_number (s, run_mode);
             }
           else if (run_mode & SKIP_BLANKCUT)
             {
              x_pointer_array[k] = x_address[i].text;
             }
           else
             {
              StrNCpy (s, x_address[i].text, LENGTH_INTERNAL_VAR+1);
              x_pointer_array[k] = s;
              blank_cut (s, LENGTH_INTERNAL_VAR+1);
             }
          }
       }

     if (x_pointer_array [IS_FULL_BIRTHDAY] != NULL)
       {
        /****  split full birthday  ****/
        x_pointer_array [IS_BIRTH_DAY]   = temp_birth;
        x_pointer_array [IS_BIRTH_MONTH] = temp_birth + 3;
        x_pointer_array [IS_BIRTH_YEAR]  = temp_birth + 6;

        split_birthday (x_pointer_array [IS_FULL_BIRTHDAY],
              x_pointer_array [IS_DATE_FORMAT],
              x_pointer_array [IS_BIRTH_DAY],
              x_pointer_array [IS_BIRTH_MONTH],
              x_pointer_array [IS_BIRTH_YEAR]);
       }
    }
}





/************************************************************/
/****  "external" functions  ********************************/
/************************************************************/

void StrNCpy (char dest[], char src[], int n)
{
  strncpy (dest, src,n-1);
  dest[n-1] = '\0';
}



void blank_cut (char src[], int len)
{
 int i,j,k;
 char c,*s;

 while ((s = strchr (src,'\n')) != NULL)
   {
    *s = ' ';
   }
 while ((s = strchr (src,'\r')) != NULL)
   {
    *s = ' ';
   }
 while ((s = strchr (src,'\t')) != NULL)
   {
    *s = ' ';
   }

 s = src;
 i = 1;
 j = 0;
 k = 0;
 while ((c = s[k]) != '\0'  &&  (j < len-1  ||  len <= 0))
   {
    i = (c == ' ') ?  (i-1) : 2;
    if (i > 0)
      {
       s[j] = c;
       j++;
      }
    k++;
   }

 if (j > 0  &&  s[j-1] == ' ')
   {
    /****  delete trailing blank  ****/
    j--;
   }
 s[j] = '\0';
}




int check_valid_first_name (char name[])
{
  if (name[0] != '\0'
  &&  strchr (name,'.') == NULL)
    {
#ifdef USE_GENDER
     int c;
     c = get_gender (name, GENDER_MODE, GENDER_COUNTRY);

     if (c == IS_FEMALE  ||  c == IS_MOSTLY_FEMALE
     ||  c == IS_MALE    ||  c == IS_MOSTLY_MALE
     ||  c == IS_UNISEX_NAME)
       {
        return (1);
       }
#endif
     return (0);
    }

 return (-1);
}




int delete_suffix_jun (char a_name[], char a2_name[])
/****  delete suffixes in names (e.g. "jun.")  ****/
{
  int i,k,n;
  int jun_count;
  char *s,**p;

  if (a_name == NULL)
    {
     a_name = "";
    }
  if (a2_name == NULL)
    {
     a2_name = "";
    }
  n = (int)strlen (a_name);
  k = (int)strlen (a2_name);
  jun_count = 0;

  for (p=jun; *p != NULL; p++)
    {
     s = *p;
     i = (int)strlen (s);

     if (n > i+1  &&  a_name [n-i-1] == ' '
     &&  strcmp (a_name+n-i, s) == 0)
       {
        a_name [n-i-1] = '\0';
        n = 0;
        if (*s == 'j'  ||  *s == 'J')
          {
           jun_count |= 1;
          }
        if (*s == 's'  ||  *s == 'S')
          {
           jun_count |= 2;
          }
       }

     if (k > i+1  &&  a2_name [k-i-1] == ' '
     &&  strcmp (a2_name+k-i, s) == 0)
       {
        a2_name [k-i-1] = '\0';
        k = 0;
        if (*s == 'j'  ||  *s == 'J')
          {
           jun_count |= 4;
          }
        if (*s == 's'  ||  *s == 'S')
          {
           jun_count |= 8;
          }
       }
    }

 return (jun_count);
}




void format_street (char src[], char street[], int run_mode)
{
  int  i;
  char c,*s;
  char tmp_string [LENGTH_STREET+1];

  StrNCpy (street, src, LENGTH_STREET+1);

  if (strcmp (street,UNKNOWN_STREET) == 0)
    {
     strcpy (street,"");
    }

  if (! (conv_strings_initialized & CONV_STRINGS_ARE_INITIALIZED))
    {
      initialize_conv_strings (run_mode);

      if (! (conv_strings_initialized & CONV_STRINGS_ARE_INITIALIZED))
        {
         /****  internal error  ****/
         street[0] = '\0';
         return;
        }
    }

  i = 0;
  while (IS_DIGIT (street[i]))
    {
     i++;
    }
  if (i > 4    &&  street[i] != '\0'
  &&  street[i-1] != ' '
  &&  (int) strlen (street) < LENGTH_STREET)
    {
     /****  street = (e.g.) "Avenue123"  ****/
     strcpy (tmp_string, street+i);
     street[i] = ' ';
     strcpy (street+i+1, tmp_string);
    }

  if (run_mode & COMPARE_LANGUAGE_GERMAN)
    {
     i = 0;
     while ((c = street[i]) != '\0'  &&  i < LENGTH_STREET)
       {
        if (IS_DIGIT (c)
        &&  street[i+1] == ' '  &&  IS_DIGIT (street[i+2]))
          {
           /****  compress numbers (e.g. "Postfach 1 23 45")  ****/
           strcpy (street+i+1, street+i+2);
          }
        i++;
       }
     street[i] = '\0';

     while (i > 4  &&  IS_DIGIT (street[i-1]))
       {
        i--;
       }
     if (i > 4
     &&  street[i] == ' '  &&  IS_DIGIT (street[i-1]))
       {
        /****  street = (e.g.) "Avenue 123 a"  ****/
        strcpy (street+i, street+i+1);
       }
    }

  if (run_mode & COMPARE_LANGUAGE_GERMAN)
    {
     if (strncmp (street, POSTF_1, strlen (POSTF_1)) == 0
     ||  strncmp (street, POSTF_2, strlen (POSTF_2)) == 0
     ||  strncmp (street, POSTF_3, strlen (POSTF_3)) == 0
     ||  strncmp (street, POSTF_4, strlen (POSTF_4)) == 0
     ||  strncmp (street, POSTF_5, strlen (POSTF_5)) == 0
     ||  strncmp (street, POSTF_FORMATTED, strlen (POSTF_FORMATTED)) == 0)
       {
        s = strchr (street, ' ');
        if (s != NULL
        && (int)strlen (POSTF_FORMATTED) + (int)strlen(s) < LENGTH_STREET)
          {
            sprintf (tmp_string, "%s %s", POSTF_FORMATTED, s);
            strcpy (street, tmp_string);
            blank_cut (street, LENGTH_STREET+1);
          }
       }

     if ((s=strstr (street, STRASSE_1+1)) != NULL)
       {
        sprintf (tmp_string, "%s     ", STRASSE_FORMATTED+1);
        strncpy (s, tmp_string, 5);
        blank_cut (street, LENGTH_STREET+1);
       }
     if ((s=strstr (street, STRASSE_2+1)) != NULL)
       {
        sprintf (tmp_string, "%s     ", STRASSE_FORMATTED+1);
        strncpy (s, tmp_string, 6);
        blank_cut (street, LENGTH_STREET+1);
       }
    }
}




void split_birthday (char *full_date,
       char *date_format, char day[], char month[], char year[])

/****  "date_format" may contain delimiters plus  ****/
/****  the letters "d", "m", "y" in either case,  ****/
/****  specifying "day", "month" and "year".      ****/

/****  If "date_format" is null or empty,         ****/
/****  this function expects one of the formats   ****/
/****  <year>-<month>-<day>  (= ISO standard)     ****/
/****  <month>/<day>/<year>  (American format),   ****/
/****  or <day>.<month>.<year> (European format)  ****/
/****  (e.g.  "1999-06-19" or "99-6-19"           ****/
/****    or   "19.06.1999" or "19.6.99")          ****/
{
  int  i,k, delim,lengths;
  char temp[11];
  char c1,c2, *s,*s2,*sy;

  strcpy (day,  "");
  strcpy (month,"");
  strcpy (year, "");

  if (! (conv_strings_initialized & CONV_STRINGS_ARE_INITIALIZED))
    {
      initialize_conv_strings(0);

      if (! (conv_strings_initialized & CONV_STRINGS_ARE_INITIALIZED))
        {
         /****  internal error  ****/
         return;
        }
    }

  if ((int) strlen (full_date) > 10)
    {
     /****  date is invalid (too long)  ****/
     return;
    }

  if (date_format == NULL  ||  *date_format == '\0')
    {
     /****  date format is missing                  ****/
     /****  --> look for formats "year-month-day",  ****/
     /****  "month/day/year" and "day.month.year"   ****/
     StrNCpy (temp,full_date,11);
     s = NULL;
     s2 = NULL;
     k = 0;
     for (i=0; (c1=temp[i]) != '\0'; i++)
       {
         if (c1 == '-'  ||  c1 == '/'  ||  c1 == '.')
           {
             k++;
             if (k == 1)
               {
                 s = temp + i;
               }
             if (k == 2)
               {
                 s2 = temp + i;
               }
           }
       }

     if (k == 2  &&  *s == *s2)
       {
        c1 = *s;
        *s = '\0';
        *s2 = '\0';

        switch (c1)
          {
            case '-':  /****  date format is "year-month-day"  ****/
                       sy = s;
                       s  = s2+1;
                       s2 = sy+1;
                       sy = temp;
                  break;

            case '/':  /****  date format is "month/day/year"  ****/
                       sy = s2+1;
                       s  = s+1;
                       s2 = temp;
                  break;

            case '.':  /****  date format is "day.month.year"  ****/
                       sy = s2+1;
                       s2 = s+1;
                       s  = temp;
                  break;

            default :  /****  date format is unknown  ****/
                  return;
          }

        if ((int) strlen (sy) == LENGTH_BIRTH_YEAR +2)
          {
            /****  ignore century  ****/
            sy += 2;
          }
        StrNCpy (day,   s,  LENGTH_BIRTH_DAY +1);
        StrNCpy (month, s2, LENGTH_BIRTH_MONTH +1);
        StrNCpy (year,  sy, LENGTH_BIRTH_YEAR +1);
       }
    }
  else
    {
     /****  evaluate date format  ****/
     strcpy (temp,"");
     delim = 0;
     lengths = 0;

     while (*date_format != '\0'  &&  delim < 100)
       {
        switch (*date_format)
          {
           case 'D':
           case 'd':  s = day;
                      k = LENGTH_BIRTH_DAY;
                      c1 = 'D';
                      c2 = 'd';
                 break;
           case 'M':
           case 'm':  s = month;
                      k = LENGTH_BIRTH_MONTH;
                      c1 = 'M';
                      c2 = 'm';
                 break;
           case 'Y':
           case 'y':  s = temp;
                      k = LENGTH_BIRTH_YEAR;
                      c1 = 'Y';
                      c2 = 'y';
                 break;

           default :  /****  delimiter found  ****/
                      if (*full_date != *date_format)
                        {
                          /****  invalid format  ****/
                          delim = 100;
                        }
                      date_format++;
                      full_date++;
                      s = NULL;
                 break;
          }

        if (s != NULL)
          {
           if (*s != '\0')
             {
               /****  field is doubly defined  ****/
               delim = 100;
               break;
             }

           i = 0;
           while (i < k  &&  IS_DIGIT (*full_date)
           &&  (*date_format == c1  ||  *date_format == c2))
             {
              i++;
              *s = *full_date;
              s++;
              date_format++;
              full_date++;
             }

           if ((i == k  &&  *date_format != c1  &&  *date_format != c2)
           ||  (*date_format == '\0'  &&  *full_date == '\0')
           ||  (IS_DIGIT (*full_date)
            &&  strchr ("DdMMYy", *date_format) == NULL))
             {
              /****  field length is matching  ****/
              lengths++;
             }

           while (*date_format == c1  ||  *date_format == c2)
             {
              date_format++;
             }

           if (*date_format == '\0'
           ||  strchr ("DdMMYy", *date_format) == NULL)
             {
              /****  end of string or delimiter found  ****/
              if (*date_format != '\0')
                {
                 delim++;
                }

              while (i < k  &&  IS_DIGIT (*full_date))
                {
                 i++;
                 *s = *full_date;
                 s++;
                 /****  do not increment "date_format"  ****/
                 full_date++;
                }
             }
           *s = '\0';
          }
       }

     if ((int) strlen (temp) == LENGTH_BIRTH_YEAR +2)
       {
        /****  ignore century  ****/
        strcpy (temp,temp+2);
       }
     strcpy (year,temp);

     if (lengths != 3  &&  delim != 2)
       {
        /****  invalid format  ****/
        strcpy (day,  "");
        strcpy (month,"");
        strcpy (year, "");
       }
    }

  /****  do some formatting  ****/
  blank_cut (day,  LENGTH_BIRTH_DAY +1);
  blank_cut (month,LENGTH_BIRTH_MONTH +1);
  blank_cut (year, LENGTH_BIRTH_YEAR +1);

  /****  convert one-digit strings  into       ****/
  /****  two-digit strings (e.g. "5" -> "05")  ****/
  s = day;
  if (IS_DIGIT (*s)  &&  *(s+1) == '\0')
    {
     *(s+1) = *s;
     *s = '0';
     *(s+2) = '\0';
    }
  s = month;
  if (IS_DIGIT (*s)  &&  *(s+1) == '\0')
    {
     *(s+1) = *s;
     *s = '0';
     *(s+2) = '\0';
    }
  s = year;
  if (IS_DIGIT (*s)  &&  *(s+1) == '\0')
    {
     *(s+1) = *s;
     *s = '0';
     *(s+2) = '\0';
    }
}




double calculate_ln (double d)
/****  Simple algorithm for approximately calculating the  ****/
/****  natural logarithm (ln) of a given number >= 1.0     ****/
/****  (The accuracy (== error) should be about 0.00001)   ****/
{
  double log_table[10] = {
                           0.000000,
                           0.693147,
                           1.098612,
                           1.386294,
                           1.609438,
                           1.791759,
                           1.945910,
                           2.079442,
                           2.197225,
                           2.302585
                         };

  int  n;
  double ln;
  ln = 0.0;

  if (d <= 1.0)
    {
     return (0.0);
    }

  while (d > 10.0)
    {
     d /= 10.0;
     ln += log_table[10-1];
    }

  if (d <= 2.0)
    {
     d *= 5.0;
     ln -= log_table[5-1];
    }
  else if (d <= 3.33)
    {
     d *= 3.0;
     ln -= log_table[3-1];
    }
  else if (d <= 5.0)
    {
     d *= 2.0;
     ln -= log_table[2-1];
    }

  n = 5;
  while ((double)n < d - 0.5)
    {
     n++;
    }

  d /= (double) n;
  d -= 1.0;
  ln += log_table[n-1] + d - (0.5 * d * d);

  return (ln);
}




void cleanup_addr (void)
{
  #ifdef USE_GENDER
     cleanup_gender();
  #endif
}





/************************************************************/
/****  mail address comparison  *****************************/
/************************************************************/

int compare_addr (struct MAIL_ADDR a_address[],
      struct MAIL_ADDR a2_address[], int min_points, int run_mode)

/****  This function compares two addresses.    ****/
/****  A match requires at least "min_points".  ****/
/****  The return value "points" is a measure   ****/
/****  for the similarity of two addresses.     ****/
/****  Since "MAX_POINTS" is 100, this amounts  ****/
/****  to a result in percent.                  ****/

/****  If "points < min_points", no match was   ****/
/****  found and zero is returned. Exception:   ****/
/****  If "min_points > IS_FAMILY_MEMBER", the  ****/
/****  special return value "IS_FAMILY_MEMBER"  ****/
/****  indicates that a family member with the  ****/
/****  "same" address has been found.           ****/
{
 struct LEV_RESULT l_res = { 0, 0, 0, 0 };
 struct LEV_RESULT l_nam = { 0, 0, 0, 0 };
 struct LEV_RESULT l_place = { 0, 0, 0, 0 };
 struct LEV_RESULT l_birth = { 0, 0, 0, 0 };
 struct LEV_RESULT l_cust = { 0, 0, 0, 0 };

 struct WEIGHTS_COUNTRY *w, *w2;
 int weight_first_name = AC_WEIGHT_FIRST_NAME;
 int weight_fam_name = AC_WEIGHT_FAM_NAME;
 int weight_street   = AC_WEIGHT_STREET;
 int weight_zip_code = AC_WEIGHT_ZIP_CODE;
 int weight_city    = AC_WEIGHT_CITY;

 int weight_gender  = WEIGHT_GENDER;
 int weight_country = WEIGHT_COUNTRY;
 int weight_b_day   = WEIGHT_B_DAY;
 int weight_b_month = WEIGHT_B_MONTH;
 int weight_b_year  = WEIGHT_B_YEAR;

 char *s,*s2;
 char a_temp [LENGTH_WHOLE_NAME+1];
 char a2_temp [LENGTH_WHOLE_NAME +1];

 double is_family_member = 1.0;
 int i,k,n;
 int gender_country = GC_ANY_COUNTRY;
 int comp_gender = GENDER_MAY_BE_EQUAL;
 int jun_count = 0;

 int max_points_first_name = 0;
 int max_points_fam_name = 0;
 int empty_diff_first_name = 0;
 int empty_diff_fam_name = 0;
 int diff_first_name = 0;
 int diff_fam_name = 0;

 char wildcard_any_string = MATCHES_ANY_STRING;
 char wildcard_any_char = MATCHES_ANY_CHAR;

 if (run_mode & DB_WILDCARDS_FOR_LIKE)
   {
    wildcard_any_string = DB_LIKE_ANY_STRING;
    wildcard_any_char = DB_LIKE_ANY_CHAR;
   }

 if (! (conv_strings_initialized & CONV_STRINGS_ARE_INITIALIZED))
   {
    initialize_conv_strings (run_mode);

    if (! (conv_strings_initialized & CONV_STRINGS_ARE_INITIALIZED))
      {
       /****  internal error  ****/
       return (ADDR_INTERNAL_ERROR);
      }
   }

 /****  map mail addresses on internal variables  ****/
 get_internal_addr (a_address, a_pointer_array,
           a_temp_array, a_temp_birth, run_mode);
 get_internal_addr (a2_address, a2_pointer_array,
           a2_temp_array, a2_temp_birth, run_mode);

 if (min_points >= MAX_POINTS)
   {
    min_points = MAX_POINTS -1;
   }

 if (run_mode & TRACE_ADDR)
   {
    printf ("first address:\n");
    printf ("gender = '%s', customer number = '%s'\n",
        a_gender, a_cust_number);
    printf ("first name = '%s', family name = '%s', c/o name = '%s'\n",
        a_first_name, a_fam_name, a_c_o_name);
    printf ("street = '%s', city = '%s'%s', country = '%s'\n",
        a_street, a_zip_code, a_city, a_country);
    printf ("birthday (yyyy-mm-dd) = '%1s-%1s-%1s'\n",
        a_b_year, a_b_month, a_b_day);
    printf ("phone = '%s', mobile = '%s', email = '%s'\n",
        a_phone_number, a_mobile_number, a_email_addr);
    printf ("IBAN code = '%s', bank account = '%s'\n",
        a_iban_code, a_bank_account);
    printf ("\n");

    printf ("second address:\n");
    printf ("gender = '%s', customer number = '%s'\n",
        a2_gender, a2_cust_number);
    printf ("first name = '%s', family name = '%s', c/o name = '%s'\n",
        a2_first_name, a2_fam_name, a2_c_o_name);
    printf ("street = '%s', city = '%s'%s', country = '%s'\n",
        a2_street, a2_zip_code, a2_city, a2_country);
    printf ("birthday (yyyy-mm-dd) = '%1s-%1s-%1s'\n",
        a2_b_year, a2_b_month, a2_b_day);
    printf ("phone = '%s', mobile = '%s', email = '%s'\n",
        a2_phone_number, a2_mobile_number, a2_email_addr);
    printf ("IBAN code = '%s', bank account = '%s'\n",
        a2_iban_code, a2_bank_account);
    printf ("\n");
    printf ("min_points = %d\n", min_points);
   }

 /****  change first name (e.g.) "A" to "A."  ****/
 if (IS_LETTER (a_first_name[0])  &&  a2_first_name[1] == '\0')
   {
    strcpy (a_first_name+1,".");
   }
 if (IS_LETTER (a2_first_name[0])  &&  a2_first_name[1] == '\0')
   {
    strcpy (a2_first_name+1,".");
   }

 /****  delete dummy birthdays  ****/
 if (strcmp (a_b_day, a_b_month) == 0
 &&  strncmp (a_b_day, DUMMY_BIRTHDAY,2) == 0)
   {
    strcpy (a_b_day,  "");
    strcpy (a_b_month,"");
    strcpy (a_b_year, "");
   }
 if (strcmp (a2_b_day, a_b_month) == 0
 &&  strncmp (a2_b_day, DUMMY_BIRTHDAY,2) == 0)
   {
    strcpy (a2_b_day,  "");
    strcpy (a2_b_month,"");
    strcpy (a2_b_year, "");
   }

 /****  set country  ****/
 s = DEFAULT_COUNTRY;
 if (s == NULL  ||  *s == ' ')
   {
    s = "";
   }

 if (*a_country == '\0')
   {
    if ((run_mode & TRACE_ADDR)  &&  *s != '\0')
      {
       printf ("Note:  country for first address is not set;\n");
       printf ("       country will be set to default country (\"%s\").\n\n", s);
      }
    a_country = s;
   }
 if (*a2_country == '\0')
   {
    if ((run_mode & TRACE_ADDR)  &&  *s != '\0')
      {
       printf ("Note:  country for second address is not set;\n");
       printf ("       country will be set to default country (\"%s\").\n\n", s);
      }
    a2_country = s;
   }

 /****  set weights according to country  ****/
 w = NULL;
 w2 = NULL;
 k = 0;
 n = 0;
 for (i=0; WEIGHT_LIST[i].country != NULL; i++)
   {
    if (k < 2  &&  strcmp (a_country, WEIGHT_LIST[i].country) == 0)
      {
        w = & (WEIGHT_LIST[i]);
        k = 2;
      }
    if (k == 0  &&  WEIGHT_LIST[i].country[0] == '*')
      {
        w = & (WEIGHT_LIST[i]);
        k = 1;
      }

    if (n < 2  &&  strcmp (a2_country, WEIGHT_LIST[i].country) == 0)
      {
        w2 = & (WEIGHT_LIST[i]);
        n = 2;
      }
    if (n == 0  &&  WEIGHT_LIST[i].country[0] == '*')
      {
        w2 = & (WEIGHT_LIST[i]);
        n = 1;
      }
   }

 if (w != NULL  &&  w2 != NULL)
   {
    if (run_mode & TRACE_ADDR)
      {
       if (w == w2)
         {
          printf ("set weights according to country:  \"%s\"\n",
              w->country);
         }
       else
         {
          printf ("Set weights according to country:  \"%s\" or \"%s\"\n",
              w->country, w2->country);
         }
      }

    weight_first_name = (int) ((w->w_first_name + w2->w_first_name) / 2);
    weight_fam_name = (int) ((w->w_fam_name + w2->w_fam_name) / 2);
    weight_street   = (int) ((w->w_street + w2->w_street) / 2);
    weight_zip_code = (int) ((w->w_zip_code + w2->w_zip_code) / 2);

    if (w == w2)
      {
        if (strcmp (w->country, "D") == 0)
          {
           gender_country = GC_GERMANY;
          }
        if (strcmp (w->country, "A") == 0)
          {
           gender_country = GC_AUSTRIA;
          }
        if (strcmp (w->country, "CH") == 0)
          {
           gender_country = GC_SWISS;
          }
      }
   }

 if (run_mode & DO_UNWEIGHTED_COMPARISON)
   {
    /****  weights are proportional to       ****/
    /****  estimated average string lengths  ****/
    weight_gender   = 100;
    weight_first_name = 900;
    weight_fam_name = 800;
    weight_street   = 1100;
    weight_zip_code =  450;
    weight_city    = 800;
    weight_country = 140;
    weight_b_day   = 200;
    weight_b_month = 160;
    weight_b_year  = 200;
   }


 /****  Look for "inverted names"  (e.g.      ****/
 /****  "Richard, Paul" vs. "Paul, Richard")  ****/
 i = 1000;
 k = 0;
 if (a_first_name[0] != '\0'  &&  a2_first_name[0] != '\0')
   {
    /****  do not change the Levenshtein calls  ****/
    l_res.points = 0;
    n = (int)strlen (a_first_name) - (int)strlen (a2_fam_name);
    if (n >= -2  &&  n <= 2)
      {
       l_res = lev_x (a_first_name, a2_fam_name,
                  "",1000, (run_mode & ~TRACE_ADDR));
       i = l_res.diff;
      }

    if (i > 100  ||  l_res.points < 800)
      {
       n = (int)strlen (a2_first_name) - (int)strlen (a_fam_name);
       if (n >= -2  &&  n <= 2)
         {
          l_res = lev_x (a2_first_name, a_fam_name,
                     "",1000, (run_mode & ~TRACE_ADDR));
          i = l_res.diff;
         }
      }

    if (i <= 100  &&  l_res.points >= 800)
      {
       l_res = lev_x (a_fam_name, a2_fam_name,
                  "",1000, (run_mode & ~TRACE_ADDR));
       k = l_res.diff;
      }
   }

 if (i <= 100  &&  k > 300)
   {
    /****  switch first name and family name  ****/
    if ((int)strlen (a_first_name) > (int)strlen (a2_first_name))
      {
       s = a_first_name;
       a_first_name = a_fam_name;
       a_fam_name = s;

       s = a_first_name_ph;
       a_first_name_ph = a_fam_name_ph;
       a_fam_name_ph = s;
      }
    else
      {
       s = a2_first_name;
       a2_first_name = a2_fam_name;
       a2_fam_name = s;

       s = a2_first_name_ph;
       a2_first_name_ph = a2_fam_name_ph;
       a2_fam_name_ph = s;
      }

    i = (int) ((weight_first_name + weight_fam_name) / 2);
    weight_first_name = i;
    weight_fam_name = i;

    if (run_mode & TRACE_ADDR)
      {
       printf ("First name and family name have been switched\n");
      }
   }


 /****  if gender is empty, use the function "get_gender"  ****/
 if (a_gender[0] == '\0'  &&  a_first_name[0] != '\0')
   {
    i = get_gender (a_first_name, GENDER_ALLOW_COUPLE, gender_country);

    if (i == IS_FEMALE  ||  i == IS_MALE)
      {
       sprintf (a_temp_gender, "%c",i);
       a_gender = a_temp_gender;
      }
   }

 if (a2_gender[0] == '\0'  &&  a2_first_name[0] != '\0')
   {
    i = get_gender (a2_first_name, GENDER_ALLOW_COUPLE, gender_country);

    if (i == IS_FEMALE  ||  i == IS_MALE)
      {
       sprintf (a2_temp_gender, "%c",i);
       a2_gender = a2_temp_gender;
      }
   }

 /****  compare gender  ****/
 n = 0;
 k = weight_gender;
 i = 0;
 if ((run_mode & DATABASE_SELECT)
 && (a_gender[0] == '\0'
 ||  a_gender[0] == wildcard_any_string
 ||  a_gender[0] == wildcard_any_char))
   {
    comp_gender = GENDER_IS_EQUAL;
    k = 0;
   }
 else if (a_gender[0] == '\0'  ||  a2_gender[0] == '\0')
   {
    comp_gender = GENDER_MAY_BE_EQUAL;

    if (a_gender[0] == a2_gender[0])
      {
       k = 0;
      }
    else
      {
       k = (int) ((weight_gender + 1) / 5);
       i = k;
      }
   }
 else if (a_gender[0] == a2_gender[0])
   {
    comp_gender = GENDER_IS_EQUAL;
    n = k;
   }
 else
   {
    comp_gender = GENDER_IS_DIFFERENT;
    l_nam.diff += 100;
   }

 l_nam.points += n;
 l_nam.max_points += k;
 l_nam.empty_diff += i;

 if (run_mode & TRACE_ADDR)
   {
    switch (comp_gender)
      {
       case GENDER_IS_DIFFERENT :  s = "GENDER_IS_DIFFERENT";  break;
       case GENDER_IS_EQUAL     :  s = "GENDER_IS_EQUAL";      break;
       default                  :  s = "GENDER_MAY_BE_EQUAL";  break;
      }

    printf ("\n");
    printf ("gender:  result = %s,\n", s);

    print_number ("gender:  points", l_nam.points, -1);
    print_number (",  max_points", l_nam.max_points, -1);
    print_number (",  diff", l_nam.diff, -1);
    print_number (",  empty_diff", l_nam.empty_diff, -1);
    printf ("\n");
    printf ("\n");
   }


 /****  delete suffixes (e.g. "jun.") from family names  ****/
 StrNCpy (a_temp, a_fam_name, LENGTH_FAM_NAME +1);
 StrNCpy (a2_temp, a2_fam_name, LENGTH_FAM_NAME +1);

 jun_count |= delete_suffix_jun (a_temp, a2_temp);

 /****  compare family names  ****/
 l_res = lev_2_ph (a_temp, a2_temp, a_fam_name_ph, a2_fam_name_ph,
            "family name", weight_fam_name, run_mode);

 max_points_fam_name = l_res.max_points;
 empty_diff_fam_name = l_res.empty_diff;

 l_nam.points += l_res.points;
 l_nam.max_points += l_res.max_points;
 l_nam.diff  += l_res.diff;
 l_nam.empty_diff += l_res.empty_diff;

 diff_fam_name = l_res.diff;

 if (6 * l_res.diff <= l_res.max_points
 &&  l_res.empty_diff == 0
 &&  3 * l_res.max_points >= 2 * weight_fam_name)
   {
    is_family_member *= (double)l_res.points / (double)l_res.max_points;
   }
 else
   {
    /****  no family member  ****/
    is_family_member = 0.0;
   }

 if (run_mode & TRACE_ADDR)
   {
    print_number ("total for \"nam\":  points", l_nam.points, -1);
    print_number (",  max_points", l_nam.max_points, -1);
    print_number (",  diff", l_nam.diff, -1);
    print_number (",  empty_diff", l_nam.empty_diff, -1);
    printf ("\n");
    printf ("\n");
   }


 /****  delete suffixes (e.g. "jun.") from first names  ****/
 StrNCpy (a_temp, a_first_name, LENGTH_FIRST_NAME +1);
 StrNCpy (a2_temp, a2_first_name, LENGTH_FIRST_NAME +1);

 jun_count |= delete_suffix_jun (a_temp, a2_temp);

 /****  count number of first names  ****/
 i = count_names (a_temp);
 k = count_names (a2_temp);

 n = weight_first_name;
 if (i >= 2  &&  k >= 2)
   {
    n = 2 * weight_first_name;
    if (n > MAX_POINTS_FIRST_NAME)
      {
       n = MAX_POINTS_FIRST_NAME;
      }
   }

 /****  compare first names  ****/
 if (! (run_mode & DO_UNWEIGHTED_COMPARISON))
   {
    /****  "normal" comparison  ****/
    l_res = lev_2_ph (a_temp, a2_temp, a_first_name_ph, a2_first_name_ph,
              "first name", n, run_mode);
   }
 else
   {
    l_res = lev_x (a_temp, a2_temp, "first name", n, run_mode);
   }

 if (l_res.diff >= 500
 &&  l_res.max_points == n  &&  l_res.empty_diff == 0)
   {
    n = check_nickname (a_first_name, a2_first_name, GENDER_MODE, 
                 GENDER_COUNTRY);

    if (n == EQUIVALENT_NAMES)
      {
       /****  first names are "equivalent"  ****/
       l_res.diff = 200;
       l_res.points = 7 * (int) (n / 10);
      }
   }

 diff_first_name = l_res.diff;
 max_points_first_name = l_res.max_points;
 empty_diff_first_name = l_res.empty_diff;

 l_nam.points += l_res.points;
 l_nam.max_points += l_res.max_points;
 l_nam.diff  += l_res.diff;
 l_nam.empty_diff += l_res.empty_diff;

 if (run_mode & TRACE_ADDR)
   {
    print_number ("total for \"nam\":  points", l_nam.points, -1);
    print_number (",  max_points", l_nam.max_points, -1);
    print_number (",  diff", l_nam.diff, -1);
    print_number (",  empty_diff", l_nam.empty_diff, -1);
    printf ("\n");
    printf ("\n");
   }


 if ((empty_diff_first_name > 0  &&  diff_fam_name > 0)
 || (a_gender[0] == IS_COMPANY  &&  a2_gender[0] == IS_COMPANY
   &&  diff_fam_name >= 500)
 || (diff_first_name > 500  &&  diff_fam_name > 500))
   {
    /****  compare whole name  ****/
    sprintf (a_temp, "%s %s", a_first_name, a_fam_name);
    blank_cut (a_temp, LENGTH_WHOLE_NAME +1);
    sprintf (a2_temp, "%s %s", a2_first_name, a2_fam_name);
    blank_cut (a2_temp, LENGTH_WHOLE_NAME +1);

    /****  don't call "lev_2_ph" here  ****/
    l_res = lev_x (a_temp,a2_temp, "whole name", 800, run_mode);
    k = l_res.diff;

    if (diff_fam_name > 500)
      {
       if (k <= 250)
         {
          /****  add bonus points  ****/
          l_nam.points += 2 * l_res.points;
         }

       if (k > 500
       && (max_points_first_name == 0  ||  empty_diff_first_name > 0))
         {
          /****  first name is missing and     ****/
          /****  family names differ too much  ****/
          l_nam.points -= (int) (weight_fam_name / 2);
         }
      }

    if (k < diff_fam_name)
      {
       diff_fam_name = k;
      }
    if (k < diff_first_name)
      {
       diff_first_name = k;
      }

    if (k <= 100  &&  empty_diff_first_name > 0)
      {
       l_nam.empty_diff -= 100;    /**  "-=" !!  **/
      }
   }

 if (comp_gender == GENDER_IS_DIFFERENT)
   {
    if (diff_first_name >= 200)
      {
       l_nam.points -= 400;
      }
    else if (diff_first_name >= 100)
      {
       l_nam.points -= 200;
      }
   }

 if (diff_first_name > 500  &&  diff_fam_name > 500)
   {
    /****  both first name and family name differ too much  ****/
    l_nam.points -= (int) (weight_fam_name / 2);
   }
 if (jun_count == 6  ||  jun_count == 9)
   {
    /****  this is a junior and a senior  ****/
    l_nam.points -= weight_first_name;
   }

 if (strncmp (a_c_o_name, "C/O ",4) == 0
 ||  strncmp (a_c_o_name, "C/o ",4) == 0
 ||  strncmp (a_c_o_name, "c/o ",4) == 0
 || ((run_mode & COMPARE_LANGUAGE_GERMAN)
  &&  strncmp (a_c_o_name, "z.Hd. ",5) == 0))
   {
    /****  evaluate c/o name  ****/
    i = (int) strlen (a2_first_name);
    n = (int) strlen (a2_fam_name);
    k = (int) strlen (a_c_o_name) -4 -1;

    if (i >= 4  &&  n >= 4  &&  i+n >= k-2  &&  i+n <= k+2)
      {
       sprintf (a_temp, "%s %s", a2_first_name, a2_fam_name);
       blank_cut (a_temp, LENGTH_WHOLE_NAME +1);
       sprintf (a2_temp, "%s", a_c_o_name +4);
       blank_cut (a2_temp, LENGTH_WHOLE_NAME +1);

       /****  don't call "lev_2_ph" here  ****/
       l_res = lev_x (a2_temp,a_temp, "whole name", 1000, run_mode);

       if (l_res.diff <= 100  &&  l_res.points >= 500)
         {
          l_nam.points -= l_res.points;
         }
      }
   }

 if (strncmp (a2_c_o_name, "C/O ",4) == 0
 ||  strncmp (a2_c_o_name, "C/o ",4) == 0
 ||  strncmp (a2_c_o_name, "c/o ",4) == 0
 || ((run_mode & COMPARE_LANGUAGE_GERMAN)
  &&  strncmp (a2_c_o_name, "z.Hd. ",5) == 0))
   {
    /****  evaluate c/o name  ****/
    i = (int) strlen (a_first_name);
    n = (int) strlen (a_fam_name);
    k = (int) strlen (a2_c_o_name) -4 -1;

    if (i >= 4  &&  n >= 4  &&  i+n >= k-2  &&  i+n <= k+2)
      {
       sprintf (a_temp, "%s %s", a_first_name, a_fam_name);
       blank_cut (a_temp, LENGTH_WHOLE_NAME +1);
       sprintf (a2_temp, "%s", a2_c_o_name +4);
       blank_cut (a2_temp, LENGTH_WHOLE_NAME +1);

       /****  don't call "lev_2_ph" here  ****/
       l_res = lev_x (a2_temp,a_temp, "whole name", 1000, run_mode);

       if (l_res.diff <= 100  &&  l_res.points >= 500)
         {
          l_nam.points -= l_res.points;
         }
      }
   }

 if (run_mode & TRACE_ADDR)
   {
    printf ("total points for \"nam\" (after \"extra checks with name\"):\n");
    
    print_number ("      points", l_nam.points, -1);
    print_number (",  max_points", l_nam.max_points, -1);
    print_number (",  diff", l_nam.diff, -1);
    print_number (",  empty_diff", l_nam.empty_diff, -1);
    printf ("\n");
    printf ("\n");
   }


 /****  compare ZIP codes  ****/
 if (! (run_mode & DO_UNWEIGHTED_COMPARISON))
   {
    /****  "normal" comparison  ****/
    l_res = lev_zipcode_city (a_zip_code, a2_zip_code,
               a_city, a2_city, "ZIP code/city",
               weight_zip_code, weight_city, run_mode);
   }
 else
   {
    l_res = lev_x (a_zip_code, a2_zip_code,
               "ZIP code", weight_zip_code, run_mode);
   }

 i = 0;
 if (l_res.points == 0  ||  l_res.empty_diff != 0)
   {
    i = 1;
    l_res = lev_x (a_city,a2_city, "city", weight_city, run_mode);
   }

 k = l_res.points;  /****  "k" will be used in "compare streets"  ****/
 n = l_res.diff;    /****  "n" will be used in "compare streets"  ****/

 l_place.points += l_res.points;
 l_place.max_points += l_res.max_points;
 l_place.diff += l_res.diff;
 l_place.empty_diff += l_res.empty_diff;

 if (6 * l_res.diff <= l_res.max_points
 &&  l_res.empty_diff == 0
 &&  3 * l_res.max_points >= 2 * weight_zip_code)
   {
    is_family_member *= (double)l_res.points / (double)l_res.max_points;
   }
 else
   {
    /****  no family member  ****/
    is_family_member = 0.0;
   }

 if ((run_mode & COMPARE_LANGUAGE_GERMAN)
 &&  i == 0
 &&  strcmp (a_city, a2_city) != 0)
   {
    l_res = lev_x (a_city,a2_city, "city", weight_city, run_mode);

    if (l_res.diff > 200)
      {
       /****  city names are different --> subtract points  ****/
       l_place.points -= weight_city - l_res.points;
       l_place.diff += 100;

       /****  no family member  ****/
       is_family_member = 0.0;
      }
   }

 if (run_mode & TRACE_ADDR)
   {
    print_number ("total points for \"place\":  points", l_place.points, -1);
    print_number (",  max_points", l_place.max_points, -1);
    print_number (",  diff", l_place.diff, -1);
    print_number (",  empty_diff", l_place.empty_diff, -1);
    printf ("\n");
    printf ("\n");
   }


 /****  compare countries  ****/
 s = DEFAULT_COUNTRY;
 if (s != NULL
 &&  strcmp (s, a_country) == 0  &&  strcmp (s, a2_country) == 0)
   {
    a_country  = "";
    a2_country = "";
   }

 l_res = lev_x (a_country, a2_country,
            "country", weight_country, run_mode);

 l_place.points += l_res.points;
 l_place.max_points += l_res.max_points;
 l_place.diff += l_res.diff;
 l_place.empty_diff += l_res.empty_diff;

 if (l_res.diff > 0)    /****  do not add  "||  l_res.empty_diff > 0"  ****/
   {
    /****  no family member  ****/
    is_family_member = 0.0;
   }

 if (run_mode & TRACE_ADDR)
   {
    print_number ("total points for \"place\":  points", l_place.points, -1);
    print_number (",  max_points", l_place.max_points, -1);
    print_number (",  diff", l_place.diff, -1);
    print_number (",  empty_diff", l_place.empty_diff, -1);
    printf ("\n");
    printf ("\n");
   }


 /****  compare streets  ****/
 format_street (a_street, a_temp, run_mode);
 format_street (a2_street, a2_temp, run_mode);

 if ((run_mode & COMPARE_LANGUAGE_GERMAN)
 &&  strncmp (a_temp, POSTF_FORMATTED, strlen (POSTF_FORMATTED)) == 0
 &&  strncmp (a2_temp,POSTF_FORMATTED, strlen (POSTF_FORMATTED)) == 0)
   {
    s = a_temp + 4;
    s2 = a2_temp + 4;

    if (run_mode & TRACE_ADDR)
      {
       printf ("street: compare '%s' with '%s'\n", s,s2);
      }

    l_res = lev_x (s,s2, "street", weight_street, run_mode);
   }
 else if (! (run_mode & DO_UNWEIGHTED_COMPARISON))
   {
    /****  "normal" comparison  ****/
    l_res = lev_2_ph (a_temp, a2_temp, a_street_ph, a2_street_ph,
              "street", weight_street, run_mode);
   }
 else
   {
    /****  "unweighted" comparison  ****/
    l_res = lev_x (a_temp,a2_temp, "street", weight_street, run_mode);
   }

 if ((run_mode & COMPARE_LANGUAGE_GERMAN)
 &&  l_res.points == 0
 &&  n <= 100  &&  k > 0  &&  4 * k >= 3 * weight_zip_code)
   {
    /****  ZIP codes are similar, streets are different:  ****/
    /****  check for  street == ("Dorfstr.*" or "Nr.*")   ****/
    /****      and     city  == street_2                  ****/
    s = NULL;
    i = (int)strlen (a2_city);
    if (i >= 4  &&  strncmp (a_temp, a2_city,i) == 0)
      {
       s = a_temp + i;
      }
    if (strncmp (a_temp, DORFSTR_1, strlen (DORFSTR_1)) == 0
    ||  strncmp (a_temp, DORFSTR_2, strlen (DORFSTR_2)) == 0
    ||  strncmp (a_temp, DORFSTR_3, strlen (DORFSTR_3)) == 0
    ||  strncmp (a_temp, DORFSTR_4, strlen (DORFSTR_4)) == 0)
      {
       s = strchr (a_temp,'.');
      }

    s2 = NULL;
    i = (int)strlen (a_city);
    if (i >= 4  &&  strncmp (a2_temp, a_city,i) == 0)
      {
       s2 = a2_temp + i;
      }
    if (strncmp (a2_temp, DORFSTR_1, strlen (DORFSTR_1)) == 0
    ||  strncmp (a2_temp, DORFSTR_2, strlen (DORFSTR_2)) == 0
    ||  strncmp (a2_temp, DORFSTR_3, strlen (DORFSTR_3)) == 0
    ||  strncmp (a2_temp, DORFSTR_4, strlen (DORFSTR_4)) == 0)
      {
       s2 = strchr (a2_temp,'.');
      }

    if (s != NULL  &&  s2 != NULL)
      {
       while (*s == '.' ||  *s == ' ')
         {
          s++;
         }
       while (*s2 == '.' ||  *s2 == ' ')
         {
          s2++;
         }

       strncpy (a_temp, "Nr ",3);
       StrNCpy (a_temp+3, s, LENGTH_WHOLE_NAME+1-3);
       strncpy (a2_temp, "Nr ",3);
       StrNCpy (a2_temp+3, s2, LENGTH_WHOLE_NAME+1-3);
       
       if (run_mode & TRACE_ADDR)
         {
          printf ("street: compare '%s' with '%s'\n", a_temp,a2_temp);
         }

       l_res = lev_x (a_temp+1,a2_temp+1, "street",
                  (int) ((2*weight_street+1)/3), run_mode);
      }
   }

 l_place.points += l_res.points;
 l_place.max_points += l_res.max_points;
 l_place.diff += l_res.diff;
 l_place.empty_diff += l_res.empty_diff;

 if (6 * l_res.diff <= l_res.max_points
 &&  l_res.empty_diff == 0
 &&  3 * l_res.max_points >= 2 * weight_street)
   {
    is_family_member *= (double)l_res.points / (double)l_res.max_points;
   }
 else
   {
    /****  no family member  ****/
    is_family_member = 0.0;
   }

 if (run_mode & TRACE_ADDR)
   {
    print_number ("total points for \"place\":  points", l_place.points, -1);
    print_number (",  max_points", l_place.max_points, -1);
    print_number (",  diff", l_place.diff, -1);
    print_number (",  empty_diff", l_place.empty_diff, -1);
    printf ("\n");
    printf ("\n");
   }


 /****  compare day, month and year of birth  ****/
 l_res = lev_x (a_b_day, a2_b_day,
            "day of birth", weight_b_day, run_mode);

 if (l_res.diff > 200)
   {
    l_res.diff = 200;
   }
 l_birth.points += l_res.points;
 l_birth.max_points += l_res.max_points;
 l_birth.diff += l_res.diff;
 l_birth.empty_diff += l_res.empty_diff;


 /****  compare month of birth  ****/
 s = a_b_month;
 if (strcmp (a_b_day, a2_b_month) == 0
 &&  strcmp (a_b_month, a2_b_day) == 0
 &&  IS_DIGIT (a_b_day[0])  &&  IS_DIGIT (a_b_month[0]))
   {
    /****  "inversion" between day and month  ****/
    s = a_b_day;
   }

 l_res = lev_x (s, a2_b_month,
            "month of birth", weight_b_month, run_mode);

 if (l_res.diff > 200)
   {
    l_res.diff = 200;
   }
 l_birth.points += l_res.points;
 l_birth.max_points += l_res.max_points;
 l_birth.diff += l_res.diff;
 l_birth.empty_diff += l_res.empty_diff;


 if (l_birth.points > 0
 ||  l_birth.diff == 0  ||  l_birth.empty_diff == 0)  
   {
    /****  formatting strings for year of birth  ****/
    i = (int) strlen (a_b_year);
    k = (int) strlen (a2_b_year);
    if (i == 4  &&  k == 2  &&  atoi (a_b_year) > 1000)
      {
       a_b_year += 2;
      }
    if (k == 4  &&  i == 2  &&  atoi (a2_b_year) > 1000)
      {
       a2_b_year += 2;
      }

    /****  compare year of birth  ****/
    l_res = lev_x (a_b_year, a2_b_year,
               "year of birth", weight_b_year, run_mode);

    if (l_res.diff > 200)
      {
       l_res.diff = 200;

       if (i >= 2  &&  k >= 2
       &&  IS_DIGIT (*a_b_year)  &&  IS_DIGIT (*a2_b_year))
         {
          n = atoi (a_b_year) - atoi (a2_b_year);
          if (n < 0)
            {
             n = -n;
            }
          if (i == 2  &&  k == 2  &&  n > 50)
            {
             n = 100 - n;
            }

          if (n > 20)
            {
             l_res.diff = 400;
             l_birth.points -= 200;    /**  "-=" !!  **/

             if (l_birth.points < 0)
               {
                l_birth.points = 0;
               }
            }
         }
      }

    l_birth.points += l_res.points;
    l_birth.max_points += l_res.max_points;
    l_birth.diff += l_res.diff;
    l_birth.empty_diff += l_res.empty_diff;
   }


 /****  evaluate results for birthday  ****/
 if ((run_mode & ACCEPT_SIMILAR_BIRTHDAYS)
 &&  l_birth.diff >= 500
 &&  l_birth.empty_diff == 0
 && (strcmp (a2_b_day,"01") == 0  ||  strcmp (a_b_day,"01") == 0))
   {
    /****  birthdays differ much  -->  ***/
    /****  calculate difference in months  ****/
    k = 12 * (atoi (a2_b_year) - atoi (a_b_year))
         + atoi (a2_b_month) - atoi (a_b_month);

    k = (k + 1200) % 1200;
    if (k > 600)
      {
       k = k - 1200;
      }

    if ((k >= 0  &&  k <= 1  &&  strcmp (a2_b_day,"01") == 0)
    ||  (k >= -1  &&  k <= 0  &&  strcmp (a_b_day,"01") == 0))
      {
       /****  One birthday is (e.g.) "May 19" and  ****/
       /****  the other "near" it (e.g. "June 1")  ****/
       l_birth.points = weight_b_year + 50;
       l_birth.diff = 300;
      }
   }

 if (l_birth.points <= weight_b_day
 &&  l_birth.diff >= 250
 &&  3 * min_points >= 2 * MAX_POINTS)
   {
    /****  birthdays differ too much  ****/
    l_birth.points = 0;
   }

 if (run_mode & TRACE_ADDR)
   {
    print_number ("total points for birhday:  points", l_birth.points, -1);
    print_number (",  max_points", l_birth.max_points, -1);
    print_number (",  diff", l_birth.diff, -1);
    print_number (",  empty_diff", l_birth.empty_diff, -1);
    printf ("\n");
    printf ("\n");
   }


 if (l_birth.diff > 0)
   {
    /****  birthdays differ:    look at               ****/
    /****  first name, family name, street, ZIP code  ****/
    k = diff_first_name;
    n = 400;

    if (k <= 0
    && (max_points_first_name == 0  ||  empty_diff_first_name != 0))
      {
       /****  first name is empty: compare family names  ****/
       k = diff_fam_name;
       n = 200;
      }

    if (k > 100  ||  comp_gender == GENDER_IS_DIFFERENT)
      {
       /****  first names and birthdays differ  ****/
       if (k < n + 50)
         {
          l_nam.points -= (int) (((long) k * l_birth.diff) / (50L + n));
         }
       else
         {
          l_nam.points -= l_birth.diff;
         }
      }

    k = diff_fam_name;
    n = 400;

    if (k <= 0
    && (max_points_fam_name == 0  ||  empty_diff_fam_name != 0))
      {
       /****  family name is empty: compare first names  ****/
       k = diff_first_name;
       n = 200;
      }

    if (k > 100)
      {
       /****  family names and birthdays differ  ****/
       if (k < n + 50)
         {
          l_nam.points -= (int) (((long) k * l_birth.diff) / (50L + n));
         }
       else
         {
          l_nam.points -= l_birth.diff;
         }
      }

    if (run_mode & TRACE_ADDR)
      {
       printf ("points for \"nam\" (after \"extra checks with birthday\"):\n");
       print_number ("      points", l_nam.points, -1);
       print_number (",  max_points", l_nam.max_points, -1);
       print_number (",  diff", l_nam.diff, -1);
       print_number (",  empty_diff", l_nam.empty_diff, -1);
       printf ("\n");
       printf ("\n");
      }
   }


 /****  compare phone number  ****/
 if (a_phone_number[0] != '\0'
 && (a_phone_number[0] != wildcard_any_string
   ||  a_phone_number[1] != '\0')
 &&  IS_XLETTER (a2_phone_number[0]))    /***  a2_<*> !!  ***/
   {
    l_res.max_points = WEIGHT_PHONE_NUMBER;
    k = 0;

    if (strcmp (a_phone_number, a2_phone_number) == 0)
      {
       /****  phone numbers are equal  ****/
       if (run_mode & TRACE_ADDR)
         {
          printf ("Phone numbers are equal.\n");
         }

       i = (int)strlen (a_phone_number);

       if (i >= 6)
         {
          k = l_res.max_points;
         }
       else if (i >= 5)
         {
          k = 3 * (int) (l_res.max_points / 4);
         }
       else if (i >= 4)
         {
          k = (int) (l_res.max_points / 2);
         }
      }
    else
      {
       l_res = lev_x (a_phone_number, a2_phone_number,
                  "phone number", WEIGHT_PHONE_NUMBER, run_mode);
       i = l_res.diff;

       if (i == 0)
         {
          /****  phone numbers are matching  ****/
          k = l_res.points;

          if (comp_gender == GENDER_IS_DIFFERENT)
            {
             /****  this might be a couple  ****/
             k = (int) (k / 2);
            }
         }
      }

    if (k > l_cust.points)
      {
       l_cust.points = k;
       l_cust.max_points = l_res.max_points;
      }
    if (run_mode & TRACE_ADDR)
      {
       print_number ("phone number:  cust_points = ", l_cust.points, -1);
       print_number (",  k", k, -1);
       printf ("\n");
       printf ("\n");
       printf ("\n");
      }
   }


 /****  compare mobile number  ****/
 if (a_mobile_number[0] != '\0'
 && (a_mobile_number[0] != wildcard_any_string
   ||  a_mobile_number[1] != '\0')
 &&  IS_XLETTER (a2_mobile_number[0]))    /***  a2_<*> !!  ***/
   {
    l_res.max_points = WEIGHT_MOBILE_NUMBER;
    k = 0;

    if (strcmp (a_mobile_number, a2_mobile_number) == 0)
      {
       /****  mobile numbers are equal  ****/
       if (run_mode & TRACE_ADDR)
         {
          printf ("Mobile numbers are equal.\n");
         }

       i = (int)strlen (a_mobile_number);

       if (i >= 8)
         {
          k = l_res.max_points;
         }
       else if (i >= 7)
         {
          k = 3 * (int) (l_res.max_points / 4);
         }
       else if (i >= 6)
         {
          k = (int) (l_res.max_points / 2);
         }
      }
    else
      {
       l_res = lev_x (a_mobile_number, a2_mobile_number,
                  "mobile number", WEIGHT_MOBILE_NUMBER, run_mode);
       i = l_res.diff;

       if (i == 0)
         {
          /****  mobile numbers are matching  ****/
          k = l_res.points;

          if (comp_gender == GENDER_IS_DIFFERENT)
            {
             /****  this might be a couple  ****/
             k = (int) (k / 2);
            }
         }
      }

    if (k > l_cust.points)
      {
       l_cust.points = k;
       l_cust.max_points = l_res.max_points;
      }
    if (run_mode & TRACE_ADDR)
      {
       print_number ("mobile number:  cust_points = ", l_cust.points, -1);
       print_number (",  k", k, -1);
       printf ("\n");
       printf ("\n");
      }
   }


 /****  compare email addresses  ****/
 if (a_email_addr[0] != '\0'  &&  a2_email_addr[0] != '\0'
 &&  l_cust.points < WEIGHT_EMAIL_ADDR)
   {
    l_res.max_points = WEIGHT_EMAIL_ADDR;
    k = 0;

    if (strcmp (a_email_addr, a2_email_addr) == 0)
      {
       /****  email addresses are equal  ****/
       if (run_mode & TRACE_ADDR)
         {
          printf ("Email addresses are equal.\n");
         }

       i = (int)strlen (a_email_addr);

       if (i >= 8)
         {
          k = l_res.max_points;
         }
       else if (i >= 5)
         {
          k = (int) (l_res.max_points / 2);
         }
      }
    else
      {
       l_res = lev_x (a_email_addr, a2_email_addr,
                  "email address", WEIGHT_EMAIL_ADDR, run_mode);
       i = l_res.diff;

       if (i == 0)
         {
          /****  email addresses are matching  ****/
          k = l_res.points;

          if (comp_gender == GENDER_IS_DIFFERENT)
            {
             /****  this might be a couple  ****/
             k = (int) (k / 2);
            }
         }
      }

    if (k > l_cust.points)
      {
       l_cust.points = k;
       l_cust.max_points = l_res.max_points;
      }
    if (run_mode & TRACE_ADDR)
      {
       print_number ("email address:  cust_points = ", l_cust.points, -1);
       print_number (",  k", k, -1);
       printf ("\n");
       printf ("\n");
      }
   }


 /****  compare customer numbers  ****/
 if (a_cust_number[0] != '\0'
 && (a_cust_number[0] != wildcard_any_string
   ||  a_cust_number[1] != '\0')
 &&  IS_XLETTER (a2_cust_number[0]))    /***  a2_<*> !!  ***/
   {
    l_cust.max_points = WEIGHT_CUST_NUMBER;
    k = 0;

    if (strcmp (a_cust_number, a2_cust_number) == 0)
      {
       /****  customer numbers are equal  ****/
       if (run_mode & TRACE_ADDR)
         {
          printf ("Customer numbers are equal.\n");
         }

       i = (int)strlen (a_cust_number);

       if (2 * i >= LENGTH_CUST_NUMBER)
         {
          k = l_res.max_points;
         }
       else if (i > 2)
         {
          k = (int) (l_res.max_points / 2)
             + i * (int) (l_res.max_points / LENGTH_CUST_NUMBER);
         }
      }
    else
      {
       l_res = lev_x (a_cust_number, a2_cust_number,
                  "customer number", WEIGHT_CUST_NUMBER, run_mode);
       i = l_res.diff;

       if (i == 0)
         {
          /****  customer numbers are matching  ****/
          k = l_res.points;

          if (comp_gender == GENDER_IS_DIFFERENT)
            {
             /****  this might be a couple  ****/
             k = (int) (k / 2);
            }
         }
      }

    if (k > l_cust.points)
      {
       l_cust.points = k;
       l_cust.max_points = l_res.max_points;
      }
    if (run_mode & TRACE_ADDR)
      {
       print_number ("customer number:  cust_points = ", l_cust.points, -1);
       print_number (",  k", k, -1);
       printf ("\n");
       printf ("\n");
      }
   }


 /****  compare bank account plus IBAN code  ****/
 if (a_bank_account[0] != '\0'
 && (a_bank_account[0] != wildcard_any_string
   ||  a_bank_account[1] != '\0')
 &&  IS_XLETTER (a2_bank_account[0]))    /***  a2_<*> !!  ***/
   {
    l_res.max_points = WEIGHT_BANK_ACCOUNT;
    k = 0;

    if (strcmp (a_iban_code, a2_iban_code) == 0
    &&  strcmp (a_bank_account, a2_bank_account) == 0)
      {
       /****  bank accounts are equal  ****/
       if (run_mode & TRACE_ADDR)
         {
          printf ("Bank accounts are equal.\n");
         }

       i = (int)strlen (a_iban_code);
       n = (int)strlen (a_bank_account);

       if (i >= 6  &&  n >= 7)
         {
          k = l_res.max_points;
         }
       else if (i >= 5  &&  n >= 6)
         {
          k = 3 * (int) (l_res.max_points / 4);
         }
       else if (i >= 4  &&  n >= 6)
         {
          k = (int) (l_res.max_points / 2);
         }
      }
    else
      {
       l_res = lev_x (a_iban_code, a2_iban_code,
                  "iban_code", WEIGHT_BANK_ACCOUNT, run_mode);
       i = l_res.diff;

       if (i == 0)
         {
          /****  iban codes are matching  ****/
          k = l_res.points;

          l_res = lev_x (a_bank_account, a2_bank_account,
                     "bank account", WEIGHT_BANK_ACCOUNT, run_mode);
          i = l_res.diff;
         }

       if (i == 0)
         {
          /****  bank accounts are matching  ****/
          if (l_res.points > k)
            {
             k = l_res.points;
            }

          if (comp_gender == GENDER_IS_DIFFERENT)
            {
             /****  this might be a couple  ****/
             k = (int) (k / 2);
            }
         }
      }

    if (k > l_cust.points)
      {
       l_cust.points = k;
       l_cust.max_points = l_res.max_points;
      }
    if (run_mode & TRACE_ADDR)
      {
       print_number ("bank account:  cust_points = ", l_cust.points, -1);
       print_number (",  k", k, -1);
       printf ("\n");
       printf ("\n");
      }
   }


 /****  calculate final result  ****/
 l_res.points  =  l_nam.points  + l_place.points  + l_birth.points;
 l_res.max_points = l_nam.max_points + l_place.max_points + l_birth.max_points;
 l_res.empty_diff = l_nam.empty_diff + l_place.empty_diff + l_birth.empty_diff;

 if (l_cust.points > 0)
   {
    /****  add "l_cust.points"  ****/
    l_res.points  +=  l_cust.points;
    l_res.max_points += l_cust.max_points;
    l_res.empty_diff += l_cust.empty_diff;
   }

 if (5 * l_nam.points <= l_nam.max_points)
   {
    l_res.points -= 5 * (l_nam.max_points - l_nam.points) / 4;
   }
 else if (5 * l_nam.points < 2 * l_nam.max_points)
   {
    l_res.points -= 2 * l_nam.max_points - 5 * l_nam.points;
   }

 if (5 * l_res.empty_diff > l_res.points)
   {
    if (5 * l_res.empty_diff >= 2 * l_res.points)
      {
       l_res.max_points += 2 * l_res.empty_diff;
      }
    else
      {
       i = 5 * l_res.empty_diff - l_res.points;
       l_res.max_points += 2 * i;
      }
   }


 if (run_mode & TRACE_ADDR)
   {
    print_number ("total:  points", l_res.points, -1);
    print_number (",  max_points", l_res.max_points, -1);
    printf ("\n");
   }

 if (! (run_mode & DO_UNWEIGHTED_COMPARISON))
   {
    /****  "normalize" points  ****/
    i = l_res.points - MAX_POINTS * 100;
    if (i > 0)
      {
       i = (int) (700.0 * calculate_ln (1.0 + ((double)i / 700.0)));
       l_res.points = i + MAX_POINTS * 100;
      }

    n = l_res.max_points - MAX_POINTS * 100;
    if (n > 0)
      {
       if (n > 3500)
         {
          n = 3500;
         }
       n = (int) (500.0 * calculate_ln (1.0 + ((double)n / 500.0)));
       l_res.max_points = n + MAX_POINTS * 100;
      }
   }

 /****  calculate final points  ****/
 if (l_res.max_points <= 0
 ||  l_res.points >= l_res.max_points)
   {
    l_res.points = 100 * MAX_POINTS;
   }
 else if (l_res.points <= 0)
   {
    l_res.points = 0;
   }
 else
   {
    l_res.points = (int) (100L * (long)MAX_POINTS * (long)l_res.points / (long)l_res.max_points);
   }

 if (run_mode & TRACE_ADDR)
   {
    print_number ("normalized total points", l_res.points, -1);
    printf ("\n");
   }

 if (l_res.points >= 100 * min_points  &&  l_res.points > 0)
   {
    /****  convert points to percent  ****/
    is_family_member = 0.0;
    l_res.points = (int) ((l_res.points + 30) / 100);
   }
 else if (is_family_member >= 0.01 * MIN_IS_FAMILY_MEMBER
 &&  min_points > IS_FAMILY_MEMBER)
   {
    l_res.points = IS_FAMILY_MEMBER;
   }
 else
   {
    l_res.points = 0;
   }

 if (run_mode & TRACE_ADDR)
   {
    printf ("\nfinal result:\n");
    printf ("points = %d,  min_points = %d", l_res.points, min_points);
    printf (",  is_family_member = %d\n", (int)(is_family_member/(0.01*MIN_IS_FAMILY_MEMBER)));
    printf ("\n");
   }

 return (l_res.points);
}

/************************************************************/
/****  end of file "addr.c"  ********************************/
/************************************************************/
