/*
 * dbselect.c
 * ----------
 *
 * Program for error-tolerant database selects.
 *
 * Copyright (c):
 * 2007:  Joerg MICHAEL, Adalbert-Stifter-Str. 11, 30655 Hannover, Germany
 *
 * SCCS: @(#) dbselect.c  1.1  2007-11-15
 *
 * This file is subject to the GNU Lesser General Public License (LGPL)
 * (formerly known as GNU Library General Public Licence)
 * as published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this file; if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Actually, the LGPL is __less__ restrictive than the better known GNU General
 * Public License (GPL). See the GNU Library General Public License or the file
 * LIB_GPLA.TXT for more details and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * There is one important restriction: If you modify this program in any way
 * (e.g. modify the underlying logic or translate this program into another
 * programming language), you must also release the changes under the terms
 * of the LGPL.
 * (However, since __this__ program is intended to be customized, all changes
 * covered by TO-DO comments are free.)
 *
 * That means you have to give out the source code to your changes,
 * and a very good way to do so is mailing them to the address given below.
 * I think this is the best way to promote further development and use
 * of this software.
 *
 * If you have any remarks, feel free to e-mail to:
 *     ct@ct.heise.de
 *
 * The author's email address is:
 *    astro.joerg@googlemail.com
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "addr_ext.h"
#define WORD_SEPARATORS   "+-/*( )&,'`.:"



/****  TO-DO: If you want to use this program      ****/
/****  in a production environment, delete the     ****/
/****  following macro:                            ****/
#define  FILE_FOR_STANDALONE_TEST  "samples.txt"

/****  TO-DO:  If you want to use this program     ****/
/****  in a library, delete the following macro:   ****/
  #define DBSELECT_EXECUTABLE
/****  (and do not forget to call the function      ****/
/****  "cleanup_addr()" when you exit the program)  ****/


/****  TO-DO:  In a non-German environment, delete        ****/
/****         "| COMPARE_LANGUAGE_GERMAN"  from RUN_MODE  ****/
#define RUN_MODE  (COMPARE_NORMAL | COMPARE_LANGUAGE_GERMAN | DATABASE_SELECT | SEARCH_FAMILY_MEMBERS | DB_WILDCARDS_FOR_LIKE | TRACE_ERRORS)

/****  TO-DO: If you database is very big, raise "MIN_POINTS_SELECT"  ****/
/****  (recommended values are between 80 and 90)                     ****/
#define MIN_POINTS_SELECT    81

/****  TO-DO: If you can supply phonetic variables for one     ****/
/****    or all three fields, activate the corresonding macro  ****/
/**  #define ACTIVATE_FIRST_NAME_PHONET  **/
/**  #define ACTIVATE_FAM_NAME_PHONET    **/
/**  #define ACTIVATE_STREET_PHONET      **/


/****  Position of first allowed wildcard in select      ****/
/****  predicates (first char = 0; values larger         ****/
/****  than  strlen(field)  mean: no wildcards allowed)  ****/
/****  TO-DO: Change the following macro, if necessary:  ****/
#define  MIN_POS_WILDCARD_FAM_NAME      4
#define  MIN_POS_WILDCARD_FIRST_NAME    2
/****  default: do not allow wildcards in the following fields:  ****/
#define  MIN_POS_WILDCARD_ZIP_CODE     100
#define  MIN_POS_WILDCARD_BIRTHDAY     100
#define  MIN_POS_WILDCARD_CUST_NUMBER  100

/****  TO-DO: change this value, if necessary  ****/
#define MAX_USEFUL_FOUND   40
#define MAX_FIELDS_SELECT  30


/****  basic macros and database variables  ****/
/****  TO-DO: change them, if necessary     ****/
#define SQL_SUCCESS        0L
#define SQL_NOT_FOUND     100L

/****  TO-DO: If your database uses a different        ****/
/****         date format,change the following macro:  ****/  
#define NON_EXISTING_BIRTHDAY   "01.01.0001"

#define NON_EXISTING_ENTRY    "##<<<++>>:::==###<<+++>>>::==="


/****  SQL result (error code)  ****/
static long sql_err_code;    

/****  database matchcode (MUST be unique !!)  ****/
static long db_matchcode;


/****  variables for the select "where"-clause  ****/
/****  TO-DO: change them, if necessary         ****/
static char s1_first_name [2*LENGTH_FIRST_NAME +1];
static char s2_first_name [2*LENGTH_FIRST_NAME +1];
static char s3_first_name [2*LENGTH_FIRST_NAME +1];
static char s4_first_name [2*LENGTH_FIRST_NAME +1];
static char s1_fam_name [2*LENGTH_FAM_NAME +1];
static char s2_fam_name [2*LENGTH_FAM_NAME +1];
static char s3_fam_name [2*LENGTH_FAM_NAME +1];
static char s4_fam_name [2*LENGTH_FAM_NAME +1];
static char s_zip_code [2*LENGTH_FAM_NAME +1];
static char s_full_birthday [2*LENGTH_FULL_BIRTHDAY +1];
static char s_cust_number [2*LENGTH_FAM_NAME +1];


/****  database variables for address comparison  ****/
/****  TO-DO: change them, if necessary           ****/
static char db_gender [LENGTH_GENDER+1];
static char db_first_name [LENGTH_FIRST_NAME +1];
static char db_fam_name [LENGTH_FAM_NAME +1];
static char db_c_o_name [LENGTH_C_O_NAME +1];
static char db_street [LENGTH_STREET +1];
static char db_zip_code [LENGTH_ZIP_CODE +1];
static char db_city [LENGTH_CITY +1];
static char db_country [LENGTH_COUNTRY +1];
static char db_full_birthday [LENGTH_FULL_BIRTHDAY +1];
static char db_phone_number [LENGTH_PHONE_NUMBER +1];
static char db_mobile_number [LENGTH_MOBILE_NUMBER +1];
static char db_email_addr [LENGTH_EMAIL_ADDR +1];
static char db_cust_number [LENGTH_CUST_NUMBER +1];
static char db_iban_code [LENGTH_IBAN_CODE +1];
static char db_bank_account [LENGTH_BANK_ACCOUNT +1];

#ifdef ACTIVATE_FIRST_NAME_PHONET
   static char db_first_name_ph [LENGTH_FIRST_NAME +1];
#endif
#ifdef ACTIVATE_FAM_NAME_PHONET
   static char db_fam_name_ph [LENGTH_FAM_NAME +1];
#endif
#ifdef ACTIVATE_STREET_PHONET
   static char db_street_ph [LENGTH_STREET +1];
#endif


static struct MAIL_ADDR db_addr[] =
   { { db_gender,       IS_GENDER      },
     { db_first_name,   IS_FIRST_NAME  },
     { db_fam_name,     IS_FAM_NAME    },
     { db_c_o_name,     IS_C_O_NAME    },
     { db_street,       IS_STREET      },

  #ifdef ACTIVATE_FIRST_NAME_PHONET
     { db_first_name_ph, IS_FIRST_NAME_PHONET },
  #endif
  #ifdef ACTIVATE_FAM_NAME_PHONET
     { db_fam_name_ph,  IS_FAM_NAME_PHONET },
  #endif
  #ifdef ACTIVATE_STREET_PHONET
     { db_street_ph,    IS_STREET_PHONET },
  #endif
     { db_zip_code,     IS_ZIP_CODE    },
     { db_city,         IS_CITY        },
     { db_country,      IS_COUNTRY     },
     { db_full_birthday,IS_FULL_BIRTHDAY },
     { db_phone_number, IS_PHONE_NUMBER },
     { db_mobile_number,IS_MOBILE_NUMBER },
     { db_email_addr,   IS_EMAIL_ADDR  },
     { db_cust_number,  IS_CUST_NUMBER },
     { db_iban_code,    IS_IBAN_CODE   },
     { db_bank_account, IS_BANK_ACCOUNT },
     {   NULL,             0           }
   };

#ifdef ACTIVATE_FIRST_NAME_PHONET
   static char s_first_name_ph [LENGTH_FIRST_NAME +1];
#endif
#ifdef ACTIVATE_FAM_NAME_PHONET
   static char s_fam_name_ph [LENGTH_FAM_NAME +1];
#endif
#ifdef ACTIVATE_STREET_PHONET
   static char s_street_ph [LENGTH_STREET +1];
#endif

static struct MAIL_ADDR search_addr_2 [MAX_FIELDS_SELECT];




/************************************************************/
/****  database functions  **********************************/
/************************************************************/

#ifdef FILE_FOR_STANDALONE_TEST

   /****  standalone test  ****/
   #include "dummy_db.h"

#else


/****  we are working on a real database  ****/

static void open_database (void)
{
 /**** TO-DO: Add "open" command for your database
  *           (e.g. similar to):
  *
  *  open_db (my_database);
  *
  ****/
}


static void close_database (void)
{
 /**** TO-DO: Add "close" command for your database
  *           (e.g. similar to):
  *
  *  close_db (my_database);
  *
  ****/
}


static void declare_cursor (void)
{
 /**** TO-DO: Add your SQL "declare"-statement here
  *           (e.g. similar to):
  *
  **** TO-DO: Delete unused fields
  *
  *  declare cursor_s cursor for
  *  select
  *     [database fields] matchcode, customer_number,
  *     gender, title, first_name, fam_name, c_o_name, 
  *     street, zip_code, country, full_birthday,
  *     phone_number, mobile_number, iban_code, bank_account
  *  into
  *     [strings] db_matchcode, db_cust_number,
  *     db_gender, db_title, db_first_name, db_fam_name,
  *     db_c_o_name, db_street, db_zip_code, db_country,
  *     db_full_birthday, db_phone_number, db_mobile_number,
  *     db_iban_code, db_bank_account
  *   from  my_table
  *  where (zip_code = s_zip_code
  *     and (fam_name like s1_fam_name
  *      or  fam_name like s2_fam_name
  *      or  fam_name like s3_fam_name
  *      or  fam_name like s4_fam_name))
  *   or (full_birthday = s_full_birthday
  *     and (first_name like s1_first_name
  *      or  first_name like s2_first_name
  *      or  first_name like s3_first_name
  *      or  first_name like s4_first_name)
  *   or  customer_number like s_cust_number;
  *
  **** TO-DO: If your database is not too large, you might use
  *           simpler select predicates (e.g.):
  *
  *  where  fam_name like s1_fam_name
  *     or  fam_name like s2_fam_name
  *     or  fam_name like s3_fam_name
  *     or  fam_name like s4_fam_name
  *     or  zip_code like s_zip_code
  *     or  full_birthday = s_full_birthday
  *     or  customer_number like s_cust_number;
  *
  **** TO-DO: If you use phonetic variables, you can either:
  *           a) store and extract them from your database
  *              (this means adding them to the above
  *               "declare_cursor" function)
  *           or
  *           b) create them "on the fly" in the "fetch_cursor"
  *              function. You can create them using "phonet"
  *              or any other program for phonetic conversion
  *
  **** TO-DO: Create all the necessary indexes in your database.
  *
  **** TO-DO: Change the function "check_lengths_of_search_strings",
  ****        if necessary.
  *
  ****/

 /**** TO-DO: Add your SQL "open"-statement here
  *           (e.g. similar to):
  *
  *  open (cursor_s);
  *
  ****/
}


static void fetch_cursor (void)
{
 /**** TO-DO: Add your SQL "fetch"-command here
  *           (e.g. similar to):
  *
  *  fetch (cursor_s);
  *
  ****/
}


static void close_cursor (void)
{
 /**** TO-DO: Add your SQL "close"-command here
  *           (e.g. similar to):
  *
  *  close (cursor_s);
  *
  ****/
}

#endif




static void copy_db_fields_to_search_result
                        (struct DB_SEARCH_RESULT *sr)
{
  /****  TO-DO:  Change this function, if necessary  ****/
  StrNCpy (sr->first_name,db_first_name,sizeof ((*sr).first_name));
  StrNCpy (sr->fam_name,  db_fam_name,  sizeof ((*sr).fam_name));
  StrNCpy (sr->city,      db_city,      sizeof ((*sr).city));
  StrNCpy (sr->full_birthday,db_full_birthday, sizeof ((*sr).full_birthday));
  StrNCpy (sr->cust_number, db_cust_number, sizeof((*sr).cust_number));
}





/*****************************************************************/
/****  functions for creating and checking the search strings  ***/
/*****************************************************************/

int create_search_strings (char first_name[],
         char fam_name[], char text1[], char text2[],
         char text3[], char text4[], int len,
         int min_pos_first_wildcard, int run_mode)

/****  Create search strings for database query with "like":      ****/
/****  Four search strings are created, one with first umlaut     ****/
/****  in "compressed" form, and another one with the first       ****/
/****  umlaut in "expanded" form.                                 ****/
/****  All other umlauts and all word separators are converted    ****/
/****  to '%'. Finally, one '%' is appended to the string.        ****/
/****  The third search string is needed to cover certain         ****/
/****  special cases (e.g. family name is very short or is also   ****/
/****  a valid first name, like "Paul").                          ****/
/****  ("len" = size of "text1", "text2", "text3", "text4" incl. '\0') ****/

/****  Example:                                                   ****/
/****  For the family name "lschlger" the search strings        ****/
/****  "Oelschl%ger%", "lschl%ger%" and "lschl%ger%"            ****/
/****  are created.                                               ****/
{
 int  i,k,n,pos;
 struct LEV_RESULT l_res;
 char c,c2,*s;
 char *t1,*t2,*t3,*t4;

 if (! (conv_strings_initialized & CONV_STRINGS_ARE_INITIALIZED))
   {
    initialize_conv_strings (run_mode);

    if (! (conv_strings_initialized & CONV_STRINGS_ARE_INITIALIZED))
      {
       /****  internal error  ****/
       return (ADDR_INTERNAL_ERROR);
      }
   }

 (void) delete_suffix_jun (fam_name, NULL);

 t1 = text1;
 t2 = text2;
 t3 = text3;
 t4 = text4;
 strcpy (text1,"");
 strcpy (text2,"");
 strcpy (text3,"");
 strcpy (text4,"");

 /****  check length of family name  ****/
 pos = -1;
 i = 0;
 while (IS_LETTER (fam_name[i]))
   {
    i++;
   }
 if (i > 0  &&  (i <= 2  ||  i < min_pos_first_wildcard)
 &&  fam_name[i] == '\0'  &&  len > i+2)
   {
    /****  If name is very short (e.g. "Li"), create  ****/
    /****  search strings "Li", "Li %" and "Li-%"     ****/
    strcpy (text1, fam_name);
    strcpy (text2, fam_name);

    text2[i] = ' ';
    text2[i+1] = DB_LIKE_ANY_STRING;
    text2[i+2] = '\0';
    strcpy (text3, text2);
    text3[i] = '-';
    strcpy (text4, text2);

    return (0);
   }

 if (run_mode & TRACE_ADDR)
   {
    printf ("Step 1:  search strings = '%s'%s'%s'%s'\n", t1,t2,t3,t4);
   }

 if (pos != -10  &&  i <= 1  &&  len > i+1
 && (fam_name[i] == '\''
  ||  fam_name[i] == ''  ||  fam_name[i] == '`'))
   {
    /****  name = (e.g.) "'t Hoff" (Dutch name)  ****/
    /****     or   "d'Alembert"  (French name)   ****/
    strncpy (text1, fam_name,i+1);
    strncpy (text2, text1,i+1);
    strncpy (text3, text1,i+1);
    strncpy (text4, text1,i+1);

    text1[i] = '\'';
    text2[i] = '';
    text3[i] = '`';

    fam_name += i+1;
    text1 += i+1;
    text2 += i+1;
    text3 += i+1;
    text4 += i+1;

    len -= i+1;
    pos = -10;
   }

 if (run_mode & TRACE_ADDR)
   {
    printf ("Step 2:  search strings = '%s'%s'%s'%s'\n", t1,t2,t3,t4);
   }

 if (run_mode & COMPARE_LANGUAGE_GERMAN)
   {
    if (pos != -10  &&  len > 5
    && (strncmp (fam_name,"von ",4) == 0
    ||  strncmp (fam_name,"v.",2) == 0))
      {
        strncpy (text1, "von ",4);
        strncpy (text2, "v.",2);
        text2[2] = DB_LIKE_ANY_STRING;
        strncpy (text3, text2,3);
        strncpy (text4, text2,3);

        if (strncmp (fam_name,"von ",4) == 0)
          {
           fam_name += 2;
          }
        fam_name += 2;

        text1 += 4;
        text2 += 3;
        text3 += 3;
        text4 += 3;
        len -= 4;
        pos = -10;
      }

    if (run_mode & TRACE_ADDR)
      {
       printf ("Step 3:  search strings = '%s'%s'%s'%s'\n", t1,t2,t3,t4);
      }

    if (pos != -10  &&  len > 4
    &&  IS_LETTER (fam_name[0])
    && (fam_name[1] == 'a'  ||  fam_name[1] == 'e')
    && (fam_name[2] == 'i'  ||  fam_name[2] == 'y'))
      {
        /****  Search for (e.g.) "Meier" and "Meyer"  ****/
        strncpy (text1, fam_name,2);
        text1[2] = 'i';
        strncpy (text2, fam_name,2);
        text2[2] = 'y';
        strncpy (text3, text2,3);
        text3[1] = 'e';
        strncpy (text4, text3,3);

        fam_name += 3;
        text1 += 3;
        text2 += 3;
        text3 += 3;
        text4 += 3;
        len -= 3;
        pos = -10;
      }
   }

 if (run_mode & TRACE_ADDR)
   {
    printf ("Step 4:  search strings = '%s'%s'%s'%s'\n", t1,t2,t3,t4);
   }

 if (pos != -10  &&  i == 1  &&  len > i+4
 &&  fam_name[i] != '\0'  &&  ! IS_LETTER (fam_name[i]))
   {
    /****  search strings like (e.g.) "A%B%Inc%" would  ****/
    /****  make for an extremely slow database select   ****/
    if (fam_name[i] == '.')
      {
       /****  name = (e.g.) "A. Miller Inc."  ****/
       strncpy (text1, fam_name,i+1);
       text1[i+1] = DB_LIKE_ANY_STRING;
       text1[i+2] = '\0';
       strcpy (text2, text1);
       strcpy (text3, text1);
       strcpy (text4, text1);

       fam_name += i+1;
       text1 += i+2;
       text2 += i+2;
       text3 += i+2;
       text4 += i+2;
       len -= i+2;
       /****  no "pos = -10;"  ****/
      }
    else if (fam_name[i] == ' ')
      {
       if (fam_name[i+1] != '\0'  &&  ! IS_LETTER (fam_name[i+1]))
         {
          /****  name = (e.g.) "A & B Inc."  ****/
          strncpy (text1, fam_name,i+2);
          text1[i+2] = DB_LIKE_ANY_STRING;
          text1[i+3] = '\0';
          strncpy (text2, fam_name,i);
          text2[i] = fam_name[i+1];
          text2[i+1] = DB_LIKE_ANY_STRING;
          text1[i+2] = '\0';
          strcpy (text3, text2);
          strcpy (text4, text2);

          fam_name += i+2;
          text1 += i+3;
          text2 += i+2;
          text3 += i+2;
          text4 += i+2;
          len -= i+2;
          pos = -10;
         }
      }
    else if (fam_name[i] != DB_LIKE_ANY_STRING
    &&  fam_name[i] != DB_LIKE_ANY_CHAR)
      {
       /****  name = (e.g.) "A&B Inc."  ****/
       strncpy (text1, fam_name,i);
       text1[i] = ' ';
       text1[i+1] = fam_name[i];
       text1[i+2] = DB_LIKE_ANY_STRING;
       text1[i+3] = '\0';
       strncpy (text2, fam_name,i+1);
       text2[i+1] = DB_LIKE_ANY_STRING;
       text2[i+2] = '\0';
       strcpy (text3, text2);
       strcpy (text4, text2);

       fam_name += i+1;
       text1 += i+3;
       text2 += i+2;
       text3 += i+2;
       text4 += i+2;
       len -= i+3;
       pos = -10;
      }
   }

 StrNCpy (text1, fam_name, len);

 if (run_mode & TRACE_ADDR)
   {
    printf ("Step 5:  search strings = '%s'%s'%s'%s'\n", t1,t2,t3,t4);
   }

 i = 0;
 while (text1[i] != '\0')
   {
    if (IS_LETTER (text1[i])  &&  ! IS_UMLAUT (text1[i])
    &&  IS_LETTER (text1[i+1])  &&  ! IS_UMLAUT (text1[i+1]))
      {
       c  = upperchar [(unsigned char) text1[i]];
       c2 = upperchar [(unsigned char) text1[i+1]];

       k = 0;
       if (IS_SORTCHAR (c)  &&  IS_SORTCHAR2 (c2))
         {
          for (k=0; k < HASH_COUNT; k++)
            {
             if (sortchar[k] == c  &&  sortchar2[k] == c2)
               {
                k = -1;
                break;
               }
            }
         }

       if (k < 0  &&  IS_SORTCHAR (c)  &&  IS_SORTCHAR2 (c2))
         {
          if (pos < 0  &&  pos != -10)
          /****  this is the first "expanded umlaut"  ****/
            {
             for (k=0; k < HASH_COUNT; k++)
               {
                if (sortchar[k] == c  &&  sortchar2[k] == c2)
                /****  "compress" expanded umlaut  ****/
                  {
                   c = (char) k;
                   if (IS_LOWER (text1[i]))
                     {
                      c = lowerchar [(unsigned char) c];
                     }

                   pos = i;
                   text1[i] = c;
                   break;
                  }
               }
            }
          else
            {
             text1[i] = DB_LIKE_ANY_STRING;
            }
          strcpy (text1+i+1, text1+i+2);
         }
      }
    else if (strchr (WORD_SEPARATORS, text1[i]) != NULL)
      {
       text1[i] = DB_LIKE_ANY_STRING;
      }
    else if (IS_UMLAUT (text1[i]))
      {
       if (pos < 0  &&  pos != -10)
         {
          /****  this is the first umlaut  ****/
          pos = i;
          k = (unsigned char) text1[i];
         }
       else
         {
          text1[i] = DB_LIKE_ANY_STRING;
         }
      }

    if (i > 0  &&  text1[i] == DB_LIKE_ANY_STRING
    &&  text1[i-1] == DB_LIKE_ANY_STRING)
      {
       /****  compress multiple '%' (e.g. "%%")  ****/
       strcpy (text1+i, text1+i+1);
       i--;
      }
    i++;
   }

 if (run_mode & TRACE_ADDR)
   {
    printf ("Step 6:  search strings = '%s'%s'%s'%s',  pos = %d\n",
               t1,t2,t3,t4, pos);
   }

 if (i >= len-1  &&  len >= 6)
   {
    i = len-2;
   }
 if (i == 0
 ||  (i > 0  &&  i < len-1  &&  text1[i-1] != DB_LIKE_ANY_STRING))
   {
    /****  add char DB_LIKE_ANY_STRING  ****/
    text1[i] = DB_LIKE_ANY_STRING;
    i++;
   }
 text1[i] = '\0';

 if (run_mode & COMPARE_LANGUAGE_GERMAN)
   {
    if (i > 5
    && (strcmp (text1+i-2, "e%") == 0
    ||  strcmp (text1+i-2, "s%") == 0
    ||  text1[i-3] == text1[i-2]
    ||  strcmp (text1+i-3, "dt%") == 0))
      {
       /****  shorten the search pattern  ****/
       text1[i-2] = DB_LIKE_ANY_STRING;
       text1[i-1] = '\0';
       i--;
      }
   }

 strcpy (text2,text1);
 strcpy (text3,text1);
 strcpy (text4,text1);

 if (run_mode & TRACE_ADDR)
   {
    printf ("Step 7:  search strings = '%s'%s'%s'%s'\n", t1,t2,t3,t4);
   }

 if (pos >= 0  &&  pos < len-1  &&  IS_UMLAUT (text1[pos]))
   {
    /****  change first umlaut  ****/
    n = (unsigned char) text1[pos];
    c = sortchar[n];
    c2 = sortchar2[n];

    if (IS_LOWER (text1[pos]))
      {
       c = lowerchar [(unsigned char) c];
      }
    if (! IS_UPPER (text1[pos+1]))
      {
       c2 = lowerchar [(unsigned char) c2];
      }

    if (! IS_LETTER (sortchar2[n]))
      {
       /****  accent (e.g. '') found  ****/
       strcpy (text2,text1);
       text2[pos] = c;
      }
    else
      {
       /****  "extendable" umlaut (e.g. '') found  ****/
       text2[pos] = c;
       text2[pos+1] = c2;

       strncpy (text2+pos+2, text1+pos+1, len-pos-3);
       if (i < len-1)
         {
          i++;
         }
       text2[i-1] = DB_LIKE_ANY_STRING;
       text2[i] = '\0';

       n = (unsigned char) text1[pos];
       c = up_and_conv[n];
       c2 = c;

       if (IS_LOWER (text1[pos]))
         {
          c2 = lowerchar [(unsigned char) c2];
         }

       k = 0;
       for (n=0; n < HASH_COUNT; n++)
         {
          if (up_and_conv [(unsigned char) n] == c)
            {
             s = text1;
             if (k == 1)
               {
                s = text3;
               }
             if (k == 2)
               {
                s = text4;
               }

             if (c2 == c)
               {
                *(s+pos) = (char) n;
               }
             else
               {
                *(s+pos) = lowerchar [n];
               }
             k++;

             if (run_mode & TRACE_ADDR)
               {
                printf ("Step 8.%d:  search strings = '%s'%s'%s'%s'\n",
                            k, t1,t2,t3,t4);
               }
            }
         }
      }
   }

 if (run_mode & TRACE_ADDR)
   {
    printf ("Step 9:  search strings = '%s'%s'%s'%s'\n", t1,t2,t3,t4);
   }

 /****  check for double name (e.g. "Miller-Smith")  ****/
 k = 0;
 while (fam_name[k] != '\0'
 &&  fam_name[k] != ' '  &&  fam_name[k] != '-'
 &&  fam_name[k] != DB_LIKE_ANY_STRING
 &&  fam_name[k] != DB_LIKE_ANY_CHAR)
   {
    k++;
   }

 if (fam_name[k] == ' '  ||  fam_name[k] == '-')
   {
    i = k;
    while (fam_name[i] == ' '  ||  fam_name[i] == '-')
      {
       i++;
      }

    if (k >= 4
    &&  IS_LOWER (fam_name[k-1])
    &&  IS_UPPER (fam_name[i])  &&  IS_LETTER (fam_name[i+1]))
      {
       n = 0;
       l_res = lev_x (text2, fam_name, "", 1000,
              (COMPARE_NORMAL | DATABASE_SELECT | DB_WILDCARDS_FOR_LIKE));

       if (l_res.diff != 0)
         {
          n++;
          l_res = lev_x (text3, fam_name, "", 1000,
                 (COMPARE_NORMAL | DATABASE_SELECT | DB_WILDCARDS_FOR_LIKE));

          if (l_res.diff != 0)
            {
             n += 2;
            }
         }

       if (n < 3)
         {
          if (n == 1)
            {
             strcpy (text2, text3);
            }

          k = 0;
          while (text2[k] != '\0')
            {
             l_res = lev_x (text2+k, fam_name+i, "", 1000,
                    (COMPARE_NORMAL | DATABASE_SELECT | DB_WILDCARDS_FOR_LIKE));

             if (l_res.diff == 0)
               {
                break;
               }
             k++;
            }

          /****  search for both names  ****/
          text2[k] = DB_LIKE_ANY_STRING;
          text2[k+1] = '\0';
          StrNCpy (text3, fam_name+i, len);
          blank_cut (text3, len);
         }
      }
   }

 if (run_mode & TRACE_ADDR)
   {
    printf ("Step 10: search strings = '%s'%s'%s'%s'\n", t1,t2,t3,t4);
   }

 if (strcmp (text2,text3) == 0
 &&  strcmp (fam_name, first_name) != 0
 &&  first_name[0] != '\0'
 &&  check_valid_first_name (fam_name) > 0        /**  > 0  !! **/
 &&  check_valid_first_name (first_name) >= 0)    /**  >= 0  !! **/
   {
    /****  Is it necessary to look for "name inversions" ?  ****/
    i = 0;
    while (IS_LETTER (first_name[i]))
      {
       i++;
      }
    if (i > 4  ||  (i > 2  &&  first_name[i] == '\0'))
      {
       /****  search for "name inversions"  *****/
       StrNCpy (t3, first_name, len);    /***  t3 !!  ***/
      }
   }

 if (run_mode & TRACE_ADDR)
   {
    printf ("Step 11: search strings = '%s'%s'%s'%s'\n", t1,t2,t3,t4);
   }

 return (0);
}




int check_search_field (char *field, int size_of_string,
            int min_pos_first_wildcard, int run_mode)
/****  check length of string for DB select  ****/
{
  int i,k;

  if (! (conv_strings_initialized & CONV_STRINGS_ARE_INITIALIZED))
    {
     initialize_conv_strings (run_mode);

     if (! (conv_strings_initialized & CONV_STRINGS_ARE_INITIALIZED))
       {
        /****  internal error  ****/
        return (ADDR_INTERNAL_ERROR);
       }
    }

  k = 0;
  i = 0;
  while (field[i] != '\0'
  &&  field[i] != DB_LIKE_ANY_STRING  &&  field[i] != DB_LIKE_ANY_CHAR)
    {
     if (IS_XLETTER (field[i]))
       {
        k++;
       }
     else
       {
        k += 2;
       }
     i++;
    }

  if (i > 0  &&  field[i] == '\0')
    {
     k += min_pos_first_wildcard;
    }

#ifdef FILE_FOR_STANDALONE_TEST
   k++;
#endif

  if ((run_mode & COMPARE_LANGUAGE_GERMAN)
  &&  k <= 3  &&  k == i
  && (strncmp (field,"Sch",k) == 0  ||  strncmp (field,"SCH",k) == 0))
    {
     /****  searching for (e.g.) "Sch%"  ****/
     k--;
    }

  if (k < min_pos_first_wildcard)
    {
     /****  this select would take too much time  ****/
     StrNCpy (field, NON_EXISTING_ENTRY, size_of_string);
     return (0);
    }

  return (1);
}




int check_lengths_of_search_strings (int run_mode)
{
  /****  TO-DO:  Change this function, if necessary  ****/
  int i,n,k;

  k = 0;
  i = 0;
  i += check_search_field (s1_fam_name, sizeof (s1_fam_name),
                    MIN_POS_WILDCARD_FAM_NAME, run_mode);
  i += check_search_field (s2_fam_name, sizeof (s2_fam_name),
                    MIN_POS_WILDCARD_FAM_NAME, run_mode);
  i += check_search_field (s3_fam_name, sizeof (s3_fam_name),
                    MIN_POS_WILDCARD_FAM_NAME, run_mode);
  i += check_search_field (s4_fam_name, sizeof (s4_fam_name),
                    MIN_POS_WILDCARD_FAM_NAME, run_mode);

  n = check_search_field (s_zip_code, sizeof (s_zip_code),
                    MIN_POS_WILDCARD_ZIP_CODE, run_mode);

  k += i + n;
  if (i > 0  &&  n > 0)
    {
      /****  select predicates are "good" enough  ****/
      return (1);
    }

  i = 0;
  i += check_search_field (s1_first_name, sizeof (s1_first_name),
                    MIN_POS_WILDCARD_FIRST_NAME, run_mode);
  i += check_search_field (s2_first_name, sizeof (s2_first_name),
                    MIN_POS_WILDCARD_FIRST_NAME, run_mode);
  i += check_search_field (s3_first_name, sizeof (s3_first_name),
                    MIN_POS_WILDCARD_FIRST_NAME, run_mode);
  i += check_search_field (s4_first_name, sizeof (s4_first_name),
                    MIN_POS_WILDCARD_FIRST_NAME, run_mode);

  n = check_search_field (s_full_birthday, sizeof (s_full_birthday),
                    MIN_POS_WILDCARD_BIRTHDAY, run_mode);
  if (n == 0)
    {
     /****  important  ****/
     StrNCpy (s_full_birthday, NON_EXISTING_BIRTHDAY,
                   sizeof (s_full_birthday));
    }

  k += i + n;
  if (i > 0  &&  n > 0)
    {
      /****  select predicates are "good" enough  ****/
      return (1);
    }

  n = check_search_field (s_cust_number, sizeof(s_cust_number),
                    MIN_POS_WILDCARD_CUST_NUMBER, run_mode);

  k += n;
  if (n > 0)
    {
      /****  select predicates are "good" enough  ****/
      return (1);
    }

#ifdef FILE_FOR_STANDALONE_TEST
   return (k);
#endif

  return (0);
}




/************************************************************/
/****  "external" functions  ********************************/
/************************************************************/

int database_select (struct MAIL_ADDR search_addr[],
       int min_points, struct DB_SEARCH_RESULT *search_results[],
       struct DB_SEARCH_RESULT storage_area[],
       int max_useful_found, int run_mode)

/****  Do a database select. Results are     ****/
/****  stored in the array "search_results". ****/

/****  This function returns the number of matches   ****/
/****  (In case of database errors, < 0 is returned) ****/
{
 struct DB_SEARCH_RESULT *sr;
 int i,k,points,diff;
 int number_of_matches = 0;
 int num_fam = 0;

 /****  TO-DO: Add more fields, if you need them as     ****/
 /****  select predicates in function "declare_cursor"  ****/
 char *p_first_name = "";
 char *p_fam_name = "";
 char *p_street   = "";
 char *p_zip_code = "";
 char *p_full_birthday = "";
 char *p_cust_number = "";

 if (! (conv_strings_initialized & CONV_STRINGS_ARE_INITIALIZED))
   {
    initialize_conv_strings (run_mode);

    if (! (conv_strings_initialized & CONV_STRINGS_ARE_INITIALIZED))
      {
       /****  internal error  ****/
       if (run_mode & (TRACE_ADDR | TRACE_ERRORS))
         {
           printf ("Internal error: Initialization failed\n");
         }
       return (ADDR_INTERNAL_ERROR);
      }
   }

 if (search_addr == NULL  ||  search_addr[0].info == 0)
   {
    /****  error  ****/
    if (run_mode & (TRACE_ADDR | TRACE_ERRORS))
      {
        printf ("Error: No search criteria given.\n");
      }
    return (ADDR_INSUFFICIENT_SEL_CRITERIA);
   }

 /****  read search address  ****/
 for (i=0; search_addr[i].info > 0; i++)
   {
    switch (search_addr[i].info)
      {
       case IS_FIRST_NAME:  p_first_name = search_addr[i].text;
                     break;
       case IS_FAM_NAME  :  p_fam_name = search_addr[i].text;
                     break;

       case IS_STREET    :  p_street = search_addr[i].text;
                     break;
       case IS_ZIP_CODE  :  p_zip_code = search_addr[i].text;
                     break;

       case IS_FULL_BIRTHDAY :  p_full_birthday = search_addr[i].text;
                     break;

       case IS_CUST_NUMBER :  p_cust_number = search_addr[i].text;
                     break;

       /****  TO-DO: Add more fields, if you need them as     ****/
       /****  select predicates in function "declare_cursor"  ****/
      }
   }

 /****  copy ZIP code, birthday and customer number for DB select  ****/
 StrNCpy (s_zip_code, p_zip_code, LENGTH_ZIP_CODE +1);
 StrNCpy (s_full_birthday, p_full_birthday, LENGTH_FULL_BIRTHDAY +1);
 StrNCpy (s_cust_number, p_cust_number, LENGTH_CUST_NUMBER+1);

 /****  create family names for database select  ****/
 (void) create_search_strings (p_first_name, p_fam_name,
            s1_fam_name, s2_fam_name, s3_fam_name, s4_fam_name,
            2*LENGTH_FAM_NAME+1, MIN_POS_WILDCARD_FAM_NAME,
            (run_mode & ~TRACE_ADDR));

 /****  create first names for database select  ****/
 (void) create_search_strings ("", p_first_name,
            s1_first_name, s2_first_name, s3_first_name, s4_first_name,
            2*LENGTH_FIRST_NAME+1, MIN_POS_WILDCARD_FIRST_NAME,
            (run_mode & ~TRACE_ADDR));

 /****  check lengths of search strings  ****/
 k = check_lengths_of_search_strings (run_mode);

 if (run_mode & TRACE_ADDR)
   {
     /****  TO-DO: Add more fields, if you use them as      ****/
     /****  select predicates in function "declare_cursor"  ****/
     printf ("\n");
     printf ("Doing \"primary\" DB select\n");
     printf ("   for (family name \"%s\"  and  ZIP code \"%s\")\n",
               p_fam_name, p_zip_code);
     printf ("    or (birthday \"%s\"  and  first name \"%s\")\n",
               p_full_birthday, p_first_name);
     printf ("    or customer number \"%s\"\n", p_cust_number);
     printf ("\n");
   } 

 if (k == 0)
   {
    /****  this select would take too much time  ****/
    if (run_mode & (TRACE_ADDR | TRACE_ERRORS))
      {
        printf ("Error: This select would take too much time.\n");
      }
    return (ADDR_INSUFFICIENT_SEL_CRITERIA);
   }

 if (run_mode & COMPARE_LANGUAGE_GERMAN)
   {
    i = 0;
    while (search_addr[i].info > 0  &&  i < MAX_FIELDS_SELECT -5)
      {
       search_addr_2[i].text = search_addr[i].text;
       search_addr_2[i].info = search_addr[i].info;
       i++;
      }

 #ifdef ACTIVATE_FIRST_NAME_PHONET
    (void) phonet (p_first_name, s_first_name_ph,
               LENGTH_FIRST_NAME+1, PHONET_MODE);

    search_addr_2[i].text = s_first_name_ph;
    search_addr_2[i].info = IS_FIRST_NAME_PHONET;
    i++;
 #endif

 #ifdef ACTIVATE_FAM_NAME_PHONET
    (void) phonet (p_fam_name, s_fam_name_ph,
               LENGTH_FAM_NAME+1, PHONET_MODE);

    search_addr_2[i].text = s_fam_name_ph;
    search_addr_2[i].info = IS_FAM_NAME_PHONET;
    i++;
 #endif

 #ifdef ACTIVATE_STREET_PHONET
    (void) phonet (p_street, s_street_ph,
               LENGTH_STREET+1, PHONET_MODE);

    search_addr_2[i].text = s_street_ph;
    search_addr_2[i].info = IS_STREET_PHONET;
    i++;
 #endif

    search_addr_2[i].text = NULL;
    search_addr_2[i].info = 0;

    search_addr = search_addr_2;
   }

 /****  do the database search  ****/
 sql_err_code = SQL_SUCCESS -1;
 run_mode |= DATABASE_SELECT | DB_WILDCARDS_FOR_LIKE;

 declare_cursor();

 if (sql_err_code != SQL_SUCCESS)
   {
    if (run_mode & (TRACE_ADDR | TRACE_ERRORS))
      {
        printf ("Error when doing SQL 'declare'-command.\n");
      }
    return (ADDR_SQL_DECLARE_ERROR);
   }

 if (min_points >= MAX_POINTS)
   {
    min_points = MAX_POINTS -5;
   }
 for (i=0; i< max_useful_found; i++)
   {
    storage_area[i].matchcode = -1L;
    storage_area[i].points = 0;
    search_results[i] = &(storage_area[i]);
   }

 /****  reset database variables  ****/
 for (i=0; db_addr[i].info > 0; i++)
   {
    strcpy (db_addr[i].text, "");
   }


 /****  search for matches  ****/
 for (;;)
   {
    fetch_cursor();

    if (sql_err_code == SQL_SUCCESS)
      {
       /****  format database address  ****/
       for (i=0; db_addr[i].info != 0; i++)
         {
          blank_cut (db_addr[i].text, 0);
         }

   #ifdef ACTIVATE_FIRST_NAME_PHONET
       (void) phonet (db_first_name, db_first_name_ph,
                  LENGTH_FIRST_NAME+1, PHONET_MODE);
   #endif

   #ifdef ACTIVATE_FAM_NAME_PHONET
       (void) phonet (db_fam_name, db_fam_name_ph,
                  LENGTH_FAM_NAME+1, PHONET_MODE);
   #endif

   #ifdef ACTIVATE_STREET_PHONET
       (void) phonet (db_street, db_street_ph,
                  LENGTH_STREET+1, PHONET_MODE);
   #endif

       /****  compare addresses  ****/
       points = compare_addr (search_addr, db_addr, min_points,
                        ((run_mode | SKIP_BLANKCUT) & ~TRACE_ADDR));

       if (points >= min_points  ||  points == IS_FAMILY_MEMBER)
         {
          if (points == IS_FAMILY_MEMBER
          &&  min_points > IS_FAMILY_MEMBER)
            {
             if (min_points > MAX_POINTS - 15)
               {
                i = compare_addr (search_addr, db_addr, MAX_POINTS-15,
                         ((run_mode | SKIP_BLANKCUT) & ~TRACE_ADDR));
                if (i > points)
                  {
                   points = i;
                  }
               }

             if (points == IS_FAMILY_MEMBER
             &&  ! (run_mode & SEARCH_FAMILY_MEMBERS))
               {
                points = -1;
               }
            }

          if (points > 0)
            {
             /****  add database entry to list of matches  ****/
             number_of_matches++;
             if (number_of_matches > max_useful_found)
               {
                /****  too many matches found  ****/
                break;
               }

             /****  sort list of matches (best match first)  ****/
             i = number_of_matches - 1;
             sr = search_results[i];
             while (i > 0  &&  points > search_results[i-1]->points)
               {
                search_results[i] = search_results[i-1];
                i--;
               }
             search_results[i] = sr;
             sr->matchcode = db_matchcode;
             sr->points = points;
             copy_db_fields_to_search_result (sr);

             if ((run_mode & SEARCH_FAMILY_MEMBERS)
             &&  sr->points == IS_FAMILY_MEMBER)
               {
                num_fam++;
               }

             if (number_of_matches >= max_useful_found)
               {
                /****  drop least-matching entries from list  ****/
                diff = 0;
                i = number_of_matches - num_fam -1;
                if (i > 0)
                  {
                   diff = search_results[0]->points - search_results[i]->points;
                  }

                if (diff >= 15)
                  {
                   min_points = search_results[i]->points + 1;

                   k = i-1;
                   while (k > 1
                   &&  search_results[k]->points < min_points)
                     {
                      k--;
                     }
                   k++;

                   diff = i - k + 1;
                   while (k < number_of_matches - diff)
                     {
                      search_results[k]->points = 0;
                      sr = search_results[k];
                      search_results[k] = search_results[k+diff];
                      search_results[k+diff] = sr;
                      k++;
                     }
                   number_of_matches -= diff;
                  }

                if (2 * num_fam > max_useful_found
                &&  number_of_matches >= max_useful_found)
                  {
                   /****  too many family members found  ****/
                   i = number_of_matches - num_fam;
                   for (k=0; k< num_fam; k++)
                     {
                      search_results[i+k]->points = 0;
                     }
                   number_of_matches = i;
                   num_fam = 0;
                   run_mode &= ~SEARCH_FAMILY_MEMBERS;
                  }
               }
            }
         }
      }

    else if (sql_err_code == SQL_NOT_FOUND)
      {
       break;
      }
    else
      {
       if (run_mode & (TRACE_ADDR | TRACE_ERRORS))
         {
           printf ("Error when doing SQL 'fetch'-command.\n");
         }
       return (ADDR_SQL_FETCH_ERROR);
      }
   }

 close_cursor();

 if (sql_err_code != SQL_SUCCESS)
   {
    if (run_mode & (TRACE_ADDR | TRACE_ERRORS))
      {
        printf ("Error when doing SQL 'close'-command.\n");
      }
    return (ADDR_SQL_CLOSE_ERROR);
   }

 if (number_of_matches > max_useful_found)
   {
    /****  too many matches found  ****/
    return (ADDR_TOO_MANY_MATCHES_FOUND);
   }
 return (number_of_matches);
}





#ifdef DBSELECT_EXECUTABLE

/****  array to store the search results:  ****/
static struct DB_SEARCH_RESULT *search_results [MAX_USEFUL_FOUND];
static struct DB_SEARCH_RESULT storage_area [MAX_USEFUL_FOUND];


int main (int argc, char *argv[])
{
 struct DB_SEARCH_RESULT *sr;
 int  i,k,n;

 struct MAIL_ADDR arg_addr[] =
   { { NULL,  IS_FIRST_NAME },
     { NULL,  IS_FAM_NAME   },
     { NULL,  IS_STREET     },
     { NULL,  IS_ZIP_CODE   },
     { NULL,  IS_FULL_BIRTHDAY },
     { NULL,     0          }
   };

 if (argc < 3
 ||  strcmp (argv[1], "-?") == 0
 ||  strcmp (argv[1], "-h") == 0
 ||  strcmp (argv[1], "-help") == 0)
   {
    printf ("Usage:  %s  -search  <address>  [-trace]\n", argv[0]);
    printf ("\n");
    printf ("Program for database queries\n");
    printf ("(searching an address in a database)\n");
    printf ("\n");
    printf ("<address> must be given as:\n");
    printf ("   <first_name> <fam_name> <street> <zip_code> <birthday>\n");
    printf ("Supported date formats for birthday:\n");
    printf ("   <year>-<month>-<day>, <day>.<month>.<year>  and  <month>/<day>/<year>\n");
    printf ("\n");
    printf ("Note:\n");
    printf ("In this program, you must use DB wildcards for like (i.e. '%c' and '%c')\n",
                    DB_LIKE_ANY_STRING, DB_LIKE_ANY_CHAR);
    return (1);
   }

 if (strncmp (argv[1],"-search",7) == 0
 ||  strncmp (argv[1],"-select",7) == 0)
   {
    if (argc <= 6)
      {
        printf ("Error: wrong # of arguments.\n");
        return (1);
      }

    /****  do a database query  ****/
    for (k=0; k<5; k++)
      {
       arg_addr[k].text = argv[k+2];
      }

    i = RUN_MODE;
    if (argc >= 8  &&  strcmp (argv[7],"-trace") == 0)
      {
       i |= TRACE_ADDR;
      }

    open_database();
    if (sql_err_code != SQL_SUCCESS)
      {
       printf ("Error when opening database\n");
       cleanup_addr();
       return (1);
      }

    n = database_select (arg_addr, MIN_POINTS_SELECT,
             search_results, storage_area, MAX_USEFUL_FOUND, i);

    if (n == 0)
      {
       printf ("No matches found.\n");
      }
    else if (n == ADDR_TOO_MANY_MATCHES_FOUND)
      {
       printf ("Too many matches found.\n");
       printf ("Narrow down your search criteria.\n");
       printf ("\n");
       printf ("First %d matches are:\n", MAX_USEFUL_FOUND);

       for (i=0; i< MAX_USEFUL_FOUND; i++)
         {
          sr = search_results[i];
          printf ("Points = %d,   matchcode = %ld,   name = '%s'%s',",
              sr->points, sr->matchcode, sr->first_name, sr->fam_name);
          printf ("  birthday ='%1s', customer number = '%s'\n",
              sr->full_birthday, sr->cust_number);
         }
      }
    else if (n > 0)
      {
       printf ("%d match(es) found:\n", n);
       for (i=0; i<n; i++)
         {
          sr = search_results[i];
          printf ("Points = %d,   matchcode = %ld,   name = '%s'%s',",
              sr->points, sr->matchcode, sr->first_name, sr->fam_name);
          printf ("  birthday ='%1s', customer number = '%s'\n",
              sr->full_birthday, sr->cust_number);
         }
      }
    else
      {
       /****  n < 0:  we have an error  ****/
       if (n == ADDR_INSUFFICIENT_SEL_CRITERIA)
         {
          printf ("Insufficient select criteria given.\n");
         }
       else if (n == ADDR_INTERNAL_ERROR)
         {
          printf ("Internal error (initialization failed).\n");
         }
       else
         {
          printf ("SQL error.\n");
         }
      }

    close_database();
    if (sql_err_code != SQL_SUCCESS)
      {
       printf ("Error when closing database\n");
       cleanup_addr();
       return (1);
      }
   }

 cleanup_addr();
 return (0);
}

#endif

/************************************************************/
/****  end of file "dbselect.c"  ****************************/
/************************************************************/
