// $Id: ct_spirit.cpp 2015 2007-12-11 09:41:03Z olau $
// Copyright (c) 2007 Torsten T. Will
// Copyright (c) 2007 Heise Zeitschriften Verlag, Hannover
// Alle Rechte vorbehalten. All rights reserved.

#include <boost/spirit.hpp>

using namespace boost::spirit;

//////////////////////////////////////////////////////////////////////////
// basic helpers

template<typename PARSER>
void do_parse(std::string const &name, PARSER &p, std::string const &str)
{
    std::cout << "Parser '" << name << "' parses '" << str << "'... "
        << ( parse(str.c_str(), p).full ? "ok." : "fail!" )
        << std::endl;
}
template<typename PARSER>
void dont_parse(std::string const &name, PARSER &p, std::string const &str)
{
    std::cout << "Parser '" << name << "' parses not '" << str << "'... "
        << ( parse(str.c_str(), p).full ? "ok=wrong!" : "fail=correct." )
        << std::endl;
}

//////////////////////////////////////////////////////////////////////////

void do_01_numbers()
{
    std::cout << "=== do_01_numbers ===" << std::endl;
    std::string const name = "numbers";

    rule<> my_p = uint_p >> *(space_p >> uint_p);

    do_parse(name, my_p, "1 2 3 4");
    do_parse(name, my_p, "9");
    do_parse(name, my_p, "101 2002 31 4 0 2");

    dont_parse(name, my_p, "");
    dont_parse(name, my_p, "wort");
    dont_parse(name, my_p, "1 2 3 hello 4");
    dont_parse(name, my_p, "3.4");
    dont_parse(name, my_p, "1 2 3 3.14 4");
}

//////////////////////////////////////////////////////////////////////////

void do_02_numbers_list()
{
    std::cout << "=== do_02_numbers_list ===" << std::endl;
    std::string const name = "num_list";

    rule<> my_p = list_p(uint_p, space_p);

    do_parse(name, my_p, "1 2 3 4");
    do_parse(name, my_p, "9");
    do_parse(name, my_p, "101 2002 31 4 0 2");

    dont_parse(name, my_p, "");
    dont_parse(name, my_p, "dhdn");
    dont_parse(name, my_p, "1 2 3 hello 4");
    dont_parse(name, my_p, "3.4");
    dont_parse(name, my_p, "1 2 3 3.14 4");
}

//////////////////////////////////////////////////////////////////////////

void do_03_beispiel_oneliner()
{
    std::cout << "=== do_03_beispiel_oneliner ===" << std::endl;
    //ct-start
    bool ok = parse("2 3 5 7 11 13",
        uint_p >> *(space_p >> uint_p) ).full;
    //ct-end
    std::cerr << "parse should be ok: " << ( ok ? "fine." : "bad!") << std::endl;
}

//////////////////////////////////////////////////////////////////////////

void do_04_beispiel_ebnfig()
{
    std::cout << "=== do_04_beispiel_ebnfig ===" << std::endl;
    std::string const name = "ebnfig";

    rule<> expr, term, factor;
    term = '(' >> expr >> ')'
        | '-' >> expr
        | uint_p;
    factor = term >> !( ('+' | ch_p('-')) >> term);
    expr = factor >> !( ('*' | ch_p('/')) >> factor);
    bool ok = parse("(3+4)*5", expr).full;

    std::cerr << "parse should be ok: " << ( ok ? "fine." : "bad!") << std::endl;

    do_parse(name, expr, "(3+4)*5");
    do_parse(name, expr, "3+4*5");
    do_parse(name, expr, "3+4");
    do_parse(name, expr, "4*5");
    do_parse(name, expr, "4");
    do_parse(name, expr, "(2-4)*-(1+12/3)");

    dont_parse(name, expr, "( 3 + 4 ) * 5");
}

//////////////////////////////////////////////////////////////////////////

void do_05_beispiel_real()
{
    std::cout << "=== do_05_beispiel_real ===" << std::endl;
    bool ok = parse("2.1 3.14 5 -2.0",
        real_p >> *(space_p >> real_p) ).full;
    std::cerr << "parse should be ok: " << ( ok ? "fine." : "bad!") << std::endl;
}

//////////////////////////////////////////////////////////////////////////

struct Inc_A
{
    int &m_value;
    Inc_A(int &value)
        : m_value(value) { }
    template<typename T>
    void operator()(const T&) const
    { ++m_value; }
};

template<typename RULE>
void parse_06(const char* str, RULE &rule_p, int &counter)
{
    counter = 0;
    parse(str, rule_p);
    std::cerr << "Count '"<<str<<"' is:" << counter << std::endl;
}

void do_06_intlist_count()
{
    std::cout << "=== do_06_intlist_count ===" << std::endl;
    int counter;
    Inc_A inc(counter);
    rule<> int_list_p = uint_p[inc] >> *(space_p >> uint_p[inc]) ;
    parse_06("1 2 3 4", int_list_p, counter);
    parse_06("99",      int_list_p, counter);
    parse_06("3 1 12",  int_list_p, counter);

}

//////////////////////////////////////////////////////////////////////////

template<typename RULE>
void parse_07(const char* str, RULE &rule_p, int &ci, int &cf)
{
    ci = 0;
    cf = 0;
    bool ok = parse(str, rule_p).full;
    std::cerr <<"Counting '"<<str <<"' ci:"<<ci <<" cf:"<<cf
        << " with ok:" << (ok ? "yes" : "NO")
        << std::endl;
}

void do_07_numlist_count()
{
    std::cout << "=== do_07_numlist_count ===" << std::endl;
    int ci;
    int cf;
    rule<> num_list_p = list_p( real_p[Inc_A(ci)] | uint_p[Inc_A(cf)], space_p);
    parse_07("1 2.1 3.52 4 5", num_list_p, ci, cf);
    parse_07("99",             num_list_p, ci, cf);
    parse_07("3.141592",       num_list_p, ci, cf);
    parse_07("3.1 1.0 12",     num_list_p, ci, cf);

}

//////////////////////////////////////////////////////////////////////////

struct Sum_A
{
    int &m_value;
    Sum_A(int &value)
        : m_value(value) { }
    void operator()(const char* beg, const char*) const
    {
        m_value += std::strtol(beg, NULL, 10);
    }
};

template<typename RULE>
void parse_08(const char* str, RULE &rule_p, int &sum)
{
    sum = 0;
    parse(str, rule_p);
    std::cerr << "Sum '"<<str<<"' is:" << sum << std::endl;
}

void do_08_rangenumlist_sum()
{
    std::cout << "=== do_08_rangenumlist_sum ===" << std::endl;
    int sum;
    rule<> my_num_p = +( range<>('0','9') );
    rule<> my_list_p = list_p( my_num_p[Sum_A(sum)]  ,  space_p);
    parse_08("1 2 3 4", my_list_p, sum);
    parse_08("99",      my_list_p, sum);
    parse_08("3 1 12",  my_list_p, sum);

}

//////////////////////////////////////////////////////////////////////////

void do_10_geoparse()
{
    std::cout << "=== do_10_geoparse ===" << std::endl;
    // inputs:   856'32.4" ,  856'32.4", 856'  , 8   , 856'32.4"O

    rule<> compass_p = chset_p("NE") | chset_p("SW");
    rule<> deg_p = ureal_p >> ch_p('');
    rule<> min_p = ureal_p >> ch_p('\'');
    rule<> sec_p = ureal_p >> ch_p('"');
    rule<> geo_p = deg_p >> !( min_p >> !( sec_p ) ) >> !compass_p;

    do_parse("geo", geo_p, "8");
    do_parse("geo", geo_p, "8.3");
    do_parse("geo", geo_p, "856'");
    do_parse("geo", geo_p, "856.4'");
    do_parse("geo", geo_p, "856'12\"");
    do_parse("geo", geo_p, "856.4'12.9\"");
    do_parse("geo", geo_p, "8N");
    do_parse("geo", geo_p, "8.3E");
    do_parse("geo", geo_p, "856'S");
    do_parse("geo", geo_p, "856.4'W");
    do_parse("geo", geo_p, "856.4'12\"N");
    do_parse("geo", geo_p, "856.4'12.9\"E");

    dont_parse("geo", geo_p, "8.3'");
    dont_parse("geo", geo_p, "-8");
}

//////////////////////////////////////////////////////////////////////////

template<int DIVISOR>
struct Factor_A
{
    double &m_val;
    Factor_A(double&val) : m_val(val) {}
    void operator()(const double val) const { m_val += val/DIVISOR; }
};

typedef Factor_A<1>     Deg_A;
typedef Factor_A<60>    Min_A;
typedef Factor_A<60*60> Sec_A;

struct Neg_A
{
    double &m_val;
    Neg_A(double&val) : m_val(val) {}
    void operator()(const char) const { m_val *= -1; }
};

template<typename RULE>
void parse_11(const char* str, RULE &rule_p, double &val, double expected)
{
    val = 0;
    parse(str, rule_p);
    std::cerr << "Deg '"<<str<<"' is:" << val << " exp:"<<expected
        << ( abs(expected-val) > 0.00001 ? "  ERROR" : "  OK" )
        << std::endl;
}

void do_11_geoactors()
{
    std::cout << "=== do_11_geoactors ===" << std::endl;
    // inputs:   856'32.4" ,  856'32.4", 856'  , 8   , 856'32.4"O
    // outputs:  8.93323    ,  8.9322    , 8.9320 , 8.0  , -8.9322
    // see: http://maps.google.com/?q=N53%B056'0%22,E0%B01'4%22

    double val;

    rule<> compass_p = chset_p("NE") | chset_p("SW")[Neg_A(val)];
    rule<> deg_p = ureal_p[Deg_A(val)] >> ch_p('');
    rule<> min_p = ureal_p[Min_A(val)] >> ch_p('\'');
    rule<> sec_p = ureal_p[Sec_A(val)] >> ch_p('"');
    rule<> geo_p = deg_p >> !( min_p >> !( sec_p ) ) >> !compass_p;

    parse_11("52"        , geo_p, val, 52.0     );
    parse_11("52.30"     , geo_p, val, 52.3     );
    parse_11("52.3"      , geo_p, val, 52.3     );
    parse_11("52.3S"     , geo_p, val,-52.3     );
    parse_11("5230'"     , geo_p, val, 52.5     );
    parse_11("523'"      , geo_p, val, 52.05    );
    parse_11("5203'"     , geo_p, val, 52.05    );
    parse_11("5233'"     , geo_p, val, 52.55    );
    parse_11("5256'"     , geo_p, val, 52.933333); // http://maps.google.com/?q=N52%B056'0%22,E0%B01'4%22
    parse_11("5256'12\"" , geo_p, val, 52.936667); // http://maps.google.com/?q=N52%B056'12%22,E0%B01'4%22
    parse_11("5256'12\"S", geo_p, val,-52.936667); // ?
    parse_11("5256'12\"N", geo_p, val, 52.936667);
}

//////////////////////////////////////////////////////////////////////////

using phoenix::arg1;
using phoenix::var;

struct Dbl_Closure12 : closure<Dbl_Closure12, double>
{
    member1 val;
};

struct GeoParser12 : public grammar<GeoParser12, Dbl_Closure12::context_t>
{
    template<typename T>
    struct definition
    {
        definition(GeoParser12 const& self)
        {
            top = geo_p[self.val = arg1];
            compass_p = chset_p("NE")[compass_p.val = 1]
            | chset_p("SW")[compass_p.val = -1]
            | eps_p[compass_p.val = 1] ;
            deg_p = ureal_p[deg_p.val = arg1]       >> ch_p('') ;
            min_p = ureal_p[min_p.val = arg1/60]    >> ch_p('\'') ;
            sec_p = ureal_p[sec_p.val = arg1/60/60] >> ch_p('"') ;
            geo_p = deg_p[geo_p.val = arg1]
            >> !( min_p[geo_p.val += arg1]
            >> !( sec_p[geo_p.val += arg1] )
                )
                >> compass_p[geo_p.val *= arg1] ;
        };

        typedef rule<T, Dbl_Closure12::context_t> rule_t;
        rule<T> top;
        rule_t compass_p, deg_p, min_p, sec_p, geo_p;

        rule<T> const& start() const { return top; }
    };
};


template<typename RULE>
void parse_12(const char* str, RULE &rule_p, double expected)
{
    double val;
    parse(str, rule_p[var(val) = arg1]);
    std::cerr << "Deg '"<<str<<"' is:" << val << " exp:"<<expected
        << ( abs(expected-val) > 0.00001 ? "  ERROR" : "  OK" )
        << std::endl;
}


void do_12_closure()
{
    std::cout << "=== do_12_closure ===" << std::endl;

    GeoParser12 geo_p;

    parse_12("52"        , geo_p, 52.0     );
    parse_12("52.30"     , geo_p, 52.3     );
    parse_12("52.3"      , geo_p, 52.3     );
    parse_12("52.3S"     , geo_p,-52.3     );
    parse_12("5230'"     , geo_p, 52.5     );
    parse_12("523'"      , geo_p, 52.05    );
    parse_12("5203'"     , geo_p, 52.05    );
    parse_12("5233'"     , geo_p, 52.55    );
    parse_12("5256'"     , geo_p, 52.933333);
    parse_12("5256'12\"" , geo_p, 52.936667);
    parse_12("5256'12\"S", geo_p,-52.936667);
    parse_12("5256'12\"N", geo_p, 52.936667);
}


//////////////////////////////////////////////////////////////////////////

void main()
{
    do_01_numbers();
    do_02_numbers_list();
    do_03_beispiel_oneliner();
    do_04_beispiel_ebnfig();
    do_05_beispiel_real();
    do_06_intlist_count();
    do_07_numlist_count();
    do_08_rangenumlist_sum();
    do_10_geoparse();
    do_11_geoactors();
    do_12_closure();
}
