// $Id: LogClient.java 1cff25c4ca8f 2009/05/12 15:18:36 Oliver Lau <oliver@von-und-fuer-lau.de> $
// Copyright (c) 2009 Oliver Lau <oliver@von-und-fuer-lau.de>
// All rights reserved.

package de.heise.demo;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.lang.Exception;
import java.lang.Object;
import java.lang.Boolean;
import java.lang.Runtime;
import java.util.LinkedList;
import java.util.ListIterator;
import android.app.Notification;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.app.Service;
import android.content.Context;
import android.content.Intent;
import android.os.RemoteCallbackList;
import android.os.IBinder;
import android.os.Bundle;
import android.os.RemoteException;
import android.os.Handler;
import android.os.Message;
import android.util.Log;
import android.location.LocationListener;
import android.location.LocationProvider;
import android.location.LocationManager;
import android.location.Location;

import org.json.JSONException;
import org.json.JSONObject;
import org.json.JSONArray;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpRequestInterceptor;
import org.apache.http.entity.StringEntity;
import org.apache.http.params.HttpConnectionParams;
import org.apache.http.params.BasicHttpParams;
import org.apache.http.auth.AuthScheme;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.AuthState;
import org.apache.http.auth.Credentials;
import org.apache.http.conn.params.ConnRoutePNames;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.protocol.ClientContext;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.CredentialsProvider;
import org.apache.http.impl.auth.BasicScheme;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.impl.client.BasicResponseHandler;
import org.apache.http.protocol.BasicHttpContext;
import org.apache.http.protocol.ExecutionContext;
import org.apache.http.protocol.HttpContext;


public class LogClient extends Service implements LocationListener {

	static class PreemptiveAuth implements HttpRequestInterceptor {

		public void process(
				final HttpRequest request, 
				final HttpContext context) throws HttpException, IOException {

			AuthState authState = (AuthState) context.getAttribute(
					ClientContext.TARGET_AUTH_STATE);
			if (authState.getAuthScheme() == null) {
				AuthScheme authScheme = (AuthScheme) context.getAttribute("preemptive-auth");
				CredentialsProvider credsProvider = (CredentialsProvider) context.getAttribute(
						ClientContext.CREDS_PROVIDER);
				HttpHost targetHost = (HttpHost) context.getAttribute(
						ExecutionContext.HTTP_TARGET_HOST);
				if (authScheme != null) {
					Credentials creds = credsProvider.getCredentials(
							new AuthScope(
									targetHost.getHostName(), 
									targetHost.getPort()));
					if (creds == null) {
						throw new HttpException("No credentials for preemptive authentication");
					}
					authState.setAuthScheme(authScheme);
					authState.setCredentials(creds);
				}
			}

		}
	}

	private static final String R_RESPONSE = "response";
	private static final String TAG = "LogClient";
	private static final String DEFAULT_API_USERNAME = "trcr";
	private static final String DEFAULT_API_PASSWORD = "D3M0demo";
	private static final PreemptiveAuth mPreemptiveAuth;
	private static final SimpleDateFormat mDateFormatter;
	private static final UsernamePasswordCredentials mCred;
	private static final AuthScope mAuthScope; 
	private static final String mUriPath;
	private static final HttpHost mTargetHost; 
	private static final HttpHost mProxyHost;
	private static final BasicHttpParams mHttpParams;
	private static final int mConnectionTimeout = 10*1000;
	
	private String mLastError;
	private Thread mPostLocationsThread;
	private final RemoteCallbackList<ILogClientCallback> mCallbacks = new RemoteCallbackList<ILogClientCallback>();
	private NotificationManager mNM;
	private LinkedList<Location> mLocList = new LinkedList<Location>();
	private Object mLock = new Object();
	private int mTourId = 1;
	private int mParticipantId = 1;
	private long mMinLocationChangeTime = 60*1000; // 1 Minute
	private float mMinLocationChangedDistance = 50; // 50 Meter

	static {
		mDateFormatter = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
		mPreemptiveAuth = new PreemptiveAuth();
		mCred = new UsernamePasswordCredentials(DEFAULT_API_USERNAME, DEFAULT_API_PASSWORD);
		mAuthScope = new AuthScope(AuthScope.ANY_HOST, AuthScope.ANY_PORT);
		mUriPath = "/trcr/add_trackpoint.php5";
		mTargetHost = new HttpHost("von-und-fuer-lau.de", 80, "http");
		mHttpParams = new BasicHttpParams();
		mProxyHost = new HttpHost("squid.heise.de", 8080); 
		HttpConnectionParams.setConnectionTimeout(mHttpParams, mConnectionTimeout);
		HttpConnectionParams.setSoTimeout(mHttpParams, mConnectionTimeout);
	}

	private final ILogClient.Stub mBinder = new ILogClient.Stub() {
		public Location getLastKnownLocation() {
			Location location = null;
			synchronized(mLock) {
				location = mLocList.getLast();
			}
			return location;
		}
		public synchronized int getPendingCount() {
			int size = 0;
			synchronized(mLock) {
				size = mLocList.size();
			}
			return size;
		}
		public String getLastError() {
			return mLastError;
		}
		public void setUpdateIntervalSeconds(int seconds) {
			mMinLocationChangeTime = 1000*seconds;
			requestLocationUpdates();
		}
		public void setUpdateIntervalMeters(float meters) {
			mMinLocationChangedDistance = meters;
			requestLocationUpdates();
		}
		public void setParticipantId(int participant_id) {
			mParticipantId = participant_id;
		}
		public void setTourId(int tour_id) {
			mTourId = tour_id;
		}
		public void registerCallback(ILogClientCallback cb) {
			if (cb != null)
				mCallbacks.register(cb);
		}
		public void unregisterCallback(ILogClientCallback cb) {
			if (cb != null)
				mCallbacks.unregister(cb);
		}
	};


	@Override
	public IBinder onBind(Intent intent) {
		Log.d(TAG, "onBind() called");
		return mBinder;
	}


	@Override
	public void onCreate() {
		Log.d(TAG, "onCreate() called");
		super.onCreate();
		showNotification(R.string.local_service_started);
	}


	@Override
	public void onStart(Intent intent, int startId) {
		Log.d(TAG, "onStart() called");
		super.onStart(intent, startId);
		requestLocationUpdates();
	}


	@Override
	public void onDestroy() {
		Log.d(TAG, "onDestroy() called");
		super.onDestroy();
		LocationManager mLocMgr = (LocationManager) getSystemService(Context.LOCATION_SERVICE);
		mLocMgr.removeUpdates(this);
		mNM.cancel(R.string.local_service_started);
		showNotification(R.string.local_service_stopped);
		mCallbacks.kill();
		mNM = null;
	}


	@Override
	public void onProviderDisabled(String provider) {
		Log.d(TAG, provider + " location provider disabled");
	}


	@Override
	public void onProviderEnabled(String provider) {
		Log.d(TAG, provider + " location provider enabled");
	}


	@Override
	public void onStatusChanged(String provider, int status, Bundle extras) {
		String msg = provider + " location provider changed status to ";
		switch (status) {
		case LocationProvider.AVAILABLE:
			msg += "available";
			break;
		case LocationProvider.TEMPORARILY_UNAVAILABLE:
			msg += "temporarily unavailable";
			break;
		case LocationProvider.OUT_OF_SERVICE:
			msg += "out of service";
			break;
		default:
			msg += "<unknown>";
		}
		Log.d(TAG, msg);
	}


	@Override
	public void onLocationChanged(Location location) {
		Log.d(TAG, "onLocationChanged(" + location.toString() + ")");
		synchronized(mLock) {
			mLocList.add(location);
			Log.d(TAG, "mLocList.size() = " + mLocList.size());
		}
		final int N = mCallbacks.beginBroadcast();
		for (int i = 0; i < N; ++i) {
			try {
				mCallbacks.getBroadcastItem(i).locationChanged();
			} catch (RemoteException e) {
				Log.e(TAG, e.getMessage());
			}
		}
		mCallbacks.finishBroadcast();

		if (mPostLocationsThread != null) {
			Log.d(TAG, "sending in progress (mPostLocationsThread != null) ...");
			return;
		}

		mPostLocationsThread = new Thread() {
			public void run() {
				Log.d(TAG, Thread.currentThread().getName() + " is running ...");
				JSONArray jsonRequestArray = new JSONArray();
				synchronized(mLock) {
					ListIterator<Location> i = mLocList.listIterator();
					while (i.hasNext()) {
						Location location = i.next();
						if (location.getTime() > 0) {
							try {
								JSONObject obj = new JSONObject();
								obj.put("participant_id", mParticipantId);
								obj.put("tour_id", mTourId);
								obj.put("lat", location.getLatitude());
								obj.put("lon", location.getLongitude());
								obj.put("ele", location.getAltitude());
								obj.put("t", mDateFormatter.format(new Date(location.getTime())));
								jsonRequestArray.put(obj);
							}
							catch (JSONException e) {
								Log.e(TAG, "JSONException" + e.getMessage());
							}
						}
					}
				}

				if (jsonRequestArray.length() > 0) {
					DefaultHttpClient httpClient = new DefaultHttpClient();
					httpClient.getCredentialsProvider().setCredentials(mAuthScope, mCred);
					if (mProxyHost != null)
						httpClient.getParams().setParameter(ConnRoutePNames.DEFAULT_PROXY, mProxyHost); 
					
					BasicHttpContext context = new BasicHttpContext();
					BasicScheme basicAuth = new BasicScheme();
					context.setAttribute("preemptive-auth", basicAuth);

					httpClient.addRequestInterceptor(mPreemptiveAuth, 0);

					HttpPost httpPost = new HttpPost(mUriPath);
					httpPost.setParams(mHttpParams);

					JSONObject jsonRequestData = new JSONObject();
					try {
						jsonRequestData.put("l", jsonRequestArray);
						Log.d(TAG, jsonRequestData.toString());
						StringEntity ent = new StringEntity(jsonRequestData.toString());
						ent.setContentType("text/x-json");
						httpPost.setEntity(ent);
					}
					catch (JSONException e) {
						Log.e(TAG, "JSONException" + e.getMessage());
					}
					catch (UnsupportedEncodingException e) {
						Log.e(TAG, "UnsupportedEncodingException: " + e.getMessage());
					}

					mLastError = null;
					ResponseHandler<String> responseHandler = new BasicResponseHandler();
					try {
						String jsonString = httpClient.execute(mTargetHost, httpPost, responseHandler, context);
						Log.d(TAG, "Response from server: " + jsonString);
						synchronized(mLock) {
							mLocList.clear();
						}
						if (jsonString != null && jsonString.length() > 0) {
							JSONObject json = new JSONObject(jsonString);
							if (json != null) {
								JSONArray jsonResponse = json.getJSONArray(R_RESPONSE);
								if (jsonResponse.length() > 0) {
									Log.d(TAG, "Trackpoints successfully added on server");
								}
							}
						}
					}
					catch (Exception e) {
						mLastError = e.getMessage();
						Log.e(TAG, e.getMessage());
					}
					httpClient.getConnectionManager().shutdown();
				}
				postReady.sendEmptyMessage(0);
			}
		};

		mPostLocationsThread.start();
		Log.d(TAG, "Java memory in use = " + (Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory()));
	}


	private final Handler postReady = new Handler() {
		@Override
		public void handleMessage(Message msg) { 
			Log.d(TAG, "postReady called");
			try {
				mPostLocationsThread.join();
			}
			catch (InterruptedException e) {
				Log.d(TAG, e.getMessage());
			}
			mPostLocationsThread = null;
		}
	};


	private void requestLocationUpdates() {
		LocationManager locMgr = (LocationManager) getSystemService(Context.LOCATION_SERVICE);
		if (locMgr.isProviderEnabled(LocationManager.GPS_PROVIDER)) {
			Log.d(TAG, "requestLocationUpdates(): " + LocationManager.GPS_PROVIDER + " is enabled");
			locMgr.requestLocationUpdates(LocationManager.GPS_PROVIDER, mMinLocationChangeTime, mMinLocationChangedDistance, this);
		}
		else if (locMgr.isProviderEnabled(LocationManager.NETWORK_PROVIDER)) {
			Log.d(TAG, "requestLocationUpdates(): " + LocationManager.NETWORK_PROVIDER + " is enabled");
			locMgr.requestLocationUpdates(LocationManager.NETWORK_PROVIDER, mMinLocationChangeTime, mMinLocationChangedDistance, this);
		}
		else {
			Log.d(TAG, "requestLocationUpdates(): no LocationManager is enabled");
		}
	}


	private void showNotification(int note) {
		CharSequence text = getText(note);
		Notification notification = new Notification(R.drawable.icon, text, System.currentTimeMillis());
		PendingIntent contentIntent = PendingIntent.getActivity(this, 0, new Intent(this, TrackAndLog.class), 0);
		notification.setLatestEventInfo(this, getText(R.string.local_service_label), text, contentIntent);
		mNM = (NotificationManager) getSystemService(NOTIFICATION_SERVICE);
		mNM.notify(note, notification);
	}

}
