//$Id: TrackAndLog.java 1cff25c4ca8f 2009/05/12 15:18:36 Oliver Lau <oliver@von-und-fuer-lau.de> $
//Copyright (c) 2009 Oliver Lau <oliver@von-und-fuer-lau.de>
//All rights reserved.

package de.heise.demo;

import java.util.Date;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import android.app.Activity;
import android.app.AlertDialog;
import android.app.Dialog;
import android.content.DialogInterface;
import android.content.ServiceConnection;
import android.content.Intent;
import android.os.IBinder;
import android.content.Context;
import android.content.ComponentName;
import android.os.Bundle;
import android.os.RemoteException;
import android.location.Location;
import android.widget.Button;
import android.widget.Toast;
import android.widget.TextView;
import android.widget.EditText;
import android.util.Log;
import android.view.View;
import android.view.View.OnClickListener;
import android.view.LayoutInflater;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;

public class TrackAndLog extends Activity {

	private static final String TAG = "TrackAndLog";
	private static final SimpleDateFormat DateFormatter = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
	private static final DecimalFormat GeoCoordFormat = new DecimalFormat("#,##0.0000000");
	private static final DecimalFormat AltitudeFormat = new DecimalFormat("#,##0.0");

	private static final int DIALOG_TOUR_ID = 1;
	private static final int DIALOG_PARTICIPANT_ID = 2;
	private static final int DIALOG_UPDATE_INTERVAL_METERS = 3;
	private static final int DIALOG_UPDATE_INTERVAL_SECONDS = 4;

	private Menu mMenu;
	private boolean mMakeToast = true;
	private Toast mToast = null;
	private ILogClient mBoundService;
	private Intent mLogClientIntent;
	private boolean mLogClientRunning = false;
	private int mParticipantId = 1;
	private int mTourId = 1;
	private float mUpdateIntervalMeters = 50;
	private int mUpdateIntervalSeconds = 60;

	private ServiceConnection mLogClientConn = new ServiceConnection() {
		public void onServiceConnected(ComponentName className, IBinder service) {
			mBoundService = ILogClient.Stub.asInterface(service);
			try {
				mBoundService.registerCallback(mCallback);
				mBoundService.setParticipantId(mParticipantId);
				mBoundService.setTourId(mTourId);
				mBoundService.setUpdateIntervalMeters(mUpdateIntervalMeters);
				mBoundService.setUpdateIntervalSeconds(mUpdateIntervalSeconds);
			} catch (RemoteException e) {
				Log.d(TAG, "onServiceConnected() failed: " + e.getMessage());
			}
			toastIt("LogClient service connected", Toast.LENGTH_SHORT);
		}
		public void onServiceDisconnected(ComponentName className) {
			mBoundService = null;
			toastIt("LogClient service disconnected", Toast.LENGTH_SHORT);
		}
	};


	@Override
	public void onCreate(Bundle savedInstanceState) {
		Log.d(TAG, "onCreate() called");
		super.onCreate(savedInstanceState);
		setContentView(R.layout.main);
		restoreInstance(savedInstanceState);
		initGUI();
	}


	@Override 
	public void onSaveInstanceState(Bundle outState) {
		Log.d(TAG, "onSaveInstanceState() called");
		super.onSaveInstanceState(outState);
		saveInstance(outState);
	}


	@Override
	public void onRestoreInstanceState(Bundle savedInstanceState) {
		Log.d(TAG, "onRestoreInstanceState() called");
		super.onRestoreInstanceState(savedInstanceState);
		initGUI();
	}


	@Override
	public void onPause() {
		Log.d(TAG, "onPause() called");
		super.onPause();
		if (mLogClientConn != null)
			unbindService(mLogClientConn);
	}


	@Override
	public void onResume() {
		Log.d(TAG, "onResume() called");
		super.onResume();
		mLogClientIntent = new Intent(TrackAndLog.this, LogClient.class);
		mLogClientRunning = bindService(mLogClientIntent, mLogClientConn, Context.BIND_AUTO_CREATE);
		Log.d(TAG, (mLogClientRunning)? "successfully bound to service" : "binding to service failed");
		if (mLogClientRunning)
			initGUI();
		fillViews();
	}


	@Override
	public void onRestart() {
		Log.d(TAG, "onRestart() called");
		super.onRestart();
	}


	@Override
	public void onStart() {
		Log.d(TAG, "onStart() called");
		super.onStart();
	}


	@Override
	public void onStop() {
		Log.d(TAG, "onStop() called");
		super.onStop();
	}


	@Override
	public void onDestroy() {
		Log.d(TAG, "onDestroy() called");
		super.onDestroy();
	}


	@Override
	public boolean onCreateOptionsMenu(Menu menu) {
		mMenu = menu;
		MenuInflater inflater = getMenuInflater();
		inflater.inflate(R.menu.main, menu);
		return true;
	}


	@Override
	public boolean onOptionsItemSelected(MenuItem item) {
		switch (item.getItemId()) {
		case R.id.edit_participant:
			showDialog(DIALOG_PARTICIPANT_ID);
			return true;
		case R.id.edit_tour:
			showDialog(DIALOG_TOUR_ID);
			return true;
		case R.id.edit_update_interval_meters:
			showDialog(DIALOG_UPDATE_INTERVAL_METERS);
			return true;
		case R.id.edit_update_interval_seconds:
			showDialog(DIALOG_UPDATE_INTERVAL_SECONDS);
			return true;
		default:
			break;
		}
		return false;
	}


	@Override
	protected Dialog onCreateDialog(int id) {
		switch (id) {
		case DIALOG_PARTICIPANT_ID: {
			LayoutInflater factory = LayoutInflater.from(this);
			final View v = factory.inflate(R.layout.participantid, null);
			EditText textEditView = (EditText)v.findViewById(R.id.participant_id_edit);
			textEditView.setText(Integer.toString(mParticipantId));
			return new AlertDialog.Builder(TrackAndLog.this)
			.setTitle(R.string.menu_change_participant_id)
			.setView(v)
			.setPositiveButton(R.string.alert_dialog_ok, new DialogInterface.OnClickListener() {
				public void onClick(DialogInterface dialog, int whichButton) {
					EditText textEdit = (EditText)v.findViewById(R.id.participant_id_edit);
					mParticipantId = Integer.parseInt(textEdit.getText().toString());
					if (mBoundService != null) {
						try {
							mBoundService.setParticipantId(mParticipantId);
						} catch (RemoteException e) {
							Log.d(TAG, "mBoundService.setParticipantId() failed: " + e.getMessage());
						}
					}
				}
			})
			.setNegativeButton(R.string.alert_dialog_cancel, new DialogInterface.OnClickListener() {
				public void onClick(DialogInterface dialog, int whichButton) {
					// ignorieren
				}
			})
			.create();
		}
		case DIALOG_TOUR_ID: {
			LayoutInflater factory = LayoutInflater.from(this);
			final View v = factory.inflate(R.layout.tourid, null);
			EditText textEditView = (EditText)v.findViewById(R.id.tour_id_edit);
			textEditView.setText(Integer.toString(mTourId));
			return new AlertDialog.Builder(TrackAndLog.this)
			.setTitle(R.string.menu_change_tour_id)
			.setView(v)
			.setPositiveButton(R.string.alert_dialog_ok, new DialogInterface.OnClickListener() {
				public void onClick(DialogInterface dialog, int whichButton) {
					EditText textEdit = (EditText)v.findViewById(R.id.tour_id_edit);
					mTourId = Integer.parseInt(textEdit.getText().toString());
					if (mBoundService != null) {
						try {
							mBoundService.setTourId(mTourId);
						} catch (RemoteException e) {
							Log.d(TAG, "mBoundService.setTourId() failed: " + e.getMessage());
						}
					}
				}
			})
			.setNegativeButton(R.string.alert_dialog_cancel, new DialogInterface.OnClickListener() {
				public void onClick(DialogInterface dialog, int whichButton) {
					// ignorieren
				}
			})
			.create();
		}
		case DIALOG_UPDATE_INTERVAL_METERS: {
			LayoutInflater factory = LayoutInflater.from(this);
			final View v = factory.inflate(R.layout.updatemeters, null);
			EditText textEditView = (EditText)v.findViewById(R.id.edit_update_interval_meters);
			textEditView.setText(Float.toString(mUpdateIntervalMeters));
			return new AlertDialog.Builder(TrackAndLog.this)
			.setTitle(R.string.menu_change_update_interval_meters)
			.setView(v)
			.setPositiveButton(R.string.alert_dialog_ok, new DialogInterface.OnClickListener() {
				public void onClick(DialogInterface dialog, int whichButton) {
					EditText textEdit = (EditText)v.findViewById(R.id.edit_update_interval_meters);
					mUpdateIntervalMeters = Float.parseFloat(textEdit.getText().toString());
					if (mBoundService != null) {
						try {
							mBoundService.setUpdateIntervalMeters(mUpdateIntervalMeters);
						} catch (RemoteException e) {
							Log.d(TAG, "mBoundService.setUpdateIntervalMeters() failed: " + e.getMessage());
						}
					}
				}
			})
			.setNegativeButton(R.string.alert_dialog_cancel, new DialogInterface.OnClickListener() {
				public void onClick(DialogInterface dialog, int whichButton) {
					// ignorieren
				}
			})
			.create();
		}
		case DIALOG_UPDATE_INTERVAL_SECONDS: {
			LayoutInflater factory = LayoutInflater.from(this);
			final View v = factory.inflate(R.layout.updateseconds, null);
			EditText textEditView = (EditText)v.findViewById(R.id.edit_update_interval_seconds);
			textEditView.setText(Integer.toString(mUpdateIntervalSeconds));
			return new AlertDialog.Builder(TrackAndLog.this)
			.setTitle(R.string.menu_change_update_interval_seconds)
			.setView(v)
			.setPositiveButton(R.string.alert_dialog_ok, new DialogInterface.OnClickListener() {
				public void onClick(DialogInterface dialog, int whichButton) {
					EditText textEdit = (EditText)v.findViewById(R.id.edit_update_interval_seconds);
					mUpdateIntervalSeconds = Integer.parseInt(textEdit.getText().toString());
					if (mBoundService != null) {
						try {
							mBoundService.setUpdateIntervalSeconds(mUpdateIntervalSeconds);
						} catch (RemoteException e) {
							Log.d(TAG, "mBoundService.setUpdateIntervalSeconds() failed: " + e.getMessage());
						}
					}
				}
			})
			.setNegativeButton(R.string.alert_dialog_cancel, new DialogInterface.OnClickListener() {
				public void onClick(DialogInterface dialog, int whichButton) {
					// ignorieren
				}
			})
			.create();
		}
		default:
			Log.e(TAG, "Ups!");
		break;
		}
		return null;
	}


	private OnClickListener mStartUpdatingListener = new OnClickListener() {
		public void onClick(View v) {
			Button button = (Button) v;
			button.setOnClickListener(mStopUpdatingListener);
			button.setText(R.string.stop);
			mLogClientIntent = new Intent(TrackAndLog.this, LogClient.class);
			startService(mLogClientIntent);
			bindService(mLogClientIntent, mLogClientConn, Context.BIND_AUTO_CREATE);
			toastIt("Updating running", Toast.LENGTH_SHORT);
		}
	};


	private OnClickListener mStopUpdatingListener = new OnClickListener() {
		public void onClick(View v) {
			Button button = (Button) v;
			button.setOnClickListener(mStartUpdatingListener);
			button.setText(R.string.start);
			unbindService(mLogClientConn);
			stopService(new Intent(TrackAndLog.this, LogClient.class));
			toastIt("Service stopped", Toast.LENGTH_SHORT);
		}
	};


	private void toastIt(String msg, int duration) {
		if (mMakeToast) {
			if (mToast != null)
				mToast.cancel();
			mToast = Toast.makeText(TrackAndLog.this, msg, duration);
			mToast.show();
		}
		Log.d(TAG, msg);
	}


	private void restoreInstance(Bundle savedInstanceState) {
		if (savedInstanceState != null) {
		}
	}


	private void saveInstance(Bundle outState) {
		if (outState != null) {
		}
	}


	private void initGUI()
	{
		Button button = (Button)findViewById(R.id.StartButton);
		if (mLogClientRunning) {
			button.setOnClickListener(mStopUpdatingListener);
			button.setText(R.string.stop);
		}
		else {
			button.setOnClickListener(mStartUpdatingListener);
			button.setText(R.string.start);
		}
	}


	private void fillViews()
	{
		try {
			if (mBoundService != null) {
				Location location = mBoundService.getLastKnownLocation();
				TextView latView = (TextView) findViewById(R.id.latView);
				latView.setText(GeoCoordFormat.format(location.getLatitude()));
				TextView lonView = (TextView) findViewById(R.id.lonView);
				lonView.setText(GeoCoordFormat.format(location.getLongitude()));
				TextView altiView = (TextView) findViewById(R.id.altiView);
				altiView.setText(AltitudeFormat.format(location.getAltitude()));
				TextView queueSizeView = (TextView) findViewById(R.id.queueSizeView);
				queueSizeView.setText(Integer.toString(mBoundService.getPendingCount()));
				String err = mBoundService.getLastError();
				if (err == null) {
					if (location.getTime() > 0) {
						TextView lastUpdateView = (TextView) findViewById(R.id.lastUpdateView);
						lastUpdateView.setText(DateFormatter.format(new Date(location.getTime())));
					}
				}
				TextView logView = (TextView) findViewById(R.id.logView);
				logView.append("[" + DateFormatter.format(new Date()) + "] " + ((err == null)? "OK" : err) + "\n");
			}
		}
		catch (RemoteException e) {
			Log.d(TAG, e.getMessage());
		}
	}
	

	private ILogClientCallback mCallback = new ILogClientCallback.Stub() {
		public void locationChanged() {
			fillViews();
		}
	};

}
