#!/usr/bin/perl -w
#
# Twitter-Timeline sowie darin verlinkte Bilder, Kurz-URLs und verlinkte Posterous-Posts sichern.
# Copyright (c) 2009 Oliver Lau <oliver@von-und-fuer-lau.de>
# Alle Rechte vorbehalten.
#
# $Id: twitterbak.pl 7fe925ca75d7 2009/10/22 11:32:51 Oliver Lau <oliver@von-und-fuer-lau.de> $

use utf8;
use strict;
use warnings;
use Config::IniFiles;
use Cwd qw(cwd);
use DateTime;
use DateTime::Format::Strptime;
use DBI;
use Error qw(:try);
use Getopt::Long qw(GetOptions);
use HTML::Entities qw(decode_entities);
use Image::Magick;
use JSON qw(from_json to_json);
use LWP;
use URI::Escape qw(uri_escape);
use XML::Simple qw(XMLin);

use constant TRUE  => 1;
use constant FALSE => 0;

my $homedir = ($^O eq 'MSWin32')? $ENV{'USERPROFILE'} : $ENV{'HOME'};
my $pathsep = ($^O eq 'MSWin32')? '\\' : '/';
my $default_working_dir = "$homedir$pathsep" . 'My Tweets';
my $config_file = undef;

# Benutzername und Passwort für Twitter
my $username = undef;
my $password = undef;

# Parameter fuer den Zugriff auf die lokale Datenbank.
my $db_name = undef;
my $db_user = undef;
my $db_pass = undef;
my $db_driver = 'SQLite';

my $tweet_page = undef;
my $tweet_count = undef;

my $tweets = undef;
my $remote_tweets = undef;
my $output_tweet_url = undef;

my $no_dm = undef;

# Wenn true, das erstmalige Backup durchführen. Das Skript versucht
# dann, so weit in die Vergangenheit zurück wie möglich Tweets
# aus den Timelines einzusammeln.
my $initial_backup = FALSE;

# Wenn true, Dateien mit demselben Namen ohne Nachfrage überschreiben.
my $force_write = FALSE;

# Wenn true, kein UTF-8-BOM in die erzeugten Textdateien schreiben
my $no_bom = undef;

# Wenn true, nur die Datenbank neu anlegen
my $only_create_db = FALSE;

# Wenn true, werden nur die lokal vorhandenen Daten zum Web-Server übertragen.
# Neue Tweets werden nicht abgeholt.
my $copy_mode = FALSE;

# Wenn true, werden die fehlenden Daten und Bilder zum und vom Webserver übertragen.
my $sync_mode = FALSE;

my $input_id_url = undef;
my $input_tweet_url = undef;
my $input_map_url   = undef;
my $input_post_url  = undef;

# Namen der Dateien, die die höchste bisher eingelesene Tweet-ID
# aufnehmen (Timeline-ID und Direct-Message-ID)
my $since = {
    'user' => { 'file' => '_MAXID',
                'id'   => undef },
    'dm' =>   { 'file' => '_MAXID_DM',
                'id'   => undef }
};

my $snapshot_file = '.resume.info';

# Wenn true, die Dateien aus $max_id_file ignorieren
my $ignore_max_id = FALSE;

# Verzeichnis, in dem alle generierten Dateien landen sollen
my $output_path = undef;

# Wenn true, keine Bilder verarbeiten
my $no_pics = undef;

# URLs des Webservice, der die Bilder entgegennimmt
my $output_pic_url = undef;

# ARRAYREF mit den Namen der Bilddateien
my $pics = undef;
my $remote_pics = undef;
my $input_pic_url = undef;

# Maximale Breite der erzeugten Thumbnails
my $max_thumb_width = 256;

# Liste der unterstützten Bilddienste
my @img_services = qw(img.ly mobypicture.com twitgoo.com
    twitpic.com yfrog.com);

# Mit diesem Protokoll wird auf twitter.com zugegriffen
my $http_scheme = 'https';

my $default_time_offset = '+02:00';

# Download-Modus
my $default_fetch_mode = 'friends';
my $fetch_mode = undef;
my $no_mentions = undef;

# Liste der unterstützten Download-Modi
my @allowed_fetch_modes = qw(user friends all);

# Standardeinstellung fuer Ausgabe-Modus
my $default_output_string = 'plain';

# Liste der unterstützten Ausgabe-Modi
my @allowed_outputs = qw(plain json db url);

# Aus der Konfigurationsdatei oder aus der Kommandozeile
# ausgelesener String mit den gewünschten Ausgabe-Modi
my $output_string = undef;

# Wenn true, die Links von URL-Shortenern nicht auflösen
my $no_url_mapping = FALSE;

# Wenn true, Posterous-Posts ignorieren
my $no_posts = FALSE;

# Wenn true, Posterous-Daten nicht in lokalen XML-Dateien speichern
my $no_write_posterous_files = FALSE;

# Link zu dem Webservice, der die URL-Mappings entgegennimmt
my $output_map_url = undef;

# Liste der unterstützten URL-Shortener
my @shorteners = qw(62c.de bit.ly cli.gs digg.com doiop.com ff.im
    grin.to is.gd j.mp j2j.de lauflinx.de memurl.com migre.me ow.ly
    post.ly tr.im u.nu snipurl.com snurl.com su.pr tinyurl.com
    tiny.cc twitthis.com twurl.nl twurl.cc wong.to);

# ARRAYREF mit den URL-Mappings
my $urlmappings = undef;
my $remote_urlmappings = undef;

# URLs des Webservice, der die Posterous-Posts entgegennimmt
my $output_post_url = undef;

# ARRAYREF mit den Posterous-Posts
my $posts = undef;
my $remote_posts = undef;

my $VERSION = '1.6.0';
my $VERBOSE = 0;
my $DEBUG = FALSE;
my $HELP = FALSE;
my $QUIET = FALSE;

my %monthMap = ( Jan => 1, Feb => 2, Mar => 3, Apr => 4,
    May => 5, Jun => 6, Jul => 7, Aug => 8,
    Sep => 9, Oct => 10, Nov => 11, Dec => 12 );

sub GREETING();
sub USAGE();
sub read_local_tweets();
sub read_local_mappings();
sub read_local_posts();
sub read_local_pics();
sub fetch_remote_ids();
sub fetch_remote_tweets();
sub fetch_remote_pics();
sub fetch_remote_posts();
sub fetch_remote_urlmappings();
sub post_json_to_url($$;$);
sub post_tweets();
sub process_posterous($);
sub process_urls($);
sub deduplicate($$;$);
sub fetch_friends();
sub fetch_timeline(;$);
sub fetch_mentions();
sub fetch_dm();
sub fetch_dm_sent();
sub fetched($$$);
sub fetch($;$$);
sub copy_to_remote();
sub synchronize();
sub init_db();
sub mk_date($$$$$$$);
sub mk_date_from_twitter($);
sub mk_date_from_posterous($);
sub read_max_ids();
sub write_max_ids();
sub print_v($);
sub print_nv($);
sub caution($;$);

GetOptions(
           'config|c=s' => \$config_file,
           'copy' => \$copy_mode,
           'sync' => \$sync_mode,
           'page|p=i' => \$tweet_page,
           'count|n=i' => \$tweet_count,
           'fetch-mode|f=s' => \$fetch_mode,
           'no-bom' => \$no_bom,
           'no-pics' => \$no_pics,
           'no-dm' => \$no_dm,
           'no-mentions' => \$no_mentions,
           'no-posts' => \$no_posts,
           'force-write' => \$force_write,
           'outputs|o=s' => \$output_string,
           'path=s' => \$output_path,
           'init' => \$initial_backup,
           'ignore-max-id' => \$ignore_max_id,
           'db-name' => \$db_name,
           'only-create-db' => \$only_create_db,
           'verbose|v+' => \$VERBOSE,
           'debug' => \$DEBUG,
           'quiet|q' => \$QUIET,
           'help|?|h' => \$HELP
);

GREETING() unless $QUIET;
USAGE() if $HELP;

if ($DEBUG) {
    use Time::HiRes qw(time);
    use Data::Dumper qw(Dumper);
}

my $cwd = cwd();
my $dt_isoformatter = DateTime::Format::Strptime->new(
    pattern => '%Y-%m-%d %H:%M:%S');

# Konfigurationsdatei suchen
if (!defined $config_file) {
    foreach ('twitterbak.ini', '.twitterbak', 'twitterbak.cfg') {
        if (-f "$homedir$pathsep$_") {
            $config_file = "$homedir$pathsep$_";
            last;
        }
    }
    $config_file = 'twitterbak.ini' unless $config_file;
}
-f $config_file
    or die "Konfigurationsdatei fehlt oder kann nicht geöffnet werden.";
my $cfg = Config::IniFiles->new(-file => $config_file);
my $user_agent_string = $cfg->val(
    'http', 'useragent',
    "TwitterBak/$VERSION (compatible; Mozilla 5.0)"
    );
my $proxy = $cfg->val('http', 'proxy', undef);
my $time_offset =  $cfg->val('twitter', 'timeoffset', $default_time_offset);
my ( $offset_sign, $h_offset, $m_offset, $s_offset ) =
    $time_offset =~ /([+-])(\d{2}):(\d{2})/;
$fetch_mode = $cfg->val('twitter', 'fetchmode', $default_fetch_mode)
    unless defined $fetch_mode;
my $fetch_mode_found = grep { $_ eq $fetch_mode } @allowed_fetch_modes;
USAGE() unless $fetch_mode_found;
$no_dm = ($cfg->val('twitter', 'dm', 'yes') =~ /(false|0|no|off)/) unless defined $no_dm;
$no_mentions = ($cfg->val('twitter', 'mentions', 'yes') =~ /(false|0|no|off)/) unless defined $no_mentions;
$no_pics = ($cfg->val('outputs', 'pics', 'yes') =~ /(false|0|no|off)/) unless defined $no_pics;
$no_bom = ($cfg->val('outputs', 'bom', 'yes') =~ /(false|0|no|off)/) unless defined $no_bom;
$no_posts = ($cfg->val('twitter', 'posts', 'yes') =~ /(false|0|no|off)/) unless defined $no_posts;
$output_path      = $cfg->val('outputs', 'path', $default_working_dir) unless defined $output_path;
$output_string    = $cfg->val('outputs', 'output', $default_output_string) unless defined $output_string;
$output_tweet_url = $cfg->val('outputs', 'tweeturl');
$output_pic_url   = $cfg->val('outputs', 'picurl');
$output_map_url   = $cfg->val('outputs', 'mapurl');
$output_post_url  = $cfg->val('outputs', 'posturl');
$input_id_url    = $cfg->val('inputs', 'idurl');
$input_tweet_url = $cfg->val('inputs', 'tweeturl');
$input_map_url   = $cfg->val('inputs', 'mapurl');
$input_post_url  = $cfg->val('inputs', 'posturl');
$input_pic_url    = $cfg->val('inputs', 'picurl');
$max_thumb_width = int($cfg->val('outputs', 'maxthumbwidth', $max_thumb_width));
die "Konfigurationsparameter [outputs]/maxthumbwidth muss größer als 0 sein"
    unless $max_thumb_width > 0;
$http_scheme = $cfg->val('outputs', 'httpscheme', $http_scheme);
$tweet_count = $cfg->val('twitter', 'count') unless defined $tweet_count;
$username = $cfg->val('twitter', 'username', undef);
(defined $username && $username ne '') or die 'Bitte [twitter]/username in Konfigurationsdatei eintragen.';
$password = $cfg->val('twitter', 'password', undef);
$db_name = $cfg->val('database', 'name', undef) unless defined $db_name;
$db_name = "$username.sqlite" unless defined $db_name;

# Ausgabeverzeichnis anlegen, falls es noch nicht existiert
mkdir $output_path unless -d $output_path;
$output_path .= "$pathsep$username";
mkdir $output_path unless -d $output_path;
print "Wechseln ins Verzeichnis $output_path ...\n" if $DEBUG;
chdir $output_path;

try {
    read_max_ids() unless $ignore_max_id;
}
catch Error::Simple with {
    my $E = shift;
    caution($E);
};

my $dbh = undef;

if ($only_create_db) {
    print "Wenn Sie fortfahren, gehen alle Datenbankeinträge "
        . "unwiderruflich verloren.\n"
        . "Tippen Sie 'ja' [RETURN], um fortzufahren: ";
    my $answer = <>;
    chomp($answer);
    if ($answer eq 'ja') {
        $dbh = DBI->connect("dbi:$db_driver:dbname=$db_name", $db_user, $db_pass)
            or die $DBI::errstr;
        init_db();
    }
    exit 0;
}

if ($output_string =~ /db/) {
    print_v("Verbinden mit Datenbank ...\n");
    $dbh = DBI->connect("dbi:$db_driver:dbname=$db_name", $db_user, $db_pass)
        or die $DBI::errstr;
    my $sth = $dbh->table_info('%', '%', '%', 'TABLE');
    my $tab = $sth->fetchall_arrayref;
    init_db() if $#{$tab} < 0;
}

my $ua = LWP::UserAgent->new(agent => $user_agent_string);
$ua->proxy($proxy) if defined $proxy;
my $ua1 = LWP::UserAgent->new(agent => $user_agent_string, max_redirect => 1);
$ua1->proxy($proxy) if defined $proxy;

my $timeline_count;
my $dm_count;
my $dm_sent_count;
my $mention_count;
my $num_tweets_sent   = 0;
my $num_mappings_sent = 0;
my $num_posts_sent    = 0;
my $num_pics_sent     = 0;
my $num_tweets_received   = 0;
my $num_mappings_received = 0;
my $num_posts_received    = 0;
my $num_pics_received     = 0;

if ($copy_mode) {
    copy_to_remote();
}
elsif ($sync_mode) {
    synchronize();
}
else {
    try {
        $timeline_count = fetch(\&fetch_timeline);
        $dm_count       = fetch(\&fetch_dm) unless $no_dm;
        $dm_sent_count  = fetch(\&fetch_dm_sent) unless $no_dm;
        $mention_count  = fetch(\&fetch_mentions) unless $no_mentions;
    }
    catch Error::Simple with {
        my $E = shift;
        caution($E);
    };

    if ($fetch_mode eq 'all') {
        my $friends;
        try {
            $friends = fetch_friends();
        }
        catch Error::Simple with {
            my $E = shift;
            caution($E);
        };
        if (defined $friends) {
            $timeline_count = 0 unless defined $timeline_count;
            foreach (@{$friends}) {
                $timeline_count += fetch(\&fetch_timeline, $_->{'id'}, $_->{'name'});
            }
        }
    }

    try {
        $num_tweets_sent = post_tweets() if $output_string =~ /url/;
        write_max_ids() if defined($num_tweets_sent) && $num_tweets_sent >= 0;
        $num_mappings_sent = post_mappings() unless $no_url_mapping;
        $num_posts_sent = post_posts() unless $no_posts;
    }
    catch Error::Simple with {
        my $E = shift;
        caution($E);
    };

    $timeline_count = 0 unless $timeline_count;
    $dm_count       = 0 unless $dm_count;
    $dm_sent_count  = 0 unless $dm_sent_count;
    $mention_count  = 0 unless $mention_count;
    print_v("\n$timeline_count/$dm_count/$dm_sent_count/$mention_count " .
        "Tweets/DMs/SentDMs/Mentions heruntergeladen.\n");

}

chdir $cwd;
exit 0;


# #############################################################
#
# Unterroutinen
#
# #############################################################

sub print_nv($) {
    do { $|=1; print shift; } if !$VERBOSE && !$QUIET;
}

sub print_v($) {
    do { $|=1; print shift; } if $VERBOSE && !$QUIET;
}

# Warnung ausgeben
sub caution($;$) {
    my ($E, $message) = @_;
    warn 'WARNUNG #', $E->{'-value'}, ': ',
    (defined($message)? "$message. " : ''), $E->{'-text'},
    ' (Zeile ', $E->{'-line'}, ")\n";
}

# Namen der Thumbnail-Datei aus Originalnamen ermitteln
# und zurückgeben
sub mk_thumb_name($) {
    my ($file) = @_;
    return "thumb-$file";
}

# Tweets aus der lokalen Datenbank lesen und
# das ARRAYREF $tweets damit befüllen.
sub read_local_tweets() {
    my $sql = 'SELECT id, user_id, user_screen_name, user_name, created_at, tweet, in_reply_to_screen_name, in_reply_to_status_id FROM tweets ORDER BY id';
    my $sth = $dbh->prepare($sql) or die "$dbh->errstr ($sql)";
    my $res = $sth->execute;
    my $rows = $sth->fetchall_arrayref or die "$sth->err ($sql)";
    foreach (@{$rows}) {
        my $newTweet = {
            'id' => $_->[0],
            'user_id' => $_->[1],
            'user_screen_name' => $_->[2],
            'user_name' => $_->[3],
            'created_at' => $_->[4],
            'text' => $_->[5],
            'in_reply_to_screen_name' => $_->[6],
            'in_reply_to_status_id' => $_->[7]
        };
        push @{$tweets}, $newTweet;
    }
}

# URL-Abbildungen aus der lokalen Datenbank lesen und
# das ARRAYREF $urlmappings damit befüllen.
sub read_local_mappings() {
    print "Einlesen der lokalen URL-Abbildungen ...\n" unless $QUIET;
    my $sql = 'SELECT origin, destination FROM urlmappings ORDER BY origin';
    my $sth = $dbh->prepare($sql) or die "$dbh->errstr ($sql)";
    my $res = $sth->execute;
    my $rows = $sth->fetchall_arrayref or die "$sth->err ($sql)";
    foreach (@{$rows}) {
        my $newUrlMapping = {
            'origin' => $_->[0],
            'destination' => $_->[1]
        };
        push @{$urlmappings}, $newUrlMapping;
    }
}

# Posterous-Posts aus der lokalen Datenbank lesen und
# das ARRAYREF $posts damit befüllen.
sub read_local_posts() {
    print "Einlesen der IDs der lokalen Posterous-Posts ...\n" unless $QUIET;
    my $sql = 'SELECT id, url, created_at, author, title, body FROM posts ORDER BY id';
    my $sth = $dbh->prepare($sql) or die "$dbh->errstr ($sql)";
    my $res = $sth->execute;
    my $rows = $sth->fetchall_arrayref or die "$sth->err ($sql)";
    foreach (@{$rows}) {
        my $newPost = {
            'id' => $_->[0],
            'url' => $_->[1],
            'date' => $_->[2],
            'author' => $_->[3],
            'title' => $_->[4],
            'body' => $_->[5]
        };
        push @{$posts}, $newPost;
    }
}

# Namen der lokal gespeicherten Bilddateien lesen und
# das ARRAYREF $pics damit befüllen.
sub read_local_pics() {
    print "Einlesen der Namen der lokal gespeicherten Bilddateien ...\n" unless $QUIET;
    opendir(my $dh, '.') or die "can't opendir: $!";
    # Die Dateinamen müssen aufsteigend sortiert sein, damit die optimierte
    # Restmengenbildung in der Funktion deduplicate() funktioniert.
    @{$pics} = sort grep { /\.jpg$/ && -f "./$_" } readdir($dh);
    closedir $dh;
}

# HTML-Code von der angegebenen URL laden und nach einem
# regulären Ausdruck durchsuchen. Das Ergebnis des Match
# zurückgeben
sub extract_by_regex($$) {
    my ($url, $regex) = @_;
    my $req = HTTP::Request->new(GET => $url);
    my $res = $ua->request($req);
    throw Error::Simple($res->message, $res->code) unless $res->is_success;
    ( $url ) = $res->content =~ /$regex/;
    return $url;
}

# Das Bild aus der angegebenen Datei zum Webservice hochladen
sub post_pic($) {
    my ($filename) = @_;
    print_v("Senden des Bildes '$filename' an $output_pic_url ...\n");
    my $req = HTTP::Request->new;
    $req->content_type('image/jpeg');
    open IN, "<$filename" or die "Öffnen von $filename fehlgeschlagen.";
    binmode IN;
    my $res = $ua->post($output_pic_url,
                        [ 'filename' => $filename,
                          'data'     => do { local $/ = undef; <IN> }
                        ]);
    close IN;
    throw Error::Simple($res->message, $res->code) unless $res->is_success;
    print "$res->content\n" if $DEBUG;
    my $st = from_json($res->content);
    print $st->{'status'}, ': ', $st->{'message'}, "\n" unless $QUIET;
}

# Ein Thumbnail aus der angegebenen Datei erzeugen und
# zum Webservice hochladen
sub thumbnailize_and_post($) {
    my ($filename) = @_;
    my $thumb_filename = mk_thumb_name($filename);
    my $img = Image::Magick->new;
    $img->Read($filename);
    my ($w, $h) = $img->Get('width', 'height');
    $img->Resize( width  => int($max_thumb_width),
                  height => int($max_thumb_width*$h/$w),
                 );
    $img->Write($thumb_filename);
    post_pic($thumb_filename);
}

# JPEG-Bild von angegebener URL herunterladen und dessen
# Binärdaten zurückgeben
sub get_pic($) {
    my ($url) = @_;
    my $req = HTTP::Request->new(GET => $url);
    $req->header(Accept => 'image/jpeg');
    my $res = $ua->request($req);
    throw Error::Simple($res->message, $res->code)
        unless $res->is_success;
    return $res->content;
}

# Posterous-URL einlesen und verarbeiten
sub process_posterous($) {
    my ($url) = @_;
    my ($id) = $url =~ m#.*/(\w+)#;
    print_v("Verarbeiten von http://post.ly/$id ...\n");
    my $xml_string = fetch_from_url("http://posterous.com/api/getpost/$id");
    if ($xml_string) {
        my $xml = eval { XMLin($xml_string) };
        if ($@) {
            caution("process_posterous($url) fehlgeschlagen: $@");
        }
        elsif ($xml->{'post'}) {
            my $filename = "post.ly-$id.xml";
            if (!$no_write_posterous_files && !-f $filename || $force_write) {
                open XML, ">$filename" or die "Kann $filename nicht schreiben";
                binmode XML;
                print XML $xml_string;
                close XML;
            }
            my $sql = 'INSERT OR REPLACE INTO posts ' .
                '(id, url, created_at, author, title, body) ' .
                'VALUES (?, ?, ?, ?, ?, ?)' if defined $dbh;
            my $sth = (!$dbh)? undef : ($dbh->prepare($sql) or die $dbh->errstr);
            $xml->{'post'}->{'date'} = mk_date_from_posterous($xml->{'post'}->{'date'});
            $xml->{'post'}->{'title'} =~ s/^\s+//;
            $xml->{'post'}->{'title'} =~ s/\s+$//;
            $xml->{'post'}->{'body'} =~ s/^\s+//;
            $xml->{'post'}->{'body'} =~ s/\s+$//;
            utf8::encode($xml->{'post'}->{'title'});
            utf8::encode($xml->{'post'}->{'body'});
            utf8::encode($xml->{'post'}->{'author'});
            my $res = $sth->execute($xml->{'post'}->{'id'},
                                    $xml->{'post'}->{'url'},
                                    $xml->{'post'}->{'date'},
                                    $xml->{'post'}->{'author'},
                                    $xml->{'post'}->{'title'},
                                    $xml->{'post'}->{'body'})
                or die "$sth->errstr ($sql)";
            return $xml->{'post'};
        }
    }
    return undef;
}

# Einen Tweet nach URLs durchsuchen. Verlinkte Bilder
# herunterladen, Thumbnails daraus erzeugen, beide
# Bilder ggf. zu einem Webserver hochladen.
# Links von URL-Shortenern auflösen und
# das Mapping von Kurz- zu Lang-URL merken.
sub process_urls($) {
    my ($tx) = @_;
    while (my ($url, $remainder) =
           $tx =~ /(https?:\/\/[\w\.-]+\.[a-z\.]{2,6}[\/\w\.-]*\/?)(.*)/) {
        my $pic_url = undef;
        if (!$no_pics && grep { $url =~ /$_/ } @img_services) {
            ( my $lnk, my $src, my $code ) = $url =~ /(http:\/\/([\w\.-]+)\/)(\w+)/;
            if ($lnk =~ /twitgoo\.com/) {
                $pic_url = "$lnk$code/img";
            }
            elsif ($lnk =~ /twitpic\.com/) {
                $pic_url = extract_by_regex(
                    $url, q/<img id="photo-display".+? src="(.*?)"/);
            }
            elsif ($lnk =~ /img\.ly/) {
                $pic_url = extract_by_regex(
                    $url, q/id="the-image" src="(.*?)"/);
            }
            elsif ($lnk =~ /yfrog\.com/) {
                $pic_url = extract_by_regex(
                    $url, q/<link rel="image_src" href="(.*?)"/);
            }
            elsif ($lnk =~ /mobypicture\.com/) {
                $pic_url = extract_by_regex(
                    $url, q/<img id="photo".+? src="(.*?)"/);
            }
            if (defined $pic_url) {
                # absolute URL erzeugen, falls URL relativ
                $pic_url = "$lnk$pic_url" unless $pic_url =~ /https?:\/\//;
                my $pic_file = "$src-$code.jpg";
                if ($pic_url && !-f $pic_file || $force_write) {
                    my $picdata;
                    try {
                        $picdata = get_pic($pic_url);
                    }
                    catch Error::Simple with {
                        my $E = shift;
                        caution($E);
                    };
                    if (defined $picdata) {
                        open OUT, ">$pic_file"
                            or die "Kann $pic_file nicht zum Schreiben öffnen.";
                        binmode OUT;
                        print OUT $picdata;
                        close OUT;
                        post_pic($pic_file);
                    }
                }
                thumbnailize_and_post($pic_file)
                    unless !defined($pic_file) || -f mk_thumb_name($pic_file);
            }
            else {
                warn "Der Tweet enthielt den Bild-Link $url, der Direkt-Link auf das Bild konnte aber nicht bestimmt werden.";
            }
        }
        if (!defined($pic_url) && !$no_url_mapping && grep { $url =~ /$_/ } @shorteners) {
            my $req = HTTP::Request->new(GET => $url);
            my $res = $ua1->request($req); # max. 1 Redirect !
            if (my $prev = $res->previous) {
               my $destination = $prev->headers->header('location');
               print "URL $url REDIRECTS TO $destination\n" if $DEBUG;
               push @{$urlmappings}, {
                   'origin'      => $url,
                   'destination' => uri_escape($destination, "\x00-\x1f\x7f-\xff")
               };
            }
            if ($url =~ /post\.ly/) {
                my $post = process_posterous($url);
                push @{$posts}, $post if defined $post;
            }
        }
        last unless $remainder;
        $tx = $remainder;
    }
}

# JSON-Daten an eine URL senden und JSON-Nachricht empfangen, ob die
# Verarbeitung der Daten erfolgreich war. Im Erfolgsfall enthält das
# Feld "status" der Rückgabe den Wert "ok".
sub post_json_to_url($$;$) {
    my ($data, $url, $convert) = @_;
    my $req = HTTP::Request->new(POST => $url);
    $req->header('Content-Transfer-Encoding' => 'binary');
    $req->content_type('application/binary');
    my $json = to_json($data);
    $req->content($json);
    my $res = $ua->request($req);
    throw Error::Simple($res->message, $res->code)
        unless $res->is_success;
    if ($convert) {
        my $reply = from_json($res->content);
        print $reply->{'status'}, ': ', $reply->{'message'}, "\n" unless $QUIET;
        return defined($reply->{'ids'})? $#{$reply->{'ids'}} : undef;
    }
    else {
        return $res->content;
    }
}

sub post_tweets() {
    return unless defined $tweets && defined $output_tweet_url;
    return if $#{$tweets} < 0;
    print "Senden der Tweets an $output_tweet_url ...\n" unless $QUIET;
    return post_json_to_url($tweets, $output_tweet_url, TRUE);
}

sub post_mappings() {
    return unless defined $urlmappings && defined $output_map_url;
    return if $#{$urlmappings} < 0;
    print "Senden der URL-Mappings an $output_map_url ...\n" unless $QUIET;
    return post_json_to_url($urlmappings, $output_map_url, TRUE);
}

sub post_posts() {
    return unless defined $posts && defined $output_post_url;
    return if $#{$posts} < 0;
    print "Senden der Posts an $output_post_url ...\n" unless $QUIET;
    return post_json_to_url($posts, $output_post_url, TRUE);
}

sub post_pics() {
    foreach (@{$pics}) {
        post_pic($_);
    }
}

sub url_param() {
    my $param = '';
    $param .= "&page=$tweet_page"   if defined $tweet_page;
    $param .= "&count=$tweet_count" if defined $tweet_count;
    return $param;
}

sub fetch_from_url($) {
    my ($url) = @_;
    print "URL = $url\n" if $DEBUG;
    my $req = HTTP::Request->new(GET => $url);
    my $res = $ua->request($req);
    # wenn API-Limit erschöpft -> 400 Bad Request
    throw Error::Simple($res->message, $res->code)
        unless $res->is_success;
    return $res->content;
}

# Tweet-IDs, Kurz-URLs, Posterous-IDs und Bildernamen vom Webserver holen.
sub fetch_remote_ids() {
    print "Herunterladen der IDs vom Webserver ...\n" unless $QUIET;
    my $json = fetch_from_url($input_id_url);
    my $ids = from_json($json);
    map { push @{$remote_tweets},      { 'id'     => $_ } } @{$ids->{'tweets'}};
    map { push @{$remote_urlmappings}, { 'origin' => $_ } } @{$ids->{'urlmappings'}};
    map { push @{$remote_posts},       { 'id'     => $_ } } @{$ids->{'posts'}};
    $remote_pics = $ids->{'pics'};
}

sub fetch_remote_tweets() {
    return unless defined $remote_tweets;
    my $ids;
    map { push @{$ids}, $_->{'id'} } @{$remote_tweets};
    return if $#{$ids} < 0;
    print "Herunterladen der fehlenden Tweets von '$input_tweet_url' ...\n" unless $QUIET;
    my $newTweets = from_json(post_json_to_url($ids, $input_tweet_url));
    my $sql = 'INSERT OR REPLACE INTO tweets ' .
              '(id, user_id, user_screen_name, user_name, created_at, tweet, in_reply_to_status_id, in_reply_to_screen_name) ' .
              'VALUES (?, ?, ?, ?, ?, ?, ?, ?)' if defined $dbh;
    my $sth = (!$dbh)? undef : ($dbh->prepare($sql) or die $dbh->errstr);
    print_v('Einfügen der Tweets in lokale Datenbank ... ');
    my $msg_count = 0;
    $dbh->do('BEGIN TRANSACTION');
    foreach (@{$newTweets}) {
        my $res = $sth->execute($_->{'id'},
                                $_->{'user'}->{'id'},
                                $_->{'user'}->{'screen_name'},
                                $_->{'user'}->{'name'},
                                $_->{'created_at'},
                                $_->{'tweet'},
                                $_->{'in_reply_to_status_id'},
                                $_->{'in_reply_to_screen_name'})
            or die "$sth->errstr ($sql)";
        ++$msg_count;
        print_v(sprintf("%3d%%\b\b\b\b", 100*$msg_count/@{$newTweets}));
    }
    $dbh->do('END TRANSACTION');
    print_v("\n");
}

sub fetch_pic($) {
    my ( $filename ) = @_;
    my $url = $input_pic_url . "?filename=" . uri_escape($filename, "\x00-\x1f\x7f-\xff");
    return get_pic($url);
}

sub fetch_remote_pics() {
    return unless defined $remote_pics;
    return if $#{$remote_pics} < 0;
    print "Herunterladen der fehlenden Bilder von '$input_pic_url' ... " unless $QUIET;
    my $msg_count = 0;
    foreach (@{$remote_pics}) {
        my $jpeg = fetch_pic($_);
        open PICOUT, ">$_" or die "Kann '$_' nicht zum Schreiben öffnen: $!";
        binmode PICOUT;
        print PICOUT $jpeg;
        close PICOUT;
        ++$msg_count;
        print_v(sprintf("%3d%%\b\b\b\b", 100*$msg_count/@{$remote_pics}));
    }
    print_v("\n");
}

sub fetch_remote_posts() {
    return unless defined $remote_posts;
    my $ids;
    map { push @{$ids}, $_->{'id'} } @{$remote_posts};
    return if $#{$ids} < 0;
    print "Herunterladen der fehlenden Posterous-Posts von '$input_post_url' ...\n" unless $QUIET;
    my $newPosts = from_json(post_json_to_url($ids, $input_post_url));
    print Dumper($newPosts) if $DEBUG;
    my $sql = 'INSERT OR REPLACE INTO posts (id, url, created_at, author, title, body) ' .
              'VALUES (?, ?, ?, ?, ?, ?)' if defined $dbh;
    my $sth = (!$dbh)? undef : ($dbh->prepare($sql) or die $dbh->errstr);
    print_v('Einfügen der Posterous-Posts in lokale Datenbank ... ');
    my $msg_count = 0;
    $dbh->do('BEGIN TRANSACTION');
    foreach (@{$newPosts}) {
        my $res = $sth->execute($_->{'id'},
                                $_->{'url'},
                                $_->{'created_at'},
                                $_->{'author'},
                                $_->{'title'},
                                $_->{'body'})
            or die "$sth->errstr ($sql)";
        ++$msg_count;
        print_v(sprintf("%3d%%\b\b\b\b", 100*$msg_count/@{$newPosts}));
    }
    $dbh->do('END TRANSACTION');
    print_v("\n");
}

sub fetch_remote_mappings() {
    return unless defined $remote_urlmappings;
    my $ids;
    map { push @{$ids}, $_->{'origin'} } @{$remote_urlmappings};
    return if $#{$ids} < 0;
    print "Herunterladen der fehlenden URL-Abbildungen von '$input_map_url' ...\n" unless $QUIET;
    my $newMappings = from_json(post_json_to_url($ids, $input_map_url));
    print Dumper($newMappings) if $DEBUG;
    my $sql = 'INSERT OR REPLACE INTO urlmappings (origin, destination) ' .
              'VALUES (?, ?)' if defined $dbh;
    my $sth = (!$dbh)? undef : ($dbh->prepare($sql) or die $dbh->errstr);
    print_v('Einfügen der URL-Abbildungen in lokale Datenbank ... ');
    my $msg_count = 0;
    $dbh->do('BEGIN TRANSACTION');
    foreach (@{$newMappings}) {
        my $res = $sth->execute($_->{'origin'},
                                $_->{'destination'})
            or die "$sth->errstr ($sql)";
        ++$msg_count;
        print_v(sprintf("%3d%%\b\b\b\b", 100*$msg_count/@{$newMappings}));
    }
    $dbh->do('END TRANSACTION');
    print_v("\n");
}

# Liste der Twitter-User ermitteln, denen der angemeldete Benutzer folgt
sub fetch_friends() {
    print_v("Ermitteln der Friends von $username ...\n");
    my $url = "http://twitter.com/statuses/friends/$username.json";
    return from_json(fetch_from_url($url));
}

sub fetch_timeline(;$) {
    my ($friend_id, $friend_name) = @_;
    print 'Herunterladen der ', (($fetch_mode eq 'friends')
                                 ? 'User+Friends' : 'User'),
          "-Timeline", ((defined $friend_name)? " von $friend_name" : ''), " ...\n" unless $QUIET;
    my $credentials = (defined $username && defined $password)? "$username:$password\@" : '';
    my $url = "$http_scheme://$credentials" . 'twitter.com/statuses/' .
        (($fetch_mode eq 'friends')? 'friends' : 'user')
        . '_timeline.json?' . url_param();
    $url .= "&user_id=$friend_id" if defined $friend_id;
    $url .= "&screen_name=$username" unless $credentials;
    $url .= "&since_id=" . $since->{'user'}->{'id'}
        if defined $since->{'user'}->{'id'} && $since->{'user'}->{'id'} > -1;
    return fetch_from_url($url);
}

sub fetch_mentions() {
    print "Herunterladen der Mentions ...\n" unless $QUIET;
    my $url = "$http_scheme://$username:$password\@twitter.com/statuses/mentions.json?" . url_param();
    return fetch_from_url($url);
}

sub fetch_dm() {
    print "Herunterladen der empfangenen Direktnachrichten ...\n" unless $QUIET;
    my $url = "$http_scheme://$username:$password\@twitter.com/direct_messages.json?" . url_param();
    $url .= ("&since_id=" . $since->{'dm'}->{'id'})
        if defined $since->{'dm'}->{'id'} && $since->{'dm'}->{'id'} > -1;
    return fetch_from_url($url);
}

sub fetch_dm_sent() {
    print "Herunterladen der versendeten Direktnachrichten ...\n" unless $QUIET;
    my $url = "$http_scheme://$username:$password\@twitter.com/direct_messages/sent.json?" . url_param();
    $url .= ("&since_id=" . $since->{'dm'}->{'id'})
        if defined $since->{'dm'}->{'id'} && $since->{'dm'}->{'id'} > -1;
    return fetch_from_url($url);
}

sub fetched($$$) {
    my ($fetcher, $friend_id, $friend_name) = @_;
    my $status;
    try {
        $status = from_json($fetcher->($friend_id, $friend_name));
    }
    catch Error::Simple with {
        my $E = shift;
        caution($E);
    };
    return -1 unless $status;
    print_nv('Verarbeiten der Tweets ... ');
    my $sql = 'INSERT OR REPLACE INTO tweets ' .
            '(id, user_id, user_screen_name, user_name, created_at, tweet) ' .
            'VALUES (?, ?, ?, ?, ?, ?)' if defined $dbh;
    my $sth = (!$dbh)? undef : ($dbh->prepare($sql) or die $dbh->errstr);
    my $msg_count = 0;
    foreach (@{$status}) {
        my $created_at = mk_date_from_twitter($_->{'created_at'});
        my $userid = ($fetcher == \&fetch_timeline || $fetcher == \&fetch_mentions)
            ? $_->{'user'}->{'id'}
            : $_->{'sender'}->{'id'};
        utf8::decode($userid);
        my $userscreen = ($fetcher == \&fetch_timeline || $fetcher == \&fetch_mentions)
            ? $_->{'user'}->{'screen_name'}
            : $_->{'sender'}->{'screen_name'} . ' -> ' .
              $_->{'recipient'}->{'screen_name'};
        my $username = ($fetcher == \&fetch_timeline || $fetcher == \&fetch_mentions)
            ? $_->{'user'}->{'name'}
            : $_->{'sender'}->{'name'} . ' -> ' .
              $_->{'recipient'}->{'name'};
        $_->{'text'} = decode_entities($_->{'text'});
        utf8::encode($_->{'text'});
        print_v("#$_->{'id'} $userscreen: $_->{'text'} ($created_at)\n");
        process_urls($_->{'text'});
        if ($fetcher == \&fetch_timeline) {
            $since->{'user'}->{'id'} = $_->{'id'}
                if !$initial_backup && defined($since->{'user'}->{'id'})
                    && $since->{'user'}->{'id'} < $_->{'id'};
        } else {
            $since->{'dm'}->{'id'} = $_->{'id'}
                if defined($since->{'dm'}->{'id'}) && $since->{'dm'}->{'id'} < $_->{'id'};
        }
        my $plain_file = $_->{'id'} . '.txt';
        if ($output_string =~ /plain/ && (!-f $plain_file || $force_write)) {
            open OUT, ">$plain_file"
                or die "Schreiben von '$plain_file' fehlgeschlagen: $!\n";
            my $userlong = ($fetcher == \&fetch_timeline || $fetcher == \&fetch_mentions)
                ? $_->{'user'}->{'name'} . ' ('
                . $_->{'user'}->{'screen_name'} . ')'
                : $_->{'sender'}->{'name'} . ' ('
                . $_->{'sender'}->{'screen_name'} . ') -> '
                . $_->{'recipient'}->{'name'} . ' ('
                . $_->{'recipient'}->{'screen_name'} . ')';
            my $data = "$userlong: $_->{'text'} ($created_at)";
            print OUT "\x{ef}\x{bb}\x{bf}" unless $no_bom;
            print OUT "$data\n";
            close OUT;
        }
        my $json_file = "$_->{'id'}.json";
        if ($output_string =~ /json/ && (!-f $json_file || $force_write)) {
            my $data = to_json($_);
            open OUT, ">$json_file"
                or die "Schreiben von '$json_file' fehlgeschlagen: $!\n";
            print OUT "\x{ef}\x{bb}\x{bf}" unless $no_bom;
            print OUT "$data\n";
            close OUT;
        }
        if (defined $sql) {
            my $res = $sth->execute($_->{'id'}, $userid, $userscreen,
                                    $username, $created_at, $_->{'text'})
                or die "$sth->errstr ($sql)";
        }
        my $newTweet = {
            'id' => $_->{'id'},
            'user_id' => $userid,
            'user_screen_name' => $userscreen,
            'user_name' => $username,
            'created_at' => $created_at,
            'text' => $_->{'text'},
            'in_reply_to_screen_name' => $_->{'in_reply_to_screen_name'},
            'in_reply_to_status_id' => $_->{'in_reply_to_status_id'},
        };
        push @{$tweets}, $newTweet;
        print to_json($newTweet) if $DEBUG;
        ++$msg_count;
        print_nv(sprintf("%3d%%\b\b\b\b", 100*$msg_count/@{$status}));
    }
    print_nv("\n");
    if (defined $urlmappings) {
        print_v("Schreiben der URL-Abbildungen ...\n");
        my $sql = 'INSERT OR REPLACE INTO urlmappings ' .
                  '(origin, destination) VALUES (?, ?)' if defined $dbh;
        my $sth = (!$dbh)? undef : ($dbh->prepare($sql) or die $dbh->errstr);
        foreach (@{$urlmappings}) {
            my $res = $sth->execute($_->{'origin'}, $_->{'destination'})
                or die $sth->errstr . " ($sql)";
        }
    }
    return $msg_count;
}

sub fetch($;$$) {
    my ($fetcher, $friend_id, $friend_name) = @_;
    my $total_msg_count = 0;
    if ($initial_backup) {
        my $msg_count = 0;
        $tweet_page = 1;
        $since->{'user'}->{'id'} = -1;
        try {
            while (($msg_count = fetched($fetcher, $friend_id, $friend_name)) > 0) {
                $total_msg_count += $msg_count;
                ++$tweet_page;
            }
        }
        catch Error::Simple with {
            my $E = shift;
            caution($E);
            # TODO: Fehler wie "400, Bad Request" abfangen und
            # Infos für Wiederaufnahme des Backup-Laufs ins
            # Arbeitsverzeichnis schreiben
            open RESUMEINFO, ">$snapshot_file"
                or die "Kann $snapshot_file nicht zum Schreiben öffnen: $!";
            close RESUMEINFO;
        };

    }
    else {
        $total_msg_count = fetched($fetcher, $friend_id, $friend_name);
    }
    return $total_msg_count;
}

sub copy_to_remote() {
    print "Wenn Sie fortfahren, werden sämtliche Datenbankeinträge "
        . "in der Web-Datenbank mit den lokalen Daten überschrieben.\n"
        . "Tippen Sie 'ja' [RETURN], um fortzufahren: ";
    my $answer = <>;
    chomp($answer);
    if ($answer eq 'ja') {
        read_local_tweets();
        read_local_mappings() unless $no_url_mapping;
        read_local_posts()    unless $no_posts;
        read_local_pics()     unless $no_pics;
        try {
            $num_tweets_sent   = post_tweets();
            $num_mappings_sent = post_mappings() unless $no_url_mapping;
            $num_posts_sent    = post_posts() unless $no_posts;
            $num_pics_sent     = post_pics() unless $no_pics;
        }
        catch Error::Simple with {
            my $E = shift;
            caution($E);
        };
    }
}

sub synchronize() {
    read_local_tweets();
    fetch_remote_ids();
    deduplicate($tweets, $remote_tweets, 'id');
    unless ($no_url_mapping) {
        read_local_mappings();
        deduplicate($urlmappings, $remote_urlmappings, 'origin');
    }
    unless ($no_pics) {
        read_local_pics();
        deduplicate($pics, $remote_pics);
    }
    unless ($no_posts) {
        read_local_posts();
        deduplicate($posts, $remote_posts, 'id');
    }
    try {
        fetch_remote_tweets();
        fetch_remote_mappings() unless $no_url_mapping;
        fetch_remote_pics()     unless $no_pics;
        fetch_remote_posts()    unless $no_posts;
    }
    catch Error::Simple with {
        my $E = shift;
        caution($E);
    };
    try {
        $num_tweets_sent   = post_tweets();
        $num_mappings_sent = post_mappings() unless $no_url_mapping;
        $num_posts_sent    = post_posts()    unless $no_posts;
        $num_pics_sent     = post_pics()     unless $no_pics;
    }
    catch Error::Simple with {
        my $E = shift;
        caution($E);
    };
}

sub write_max_ids() {
    return if !defined $tweets;
    foreach my $since_id_key (keys %{$since}) {
        if (defined $since->{$since_id_key}->{'id'}) {
            my $file = "$output_path$pathsep" . $since->{$since_id_key}->{'file'};
            print "Schreiben von $file ...\n" if $DEBUG;
            open OUT, ">$file"
                or die "Kann $file nicht zum Schreiben öffnen: $!";
            print OUT $since->{$since_id_key}->{'id'};
            close OUT;
        }
    }
}

sub read_max_ids() {
    foreach my $since_id_key (keys %{$since}) {
        my $file = "$output_path$pathsep" . $since->{$since_id_key}->{'file'};
        print "Lesen von $file ..." if $DEBUG;
        my $id = -1;
        if (-f $file) {
            open IN, "<$file" or die "Kann $file nicht öffnen: $!";
            $id = <IN>;
            close IN;
        }
        $since->{$since_id_key}->{'id'} = $id if $id;
        print " $id\n" if $DEBUG;
    }
}

# Symmetrische Restmengen zweier aufsteigend sortierter ARRAYREFs ohne doppelte Elemente bilden.
sub deduplicate($$;$) {
    my ($A, $B, $id) = @_;
    print_v("  Bilden der symmetrischen Restmengen ... ");
    my ($I, $J) = ( $#{$A}, $#{$B} );
    my $j0 = 0;
    if (defined $id) {
        for (my $i = 0; $i <= $I; ++$i) {
            if (defined($A->[$i])) {
                foreach (my $j = $j0; $j <= $J; ++$j) {
                    if (defined($B->[$j]) && ($A->[$i]->{$id} eq $B->[$j]->{$id})) {
                        undef $A->[$i];
                        undef $B->[$j];
                        $j0 = $j + 1;
                        last;
                    }
                }
            }
        }
    }
    else {
        for (my $i = 0; $i <= $I; ++$i) {
            if (defined($A->[$i])) {
                for (my $j = $j0; $j <= $J; ++$j) {
                    if (defined($B->[$j]) && ($A->[$i] eq $B->[$j])) {
                        undef $A->[$i];
                        undef $B->[$j];
                        $j0 = $j + 1;
                        last;
                    }
                }
            }
        }
    }
    my @AA; # Eliminieren der undef-Elemente
    map { push @AA, $_ if defined $_ } @{$A};
    @{$A} = @AA;
    my @BB; # Eliminieren der undef-Elemente
    map { push @BB, $_ if defined $_ } @{$B};
    @{$B} = @BB;
    print_v((1+$#AA) . '/' . (1+$#BB) . " verbleiben.\n");
}


sub init_db() {
    print_v("Initialisieren der Datenbank ...\n");
    foreach (split ';', do { local $/ = undef; <DATA> }) {
        chomp;
        s/\s+$//;
        s/^\s+//;
        next unless $_;
        $dbh->do($_) or die $dbh->errstr . " ($_)";
    };
}

sub mk_iso_date($$$$$$$) {
    my ($day, $month, $yr, $hours, $mins, $secs, $offset) = @_;
    my $datetime = DateTime->new(year   => $yr,
                                 month  => $monthMap{$month},
                                 day    => $day,
                                 hour   => $hours,
                                 minute => $mins,
                                 second => $secs,
                                 formatter => $dt_isoformatter
    );
    my ($offset_sign_r, $h_offset_r, $m_offset_r) = $offset =~ /([+-])(\d{2})(\d{2})/;
    my %offset_r = (hours => $h_offset_r, minutes => $m_offset_r, seconds => 0);
    my %offset = (hours => $h_offset, minutes => $m_offset, seconds => 0);
    if ($offset_sign eq '-') {
        $datetime->subtract(%offset);
    } else {
        $datetime->add(%offset);
    }
    if ($offset_sign_r eq '+') {
        $datetime->subtract(%offset_r);
    } else {
        $datetime->add(%offset_r);
    }
    return "$datetime";
}

sub mk_date_from_posterous($) {
    # Thu, 08 Oct 2009 11:14:00 -0700
    my ($day, $month, $yr, $hours, $mins, $secs, $offset) =
        (shift =~ /[a-zA-Z]{3}, (\d+) ([a-zA-Z]{3}) (\d+) (\d{2}):(\d{2}):(\d{2}) ([+-]\d+)/);
    return mk_iso_date($day, $month, $yr, $hours, $mins, $secs, $offset);
}

sub mk_date_from_twitter($) {
    # Wed Aug 05 10:06:41 +0000 2009
    my ($month, $day, $hours, $mins, $secs, $offset, $yr) =
        (shift =~ /[a-zA-Z]{3} ([a-zA-Z]{3}) (\d+) (\d{2}):(\d{2}):(\d{2}) ([+-]\d+) (\d+)/);
    return mk_iso_date($day, $month, $yr, $hours, $mins, $secs, $offset);
}

sub GREETING() {
    print "\ntwitterbak.pl $VERSION\n",
    "Twitter-Timeline in Dateien oder in einer Datenbank sichern.\n\n",
    "Copyright (c) 2009 Heise Zeitschriften Verlag\n",
    "                 - Oliver Lau <oliver\@von-und-fuer-lau.de>\n",
    "Alle Rechte vorbehalten.\n\n";
}

sub USAGE() {
    print "Aufrufen mit: twitterbak.pl <Optionen>\n",
    "\nOptionen:\n",
    "  --config=file | -c file\n",
    "      Pfad und Name zur Konfigurationsdatei (Vorgabe: twitterbak.ini)\n",
    "  --init\n",
    "      Versuche, beim Einsammeln der Tweets so weit wie möglich in die\n",
    "      Vergangenheit zu gehen. (--page wird ignoriert)\n",
    "  --count=n | -n n\n",
    "      n Tweets abholen (max. 200, Default: 20)\n",
    "  --page=n | -p n\n",
    "      n-te Seite abrufen\n",
    "  --copy\n",
    "      Daten aus der lokalen Datenbank und Bilder zum Webserver kopieren\n",
    "  --sync\n",
    "      Daten und Bilder des Webservers mit den lokalen synchronisieren.\n",
    "      Es werden nur fehlende Datensätze/Bilder übertragen.\n",
    "  --fetch-mode=(", join('|', @allowed_fetch_modes), ")\n",
    "  -f (", join('|', @allowed_fetch_modes), ")\n",
    "      user    = nur User-Timeline lesen\n",
    "      friends = User+Friends-Timeline lesen\n",
    "      all     = User-Timelines des angemeldeten Benutzers und\n",
    "                aller seiner Friends lesen\n",
    "      Vorgabe: $default_fetch_mode\n",
    "  --no-dm\n",
    "      Direktnachrichten ignorieren\n",
    "  --outputs=(", join('|', @allowed_outputs), ")\n",
    "  -o (", join('|', @allowed_outputs), ")\n",
    "      kommaseparierte Auflistung der Formate, in denen die heruntergeladenen\n",
    "      Tweets abgespeichert werden sollen. Zum Beispiel, um die Tweets als\n",
    "      Textdatei und in der Datenbank abzuspeichern: --outputs=plain,db\n",
    "      Vorgabe: $default_output_string\n",
    "  --path=path\n",
    "      Absoluter Pfad zu dem Verzeichnis, in dem die Tweets abgelegt werden\n",
    "      sollen.\n",
    "      Vorgabe: '$default_working_dir'\n",
    "  --db-name=name\n",
    "      Datenbanknamen festlegen. Vorgabe: <username>.sqlite\n",
    "  --only-create-db\n",
    "      Nur die Datenbanktabellen neu anlegen.\n",
    "      Achtung, alle Daten gehen verloren!\n",
    "  --force-write\n",
    "      Eventuell bereits vorhandene Dateien überschreiben\n",
    "  --no-bom\n",
    "      Unterbindet das Schreiben des Unicode-BOM in die Textdatei\n",
    "  --verbose | -v\n",
    "      Detaillierte Informationen über Verarbeitungsschritte ausgeben\n",
    "  --quiet | -q\n",
    "      Sämtliche Bildschirmausgaben unterdrücken\n",
    "  --help | -? | -h\n",
    "      Diese Hilfe anzeigen\n",
    "\n";
    exit;
}

__DATA__
DROP TABLE IF EXISTS tweets;

CREATE TABLE tweets
(
 id INTEGER PRIMARY KEY,
 user_id INTEGER NOT NULL,
 user_screen_name TEXT,
 user_name TEXT,
 created_at DATETIME NOT NULL,
 tweet TEXT,
 in_reply_to_screen_name TEXT,
 in_reply_to_status_id INTEGER
);

DROP INDEX IF EXISTS tweetidx;

CREATE INDEX tweetidx ON tweets
(
 user_screen_name,
 user_name,
 user_id,
 tweet
);

DROP TABLE IF EXISTS urlmappings;

CREATE TABLE urlmappings
(
 origin TEXT PRIMARY KEY,
 destination TEXT
);


DROP TABLE IF EXISTS posts;

CREATE TABLE posts
(
 id INTEGER PRIMARY KEY,
 url TEXT NOT NULL,
 created_at DATETIME NOT NULL,
 author TEXT NOT NULL,
 title TEXT NOT NULL,
 body TEXT NOT NULL
);
