package com.linkesoft.bbingo;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import android.app.Activity;
import android.app.AlertDialog;
import android.graphics.Color;
import android.graphics.PorterDuff;
import android.os.Bundle;
import android.view.View;
import android.widget.Button;
import android.widget.LinearLayout;
import android.widget.LinearLayout.LayoutParams;

/**
 * Hauptklasse und einzige Activity der B-Bingo-Anwendung.
 * 
 * Siehe {@link http://de.wikipedia.org/wiki/Buzzword-Bingo}
 * 
 * @author Andreas Linke
 *
 */
public class Main extends Activity {
	private static final int SIZE = 5;
	
	// hard-codierte Liste von mindestens 25 Buzz-Wrtern 
	// (wird in einer spteren Version dynamisch)
	private static final List<String> words=new ArrayList<String>(
			Arrays.asList(
					"Android","Architektur",
					"Big Picture","Benchmark",
					"Context","Core",
					"Gadget",
					"Hut aufhaben",
					"Internet","iPhone",
					"Kundenorientiert",
					"Mobile irgendwas",
					"Netzwerk",
					"People","pro-aktiv",
					"Qualitt",
					"Ressourcen","Revolution","Runde drehen",
					"Sozial","Synergie",
					"Technologie",
					"berall",
					"Values","Virtuell","Vision",
					"Weltneuheit","Web 2.0","Win-Win",
					"Zielfhrend"
			));

/**
 * Startmethode, wird beim ersten Start der Activity aufgerufen,
 * auch bei nderung der Orientierung (horizontal/vertikal).
 * Definiere GUI-Layout und verknpfe Event-Handler.	
 */
	@Override
	public void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		boolean staticUI=false; // zwei Wege, um das UI zu erzeugen
		if(staticUI)
			setContentView(R.layout.main); // aus XML
		else
			createUIDynamically(); // dynamisch
		
		shuffleWords();

		// verknpfe Buttons mit Aktionen	
		for (int y = 0; y < SIZE; y++)
			for (int x = 0; x < SIZE; x++) {
				Button button = (Button) findViewById(getButtonID(x, y));				
				button.setOnClickListener(onButtonClick); 
			}
	}
/**
 * Button-Klick-Handler fr alle Buttons	
 */
	final View.OnClickListener onButtonClick=new View.OnClickListener() {
		@Override
		public void onClick(View v) {
			toggleButtonState((Button) v);
			checkForBingo();
		}
	};

/**
 * Liefere numerische Button-ID per Reflection aus den Konstanten der R-Klasse
 * @param x Spaltennummer 0..4
 * @param y Zeilennummer 0..4
 * @return
 */
	private int getButtonID(int x, int y) {
		try {
			// lookup id by reflection
			// button id is a static int defined in R.id, e.g. R.id.Button01
			String buttonid = "Button" + y + "" + (x + 1); // e.g. "Button01" 
			return R.id.class.getField(buttonid).getInt(null);
		} catch (Exception e) {
			// reflection lookup could throw exceptions e.g. if the button is
			// not found
			throw new RuntimeException("Internal error", e);
		}
	}

	private boolean isButtonSet(Button button) {
		return button.getTag() != null;
	}

/**
 * ndere Zustand eines Buttons von nicht gesetzt auf gesetzt 
 * und umgekehrt. Dabei wird die Button-Farbe entsprechend angepasst. 
 * @param button der zu ndernde Button
 */
	private void toggleButtonState(Button button) {
		if (isButtonSet(button)) {
			button.getBackground().clearColorFilter();
			button.setTag(null);
		} else {
			// Button-Hintergrund-Farbe wird ber einen Filter gendert 
			button.getBackground().setColorFilter(Color.GREEN,
					PorterDuff.Mode.MULTIPLY);
			button.setTag("x"); // verwende die frei tag-Property zum Speichern des Zustands
		}
	}

/**
 * Prfe auf vollstndige ausgewhlte Button-Reihe 
 * vertikal, horizontal und die beiden Diagonalen 
 */
	private void checkForBingo() {
		boolean bingo = false;
		int x, y;
		// vertikal
		for (x = 0; x < SIZE && !bingo; x++) {
			for (y = 0; y < SIZE; y++) {
				Button button = (Button) findViewById(getButtonID(x, y));
				if (!isButtonSet(button))
					break;
			}
			bingo = (y == SIZE);
		}
		// horizontal
		for (y = 0; y < SIZE && !bingo; y++) {
			for (x = 0; x < SIZE; x++) {
				Button button = (Button) findViewById(getButtonID(x, y));
				if (!isButtonSet(button))
					break;
			}
			bingo = (x == SIZE);
		}
		// diagonal
		for (x = 0; x < SIZE && !bingo; x++) {
			Button button = (Button) findViewById(getButtonID(x, x));
			if (!isButtonSet(button))
				break;
		}
		if (x == SIZE)
			bingo = true;
		for (x = 0; x < SIZE && !bingo; x++) {
			Button button = (Button) findViewById(getButtonID(x, SIZE - 1 - x));
			if (!isButtonSet(button))
				break;
		}
		if (x == SIZE)
			bingo = true;

		if (bingo) {
			// Gewonnen, zeige ein einfaches Popup.
			// Die Methoden geben jeweils wieder das Builder-Objekt zurck
			// und lassen sich so leicht verketten
			new AlertDialog.Builder(this)
					.setTitle(R.string.BBingoWonTitle)
					.setMessage(R.string.BBingoWon)
					.setPositiveButton(android.R.string.ok, null)
					.show();
		}
	}
/**
 * Worte der Wortliste mischen und auf die Buttons verteilen
 */
	private void shuffleWords()
	{
		assert(words.size()>=SIZE*SIZE); // Liste muss mindestens 25 Elemente enthalten
		Collections.shuffle(words);
		int i=0;
		for (int x = 0; x < SIZE; x++) 
			for (int y = 0; y < SIZE; y++)
			{
				Button button=(Button) findViewById(getButtonID(x, y));
				button.setText(words.get(i++)); 
			}				
	}
/**
 * Alternative zur XML-basierten GUI-Definition	
 */
	private void createUIDynamically() {
		// wichtig: fr LayoutParams immer die passende Layout-Klasse verwenden
		LinearLayout main = new LinearLayout(this);
		main.setOrientation(LinearLayout.VERTICAL);
		for (int y = 0; y < SIZE; y++) {
			LinearLayout row = new LinearLayout(this);
			for (int x = 0; x < SIZE; x++) {
				Button button = new Button(this);
				// die ID wird hier der Einfachheit halber aus den generierten R-Konstanten
				// gesetzt, damit das restliche Coding weiter funktioniert.
				button.setId(getButtonID(x, y)); 
				button.setWidth(0);
				button.setHeight(0);
				row.addView(button, new LinearLayout.LayoutParams(
								LayoutParams.WRAP_CONTENT,
								LayoutParams.FILL_PARENT, 1)); // weight 1
			}
			main.addView(row, new LinearLayout.LayoutParams(
					LayoutParams.FILL_PARENT, LayoutParams.WRAP_CONTENT, 1)); // jede Zeile soll komplette Breite ausfllen 
		}
		setContentView(main);
	}
}