package com.linkesoft.bbingo;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import android.app.Activity;
import android.app.AlertDialog;
import android.app.AlertDialog.Builder;
import android.content.DialogInterface;
import android.content.Intent;
import android.gesture.Gesture;
import android.gesture.GestureLibraries;
import android.gesture.GestureLibrary;
import android.gesture.GestureOverlayView;
import android.gesture.Prediction;
import android.gesture.GestureOverlayView.OnGesturePerformedListener;
import android.graphics.Color;
import android.graphics.PorterDuff;
import android.hardware.Sensor;
import android.hardware.SensorEvent;
import android.hardware.SensorEventListener;
import android.hardware.SensorManager;
import android.media.AudioManager;
import android.media.SoundPool;
import android.os.Bundle;
import android.util.Log;
import android.view.Menu;
import android.view.MenuItem;
import android.view.View;
import android.view.ViewGroup;
import android.view.animation.Animation;
import android.view.animation.AnimationUtils;
import android.view.animation.LayoutAnimationController;
import android.view.animation.TranslateAnimation;
import android.view.animation.Animation.AnimationListener;
import android.widget.Button;
import android.widget.LinearLayout;
import android.widget.Toast;
import android.widget.LinearLayout.LayoutParams;

import com.linkesoft.bbingo.BBingoDB.WordList;

/**
 * Hauptklasse-Activity der B-Bingo-Anwendung,
 * zeigt ein Gitter von 5x5 Buttons zur Auswahl von Buzzwrtern.
 * 
 * Siehe {@link http://de.wikipedia.org/wiki/Buzzword-Bingo}
 * 
 * @author Andreas Linke
 *
 */
public class Main extends Activity {
	private static final int SIZE = 5;
	// menu IDs
	private static final int EDIT = Menu.FIRST; // akt. Wortliste bearbeiten
	private static final int LISTS = Menu.FIRST+1; // Wortlisten auswhlen/anlegen/lschen
	private static final int PREFS = Menu.FIRST+2; // Einstellung fr Sound
	
	private String currentWordlist=null;

	private BBingoDB db;
	private SoundPool soundPool; // fr Erfolgsfanfare
	private int fanfareSoundID; // ID der geladenen Sound-Resource
	private int streamID; // ID des gerade gespielten Sounds (zum Stoppen)
	
/**
 * Startmethode, wird beim ersten Start der Activity aufgerufen,
 * auch bei nderung der Orientierung (horizontal/vertikal).
 * Definiere GUI-Layout und verknpfe Event-Handler.	
 */
	@Override
	public void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		setContentView(R.layout.main); // aus XML
		// verknpfe Buttons mit Aktionen	
		for (int y = 0; y < SIZE; y++)
			for (int x = 0; x < SIZE; x++) {
				Button button = (Button) findViewById(getButtonID(x, y));				
				button.setOnClickListener(onButtonClick); 
			}
		db=new BBingoDB(this); // ffne Datenbank, wird in onDestroy wieder geschlossen
		
	    // lade Soundschnipsel fr Gewinn-Fanfare
	    soundPool = new SoundPool(1, AudioManager.STREAM_MUSIC, 0);
		fanfareSoundID=soundPool.load(this, R.raw.fanfare, 1); // Verweis auf fanfare.mp3 im res/raw Ordner des Projekts

		// rechts/links Wischen zum Wechsel der Wortliste
		registerGestures();
		
		// Buttons sollen sich per Layout Animation an ihre Pltze bewegen 
		Animation animation = new TranslateAnimation(
			      Animation.RELATIVE_TO_PARENT, -1.0f, Animation.RELATIVE_TO_PARENT, 0,
			      Animation.RELATIVE_TO_PARENT, -1.0f, Animation.RELATIVE_TO_PARENT, 0);
		LayoutAnimationController controller =new LayoutAnimationController(animation,0.1f); // Verzgerung fr jedes einzelne Element in Prozent der Gesamtzeit
	    controller.setOrder(LayoutAnimationController.ORDER_RANDOM); // zufllige Reihenfolge
		controller.getAnimation().setDuration(800); // Gesamtzeit der Animation in ms
		// setze Animation fr Buttons in jeder einzelnen Zeile
		ViewGroup mainViewGroup = (ViewGroup) findViewById(R.id.LinearLayoutMain);
		for (int i = 0; i < mainViewGroup.getChildCount(); i++) {
			ViewGroup rowViewGroup = (ViewGroup) mainViewGroup.getChildAt(i);
			rowViewGroup.setLayoutAnimation(controller);
		}		
	}
	
	/**
	 * Spiele vorher im soundPool geladenen Sound ab.
	 */
	private void playFanfare() {
		AudioManager audioManager = (AudioManager) getSystemService(AUDIO_SERVICE);
		// berechne relative Lautstrke, basierend auf vom Benutzer
		// eingestellter Lautstrke und Maximum (int-Werte!)
		float volume = (float) audioManager.getStreamVolume(AudioManager.STREAM_MUSIC) / 
						(float) audioManager.getStreamMaxVolume(AudioManager.STREAM_MUSIC);
		// spiele den in onCreate geladenen Sound
		streamID = soundPool.play(fanfareSoundID, volume, volume, 1, 0, 1f);
	}
	
	/**
	 * Gesten-Erkennung: Finger nach rechts bzw. links wischen ldt vorherige bzw. nchste Wortliste. 
	 */
	private void registerGestures()	{
		// Gesten-Erkennung
		final GestureLibrary gesturelib = GestureLibraries.fromRawResource(this, R.raw.gestures);
		gesturelib.load();
		LinearLayout layout = (LinearLayout) findViewById(R.id.LinearLayoutRoot);
		GestureOverlayView gestureOverlay = new GestureOverlayView(this);
		gestureOverlay.setUncertainGestureColor(Color.TRANSPARENT); // nicht erkannte Gesten werden nicht auf den Screen gemalt
		layout.addView(gestureOverlay, new LinearLayout.LayoutParams(LayoutParams.FILL_PARENT, LayoutParams.FILL_PARENT, 1));
		// hnge den LayoutView mit den Buttons unterhalb des Overlays
		ViewGroup mainViewGroup = (ViewGroup) findViewById(R.id.LinearLayoutMain);
		layout.removeView(mainViewGroup);
		gestureOverlay.addView(mainViewGroup);
		gestureOverlay.addOnGesturePerformedListener(new OnGesturePerformedListener() {			
			@Override
			public void onGesturePerformed(GestureOverlayView overlay, Gesture gesture) {
				ArrayList<Prediction> predictions = gesturelib.recognize(gesture);
				for (Prediction prediction : predictions) {
					if (prediction.score > 1.0) {
						if (prediction.name.equals("SwipeRight"))
							prev(); // schiebe nach rechts aus dem Bild und hole den Vorgnger
						else if (prediction.name.equals("SwipeLeft"))
							next();
						break;
					}
				}
			}
		});
	}
	/**
	 * Lade nchste Wortliste, falls vorhanden, mit Animation.
	 */
	private void next() {
		long nextid = db.getAdjacentWordList(Prefs.getID(this),true); // nchste Wortliste
		if (nextid == 0)
			Toast.makeText(Main.this, R.string.NoMoreWordlists,Toast.LENGTH_SHORT).show(); // kurzes Popup
		else {
			Prefs.setID(this,nextid); // neu gesetzte Wortliste wird am Ende der Animation geladen (s.u.)
			animateSwitch(false);
		}
	}
	/**
	 * Lade vorherige Wortliste, falls vorhanden, mit Animation.
	 */
	private void prev() {
		long nextid = db.getAdjacentWordList(Prefs.getID(this),false); // vorhergehende Wortliste
		if (nextid == 0)
			Toast.makeText(Main.this, "Keine weiteren Wortlisten",Toast.LENGTH_SHORT).show();
		else {
			Prefs.setID(this,nextid); 
			animateSwitch(true);
		}
	}
	
	/**
	 * Animiere Wechsel zur nchsten Wortliste mit out- und in-Animation
	 * @param toRight true wenn nach rechts geschoben werden soll (vorherige Liste)
	 */
	private void animateSwitch(final boolean toRight) {
		final LinearLayout layout = (LinearLayout) findViewById(R.id.LinearLayoutMain);
		Animation animation = AnimationUtils.makeOutAnimation(this,toRight);
		animation.setAnimationListener(new AnimationListener() {
			@Override
			public void onAnimationStart(Animation animation) {
			}
			@Override
			public void onAnimationRepeat(Animation animation) {
			}
			@Override
			public void onAnimationEnd(Animation animation) {
				// lade neue Wortliste, wenn die out-Animation zu Ende ist 
				loadWordList();
				// und schiebe UI von der anderen Seite herein
				Animation inAnim = AnimationUtils.makeInAnimation(Main.this, toRight); 
				layout.startAnimation(inAnim);
			}
		});
		layout.startAnimation(animation); // starte ersten Teil der Animation
	}
	
	// Schttelerkennung, wird in onResume aktiviert
	private final SensorEventListener sensorListener = new SensorEventListener() {
	    public void onSensorChanged(SensorEvent se) {
	      float x = se.values[0];
	      float y = se.values[1];
	      float z = se.values[2];
	      // Log-Ausgabe fr Test-Zwecke
	      // sichtbar in DDMS-Perspektive LogCat-View
	      Log.v("BBingo", "Sensor Values x="+x+" y="+y+" z="+z);      
	      float asquaredrel=(x*x+y*y+z*z)/(SensorManager.GRAVITY_EARTH*SensorManager.GRAVITY_EARTH);	      
	      if(asquaredrel>=4) // doppelte Erdbeschleunigung
	    	  shuffleWords(true);
	    }
	    public void onAccuracyChanged(Sensor sensor, int accuracy) {
	    }
	  };

	
	/**
	 * Wird nach onCreate und nach dem Beenden einer anderen Activity aufgerufen.
	 * Lade die aktuelle Wortliste. Wenn sie nicht mit der aktuellen Liste bereinstimmt,
	 * mische und setze alle Buttons neu.
	 * Registriere fr Accelerometer-Events. 
	 */
	@Override
	protected void onResume() {
		super.onResume();
		// registriere Listener fr Beschleunigungssensordaten (Schttel-Erkennung)
		// falls der Sensor-Typ nicht untersttzt wird, gibt registerListener false zurck
		SensorManager sensorManager=(SensorManager) getSystemService(SENSOR_SERVICE);
	    sensorManager.registerListener(sensorListener, sensorManager.getDefaultSensor(Sensor.TYPE_ACCELEROMETER),
			SensorManager.SENSOR_DELAY_NORMAL); 	    
		loadWordList();
	}
	@Override
	protected void onPause() {
		// Listener deregistrieren um Strom zu sparen
		SensorManager sensorManager=(SensorManager) getSystemService(SENSOR_SERVICE);		
	    sensorManager.unregisterListener(sensorListener); 
		super.onPause();
	}
	/**
	 * Hole aktuelle Wortlist aus der DB.
	 * Falls noch keine Liste gespeichert wurde,
	 * wird auf eine eingebaute Standard-Liste zurckgegriffen.
	 */
	private void loadWordList()
	{
		// 
		WordList wordlist = db.getWordList(Prefs.getID(this));
		if (wordlist == null) {
			// keine Wortliste zur ID gefunden, nimm erste
			long id = db.getFirstWordListID();
			if (id == 0) {
				// keine Wortliste gespeichert
				id = db.createDefaultWordList(); // erzeuge Standard-Liste
			}
			Prefs.setID(this, id);
			wordlist = db.getWordList(id);
		}
		setTitle("B-Bingo: "+wordlist.title); // zeige Titel der aktuelle Wortliste im Titel der Activity
		if (currentWordlist == null || !currentWordlist.equals(wordlist.words)) {
			// Wortliste hat sich gendert, mische und setze Button-Titel entsprechend
			currentWordlist = wordlist.words;
			shuffleWords(false); // keine Animation, da schon beim Layout animiert wurde
		}
	}
	
	@Override
	protected void onDestroy() {
		db.close();
		super.onDestroy();
	}
/**
 * Button-Klick-Handler fr alle Buttons	
 */
	final View.OnClickListener onButtonClick=new View.OnClickListener() {
		@Override
		public void onClick(View v) {
			toggleButtonState((Button) v);
			checkForBingo();
		}
	};
	
/**
 * Anlegen von Mens (die per Men-Taste aufgerufen werden)	
 */
	@Override
	public boolean onCreateOptionsMenu(android.view.Menu menu) {
		menu.add(Menu.NONE, EDIT, Menu.NONE, R.string.EditWords);
		menu.add(Menu.NONE, LISTS, Menu.NONE, R.string.WordLists);
		menu.add(Menu.NONE, PREFS, Menu.NONE, R.string.Prefs).setIcon(android.R.drawable.ic_menu_preferences);
		return super.onCreateOptionsMenu(menu);
	};

/**
 * Auswahl eines Men-Eintrags	
 */
	@Override
	public boolean onOptionsItemSelected(MenuItem item) {
		switch (item.getItemId()) {
		case EDIT: {
				Intent intent = new Intent(this, EditWordList.class);
				intent.putExtra(EditWordList.ID, Prefs.getID(this)); // bergebe aktuelle ID
				startActivity(intent);
			}
			break;
		case LISTS:
			startActivityForResult(new Intent(this, WordLists.class), LISTS);
			break;
		case PREFS:
	    	startActivity(new Intent(this,PrefsActivity.class));
			break;
		}
		return super.onOptionsItemSelected(item);
	}

/**
 * Wird nach Beenden einer per <code>startActivityForResult</code> gestarteten 
 * Activity gerufen	
 */
	@Override
	protected void onActivityResult(int requestCode, int resultCode, Intent data) {
		if (requestCode == LISTS && resultCode == RESULT_OK) {
			// eine andere Liste wurde ausgewhlt
			long id = data.getExtras().getLong(WordLists.ID);
			Prefs.setID(this, id);
		}
		super.onActivityResult(requestCode, resultCode, data);
	}

/**
 * Liefere numerische Button-ID per Reflection aus den Konstanten der R-Klasse
 * @param x Spaltennummer 0..4
 * @param y Zeilennummer 0..4
 * @return
 */
	private int getButtonID(int x, int y) {
		try {
			// lookup id by reflection
			// button id is a static int defined in R.id, e.g. R.id.Button01
			String buttonid = "Button" + y + "" + (x + 1); // e.g. "Button01" 
			return R.id.class.getField(buttonid).getInt(null);
		} catch (Exception e) {
			// reflection lookup could throw exceptions e.g. if the button is
			// not found
			throw new RuntimeException("Internal error", e);
		}
	}

	private boolean isButtonSet(Button button) {
		return button.getTag() != null;
	}

/**
 * ndere Zustand eines Buttons von nicht gesetzt auf gesetzt 
 * und umgekehrt. Dabei wird die Button-Farbe entsprechend angepasst. 
 * @param button der zu ndernde Button
 */
	private void toggleButtonState(Button button) {
		if (isButtonSet(button)) {
			button.getBackground().clearColorFilter();
			button.setTag(null);
		} else {
			// Button-Hintergrund-Farbe wird ber einen Filter gendert 
			button.getBackground().setColorFilter(Color.GREEN,
					PorterDuff.Mode.MULTIPLY);
			button.setTag("x"); // verwende die frei tag-Property zum Speichern des Zustands
		}
	}
	
/**
 * Prfe auf vollstndige ausgewhlte Button-Reihe 
 * vertikal, horizontal und die beiden Diagonalen 
 */
	private void checkForBingo() {
		boolean bingo = false;
		int x, y;
		// vertikal
		for (x = 0; x < SIZE && !bingo; x++) {
			for (y = 0; y < SIZE; y++) {
				Button button = (Button) findViewById(getButtonID(x, y));
				if (!isButtonSet(button))
					break;
			}
			bingo = (y == SIZE);
		}
		// horizontal
		for (y = 0; y < SIZE && !bingo; y++) {
			for (x = 0; x < SIZE; x++) {
				Button button = (Button) findViewById(getButtonID(x, y));
				if (!isButtonSet(button))
					break;
			}
			bingo = (x == SIZE);
		}
		// diagonal
		for (x = 0; x < SIZE && !bingo; x++) {
			Button button = (Button) findViewById(getButtonID(x, x));
			if (!isButtonSet(button))
				break;
		}
		if (x == SIZE)
			bingo = true;
		for (x = 0; x < SIZE && !bingo; x++) {
			Button button = (Button) findViewById(getButtonID(x, SIZE - 1 - x));
			if (!isButtonSet(button))
				break;
		}
		if (x == SIZE)
			bingo = true;

		if (bingo) {
			// Gewonnen, zeige ein einfaches Popup und spiele Sound
			if(!Prefs.getNoSound(this))
				playFanfare(); // asynchron
			// Die Methoden geben jeweils wieder das Builder-Objekt zurck
			// und lassen sich so leicht verketten
			Builder dlg=new AlertDialog.Builder(this)
					.setTitle(R.string.BBingoWonTitle)
					.setMessage(R.string.BBingoWon)
					.setPositiveButton(android.R.string.ok, new DialogInterface.OnClickListener() {						
						@Override
						public void onClick(DialogInterface dialog, int which) {
							soundPool.stop(streamID); // stoppe Sound, falls er noch gespielt wird
						}
					})
					.setNeutralButton(R.string.New, new DialogInterface.OnClickListener() {						
						@Override
						public void onClick(DialogInterface dialog, int which) {
							soundPool.stop(streamID); // stoppe Sound
							shuffleWords(true);	
						}
					});
			dlg.show();
		}
	}
/**
 * Worte der Wortliste mischen und auf die Buttons verteilen
 */
	private void shuffleWords(boolean animate)
	{
		List<String> words=BBingoDB.stringToList(currentWordlist);
		// Liste darf nicht leer sein
		if(words.size()==0)	
			words.add(""); 
		// Liste mischen und den Buttons zuweisen
		Collections.shuffle(words);
		int i=0;
		for (int x = 0; x < SIZE; x++) 
			for (int y = 0; y < SIZE; y++)
			{
				Button button=(Button) findViewById(getButtonID(x, y));
				if(isButtonSet(button))
					toggleButtonState(button); // setze Button state zurck
				if(i>=words.size())
					{
					// overflow
					Collections.shuffle(words);
					i=0;
					}
				button.setText(words.get(i++)); 
			}		
		if(animate) {
			ViewGroup mainViewGroup = (ViewGroup) findViewById(R.id.LinearLayoutMain);
			for (int j = 0; j < mainViewGroup.getChildCount(); j++) {
				ViewGroup rowViewGroup = (ViewGroup) mainViewGroup.getChildAt(j);
				rowViewGroup.startLayoutAnimation();
			}	
		}
			
	}
}