package com.linkesoft.sensors;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.text.DecimalFormat;
import java.util.LinkedList;

import android.content.Context;
import android.content.res.Resources;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.RectF;
import android.graphics.Paint.Align;
import android.graphics.Paint.Style;
import android.hardware.SensorEvent;
import android.os.Environment;
import android.util.AttributeSet;
import android.util.Log;
import android.util.TypedValue;
import android.view.View;

public class SensorView extends View {

	private SensorEvent event;
	private float max;
	private float min;
	
	private final float historytimespan=10; // in seconds
	// History 
	// ab Android 2.3 als Deque (double-ended queue) fr einfache Lngenbegrenzung basierend auf Zeitspanne
	//private final Deque<HistorySensorEvent> history=new ArrayDeque<HistorySensorEvent>();
	private final LinkedList<HistorySensorEvent> history=new LinkedList<HistorySensorEvent>();
	
	private class HistorySensorEvent {
		public float values[];
		public long timestamp;

		public HistorySensorEvent(SensorEvent sensorEvent) {
			values = sensorEvent.values.clone();
			timestamp = sensorEvent.timestamp;
		}
	}
	
	public SensorView(Context context, AttributeSet attrs) {
		super(context, attrs);
		reset();
	}
	
	@Override
	protected void onDraw(Canvas canvas) {
		
		if(event==null)
			return; // kein Event, nichts zu zeichnen
		// berechne Betrag des Vektors, sqrt(x*x+y*y+z*z)
		// und aktualisiere min und max
		float magnitude=0; 
		for(int i=0;i<event.values.length;i++)
		{
			magnitude+=event.values[i]*event.values[i];
			if(max<event.values[i])
				max=event.values[i];
			if(event.values[i]<0 && event.values[i]<min)
				min=event.values[i];
		}
		magnitude=(float)Math.sqrt(magnitude);
		if(magnitude>max)
			max=magnitude;
		float w=getWidth();
		float h=getHeight();
		float hBar=h/(event.values.length+2);
		Paint pBackground=new Paint();
		pBackground.setColor(Color.BLACK);
		canvas.drawPaint(pBackground);

		if(max==0)
			return; // nichts zu zeichnen 

		float y=0;
		float b=dip2px(5);
		// zeichne Magnitude
		Paint pBar=new Paint();
		pBar.setStyle(Style.FILL);
		pBar.setColor(Color.GREEN);
		canvas.drawRect(0, y+b, magnitude/(max-min)*w, y+2*hBar-b, pBar);
		drawValue(canvas,y+2*hBar/2,magnitude,event.sensor.getResolution());
		drawHistory(canvas,new RectF(0f, y+b, w, y+2*hBar-b),-1);
		y+=2*hBar;
		pBar.setColor(Color.MAGENTA);
		// zeichne Vektorwerte
		for(int i=0;i<event.values.length;i++)
		{
			float zerox=-min/(max-min)*w;
			if(event.values[i]>0) {
				
				canvas.drawRect(zerox, y+b, zerox+event.values[i]/(max-min)*w, y+hBar-b, pBar);
			}
			else {
				canvas.drawRect(zerox+event.values[i]/(max-min)*w, y+b,zerox, y+hBar-b, pBar);

			}
			drawValue(canvas,y+hBar/2,event.values[i],0);
			drawHistory(canvas, new RectF(0,y+b,w,y+hBar-b), i);
			y+=hBar;
		}
	}

	private void drawValue(Canvas canvas,float y, float value,float accuracy) {
		DecimalFormat f = new DecimalFormat();
		f.setMaximumFractionDigits(2);
		String strvalue = f.format(value);
		if(accuracy!=0)
			strvalue+=" (\u00b1 "+f.format(accuracy)+")"; // +/-
		Paint paint = new Paint();
		paint.setAntiAlias(true);
		paint.setTextAlign(Align.RIGHT);
		paint.setTextSize(dip2px(15));
		paint.setColor(Color.LTGRAY);
		canvas.drawText(strvalue, getWidth(), y-paint.getFontSpacing()/2, paint);
	}
	
	
	private void drawHistory(Canvas canvas,RectF rectF, int i) {
		Paint paint = new Paint();
		paint.setAntiAlias(true);
		paint.setColor(Color.LTGRAY);
		float lastX=rectF.left;
		float lastY=rectF.top;
		synchronized (history) {
			for(HistorySensorEvent event:history)
			{
				float value;
				if(i<0)
					value=(float) Math.sqrt(event.values[0]*event.values[0]+event.values[1]*event.values[1]+event.values[2]*event.values[2]);
				else
					value=event.values[i];
				float x=(float) ((event.timestamp-history.getFirst().timestamp)/1e9/historytimespan)*rectF.width();
				float y;
				if(i<0) // magnitude hat min=0
					y=rectF.bottom-(value)/(max)*rectF.height();
				else
					y=rectF.bottom-(value-min)/(max-min)*rectF.height();

				canvas.drawLine(lastX, lastY, x, y, paint);
				lastX=x;
				lastY=y;
			}
			
		}
	}

	public void setSensorEvent(SensorEvent event) {
		this.event=event;
		// log data to file
		logToFile(event);

		
		// speichere in History, SensorEvent-Objekte werden wiederverwendet, 
		// daher mssen die Daten kopiert werden
		synchronized (history) {
			history.addLast(new HistorySensorEvent(event));
			while(history.getLast().timestamp-history.getFirst().timestamp>historytimespan*1000000000L)
				history.removeFirst();
			
		}
		invalidate();
	}

	private void logToFile(SensorEvent event) {
		if(!Environment.getExternalStorageState().equals(Environment.MEDIA_MOUNTED))
			return; // no SD card
    	File path=new File(Environment.getExternalStorageDirectory(),event.sensor.getName()+".csv"); //$NON-NLS-1$

		FileWriter writer=null;
		try {
			writer = new FileWriter(path, true); // append to existing file
			DecimalFormat formatter=new DecimalFormat();
			formatter.setMaximumFractionDigits(6);
			StringBuffer lineBuffer=new StringBuffer();
			lineBuffer.append(formatter.format(event.timestamp/1e9d));
			lineBuffer.append(';');
			for(int i=0;i<event.values.length;i++)
			{
				lineBuffer.append(formatter.format(event.values[0]));
				lineBuffer.append(';');
			}
			lineBuffer.append('\n');
		    writer.write(lineBuffer.toString());
		} catch (IOException e) {
			Log.e("Sensors","Cannot write data",e);
		} finally {
			if(writer!=null)
				try {
					writer.close();
				} catch (IOException e) {
					Log.e("Sensors","Cannot write data",e);
				}
		}
	}
	
	public void reset()
	{
		max=0;
		min=0;
	}
	
	public float dip2px(float dip)
	{
		Resources resources = getResources();
		return TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, dip, resources.getDisplayMetrics());
	}
}
