﻿using System;
using System.Threading;
using Microsoft.SPOT;
using Microsoft.SPOT.Presentation;
using Microsoft.SPOT.Presentation.Controls;
using Microsoft.SPOT.Presentation.Media;
using Microsoft.SPOT.Touch;

using GT = Gadgeteer;
using GTM = Gadgeteer.Modules;
using Gadgeteer.Modules.GHIElectronics;

namespace PuzzleGame
{
  public partial class Program
  {
    const int ROWS = 4;
    const int COLUMNS = 4;
    const int TILES = ROWS * COLUMNS;
    enum Direction
    {
      Centered = 0,
      Left = 1,
      Right = 2,
      Up = 3,
      Down = 4
    }

    GT.Picture picture;
    Bitmap[] parts = new Bitmap[TILES];
    int[] partOrder = new int[TILES]; // indexes into parts
    int emptyPart; // index of empty part within partOrder
    int tileWidth;
    int tileHeight;

    Thread inputThread;

    int shuffleState = 0;

    void ProgramStarted()
    {
      display.SimpleGraphics.DisplayRectangle(GT.Color.White, 0, GT.Color.White, 0, 0, display.Width, display.Height);
      display.SimpleGraphics.DisplayText("Willkommen beim c't Puzzle", Resources.GetFont(Resources.FontResources.NinaB), GT.Color.Blue, 68, 80);
      display.SimpleGraphics.DisplayText("Zum Beginnen bitte den Joystick druecken", Resources.GetFont(Resources.FontResources.small), GT.Color.Black, 57, 140);

      camera.PictureCaptured += new Camera.PictureCapturedEventHandler(camera_PictureCaptured);
      joystick.JoystickPressed += new Joystick.JoystickEventHandler(joystick_JoystickPressed);
      
      // Initialize touch screen
      Touch.Initialize(new TouchEventListener());
      TouchCollectorConfiguration.CollectionMode = CollectionMode.GestureOnly;
      TouchCollectorConfiguration.CollectionMethod = CollectionMethod.Native;

      inputThread = new Thread(InputProc);
      inputThread.Start();

      // Do one-time tasks here
      Debug.Print("Program Started");
    }

    void joystick_JoystickPressed(Joystick sender, Joystick.JoystickState state)
    {
      inputThread.Suspend();
      if(shuffleState != 0) {
        shuffleState++;
        // Wait for shuffle Thread to suspend itself.
        while(shuffleState != 0)
          Thread.Sleep(5);
      }
      camera.TakePicture();
    }

    void camera_PictureCaptured(Camera sender, GT.Picture picture)
    {
      this.picture = picture;
      ShowPicture();
      Bitmap bmp = picture.MakeBitmap();
      tileWidth = bmp.Width / COLUMNS;
      tileHeight = bmp.Height / ROWS;
      for(int row = 0; row < ROWS; ++row) {
        for(int column = 0; column < COLUMNS; ++column) {
          Bitmap tile = new Bitmap(tileWidth, tileHeight);
          tile.DrawImage(0, 0, bmp, column * tileWidth, row * tileHeight, tileWidth, tileHeight);
          tile.DrawRectangle(Color.Black, 1, 0, 0, tileWidth, tileHeight, 0, 0, 0, 0, 0, 0, 0, 0, 0);
          tile.Flush();
          int idx = row * COLUMNS + column;
          parts[idx] = tile;
          partOrder[idx] = idx;
        }
      }
      emptyPart = TILES - 1;

      Thread shuffleThread = new Thread(Shuffle);
      shuffleState = 1;
      shuffleThread.Start();
    }

    void Shuffle()
    {
      int shuffleCount = 0;
      Random rnd = new Random();
      Direction dir, lastDir = Direction.Centered;
      
      // Keep picture shown for 2 seconds
      Thread.Sleep(2000);
      
      if(shuffleState != 1) { // new capture requested
        shuffleState = 0;
        return;
      }
      shuffleState = 2;
      ShowTiles();
      while(shuffleState == 2 && shuffleCount < 30) {
        dir = (Direction)(rnd.Next(4) + 1);
        if((dir == Direction.Left && lastDir == Direction.Right)
          || (dir == Direction.Right && lastDir == Direction.Left)
          || (dir == Direction.Up && lastDir == Direction.Down)
          || (dir == Direction.Down && lastDir == Direction.Up))
          continue;
        if(!MoveTile(dir))
          continue;
        lastDir = dir;
        ++shuffleCount;
        Thread.Sleep(150);
      }
      if(shuffleState == 2) {
        if((inputThread.ThreadState & ThreadState.Suspended) != 0)
          inputThread.Resume();
      }
      shuffleState = 0;
    }

    void InputProc()
    {
      bool stickCentered = true;
      bool touched = false;

      inputThread.Suspend();

      while(true) {
        // read joystick position
        var pos = joystick.GetJoystickPostion();
        Direction dir = Direction.Centered;
        if(pos.X < 0.3)
          dir = Direction.Left;
        else if(pos.X > 0.7)
          dir = Direction.Right;
        else if(pos.Y > 0.7)
          dir = Direction.Up;
        else if(pos.Y < 0.3)
          dir = Direction.Down;

        if(dir == Direction.Centered) {
          stickCentered = true;
        }
        else {
          if(stickCentered) {
            MoveTile(dir);
            stickCentered = false;
            if(CheckSolution()) {
              ShowPicture();
              string message = "Geschafft!";
              display.SimpleGraphics.DisplayRectangle(GT.Color.Gray, 2, GT.Color.LightGray, 118, 105, 84, 30);
              display.SimpleGraphics.DisplayText(message, Resources.GetFont(Resources.FontResources.NinaB), GT.Color.Green, 126, 113);
              inputThread.Suspend();
            }
          }
        }

        // read touch screen
        int touchX = 0;
        int touchY = 0;
        TouchCollectorConfiguration.GetLastTouchPoint(ref touchX, ref touchY);
        if((touchX > 0 && touchX != 1022) || (touchY > 0 && touchY != 1022)) { // Pen down?
          if(!touched) {
            ShowPicture();
            touched = true;
          }
        }
        else {
          if(touched) {
            ShowTiles();
            touched = false;
          }
        }
        Thread.Sleep(50);
      }
    }

    void ShowTiles()
    {
      for(int row = 0; row < ROWS; ++row) {
        for(int column = 0; column < COLUMNS; ++column) {
          ShowTile(row, column);
        }
      }
    }

    void ShowTile(int row, int column)
    {
      int idx = row * COLUMNS + column;
      if(idx != emptyPart)
        display.SimpleGraphics.DisplayImage(parts[partOrder[idx]], (uint)(column * tileWidth), (uint)(row * tileHeight));
      else
        display.SimpleGraphics.DisplayRectangle(GT.Color.Black, 0, GT.Color.Black,
          (uint)(column * tileWidth), (uint)(row * tileHeight), (uint)tileWidth, (uint)tileHeight);
    }

    void ShowPicture()
    {
      display.SimpleGraphics.DisplayImage(picture, 0, 0);
    }

    // returns success: false if Move is impossible because emtpy part is at edge; true otherwise
    private bool MoveTile(Direction direction)
    {
      int emptyRow = emptyPart / COLUMNS;
      int emptyColumn = emptyPart % COLUMNS;
      int targetPart = -1;
      int targetRow = -1;
      int targetColumn = -1;
      switch(direction) {
        case Direction.Left:
          if(emptyColumn < COLUMNS - 1) {
            targetRow = emptyRow;
            targetColumn = emptyColumn + 1;
          }
          break;
        case Direction.Right:
          if(emptyColumn > 0) {
            targetRow = emptyRow;
            targetColumn = emptyColumn - 1;
          }
          break;
        case Direction.Up:
          if(emptyRow < ROWS - 1) {
            targetRow = emptyRow + 1;
            targetColumn = emptyColumn;
          }
          break;
        case Direction.Down:
          if(emptyRow > 0) {
            targetRow = emptyRow - 1;
            targetColumn = emptyColumn;
          }
          break;
      }
      if(targetRow != -1 && targetColumn != -1) {
        targetPart = targetRow * COLUMNS + targetColumn;
        partOrder[emptyPart] = partOrder[targetPart];
        partOrder[targetPart] = TILES - 1;
        emptyPart = targetPart;
        ShowTile(targetRow, targetColumn);
        ShowTile(emptyRow, emptyColumn);
        return true;
      }
      return false;
    }

    bool CheckSolution()
    {
      for(int i = 0; i < TILES; ++i) {
        if(partOrder[i] != i)
          return false;
      }
      return true;
    }
  }

  // Just a minimal implementation of IEventListener to satisfy API
  class TouchEventListener : IEventListener
  {
    public void InitializeForEventSource()
    {
    }

    public bool OnEvent(BaseEvent ev)
    {
      return true;
    }
  }
}
