/*
 *  GestureLibrary.cpp
 *  Gesture Library
 *
 *  Created by Andy Abgottspon on 07.01.10.
 *  Copyright 2010 aaweb. All rights reserved.
 *
 */

#include "Hand.h"
#include "GestureLibrary.h"

//--------------------------------------------------------------
void GestureLibrary::setup(int camWidth, int camHeight, int thres){
    // Cam size
	imgSize.x = camWidth;
	imgSize.y = camHeight;

    // Color threshold
	threshold = thres;   
	
    //vidGrabber.setVerbose(true);
    vidGrabber.initGrabber(imgSize.x,imgSize.y, AVCaptureDevicePositionFront); // get front camera
    //vidGrabber.setCaptureRate(30.0f);

    colorImg.allocate(imgSize.x,imgSize.y);
	grayImage.allocate(imgSize.x,imgSize.y);
	grayBg.allocate(imgSize.x,imgSize.y);
	grayDiff.allocate(imgSize.x,imgSize.y);
	grayMask.allocate(imgSize.x,imgSize.y);

	blPaused = false;
	contourMinSize = GESTLIB_CONTOUR_MIN_SIZE;
	
	blGroupTips = true; 
    blAdaptColor = false;
	
    // Create 2 hands 
    // Hand h1, h2;
    // hands.push_back(h1);
    // hands.push_back(h2);
}

//--------------------------------------------------------------
// Elements from: http://www.openframeworks.cc/forum/viewtopic.php?t=850
void GestureLibrary::update(){
	ofBackground(100,100,100);

    //bool bNewFrame = false;

    //vidGrabber.grabFrame();
    //bNewFrame = vidGrabber.isFrameNew();


    colorImg.setFromPixels(vidGrabber.getPixels(), imgSize.x,imgSize.y);
    //colorImg.mirror(false, true);
    colorImg.mirror(true, false);
    
    grayImage = colorImg;
    // if (bLearnBakground == true){
    //  grayBg = grayImage;     // the = sign copys the pixels from grayImage into grayBg (operator overloading)
    //  bLearnBakground = false;
    // }
    
    // take the abs value of the difference between background and incoming and then threshold:
    grayDiff.absDiff(grayBg, grayImage);
    grayDiff.threshold(threshold);
    
    
    // Mask depending on the pixels color
    int nbOfPixels = colorImg.width*colorImg.height*3;
    unsigned char* pix = colorImg.getPixels();
    unsigned char tmpMask[nbOfPixels];
    int diff, col;
    
    ofColor avgColor;
    int whitePixels = 0;
    for(size_t i = 0; i < nbOfPixels; i = i+3)
    {      
        diff = ABS(pix[i]   - handColor.r) + 
        ABS(pix[i+1] - handColor.g) + 
        ABS(pix[i+2] - handColor.b) ;
        
        if(diff < threshold) {
            // Color them white
            col = (int)255;
            
            // Also, take this pixels colour to calculate average colour
            avgColor.r += handColor.r;
            avgColor.g += handColor.g;
            avgColor.b += handColor.b;
            whitePixels++;
            
        } else {
            // Color them black
            col = 0;
        }
        tmpMask[i/3] = col;
    }     
    grayMask.setFromPixels(tmpMask, colorImg.width, colorImg.height);
    
    
    // Calculate average and assign it as new reference color
    if(blAdaptColor)
    {
        avgColor.r = avgColor.r / whitePixels;
        avgColor.g = avgColor.g / whitePixels;
        avgColor.b = avgColor.b / whitePixels;
        
        setHandColor(avgColor);
    }
    
    // find contours which are between the size of 20 pixels and 1/3 the w*h pixels.
    // also, find holes is set to true so we will get interior contours as well....
    // contourFinder.findContours(grayDiff, contourMinSize, (340*240)/3, 10, false);    // find holes
    contourFinder.findContours(grayMask, contourMinSize, (imgSize.x*imgSize.y)/2, 10, false);    // find holes

    // printf("%f \n", ofGetFrameRate());
    

    // Reset debug string
    sprintf(debugStr, "");

	// we could draw the whole contour finder
	//contourFinder.draw(360,540);

    // Delete hands if there are no contours
    if(contourFinder.nBlobs == 0) {
        hands.clear();
    }

    // Reset contour indexes for all hands
    for (int i = 0; i < hands.size(); i++){       
		hands[i].contourInd = -1;
    }
        
    // PHASE 1: 
    // Go look for the closest contour for each hand and assign contour blob and its index to hand
    // So for each hand ...
	for (int i = 0; i < hands.size(); i++){   
	    // Initialise the minimum distance (distance to the closest contour) with the distance
	    // to the first contour there is
        ofPoint avg = (contourFinder.nBlobs >= 1) ? contourFinder.blobs[0].centroid: ofPoint(imgSize.x/2,imgSize.y/2);
        float c_min = calcDistance(avg, hands[i].position);
        int j_min = -1;
	    
    	// Loop through contours to determine the closest 
        for (int j = 0; j < contourFinder.nBlobs; j++){   
            // First check if the contour has been assigned to a hand already
            bool usedForHand = false;
            for (int k = 0; k < hands.size(); k++){   
        	    if(hands[k].contourInd == j) { usedForHand = true; }
        	}                                                      
        	    
        	// Only continue if contour is still available for assignment
        	if(!usedForHand) {
        		// Calculate distance from blob to hand position
        		float c = calcDistance(contourFinder.blobs[j].centroid, hands[i].position);

        		// Determine c_min and set hand position
        		// Only do stuff if not paused
        		if((j == 0 || c < c_min) && !blPaused) {
        			c_min = c;
                    j_min = j;
        		}  
        	}
    	}           
    	
	    if(hands.size() > 0 && j_min != -1) {
    		hands[i].contourInd = j_min;
            hands[i].setBlob(&contourFinder.blobs[j_min]);
    	}
	}
    
    // PHASE 2:
    // Delete hands that did not get a contour in this step
    for (int i = 0; i < hands.size(); i++){   
        // Erase hands that didn't get a contour
        if(hands[i].contourInd == -1) {
            hands.erase(hands.begin()+i);
            // printf("\nErase old hand %d", i);
        }               
    }	
    
    // PHASE 3:
	// Create new hands for new contours (BE CRITICAL, only take big enough contours!!!)
	// Go trough contours ...
    for (int j = 0; j < contourFinder.nBlobs; j++){
        bool usedForHand = false;
        
    	// Check if they've been used for a hand yet
        for (int i = 0; i < hands.size(); i++){   
    	    if(hands[i].contourInd == j) { usedForHand = true; }
    	}                                                      
    	
	    // If not a hand yet, create new one   
	    // but only if it's quite big ..
    	if(!usedForHand && 
			contourFinder.blobs[j].boundingRect.width >= GESTLIB_MIN_NEW_HAND_SIZE) {
            Hand newHand;
            newHand.id = hands.size();
			newHand.contourInd = j;
            newHand.setBlob(&contourFinder.blobs[j]);
            hands.push_back(newHand);    
            
            // printf("\nCreated hand, id: %d, contour: %d", newHand.id, newHand.contourInd);
    	}
    }
	
    // PHASE 4:
	// Get gestures for hands
    for (int i = 0; i < hands.size(); i++){   
        // Workaround: This should never even happen, that a hand with
        // contourInd -1 gets here... TODO!
        if(hands[i].contourInd != -1) {
            processBlob(hands[i].contourInd);
        }
	}    

    // Debug string
    sprintf(debugStr, "%sHands: %lu", debugStr, hands.size());    
}

bool GestureLibrary::processGestures() 
{
    doubleGesture = GEST_NONE;
    doubleGestureHand1 = -1;
    doubleGestureHand2 = -1;

    for (int h = 0; h < hands.size(); h++)
    {   
        hands[h].findGesture();     
        
        // Make sure hand has a gesture, but is done (not in progress)
        if(hands[h].gesture != GEST_NONE && hands[h].gestureType != GEST_TYPE_INPROGRESS) {
                                   
            // Double gestures
            // See if another hand has the same gesture
            for (int k = 0; k < hands.size(); k++)
            {                                          
                if(h != k && hands[k].gestureType != GEST_TYPE_INPROGRESS) { // Only compare with OTHER hands and those that are done with a gesture
                    int min = hands[k].vGesture.size() - GESTLIB_DOUBLE_STEPS;
                    min = (min < 0) ? 0 : min; // make sure index is not negative
                    for(int l = hands[k].vGesture.size()-1; l >= min; l--) {
                        if(hands[h].gesture == hands[k].vGesture[l]) {
                            // Second hand with same gesture, return and be happy, we found a double gesture.
                            doubleGesture = hands[h].gesture;
                            doubleGestureHand1 = hands[h].id;
                            doubleGestureHand2 = hands[k].id;

                            printf("\nGestLib: Double Gesture (Hands %d, %d, History %d): %s", 
                                       h, k, hands[k].vGesture.size()-1-l, hands[k].gestureStr);
                            return true;
                        }                
                    }
                }
            }     
            
            // Ok, only a single gesture then
            printf("\nGestLib: Single Gesture (%d): %s", h, hands[h].gestureStr);    
            return true;
        }
    }
    return false;
}       

// Based on OpenCV sample in oF
void GestureLibrary::processBlob(int blobInd)
{
	// Vars for convexhull & convexity
	CvSeq* defects;
	CvConvexityDefect* defectArray;

	CvMemStorage* storage = cvCreateMemStorage();
	CvMemStorage* stor03;
	stor03 = cvCreateMemStorage(0);

	// 
	// Find finger tips
	// 
	// 
	// Create convex hull 
	// 
	CvSeq* ptseq = cvCreateSeq( CV_SEQ_KIND_GENERIC|CV_32SC2,
								sizeof(CvContour),
								sizeof(CvPoint),
								storage );

	int i, hullcount, count = contourFinder.blobs[blobInd].pts.size();
	CvPoint pt0;
	CvSeq* hull;
	// 
	for(i = 0; i < count; i++ )
	{
		pt0.x = contourFinder.blobs[blobInd].pts[i].x; 
		pt0.y = contourFinder.blobs[blobInd].pts[i].y; 
		cvSeqPush( ptseq, &pt0 );
	}
	hull = cvConvexHull2( ptseq, 0, CV_CLOCKWISE, 0 );
	hullcount = hull->total;

	ofNoFill();

	for( i = 0; i < count; i++ )
	{
		pt0 = *CV_GET_SEQ_ELEM( CvPoint, ptseq, i );
		// ofCircle( winPosX + pt0.x, winPosY + pt0.y, 3);
	}

	pt0 = **CV_GET_SEQ_ELEM( CvPoint*, hull, hullcount - 1 );

	ofSetColor(0x0000FF);
	for( i = 0; i < hullcount; i++ )
	{
		CvPoint pt = **CV_GET_SEQ_ELEM( CvPoint*, hull, i );
		//ofLine(imgPos.x + pt0.x, imgPos.y + pt0.y, imgPos.x + pt.x, imgPos.y + pt.y);
		pt0 = pt;
	}


	// 
	// Create convexity (finger tips)
	// 
	defects = cvConvexityDefects(ptseq, hull, stor03);

	int m=0;
	// This cycle marks all defects of convexity of current contours.
	for(;defects;defects = defects->h_next)
	{
		int nomdef = defects->total; // defect amount
		/*outlet_float( m_nomdef, nomdef );

		if(nomdef == 0)
			continue;
		 */
		// Alloc memory for defect set.   
		//fprintf(stderr,"malloc\n");
		defectArray = (CvConvexityDefect*)malloc(sizeof(CvConvexityDefect)*nomdef);

		// Get defect set.
		//fprintf(stderr,"cvCvtSeqToArray\n");
		cvCvtSeqToArray(defects,defectArray, CV_WHOLE_SEQ);

		// Convert defect array to vector with points
		vector<ofPoint> defectVec;
		for(i=0; i<nomdef; i++)
		{      
			defectVec.push_back((ofPoint(defectArray[i].start->x, defectArray[i].start->y)));
		}

		// Group points within given distance for a certain amount of recursions
		if(blGroupTips) {
			defectVec = groupPoints(defectVec, 10, 2);
		}

		// marks for all defects (finger tips) 
		/*
		for(i=0; i < defectVec.size(); i++)
		{      
			ofFill();
			ofSetColor(0, 255, 0);
			              
			if(blDebug) {
				if(blGroupTips) {
					// Draw only stuff above center
					if(defectArray[i].start->y < contourFinder.blobs[blobInd].centroid.y + 30) { // 30 could be tolerance value
						ofCircle(defectVec[i].x, defectVec[i].y, 3);
					
						ofLine(defectVec[i].x, defectVec[i].y,
							   contourFinder.blobs[blobInd].centroid.x, contourFinder.blobs[blobInd].centroid.y);
					}
				}
				else {				
					ofCircle(defectVec[i].x, defectVec[i].y, 3);
				}    			   
			}

			// ofLine(winPosX + defectArray[i].start->x, winPosY + defectArray[i].start->y, 
			//        winPosX + defectArray[i].end->x, winPosY + defectArray[i].end->y);
			
			// cvLine(rgb, *(defectArray[i].start), *(defectArray[i].depth_point),CV_RGB(0,0,255),1, CV_AA, 0 );
			// cvCircle( rgb, *(defectArray[i].depth_point), 5, CV_RGB(0,255,0), -1, 8,0);
			// cvCircle( rgb, *(defectArray[i].start), 5, CV_RGB(0,255,0), -1, 8,0);
			// cvLine(rgb, *(defectArray[i].depth_point), *(defectArray[i].end),CV_RGB(0,0,255),1, CV_AA, 0 );
			// t_atom rlist[7];
			// SETFLOAT(&rlist[0], i);
			// SETFLOAT(&rlist[1], defectArray[i].start->x);
			// SETFLOAT(&rlist[2], defectArray[i].start->y);
			// SETFLOAT(&rlist[3], defectArray[i].depth_point->x);
			// SETFLOAT(&rlist[4], defectArray[i].depth_point->y);
			// SETFLOAT(&rlist[5], defectArray[i].end->x);
			// SETFLOAT(&rlist[6], defectArray[i].end->y);
			// outlet_list( m_dataout, 0, 7, rlist );
		}    
		*/

		m++;

		// Free memory.       
		free(defectArray);
	}

    // free(hull); // causes error: pointer being freed was not allocated        
        
	// 
	// Cleaning up
	// 
	cvReleaseMemStorage( &stor03 );
	cvClearMemStorage( storage );
}


//--------------------------------------------------------------
vector<ofPoint> GestureLibrary::groupPoints(vector<ofPoint> &defectVec, float dist, int recursiveCount) {
	if (recursiveCount == 0) {
		return defectVec;
	}
	
	ofPoint pThis, pNext, pDraw;
	vector<ofPoint> defectVecNew;

	for(int i=0; i < defectVec.size(); i++)
	{
		pThis.x = defectVec[i].x;
		pThis.y = defectVec[i].y;
		pNext.x = (i+1 < defectVec.size()) ? defectVec[i+1].x : pThis.x;
		pNext.y = (i+1 < defectVec.size()) ? defectVec[i+1].y : pThis.y;
		
		if(calcDistance(pThis, pNext) < dist) {
			pDraw.x = (pThis.x + pNext.x) / 2;
			pDraw.y = (pThis.y + pNext.y) / 2;
			defectVecNew.push_back(pDraw);
			i++; // skip next
		} else {
			defectVecNew.push_back(pThis);
		}
	}
			
	return groupPoints(defectVecNew, dist, recursiveCount-1);
}

//--------------------------------------------------------------
float GestureLibrary::calcDistance(ofPoint &pA, ofPoint &pB) {
	float a = ABS(pA.x-pB.x);
	float b = ABS(pA.y-pB.y);
	return sqrt(a*a + b*b);
}	

//--------------------------------------------------------------
ofColor GestureLibrary::getColor(int x, int y)
{
    unsigned char* pix = colorImg.getPixels();
    int index = (x + (y * getImageSize().x)) * 3;
    
    ofColor color;
    color.r = (int)pix[index];
    color.g = (int)pix[index+1];
    color.b = (int)pix[index+2];
    
    return color;
}

