#!/usr/bin/perl -w
# script to let connected clients change their internet connection mode

# IP address used  by the squid proxy server running on this host
# We need this IP to toggle the iptables rule for redirecting web
# traffic to the transparent proxy.
my $server = '192.168.13.3';

use CGI;
use strict;

my $q = CGI->new;
my $action = $q->param('action');
my $table = $q->param('table');
my $policy = $q->param('policy');
my $proxy = $q->param('proxy');
my $client = $q->remote_addr();

# extract last part of the client ip
$client =~ /(\d+)$/;
my $lastBlock = $1;

# get list of valid local routing tables
my @tables = getRoutingTables();
my $tablesRegexp = join '|', @tables;

# check input
if ( defined $table && ! ($table =~ /^($tablesRegexp)$/) ) {
	dieWithMsg('no valid routing table name');
}
if ( defined $policy && ! ($policy =~ /^0|1$/) ) {
	dieWithMsg('no valid value for policy');
}
my $newTable = $table;

# determine current routing table of the client
my ($prio,$curTable) = getCurrentTable($client);

# determine if policy routing is active 
# for the current client
my $curPolicy = '1';
if (defined $prio && $prio < 20000) {
	$curPolicy = '0';
}

# get iptable rule numbers for the transparent proxy
my ($proxyRule,$disableRule) = getProxyRuleNums($client);
my $curProxy = '1';
$curProxy = '0' if ($disableRule ne '');

# apply changes
my $res = undef;
if (defined $action && $action eq 'change') {
	# remove previous routing table
	if ($curTable ne 'default') {
		$res = `sudo /sbin/ip rule del from $client table $curTable`;
	}

	# priority of new rule enables or disables policy routing
	# done through marking packets with iptables
	# priority < 20000 => new rule will be matched before any rule 
	# that handles marked packets
	# priority > 25000 => new rule will be mathced after any rule
	# that handles marked packets
	my $priority = 15000 + $lastBlock;
	if ( defined $policy && $policy == 1 )  {
		$priority = 25000 + $lastBlock;
	}

	# add new routing table
	if ( (! $res) && $newTable ne 'default') {
		$res = `sudo /sbin/ip rule add from $client table $newTable priority $priority`;
	}

	# update rule to disable the transparent proxy redirect rule
	if ($curProxy eq '0' && $proxy eq '1') {
		# enable transparent proxy by removing the circumvention rule
		$res = `sudo /sbin/iptables -t nat -D PREROUTING $disableRule`;
	}
	if ($proxyRule ne '' && $curProxy eq '1' && $proxy eq '0') {
		# disable transparant proxy by adding a circumvention rule
		$res = `sudo /sbin/iptables -t nat -I PREROUTING $proxyRule ! -d $server/32 -s $client/32 -p tcp -m tcp --dport 80 -j ACCEPT`;
	}

	# update current table
	$curTable = getCurrentTable($client);
}

# start html output
print $q->header;
print $q->start_html('Routing Table Select');
print $q->p( 'Your IP: ' . $client );
# print error message from ip rule cmd if there has been one
if ($res) {
	print $q->p( 'Error: ' . $res);
}
print $q->start_form(-method=>"POST");
print $q->hidden(-name=>'action', -default=>['change']);
print $q->p('Routing Table:');
print $q->radio_group(   -name=>'table',
			 -values=>\@tables,
			 -default=> $curTable,
			 -linebreak=>'true'
		 );
print $q->p('Enable Policy Routing:');
print $q->radio_group(   -name=>'policy',
			 -values=>['1','0'],
			 -default=> $curPolicy,
			 -linebreak=>'true',
			 -labels=>{1 => 'yes', 0 => 'no'}
		 );
print $q->p('Enable Transperant Proxy:');
print $q->radio_group(   -name=>'proxy',
			 -values=>['1','0'],
			 -default=> $curProxy,
			 -linebreak=>'true',
			 -labels=>{1 => 'yes', 0 => 'no'}
		 );
print $q->br();		 
print $q->submit(-name=>'send', -value=>'set new table');
print $q->end_form();
print $q->end_html();

# return local routing tables
sub getRoutingTables {
	my $tablesFile = '/etc/iproute2/rt_tables';
	open TABLES,'<',$tablesFile or dieWithMsg('Cannot read '.$tablesFile.'.');
	my @tables = ('default');
	while(my $line = <TABLES>) {
		# test if the current line defines a routing table
		my $ok = $line =~ /^(\d+)\s*([\w\d]+)/;
		# If it actually does and the id of the routing table
		# is not 0, 253, 254 or 255 we add it to the list of 
		# routing tables.
		# The routing tables corresponding to the excluded ids 
		# are system routing tables and not of interest for us
		if ($ok && $1 > 0 && $1 < 253) {
			push @tables, $2;
		}
	}
	return @tables;
}

# return current routing table for this client
sub getCurrentTable {
	my $client = shift;
	# read all rules
	my $rules = `sudo /sbin/ip rule list`;
	# check if there is a rule for this client
	my $status = $rules =~ /(\d+):.*$client lookup ([A-Za-z0-9]+)/;
	if ($status) { 
		return ($1,$2);
	}
	else {
		return (undef,'default');
	}
}

# get number of transparent proxy redirect rule
# and if there is a rule disabling the redirect for
# the current client return that one too
sub getProxyRuleNums {
	my $client = shift;
	# read all rules
	my $rules = `sudo /sbin/iptables -t nat --line-numbers -nL PREROUTING`;
	# check if there is a rule for the transparent proxy
	my $status = $rules =~ /(\d+).* tcp dpt:80 redir ports 3129/;
	my $proxyRule = '';
	if ($status) { 
		$proxyRule = $1;
	}
	# check if there is a rule for this client that disables the
	# transparent proxy
	$status = $rules =~ /(\d+).*\s*ACCEPT\s*tcp\s*--\s*$client\s*!192\.168\.13\.3\s*tcp dpt:80/;
	my $disableRule = '';
	if ($status) { 
		$disableRule = $1;
	}
	return ($proxyRule,$disableRule);
}

# print error message and stop
sub dieWithMsg {
	my $msg = shift;
	print $q->header;
	print $q->start_html();
	print $q->p( $msg );
	print $q->end_html();
	exit 0;
}
