﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;

using System.Xml;
using System.IO;
using System.IO.Compression;          // fuer GZip
using System.Runtime.InteropServices; // fuer DLL-Import

using System.Security.Cryptography.X509Certificates;

using System.Text.RegularExpressions;

namespace Gesundheitskarte
{

    public partial class Gesundheitskarte : Form
    {

        private List<string> Terminals = new List<string>();    // Feld von Zeigern auf Namen/Staten der Kartenleser
        private IntPtr hContext;                                // SCard-Kontext

        IntPtr hCard;       // Kartenhandle
        eGK karte;          // eGK-Objekt
        EGKtyp kartentyp;   // z.Z gibt es G1 und G1plus
        string pfadName;    // Unterverzeichnis in "Eigene Dokumente"

        PinStatus pinstatusCH = PinStatus.NoError;
        PinStatus pinstatusHOME = PinStatus.NoError;
        PinStatus pinstatusQES = PinStatus.NoError;

        // Application-IDs der verwendeten Kartenanwendungen
        byte[] AID_HCA = new byte[] { 0xD2, 0x76, 0x00, 0x00, 0x01, 0x02 };
        byte[] AID_ESIGN = new byte[] { 0xA0, 0x00, 0x00, 0x01, 0x67, 0x45, 0x53, 0x49, 0x47, 0x4E };
        byte[] AID_CIA_ESIGN = new byte[] { 0xE8, 0x28, 0xBD, 0x08, 0x0F, 0xA0, 0, 0, 0x01, 0x67, 0x45, 0x53, 0x49, 0x47, 0x4E};
        byte[] AID_QES = new byte[] { 0xD2, 0x76, 0x00, 0x00, 0x66, 0x01 };

        /// <summary>
        ///Konstruktor der Form-Klasse 
        /// </summary>
        /// <param name="i"></param>
        /// <param name="j"></param>
        /// <returns></returns>
        public Gesundheitskarte()
        {
            InitializeComponent();            

            // Hintergundfarbe der TabControls an Form-Hintergrund anpassen
            tabsEinfaerben();

            // =============================
            // Chipkarten-Kontext herstellen
            // =============================
            uint lReturn = WinSCard.SCardEstablishContext((uint)SCardContextScope.User, IntPtr.Zero, IntPtr.Zero, out hContext);
            if (lReturn != 0)
            {
                MessageBox.Show("Fehler bei SCardEstablishContext!\n\nWahrscheinlich ist kein Chipkartenterminal mit PC/SC-Schnittstelle angeschlossen.");
                return;
            }

            // Liste der im System registrierten PC/SC-Terminals aufbauen
            listTerminals();

            // Auslesenknopf aktivieren, wenn mindestens ein Terminal gefunden wurde
            if (listBoxTerminals.SelectedIndex > -1)
                buttonAuslesen.Enabled = true;
        }


        /// <summary>
        /// Teilt einen Multistring in seine Teilstrings auf
        /// </summary>
        /// <param name="multiString">Der Multistring</param>
        /// <returns>Anzahl der Teilstrings, Teilstrings stehen in der Liste Terminals</returns>
        private int splitMultiString(string multiString)
        {
            int nTemp;
            int nCount = 0;
         
            // Fuer jeden Teilstring
            for (int nIter = 0; multiString[nIter] != '\0'; nIter = nTemp + 1)
            {
                nTemp = nIter;

                // Bis Stringende
                while (multiString[nTemp] != '\0')
                    nTemp++;

                // Teilstring zur Liste Terminals zufuegen
                Terminals.Add(multiString.Substring(nIter, nTemp - nIter));                
                nCount++;
            }

            // Anzahl der Teilstrings
            return nCount;
        }


        /// <summary>
        /// Erstellt eine Liste aller registrierter Kartenterminals 
        /// </summary>
        /// <returns></returns>
        void listTerminals()
        {
            uint dwLaenge = uint.MaxValue; // SCARD_AUTOALLOCATE;
            IntPtr leserNamen = IntPtr.Zero;
            uint lReturn;

            // ================================================================
            // Liste der Terminals als Multistring in lpmszLeserNamen ablegen
            // ================================================================
            lReturn = WinSCard.SCardListReaders(hContext, null, ref leserNamen, ref dwLaenge);
            if (lReturn != 0)
            {
                MessageBox.Show("Fehler bei SCardListeReaders");
                return;
            }

            // Multistring mit Terminalnamen erzeugen 
            string multistring = Marshal.PtrToStringAuto(leserNamen, (int)dwLaenge);
            splitMultiString(multistring);

            // Listbox fuellen
            listBoxTerminals.Items.Clear();
            foreach (string cName in Terminals)
            {            
                listBoxTerminals.Items.Add(cName);
            }
            listBoxTerminals.SetSelected(0,true);            
      }


        /// <summary>
        /// Allgemeine Versicherungsdaten aus dem EF .VD auslesen 
        /// </summary>
        /// <param name="i"></param>
        /// <param name="j"></param>
        /// <returns></returns>
        private void readVD()
        {
            byte[] empfang = new byte[65536];
                       
            
            // EF .VD auslesen 
            if (karte.readBinary( (byte)HCA.VD, empfang) == false)
                return;

            // in den ersten acht Bytes stehen Anfangs- und 
            // Endeadresse der EFs .VD und GVD
            int startVD = empfang[0] * 256 + empfang[1];
            int endeVD = empfang[2] * 256 + empfang[3];
            int laengeVD = endeVD - startVD;

            int startGVD = empfang[4] * 256 + empfang[5];
            int endeGVD = empfang[6] * 256 + empfang[7];
            int laengeGVD = endeGVD - startGVD;


            //
            // ===== VD extrahieren und auswerten ============================================
            //
            byte[] vd = new byte[laengeVD];
            Array.Copy(empfang, startVD, vd, 0, laengeVD);


            // =============================
            // gezippte XML-Datei auspacken
            // =============================
            System.IO.MemoryStream memStream = new System.IO.MemoryStream(vd);
            GZipStream gzStream = new GZipStream(memStream, CompressionMode.Decompress);
            StreamReader sr = new StreamReader(gzStream, Encoding.GetEncoding("iso-8859-15"));
            string entpackt = sr.ReadToEnd();

            // XML-Dokument erzeugen
            StringReader xmlStream = new StringReader(entpackt);
            XmlDocument xmlDoc = new XmlDocument();
            xmlDoc.Load(xmlStream);


            XmlNode knoten;

            // Namensraum fuer die allgemeinen Versicherungsdaten (in Version 5.1) aufbauen
            XmlNamespaceManager nsp = new XmlNamespaceManager(xmlDoc.NameTable);
            nsp.AddNamespace("ns", "http://ws.gematik.de/fa/vsds/UC_AllgemeineVersicherungsdatenXML/v5.1");

            // --------------------------------------------------

            string Baum = "/ns:UC_AllgemeineVersicherungsdatenXML/ns:Versicherter/ns:Versicherungsschutz/ns:Kostentraeger";

            textBoxKassenname.Text = "";
            knoten = xmlDoc.SelectSingleNode(Baum + "/ns:Name", nsp);
            if (knoten != null)
                textBoxKassenname.Text = knoten.InnerText;

            textBoxLaenderCodeKasse.Text = "";
            knoten = xmlDoc.SelectSingleNode(Baum + "/ns:Kostentraegerlaendercode", nsp);
            if (knoten != null)
                textBoxLaenderCodeKasse.Text = knoten.InnerText;

            textBoxKostentraegerKennung.Text = "";
            knoten = xmlDoc.SelectSingleNode(Baum + "/ns:Name", nsp);
            if (knoten != null)
                textBoxKostentraegerKennung.Text = knoten.InnerText;

            // --------------------------------------------------

            Baum = "/ns:UC_AllgemeineVersicherungsdatenXML/ns:Versicherter/ns:Versicherungsschutz/ns:Kostentraeger/ns:AbrechnenderKostentraeger";

            textBoxKassennameABKT.Text = "";
            knoten = xmlDoc.SelectSingleNode(Baum + "/ns:Name", nsp);
            if (knoten != null)
                textBoxKassennameABKT.Text = knoten.InnerText;

            textBoxKostentraegerKennungABKT.Text = "";
            knoten = xmlDoc.SelectSingleNode(Baum + "/ns:Kostentraegerkennung", nsp);
            if (knoten != null)
                textBoxKostentraegerKennungABKT.Text = knoten.InnerText;

            // --------------------------------------------------

            Baum = "/ns:UC_AllgemeineVersicherungsdatenXML/ns:Versicherter/ns:Versicherungsschutz";

            textBoxBeginn.Text = "";
            knoten = xmlDoc.SelectSingleNode(Baum + "/ns:Beginn", nsp);
            if (knoten != null)
                textBoxBeginn.Text = knoten.InnerText.Substring(6, 2) + '.'
                                    + knoten.InnerText.Substring(4, 2) + '.'
                                    + knoten.InnerText.Substring(0, 4);

            textBoxEnde.Text = "";
            knoten = xmlDoc.SelectSingleNode(Baum + "/ns:Ende", nsp);
            if (knoten != null)
                textBoxEnde.Text = knoten.InnerText.Substring(6, 2) + '.'
                                    + knoten.InnerText.Substring(4, 2) + '.'
                                    + knoten.InnerText.Substring(0, 4);

            // --------------------------------------------------

            Baum = "/ns:UC_AllgemeineVersicherungsdatenXML/ns:Versicherter/ns:Zusatzinfos/ns:ZusatzinfosGKV";

            textBoxRechtskreis.Text = "";
            knoten = xmlDoc.SelectSingleNode(Baum + "/ns:Rechtskreis", nsp);
            if (knoten != null)
                switch (knoten.InnerText)
                {
                    case "1": textBoxRechtskreis.Text = "1 => West";
                        break;
                    case "9": textBoxRechtskreis.Text = "2 => Ost";
                        break;
                    default: textBoxRechtskreis.Text = knoten.InnerText + " (nicht erkannt)";
                        break;
                }

            textBoxVersichertenart.Text = "";
            knoten = xmlDoc.SelectSingleNode(Baum + "/ns:Versichertenart", nsp);
            if (knoten != null)
                switch (knoten.InnerText)
                {
                    case "1": textBoxVersichertenart.Text = "1 => Mitglied";
                        break;
                    case "3": textBoxVersichertenart.Text = "3 => Familienangehöriger";
                        break;
                    case "5": textBoxVersichertenart.Text = "5 => Rentner oder ihre Angehörige";
                        break;
                    default: textBoxVersichertenart.Text = knoten.InnerText + " -- ???";
                        break;
                }

            textBoxRSA.Text = "";
            knoten = xmlDoc.SelectSingleNode(Baum + "/ns:Versichertenstatus_RSA", nsp);
            if (knoten != null)
                switch (knoten.InnerText)
                {
                    case "0": textBoxRSA.Text = "0 => keine Teilnahme";
                        break;
                    case "1": textBoxRSA.Text = "1 => ohne Erwerbsminderung";
                        break;
                    case "2": textBoxRSA.Text = "2 => mit Erwerbsminderung";
                        break;
                    default: textBoxRSA.Text = knoten.InnerText + " -- ???";
                        break;
                }

            // --------------------------------------------------

            Baum = "/ns:UC_AllgemeineVersicherungsdatenXML/ns:Versicherter/ns:Zusatzinfos/ns:ZusatzinfosGKV/ns:Zusatzinfos_Abrechnung_GKV";

            textBoxWOP.Text = "";
            knoten = xmlDoc.SelectSingleNode(Baum + "/ns:WOP", nsp);
            if (knoten != null)
                switch (knoten.InnerText)
                {
                    case "01": textBoxWOP.Text = "01 => Schleswig Holstein";
                        break;
                    case "02": textBoxWOP.Text = "02 => Hamburg";
                        break;
                    case "03": textBoxWOP.Text = "03 => Bremen";
                        break;
                    case "17": textBoxWOP.Text = "17 => Niedersachsen";
                        break;
                    case "20": textBoxWOP.Text = "20 => Westfalen-Lippe";
                        break;
                    case "38": textBoxWOP.Text = "38 => Nordrhein";
                        break;
                    case "46": textBoxWOP.Text = "46 => Hessen";
                        break;
                    case "51": textBoxWOP.Text = "51 => Rheinland-Pfalz";
                        break;
                    case "52": textBoxWOP.Text = "52 => Baden-Würthenberg";
                        break;
                    case "71": textBoxWOP.Text = "71 => Bayern";
                        break;
                    case "72": textBoxWOP.Text = "72 => Berlin";
                        break;
                    case "73": textBoxWOP.Text = "73 => Saarland";
                        break;
                    case "78": textBoxWOP.Text = "78 => Mecklenburg-Vorpommern";
                        break;
                    case "83": textBoxWOP.Text = "83 => Brandenburg";
                        break;
                    case "88": textBoxWOP.Text = "88 => Sachsen-Anhalt";
                        break;
                    case "93": textBoxWOP.Text = "93 => Thüringen";
                        break;
                    case "98": textBoxWOP.Text = "98 => Sachsen";
                        break;

                    default: textBoxWOP.Text = knoten.InnerText + " -- ???";
                        break;
                }


            textBoxAmbulant.Text = "";
            knoten = xmlDoc.SelectSingleNode(Baum + "/ns:Kostenerstattung_ambulant", nsp);
            if (knoten != null)
                switch (knoten.InnerText)
                {
                    case "0": textBoxAmbulant.Text = "0 => nein";
                        break;
                    case "1": textBoxAmbulant.Text = "1 => ja";
                        break;
                    default: textBoxAmbulant.Text = knoten.InnerText + " -- ???";
                        break;
                }


            textBoxStationaer.Text = "";
            knoten = xmlDoc.SelectSingleNode(Baum + "/ns:Kostenerstattung_stationaer", nsp);
            if (knoten != null)
                switch (knoten.InnerText)
                {
                    case "0": textBoxStationaer.Text = "0 => nein";
                        break;
                    case "1": textBoxStationaer.Text = "1 => ja";
                        break;
                    default: textBoxStationaer.Text = knoten.InnerText + " -- ???";
                        break;
                }


            // --------------------------------------------------


            // XML-Datei als Text ausgeben 
            // wenn noetig, Unterverzeichnis mit "Name_Vorname" im Dokumentenordner des Programms anlegen
            string unterverzeichnis = System.IO.Path.Combine(pfadName, textBoxNachname.Text + "_" + textBoxVorname.Text);
            if (!System.IO.File.Exists(unterverzeichnis))
                System.IO.Directory.CreateDirectory(unterverzeichnis);

            // vollstaendig adressierte Zieldatei = Pfad + Dateiname
            string vdName = System.IO.Path.Combine(unterverzeichnis, "VD.XML");

            StreamWriter datei = new StreamWriter( vdName);
            datei.Write(entpackt);
            datei.Close();

            //
            // ===== geschuetzte Versichertendaten (GVD) auswerten ============================================
            //

            textBoxZuzahlungsstatus.Text = "";
            textBoxGueltigBis.Text = "";
            textBoxBesonderePersonengruppe.Text = "";
            textBoxDMPkennzeichen.Text = "";


            // Wenn der Inhalt von .GVD nicht in .VD liegt, ist die berechnete GVD-Laenge = 0
            // Dann ist hier schon Schluss
            if (laengeGVD == 0)
                return;
            
            // GVD extrahieren 
            byte[] gvd = new byte[laengeGVD];
            Array.Copy(empfang, startGVD, gvd, 0, laengeGVD);
           

            // =============================
            // gezippte XML-Datei auspacken
            // =============================     
            memStream = new System.IO.MemoryStream(gvd);
            gzStream = new GZipStream(memStream, CompressionMode.Decompress);
            sr = new StreamReader(gzStream, Encoding.GetEncoding("iso-8859-15"));
            entpackt = sr.ReadToEnd();

            xmlStream = new StringReader(entpackt);
            xmlDoc = new XmlDocument();
            xmlDoc.Load(xmlStream);


            // Namensraum fuer die geschuetzten Versichertendaten (in Version 5.1) aufbauen
            nsp = new XmlNamespaceManager(xmlDoc.NameTable);
            nsp.AddNamespace("ns", "http://ws.gematik.de/fa/vsds/UC_GeschuetzteVersichertendatenXML/v5.1");


            // --------------------------------------------------

            Baum = "/ns:UC_GeschuetzteVersichertendatenXML";
            
            knoten = xmlDoc.SelectSingleNode(Baum + "/ns:Zuzahlungsstatus/ns:Status", nsp);
            if (knoten != null)
                   switch (knoten.InnerText)
                    {
                        case "0": textBoxZuzahlungsstatus.Text = "0 => keine Befreiung";
                            break;
                        case "1": textBoxZuzahlungsstatus.Text = "1 => Befreit";
                            break;
                        default: textBoxZuzahlungsstatus.Text = knoten.InnerText + " => unbekannter Status";
                            break;
                    }
            
            knoten = xmlDoc.SelectSingleNode(Baum + "/ns:Zuzahlungsstatus/ns:Gueltig_bis", nsp);
            if (knoten != null)
                textBoxGueltigBis.Text = knoten.InnerText.Substring(6, 2) + '.'
                                       + knoten.InnerText.Substring(4, 2) + '.'
                                       + knoten.InnerText.Substring(0, 4);
            
            knoten = xmlDoc.SelectSingleNode(Baum + "/ns:Besondere_Personengruppe", nsp);
            if (knoten != null)
                switch (knoten.InnerText)
                {
                    case "4": textBoxBesonderePersonengruppe.Text = "4 => Sozialhilheempfänger";
                        break;
                    case "6": textBoxBesonderePersonengruppe.Text = "6 => BVG";
                        break;
                    case "7": textBoxBesonderePersonengruppe.Text = "7 => Pers. mit zwischenstaatlichen Krankenversicherungsrecht";
                        break;
                    case "8": textBoxBesonderePersonengruppe.Text = "8 => SVA";
                        break;
                    default: textBoxBesonderePersonengruppe.Text = knoten.InnerText + " => unbekannte Personengruppe";
                        break;
                }
            
            knoten = xmlDoc.SelectSingleNode(Baum + "/ns:DMP_Kennzeichnung", nsp);
            if (knoten != null)
                switch (knoten.InnerText)
                {
                    case "1": textBoxDMPkennzeichen.Text = "1 => Diabetes Mellitus Typ 2";
                        break;
                    case "2": textBoxDMPkennzeichen.Text = "2 => Brustkrebs";
                        break;
                    case "3": textBoxDMPkennzeichen.Text = "3 => Koronare Herzkrankheit";
                        break;
                    case "4": textBoxDMPkennzeichen.Text = "4 => Diabetes Mellitus Typ 1";
                        break;
                    case "5": textBoxDMPkennzeichen.Text = "5 => Asthma bronchiale";
                        break;
                    case "6": textBoxDMPkennzeichen.Text = "6 => COPD (chronic obstructive pulmonary disease)";
                        break;
                    default: textBoxDMPkennzeichen.Text = knoten.InnerText + " => unbekannte Personengruppe";
                        break;
                }

            // --------------------------------------------------      

            //
            // XML-Datei als Text ausgeben 
            //

            // wenn noetig, Unterverzeichnis mit "Name_Vorname" im Dokumentenordner des Programms anlegen
            unterverzeichnis = System.IO.Path.Combine(pfadName, textBoxNachname.Text + "_" + textBoxVorname.Text);
            if (!System.IO.File.Exists(unterverzeichnis))
                System.IO.Directory.CreateDirectory(unterverzeichnis);

            // vollstaendig adressierte Zieldatei = Pfad + Dateiname
            string gvdName = System.IO.Path.Combine(unterverzeichnis, "GVD.XML");

            datei = new StreamWriter( gvdName);
            datei.Write(entpackt);
            datei.Close();           
        }



        /// <summary>
        /// Status der SmartCard auslesen (div. EFs im root und PIN-Status)
        /// </summary>
        /// <returns></returns>
        private void readCardStatus()
        {
            byte[] empfang = new byte[65536];
            byte[] element = null;
            TLV tlv = new TLV();

            //
            // ===== EF .GDO auslesen
            //
            if (karte.readBinary( (byte)MF.GDO, empfang) == false)
                return;

            // Attribut body (Tag = 0x5A) auslesen
            if (tlv.sucheTag(empfang, 0x5A, ref element))
            {
                // gefundene Bytefolge in einen Hex-String umwandlen
                string text = BitConverter.ToString(element).Replace("-", "");

                // Werte aus 'body' in den Textboxen ablegen
                textBoxMII.Text = text.Substring(0, 2);
                textBoxCountryCode.Text = text.Substring(2, 3);
                textBoxKartenHerausgeberID.Text = text.Substring(5, 5);
                textBoxSeriennummer.Text = text.Substring(10, 10);
            }

            //
            // ===== EF .ATR (SID = 0x1D) auslesen ======================================
            //
            if (karte.readBinary( (byte)MF.ATR, empfang))
            {
                int groesse, pos;
                byte[] element2, rest;

                element = null;
                tlv.sucheTag(empfang, 0xE0, ref element);

                // ersten Eintrag mit Tag 0x02 suchen 
                element2 = null;
                tlv.sucheTag(element, 0x02, ref element2);
                pos = element2.Length + 2;
                groesse = 0;
                for (int i = 0; i < element2.Length; i++)
                {
                    groesse *= 256;
                    groesse += element2[i];
                }
                textBoxUngesKommAPDU.Text = groesse + " Byte";

                // zweiten Eintrag mit Tag 0x02 suchen             
                rest = new byte[element.Length - pos];
                Array.Copy(element, pos, rest, 0, rest.Length);
                tlv.sucheTag(rest, 0x02, ref element2);
                pos += element2.Length + 2;
                groesse = 0;
                for (int i = 0; i < element2.Length; i++)
                {
                    groesse *= 256;
                    groesse += element2[i];
                }
                textBoxUngesAntwAPDU.Text = groesse + " Byte";

                // dritten Eintrag mit Tag 0x02 suchen             
                rest = new byte[element.Length - pos];
                Array.Copy(element, pos, rest, 0, rest.Length);
                tlv.sucheTag(rest, 0x02, ref element2);
                pos += element2.Length + 2;
                groesse = 0;
                for (int i = 0; i < element2.Length; i++)
                {
                    groesse *= 256;
                    groesse += element2[i];
                }
                textBoxGesKommAPDU.Text = groesse + " Byte";

                // vierten Eintrag mit Tag 0x02 suchen             
                rest = new byte[element.Length - pos];
                Array.Copy(element, pos, rest, 0, rest.Length);
                tlv.sucheTag(rest, 0x02, ref element2);
                pos += element2.Length + 2;
                groesse = 0;
                for (int i = 0; i < element2.Length; i++)
                {
                    groesse *= 256;
                    groesse += element2[i];
                }
                textBoxGesAntwAPDU.Text = groesse + " Byte";

                //
                // Tag "66" = Datenobjekt DO_CardData
                // 
                tlv.sucheTag(empfang, 0x66, ref element);

                // Datenobjekt DO_CardData (0x66) MUSS ein Datenobjekt DO_PreIssuringData 
                // mit dem Tag = '46'enthalten
                tlv.sucheTag(element, 0x46, ref element2);

                textBoxChiphersteller.Text = BitConverter.ToString(element2).Replace("-", "").Substring(0, 2); // Hexwert des ersten Oktets
                textBoxKartenhersteller.Text = Encoding.UTF8.GetString(element2, 1, 5); // naechsten fuenf Oktette als String
            }

            //
            // ===== EF .VERSION auslesen ===============================================
            //

            // ersten Record von .VERSION (SID = 0x10) auslesen
            if (karte.readRecord( (byte)MF.VERSION, 1, empfang))
            {
                string text = BitConverter.ToString(empfang).Replace("-", "");
                // die drei Versionsbestandteile isolieren und danach ausgeben
                textBoxVersionEGK1.Text = Int32.Parse(text.Substring(0, 3), System.Globalization.NumberStyles.HexNumber) + "."
                    + Int32.Parse(text.Substring(3, 3), System.Globalization.NumberStyles.HexNumber) + "."
                    + Int32.Parse(text.Substring(6, 4), System.Globalization.NumberStyles.HexNumber);
            }

            // zweiten Record von .VERSION (SID = 0x10) auslesen
            if (karte.readRecord( (byte)MF.VERSION, 2, empfang))
            {

                string text = BitConverter.ToString(empfang).Replace("-", "");
                // die drei Versionsbestandteile isolieren und danach ausgeben
                textBoxVersionEGK2.Text = Int32.Parse(text.Substring(0, 3), System.Globalization.NumberStyles.HexNumber) + "."
                    + Int32.Parse(text.Substring(3, 3), System.Globalization.NumberStyles.HexNumber) + "."
                    + Int32.Parse(text.Substring(6, 4), System.Globalization.NumberStyles.HexNumber);
            }

            // dritten Record von .VERSION (SID = 0x10) auslesen
            if (karte.readRecord((byte)MF.VERSION, 3, empfang))
            {
                string text = BitConverter.ToString(empfang).Replace("-", "");
                // die drei Versionsbestandteile isolieren und danach ausgeben
                textBoxVersionEGKSpeicher.Text = Int32.Parse(text.Substring(0, 3), System.Globalization.NumberStyles.HexNumber) + "."
                    + Int32.Parse(text.Substring(3, 3), System.Globalization.NumberStyles.HexNumber) + "."
                    + Int32.Parse(text.Substring(6, 4), System.Globalization.NumberStyles.HexNumber);
            }

            // Version der Karte (G1 oder G1plus) ausgeben - resultiert aus den drei Versionen in .Version
            labelKartenVersion.Visible = true;
            labelKartenVersion.Text = "Die eingelegte Gesundheitskarte ist vom Typ ";
            switch (textBoxVersionEGK1.Text + "_" + textBoxVersionEGK2.Text + "_" + textBoxVersionEGKSpeicher.Text)
            {
                case "3.0.0_3.0.0_3.0.2": labelKartenVersion.Text += "G1.";
                    kartentyp = EGKtyp.G1;
                    break;
                case "3.0.0_3.0.1_3.0.3": labelKartenVersion.Text += "G1 plus.";
                    kartentyp = EGKtyp.G1plus;
                    break;
                default: labelKartenVersion.Text += "unbekannt.";
                    kartentyp = EGKtyp.unbekannt;
                    break;
            }

            //
            // ===== EF .DIR auslesen ===================================================
            //

            // ersten Record von .DIR (SID = 0x1E) auslesen
            if (karte.readRecord((byte)MF.DIR, 1, empfang))
                textBoxDirRecord1.Text = BitConverter.ToString(empfang).Substring(12, 6 * 2 + 5);
            else
                textBoxDirRecord1.Text = "nicht vorhanden";

            // zweiten Record von .DIR auslesen
            if (karte.readRecord((byte)MF.DIR, 2, empfang))
                textBoxDirRecord2.Text = BitConverter.ToString(empfang).Substring(12, 6 * 2 + 5);
            else
                textBoxDirRecord2.Text = "nicht vorhanden";

            // dritten Record von .DIR auslesen
            if (karte.readRecord((byte)MF.DIR, 3, empfang))
                textBoxDirRecord3.Text = BitConverter.ToString(empfang).Substring(12, 10 * 2 + 9);
            else
                textBoxDirRecord3.Text = "nicht vorhanden";

            // vierten Record von .DIR auslesen
            if (karte.readRecord((byte)MF.DIR, 4, empfang))
                textBoxDirRecord4.Text = BitConverter.ToString(empfang).Substring(12, 15 * 2 + 14);
            else
                textBoxDirRecord4.Text = "nicht vorhanden";


            // Bei G1plus sind auf den Rekords 5 bis 7 noch drei Anwendungen dazugekommen
            // Die QES-Anwendung liegt dann im Rekord 8 und nicht im Rekord 5
            switch (kartentyp)
            {
                case EGKtyp.G1plus:

                    // fuenften Record von .DIR auslesen
                    if (karte.readRecord((byte)MF.DIR, 5, empfang))
                        textBoxDirRecord5.Text = BitConverter.ToString(empfang).Substring(12, 6 * 2 + 5);
                    else
                        textBoxDirRecord5.Text = "nicht vorhanden";

                    // sechsten Record von .DIR auslesen
                    if (karte.readRecord((byte)MF.DIR, 6, empfang))
                        textBoxDirRecord6.Text = BitConverter.ToString(empfang).Substring(12, 6 * 2 + 5);
                    else
                        textBoxDirRecord6.Text = "nicht vorhanden";

                    // sieebten Record von .DIR auslesen
                    if (karte.readRecord((byte)MF.DIR, 7, empfang))
                        textBoxDirRecord7.Text = BitConverter.ToString(empfang).Substring(12, 6 * 2 + 5);
                    else
                        textBoxDirRecord7.Text = "nicht vorhanden";

                    // fuenften Record von .DIR auslesen
                    if (karte.readRecord((byte)MF.DIR, 8, empfang))
                        textBoxDirRecord8.Text = BitConverter.ToString(empfang).Substring(12, 6 * 2 + 5);
                    else
                        textBoxDirRecord8.Text = "nicht vorhanden";

                    break;

                case EGKtyp.unbekannt:
                case EGKtyp.G1:

                    textBoxDirRecord5.Text = "nicht vorhanden";
                    textBoxDirRecord6.Text = "nicht vorhanden";
                    textBoxDirRecord7.Text = "nicht vorhanden";
                    // fuenften Record von .DIR auslesen
                    if (karte.readRecord(0x1E, 5, empfang))
                        textBoxDirRecord8.Text = BitConverter.ToString(empfang).Substring(12, 6 * 2 + 5);
                    else
                        textBoxDirRecord8.Text = "nicht vorhanden";
                    break;
            }

            //
            // ===== ".C.eGK.AUTH_CVC" ==================================================
            //

            // READ BINARY der Datei mit SID = 0x03
            if (karte.readBinary((byte)MF.AUTH_CVC, empfang))
            {
                // Datenobjekt mit Tag = 0x7F21 extrahieren
                if (tlv.sucheTagCV(empfang, 0x7F, 0x21, ref element))
                {
                    textBoxAUT_CVC.Text = BitConverter.ToString(element).Replace("-", " ");

                    // Zertifikat ablegen
                    string unterverzeichnis = System.IO.Path.Combine(pfadName, textBoxNachname.Text + "_" + textBoxVorname.Text);
                    string Parameter = System.IO.Path.Combine(unterverzeichnis, "AUTH_CVC.CV");
                    FileStream fs = new FileStream(Parameter, FileMode.OpenOrCreate, FileAccess.Write);
                    BinaryWriter datei = new BinaryWriter(fs);
                    datei.Write( element);
                    datei.Close();
                }
            }

            //
            // ===== ".C.CA_eGK.CS" =====================================================
            //

            // READ BINARY der Datei mit SID = 0x04
            if (karte.readBinary((byte)MF.CA_eGK, empfang))
            {
                // Datenobjekt mit Tag = 0x7F21 extrahieren
                if (tlv.sucheTagCV(empfang, 0x7F, 0x21, ref element))
                {
                    textBoxCA_eGK.Text = BitConverter.ToString( element).Replace("-", " ");

                    // Zertifikat ablegen
                    string unterverzeichnis = System.IO.Path.Combine(pfadName, textBoxNachname.Text + "_" + textBoxVorname.Text);
                    string Parameter = System.IO.Path.Combine(unterverzeichnis, "CA_eGK.CV");
                    FileStream fs = new FileStream(Parameter, FileMode.OpenOrCreate, FileAccess.Write);
                    BinaryWriter datei = new BinaryWriter(fs);
                    datei.Write( element);
                    datei.Close();
                }
            }

            //
            // ===== PIN-Status auslesen ================================================
            //

            int fehlbed = 0;
            string ausgabe = "";

            // PIN.CH
            //buttonPinChSetzen.Enabled = false;
            if (karte.getPinStatus(0x01, ref pinstatusCH, ref fehlbed, ref ausgabe))
            {
                textBoxPinStatusCH.Text = ausgabe;

             /*   // Button fuer das Aendern der Transport-PIN einblenden
                if (pinstatusCH == PinStatus.RetryCounter)
                    buttonPinChSetzen.Text = "PIN ändern";
                else
                    buttonPinChSetzen.Text = "Transport-PIN überschreiben";
                buttonPinChSetzen.Enabled = true;*/
            }

            // PIN.home
            buttonPinHomeSetzen.Enabled = false;
            if (karte.getPinStatus(0x02, ref pinstatusHOME, ref fehlbed, ref ausgabe))
            {
                textBoxPinStatusHome.Text = ausgabe;

                // Button fuer das Aendern der PIN einblenden
                switch (pinstatusHOME)
                {
                    case PinStatus.RetryCounter:
                        buttonPinHomeSetzen.Text = "PIN ändern";
                        buttonPinHomeSetzen.Enabled = true;
                        break;

                    case PinStatus.Transport_PIN_0000:
                    case PinStatus.Transport_PIN_Leerpin_1:
                    case PinStatus.Transport_PIN_Leerpin_2:
                        buttonPinHomeSetzen.Text = "Transport-PIN überschreiben";
                        buttonPinHomeSetzen.Enabled = true;
                        break;

                    default: break;
                }
            }

        }

 
        /// <summary>
        /// Diverse EFs im DF HCA auswerten 
        /// </summary>
        /// <returns></returns>
        private void readHcaStatus()
        {
            byte[] empfang = new byte[65536];

            //
            // EF .StatusVD auslesen
            //
            if (karte.readBinary( (byte)HCA.VDStatus, empfang))
            {

                string text = ASCIIEncoding.ASCII.GetString(empfang);
                string zeitstempel = "am " +
                                     text.Substring(7, 2) + '.' + // Tag
                                     text.Substring(5, 2) + '.' + // Monat
                                     text.Substring(1, 4); // Jahr

                zeitstempel += " um " + text.Substring(9, 2) + ':' + text.Substring(11, 2) + " Uhr";

                textBoxKartenaktualisierung.Text = zeitstempel;

                // Transaktionscode auswerten
                switch ((char)empfang[0])
                {
                    case '0': textBoxTransaktion.Text = "0 -> keine Transaktion offen"; break;
                    case '1': textBoxTransaktion.Text = "1 -> Transaktion offen"; break;
                    default: textBoxTransaktion.Text =  empfang[0] + " -> unbekannter Code"; break;
                }

                // Version extrahieren
                byte[] version = new byte[5];
                Array.Copy(empfang, 15, version, 0, 5);

                // In Hexstring umwandeln und die "-"-Zeichen aus der Zeichenkette loeschen
                text = BitConverter.ToString(version).Replace("-", "");                

                // die drei Versionsbestandteile isolieren und danach ausgeben
                textBoxVSDversion.Text = Int32.Parse(text.Substring(0, 3), System.Globalization.NumberStyles.HexNumber) + "."
                    + Int32.Parse(text.Substring(3, 3), System.Globalization.NumberStyles.HexNumber) + "."
                    + Int32.Parse(text.Substring(6, 4), System.Globalization.NumberStyles.HexNumber);
            }

        }

 

        /// <summary>
        /// ESIGN-Zertifikate auslesen  
        /// </summary>
        /// <returns></returns>
        private void readCert()
        {
            byte[] empfang = new byte[65536];

            //
            // ===== "PuK.CH.AUT" =======================================================
            //

            // READ BINARY der Datei mit SID = 0x01
            if (karte.readBinary( (byte)ESIGN.PuK_CH_AUT, empfang) == false)
                return;

            // Zertifikat erstellen und Binaerdaten darin ablegen
            X509Certificate zert = new X509Certificate();
            zert.Import(empfang);

            // Ausgabe der "wichtigsten" Daten
            textBoxAutSerienNr.Text = BitConverter.ToString(zert.GetSerialNumber()).Replace("-", " ");
            textBoxAutIssuerName.Text = zert.Issuer;
            textBoxSignaturAlg.Text = zert.GetKeyAlgorithm();
            textBoxAutCertRawData.Text = BitConverter.ToString(zert.GetPublicKey()).Replace("-", " ");
            textBoxAutGueltigAb.Text = zert.GetEffectiveDateString();
            textBoxAutGueltigBis.Text = zert.GetExpirationDateString();
            textBoxAutAntragsteller.Text = zert.Subject;
            textBoxAutHash.Text = BitConverter.ToString(zert.GetCertHash()).Replace("-", " ");
            textBoxAutFormat.Text = zert.GetFormat();

            //
            // XML-Datei als Text ausgeben 
            //

            // wenn noetig, Unterverzeichnis mit "Name_Vorname" anlegen
            string unterverzeichnis = System.IO.Path.Combine(pfadName, textBoxNachname.Text + "_" + textBoxVorname.Text);
            if (!System.IO.File.Exists(unterverzeichnis))
                System.IO.Directory.CreateDirectory(unterverzeichnis);

            // vollstaendig adressierte Zieldatei = Pfad + Dateiname
            string logName = System.IO.Path.Combine(unterverzeichnis, "AUT_CERT.CER");

            FileStream fs = new FileStream( logName, FileMode.OpenOrCreate, FileAccess.Write);
            BinaryWriter datei = new BinaryWriter(fs);
            byte[] certData = zert.Export(X509ContentType.Cert);
            datei.Write( certData);
            datei.Close();
     
            buttonAutZertifikat.Enabled = true;

            //
            // ===== "PuK.CH.ENC" =======================================================
            //

            // READ BINARY der Datei mit SID = 0x02
            if (karte.readBinary( (byte)ESIGN.PuK_CH_ENC, empfang) == false)
                return;

            // Zertifikat erstellen und Binaerdaten darin ablegen
            zert = new X509Certificate();
            zert.Import(empfang);

            // Ausgabe der "wichtigsten" Daten
            textBoxEncSerienNr.Text = BitConverter.ToString(zert.GetSerialNumber()).Replace("-", " ");
            textBoxEncIssuerName.Text = zert.Issuer;
            textBoxEncSignaturAlg.Text = zert.GetKeyAlgorithm();
            textBoxEncCertRawData.Text = BitConverter.ToString(zert.GetPublicKey()).Replace("-", " ");
            textBoxEncGueltigAb.Text = zert.GetEffectiveDateString();
            textBoxEncGueltigBis.Text = zert.GetExpirationDateString();
            textBoxEncAntragsteller.Text = zert.Subject;
            textBoxEncHash.Text = BitConverter.ToString(zert.GetCertHash()).Replace("-", " ");
            textBoxEncFormat.Text = zert.GetFormat();

            // vollstaendig adressierte Zieldatei = Pfad + Dateiname
            logName = System.IO.Path.Combine(unterverzeichnis, "ENC_CERT.CER");

            fs = new FileStream(logName, FileMode.OpenOrCreate, FileAccess.Write);
            datei = new BinaryWriter(fs);
            certData = zert.Export(X509ContentType.Cert);
            datei.Write(certData);
            datei.Close();

            buttonEncZertifikat.Enabled = true;    
        }



        /// <summary>
        /// QES-Zertifikat auslesen (wenn vorhanden) 
        /// </summary>
        /// <returns></returns>
        private void leseZertifikat_QES()
        {
            byte[] empfang = new byte[65536];

            //
            // ===== "PrK.CH.QES" =======================================================
            //

            // READ BINARY der Datei mit SID = 0x10
            if (karte.readBinary(0x10, empfang) == false)
                return;

            // Zertifikat erstellen und Binaerdaten darin ablegen
            X509Certificate zert = new X509Certificate();
            zert.Import(empfang);

            // Ausgabe der "wichtigsten" Daten
            textBoxQesSerienNr.Text = BitConverter.ToString(zert.GetSerialNumber()).Replace("-", " ");
            textBoxQesIssuerName.Text = zert.Issuer;
            textBoxQesSignaturAlg.Text = zert.GetKeyAlgorithm();
            textBoxQesCertRawData.Text = BitConverter.ToString(zert.GetPublicKey()).Replace("-", " ");
            textBoxQesGueltigAb.Text = zert.GetEffectiveDateString();
            textBoxQesGueltigBis.Text = zert.GetExpirationDateString();
            textBoxQesAntragsteller.Text = zert.Subject;
            textBoxQesHash.Text = BitConverter.ToString(zert.GetCertHash()).Replace("-", " ");
            textBoxQesFormat.Text = zert.GetFormat();

            // wenn noetig, Unterverzeichnis mit "Name_Vorname" anlegen
            string unterverzeichnis = System.IO.Path.Combine(pfadName, textBoxNachname.Text + "_" + textBoxVorname.Text);
            if (!System.IO.File.Exists(unterverzeichnis))
                System.IO.Directory.CreateDirectory(unterverzeichnis);

            // vollstaendig adressierte Zieldatei = Pfad + Dateiname
            string logName = System.IO.Path.Combine(unterverzeichnis, "QES_PrK_CERT.CER");

            // Zertifikat im Verzeichnis ablegen
            FileStream fs = new FileStream(logName, FileMode.OpenOrCreate, FileAccess.Write);
            BinaryWriter datei = new BinaryWriter(fs);
            byte[] certData = zert.Export(X509ContentType.Cert);
            datei.Write(certData);
            datei.Close();

            buttonAutZertifikat.Enabled = true;
        }



        /// <summary>
        /// EF CIA_Info auslesen
        /// </summary>
        /// <returns></returns>
        private void readCIAinfo()
        {
            byte[] empfang = new byte[25];
            byte[] element = null;

            // READ BINARY der Datei .CIA_Info (SID = 0x12)
            if (karte.readBinary(0x12, empfang) == false)
                return;


            // Attribut body (Tag = 0x5A) auslesen
            TLV tlv = new TLV();
            if (tlv.sucheTag(empfang, 0x30, ref element))
            {

                byte[] version = null;
                byte[] cardflags = null;
                byte[] profileIndication = null;

                tlv.sucheTag(element, 0x02, ref version);
                tlv.sucheTag(element, 0x03, ref cardflags);
                tlv.sucheTag(element, 0xA6, ref profileIndication);

                string cfString;
                if (cardflags[0] != 0)
                    cfString = " " + cardflags[0] + " ";
                else
                    cfString = " ";

                string verString = "v" + version[0];
                string profString = Encoding.UTF8.GetString(profileIndication, 2, profileIndication.Length - 2);

                //textBoxCIA_Info_row.Text = BitConverter.ToString( empfang).Substring(0, empfang.Length-2).Replace("-", " "); 
                textBoxCIA_Info.Text = "CardInfo ::= {" + System.Environment.NewLine +
                                       "\tversion " + verString + "," + System.Environment.NewLine +
                                       "\tcardFlags {" + cfString + "}," + System.Environment.NewLine +
                                       "\tprofileIndication {" + System.Environment.NewLine + "\t\t\"" +
                                       profString + "\"" +
                                       System.Environment.NewLine + "\t}" + System.Environment.NewLine + "}";

                textBoxCIA_InfoVer.Text = verString;
                if (cfString == " ")
                    textBoxCIA_InfoCF.Text = "keine Flags";
                else
                    textBoxCIA_InfoCF.Text = cfString;
                textBoxCIA_InfoProf.Text = profString;
            }
   
        }


        /// <summary>
        /// Persoenliche Daten aus dem EF .PD auslesen
        /// </summary>
        /// <returns></returns>
        private void readPD()
        {
            byte[] empfang = new byte[1024];

            // Das transparente EF .PD auslesen 
            if (karte.readBinary( (byte)HCA.PD, empfang) == false)
                return;    

            // =============================
            // gezippte XML-Datei auspacken
            // =============================
            byte[] empfang2 = new byte[empfang.Length - 2];
            Array.Copy(empfang, 2, empfang2, 0, empfang.Length - 2);

            // Inhalt entpacken
            System.IO.MemoryStream memStream = new System.IO.MemoryStream(empfang2);
            GZipStream gzStream = new GZipStream(memStream, CompressionMode.Decompress);
            StreamReader sr = new StreamReader(gzStream, Encoding.GetEncoding("iso-8859-15"));
            string entpackt = sr.ReadToEnd();

            // XMLstream erzeugen
            StringReader xmlStream = new StringReader(entpackt);
            XmlDocument xmlDoc = new XmlDocument();
            xmlDoc.Load(xmlStream);

            // Namensraum aufbauen
            XmlNamespaceManager nsp = new XmlNamespaceManager(xmlDoc.NameTable);
            nsp.AddNamespace("ns", "http://ws.gematik.de/fa/vsds/UC_PersoenlicheVersichertendatenXML/v5.1");

            // --------------------------------------------------

            textBoxVersichertenID.Text = "";
            XmlNode knoten = xmlDoc.SelectSingleNode("/ns:UC_PersoenlicheVersichertendatenXML/ns:Versicherter/ns:Versicherten_ID", nsp);
            if (knoten != null)
                textBoxVersichertenID.Text = knoten.InnerText;

            // --------------------------------------------------

            string Baum = "/ns:UC_PersoenlicheVersichertendatenXML/ns:Versicherter/ns:Person";

            textBoxVorname.Text = "";
            knoten = xmlDoc.SelectSingleNode(Baum + "/ns:Vorname", nsp);
                if (knoten != null)
                    textBoxVorname.Text = knoten.InnerText;   


            textBoxNachname.Text = "";
            knoten = xmlDoc.SelectSingleNode(Baum + "/ns:Nachname", nsp);
            if (knoten != null)
                textBoxNachname.Text = knoten.InnerText;   


            textBoxGeburtstag.Text = "";
            knoten = xmlDoc.SelectSingleNode(Baum + "/ns:Geburtsdatum", nsp);
            if (knoten != null)
                textBoxGeburtstag.Text = knoten.InnerText.Substring(6, 2) + '.'
                                            + knoten.InnerText.Substring(4, 2) + '.'
                                            + knoten.InnerText.Substring(0, 4);

            textBoxGeschlecht.Text = "";
            knoten = xmlDoc.SelectSingleNode(Baum + "/ns:Geschlecht", nsp);
            if (knoten != null)
                switch (knoten.InnerText)
                {
                    case "M": textBoxGeschlecht.Text = "Männlich";
                        break;
                    case "W": textBoxGeschlecht.Text = "Weiblich";
                        break;
                    default: textBoxGeschlecht.Text = "Unbekannt";
                        break;
                }

            textBoxTitel.Text = "";
            knoten = xmlDoc.SelectSingleNode(Baum + "/ns:Titel", nsp);
            if (knoten != null)
                textBoxTitel.Text = knoten.InnerText;   

            // Nachname := Namenszusatz + Vorsatzwort + Nachname
            // Beispiel :  Freiherr     + von         + Schulze
            if ((knoten = xmlDoc.SelectSingleNode(Baum + "/ns:Vorsatzwort", nsp)) != null)
                textBoxNachname.Text = knoten.InnerText + ' ' + textBoxNachname.Text;
            if ((knoten = xmlDoc.SelectSingleNode(Baum + "/ns:Namenszusatz", nsp)) != null)
                 textBoxNachname.Text = knoten.InnerText + ' ' + textBoxNachname.Text;

            // --------------------------------------------------

            // Ist eine Postfach- oder Strassenadresse auf der Karte abgelegt?
            if ((knoten = xmlDoc.SelectSingleNode(Baum + "/ns:PostfachAdresse", nsp)) != null)
                Baum = Baum + "/ns:PostfachAdresse";
            else
            {
                Baum = Baum + "/ns:StrassenAdresse";

                // Die folgenden  drei Knoten sind nur in diesem Zweig vorhanden 
                textBoxStrasse.Text = "";
                knoten = xmlDoc.SelectSingleNode(Baum + "/ns:Strasse", nsp);
                if (knoten != null)
                    textBoxStrasse.Text = knoten.InnerText;

                textBoxHausNr.Text = "";
                knoten = xmlDoc.SelectSingleNode(Baum + "/ns:Hausnummer", nsp);
                if (knoten != null)
                    textBoxHausNr.Text = knoten.InnerText;

                textBoxAnschriftenzusatz.Text = "";
                knoten = xmlDoc.SelectSingleNode(Baum + "/ns:Anschriftenzusatz", nsp);
                if (knoten != null)
                    textBoxAnschriftenzusatz.Text = knoten.InnerText;
            }

            textBoxLaendercode.Text = "";
            knoten = xmlDoc.SelectSingleNode(Baum + "/ns:Land/ns:Wohnsitzlaendercode", nsp);
            if (knoten != null)
                textBoxLaendercode.Text = knoten.InnerText;

            textBoxPLZ.Text = "";
            knoten = xmlDoc.SelectSingleNode(Baum + "/ns:Postleitzahl", nsp);
            if (knoten != null)
                textBoxPLZ.Text = knoten.InnerText;

            textBoxOrt.Text = "";
            knoten = xmlDoc.SelectSingleNode(Baum + "/ns:Ort", nsp);
            if (knoten != null)
                textBoxOrt.Text = knoten.InnerText;

            // --------------------------------------------------

            // XML-Datei als Text ausgeben

            // wenn noetig, Unterverzeichnis mit "Name_Vorname" im Dokumentenordner des Programms anlegen
            string unterverzeichnis = System.IO.Path.Combine(pfadName, textBoxNachname.Text + "_" + textBoxVorname.Text);
            if ( !System.IO.File.Exists( unterverzeichnis))
                System.IO.Directory.CreateDirectory( unterverzeichnis);

            // vollstaendig adressierte Zieldatei = Pfad + Dateiname
            string pdName = System.IO.Path.Combine( unterverzeichnis, "PD.XML");

            StreamWriter datei = new StreamWriter( pdName);            
            datei.Write( entpackt);
            datei.Close();            
        }


        /// <summary>
        /// Farbe der Tab-Hintergruende auf Hintergrundfarbe der Form bringen 
        /// </summary>
        /// <returns></returns>
        private void tabsEinfaerben()
        {
            // Hintergundfarbe der TabControls an Form-Hintergrund anpassen
            tabControlEGK.TabPages[0].BackColor = Gesundheitskarte.DefaultBackColor;
            tabControlEGK.TabPages[1].BackColor = Gesundheitskarte.DefaultBackColor;
            tabControlEGK.TabPages[2].BackColor = Gesundheitskarte.DefaultBackColor;
            tabControlEGK.TabPages[3].BackColor = Gesundheitskarte.DefaultBackColor;
            tabControlEGK.TabPages[4].BackColor = Gesundheitskarte.DefaultBackColor;
            tabControlEGK.TabPages[0].UseVisualStyleBackColor = false;


            // Hintergundfarbe der TabControls an Form-Hintergrund anpassen
            tabControlHCA.TabPages[0].BackColor = Gesundheitskarte.DefaultBackColor;
            tabControlHCA.TabPages[1].BackColor = Gesundheitskarte.DefaultBackColor;
            tabControlHCA.TabPages[2].BackColor = Gesundheitskarte.DefaultBackColor;
            tabControlHCA.TabPages[3].BackColor = Gesundheitskarte.DefaultBackColor;
            tabControlHCA.TabPages[0].UseVisualStyleBackColor = false;

            // Hintergundfarbe der TabControls an Form-Hintergrund anpassen
            tabControlKartenstatus.TabPages[0].BackColor = Gesundheitskarte.DefaultBackColor;
            tabControlKartenstatus.TabPages[1].BackColor = Gesundheitskarte.DefaultBackColor;
            tabControlKartenstatus.TabPages[2].BackColor = Gesundheitskarte.DefaultBackColor;
            tabControlKartenstatus.TabPages[3].BackColor = Gesundheitskarte.DefaultBackColor;
            tabControlKartenstatus.TabPages[4].BackColor = Gesundheitskarte.DefaultBackColor;
            tabControlKartenstatus.TabPages[5].BackColor = Gesundheitskarte.DefaultBackColor;
            tabControlKartenstatus.TabPages[6].BackColor = Gesundheitskarte.DefaultBackColor;
            tabControlKartenstatus.TabPages[0].UseVisualStyleBackColor = false;

            // Hintergundfarbe der TabControls an Form-Hintergrund anpassen
            tabControlESIGN.TabPages[0].BackColor = Gesundheitskarte.DefaultBackColor;
            tabControlESIGN.TabPages[1].BackColor = Gesundheitskarte.DefaultBackColor;
            tabControlESIGN.TabPages[0].UseVisualStyleBackColor = false;
        }
        
        
        
        /// <summary>
        /// Auslesen einer eGK starten 
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="EventArgs"></param>
        /// <returns></returns>
        private void auslesen_Click(object sender, EventArgs e)
        {
            // Verzeichnis "Eigene Dokumente"
            pfadName = Environment.GetFolderPath(Environment.SpecialFolder.MyDocuments);

            // Unterverzeichnis in "Eigene Dokumente" anhaengen
            pfadName = System.IO.Path.Combine(pfadName, "Gesundheitskarte");

            // wenn noch nicht vorhanden, dieses Unterverzeichnis anlegen
            if (!System.IO.File.Exists(pfadName))
                System.IO.Directory.CreateDirectory(pfadName);


            uint dwActiveProtocol;
            uint lReturn;

            // =============================
            // Verbindung herstellen
            // =============================
            string terminal = listBoxTerminals.SelectedItem.ToString();
            lReturn = WinSCard.SCardConnect(hContext,
                                    terminal,
                                    (uint)SCardAccessMode.Shared,
                                    (uint)SCardProtocolIdentifiers.T1,
                                    out hCard,
                                    out dwActiveProtocol);

            // Aufruf von SCardConnect fehlerfrei?
            if (lReturn != 0)
            {
                string ausgabe = "Fehler beim Verbindungsaufbau mit der Karte!\n\n"
                                 + "Prüfen Sie bitte, ob sich eine Karte im ausgewählten Terminal befindet!";
                MessageBox.Show( ausgabe);
                return;
            }


            // =============================
            // Test ob eGK vorliegt
            // dazu wird der ATR ausgewertet
            // =============================
            byte[] atr = new byte[256];
            int dwCount = atr.Length;
            const uint SCARD_ATTR_ATR_STRING = 590595;

            // ATR abfragen
            WinSCard.SCardGetAttrib(
                hCard,
                SCARD_ATTR_ATR_STRING,
                atr,
                ref dwCount);

            // ATR nach bekannter Zeichenkette durchsuchen
            string atrString = BitConverter.ToString(atr).Substring(0, dwCount * 3 - 1);
            if (atrString.Contains("81-B1-FE-45-1F") == false) // erwartete Werte fuer TD1-TD2-TA3-TB3-TD3
            {
                MessageBox.Show("Keine elektronische Gesundheitskarte (eGK) gefunden!");
                return;
            }
            textBoxATR.Text = atrString.Replace("-", " ");

            karte = new eGK(hCard);

            
            // Patienteninformationen auslesen
            // dazu vorher in die Anwendung HCA wechseln
            // File-ID des DF HCA = 0xD276 00000 101
            if (karte.select_AI( AID_HCA ))
            {
                readHcaStatus();
                readPD();
                readVD();

                karte.select_MF(); // zurueck zur Wurzel
            }


            // Infos aus CIA_ESIGN auslesen
            // dazu vorher in die Anwendung CIA_ESIGN wechseln            
            if (karte.select_AI( AID_CIA_ESIGN))
            {
                readCIAinfo();
                karte.select_MF(); // zurueck zur Wurzel
            }


            // ESIGN-Zertifikate auslesen
            // dazu vorher in die Anwendung ESIGN wechseln
            // File-ID des DF ESIGN = 0xA000000167 455349474E            
            if (karte.select_AI(AID_ESIGN))
            {
                readCert();
                karte.select_MF(); // zurueck zur Wurzel
            }

            // QES-Zertifikate auslesen
            // dazu vorher in die Anwendung QES wechseln
            // File-ID des DF QES = 0xD27600006601 
            if (karte.select_AI( AID_QES))
            {
                leseZertifikat_QES();
                groupBoxQES.Visible = true;

                // Status der PIN.QES
                PinStatus pinstatus=PinStatus.NoError;
                int fehlbed = 0;
                string ausgabe = "";
                
                // 0x80+1, da es sich um ein lokales Passwort von DF QES ahndelt
                if (karte.getPinStatus(0x80 + 1, ref pinstatus, ref fehlbed, ref ausgabe))
                {
                    textBoxPinStatusQES.Text = ausgabe;

                    // Button fuer das Aendern der Transport-PIN einblenden
                    if (pinstatusQES == PinStatus.RetryCounter)
                        buttonPinQESSetzen.Text = "PIN ändern";
                    else
                        buttonPinQESSetzen.Text = "Transport-PIN überschreiben";
                    buttonPinQESSetzen.Enabled = true;
                }

                karte.select_MF(); // zurueck zur Wurzel
            }
            else
            {
                groupBoxQES.Visible = false;
                labelQESstatus.Text = "Auf dieser Karte ist die Anwendung zur qualifizierten elektronischen Signatur nicht vorhanden!";
                textBoxPinStatusQES.Text = "Auf dieser eGK gibt es keine Anwendung QES!";
            }


            // Informationen aus dem Wurzelverzeichnis auslesen
            // erst jetzt, da zur Ablage der CV-Zertifikate "Name" und "Vorname" (liegen in DF.HCA) benoetigt werden
            readCardStatus();

            // Buttons zur Anzeige der XML-Dateien aktivieren
            buttonXML_PD.Enabled = true;
            buttonXML_VD.Enabled = true;
            buttonXML_GVD.Enabled = true;


            // =============================
            // Verbindung trennen
            // =============================
            lReturn = WinSCard.SCardDisconnect(hContext, (uint)SCardDisposition.LeaveCard);
        }





        /// <summary>
        /// Reaktion auf Druecken des Buttons "buttonAutZertifikat"
        /// </summary>
        /// <returns></returns>
        private void buttonAutZertifikat_Click(object sender, EventArgs e)
        {
            string unterverzeichnis = System.IO.Path.Combine(pfadName, textBoxNachname.Text + "_" + textBoxVorname.Text);
            string Parameter = System.IO.Path.Combine(unterverzeichnis, "AUT_CERT.CER");
            System.Diagnostics.Process.Start(Parameter);
        }

        /// <summary>
        /// Reaktion auf Druecken des Buttons "buttonEncZertifikat"
        /// </summary>
        /// <returns></returns>
        private void buttonEncZertifikat_Click(object sender, EventArgs e)
        {
            string unterverzeichnis = System.IO.Path.Combine(pfadName, textBoxNachname.Text + "_" + textBoxVorname.Text);
            string Parameter = System.IO.Path.Combine(unterverzeichnis, "ENC_CERT.CER");
            System.Diagnostics.Process.Start(Parameter);
        }

        /// <summary>
        /// Reaktion auf Druecken des Buttons "buttonZertifikatQES"
        /// </summary>
        /// <returns></returns>
        private void buttonZertifikatQES_Click(object sender, EventArgs e)
        {
            string Parameter = System.Threading.Thread.GetDomain().BaseDirectory + textBoxNachname.Text + "_QES_PrK_CERT.CER";
            System.Diagnostics.Process.Start(Parameter); // Start der mit der Erweiterung verknuepften Anwendung
        }

        /// <summary>
        /// Reaktion auf Druecken des Buttons "buttonXML_PD"
        /// </summary>
        /// <returns></returns>
        private void buttonXML_PD_Click(object sender, EventArgs e)
        {
            string Programmname = "iexplore.exe";
            string unterverzeichnis = System.IO.Path.Combine(pfadName, textBoxNachname.Text + "_" + textBoxVorname.Text);
            string Parameter = System.IO.Path.Combine(unterverzeichnis, "PD.xml");
            System.Diagnostics.Process.Start(Programmname, Parameter);
        }

        /// <summary>
        /// Reaktion auf Druecken des Buttons "buttonXML_VD"
        /// </summary>
        /// <returns></returns>
        private void buttonXML_VD_Click(object sender, EventArgs e)
        {
            string Programmname = "iexplore.exe";
            string unterverzeichnis = System.IO.Path.Combine(pfadName, textBoxNachname.Text + "_" + textBoxVorname.Text);
            string Parameter = System.IO.Path.Combine(unterverzeichnis, "VD.xml");
            System.Diagnostics.Process.Start(Programmname, Parameter);        
        }

        /// <summary>
        /// Reaktion auf Druecken des Buttons "buttonXML_GVD"
        /// </summary>
        /// <returns></returns>
        private void buttonXML_GVD_Click(object sender, EventArgs e)
        {
            string Programmname = "iexplore.exe";
            string unterverzeichnis = System.IO.Path.Combine(pfadName, textBoxNachname.Text + "_" + textBoxVorname.Text);
            string Parameter = System.IO.Path.Combine(unterverzeichnis, "GVD.xml");
            System.Diagnostics.Process.Start(Programmname, Parameter);
        }

        /// <summary>
        /// Reaktion auf Druecken des Buttons "buttonPinHomeSetzen"
        /// </summary>
        /// <returns></returns>
        private void buttonPinHomeSetzen_Click(object sender, EventArgs e)
        {
            PinSetzen(0x02, pinstatusHOME, "PIN.HOME:");
            
            // Kartenstatus neu einlesen
            readCardStatus();
        }

        /// <summary>
        /// Reaktion auf Druecken des Buttons "buttonPinQESSetzen"
        /// </summary>
        /// <returns></returns>
        private void buttonPinQESSetzen_Click(object sender, EventArgs e)
        {

            // Zur Anwendung QES wechseln
            if (karte.select_AI(AID_QES))
            {
                PinSetzen(0x80 + 1, pinstatusHOME, "PIN.QES:");

                // Status der PIN.QES neu einlesen
                PinStatus pinstatus = PinStatus.NoError;
                int fehlbed = 0;
                string ausgabe = "";

                // 0x80+1, da es sich um ein lokales Passwort von DF QES ahndelt
                if (karte.getPinStatus(0x80 + 1, ref pinstatus, ref fehlbed, ref ausgabe))
                {
                    textBoxPinStatusQES.Text = ausgabe;

                    // Button fuer das Aendern der Transport-PIN einblenden
                    if (pinstatusQES == PinStatus.RetryCounter)
                        buttonPinQESSetzen.Text = "PIN ändern";
                    else
                        buttonPinQESSetzen.Text = "Transport-PIN überschreiben";
                    buttonPinQESSetzen.Enabled = true;
                }

                karte.select_MF(); // zurueck zur Wurzel
            }
        }


        /// <summary>
        /// PIN wird neu gesetzt
        /// </summary>
        /// <param name="pinID">ID der zu ueberschreibenden PIN</param>
        /// <param name="pinStatus">Aktueller Status der zu ueberschreibenden PIN</param>
        /// <param name="pinID">Klarname der zu ueberschreibenden PIN</param>
        /// <returns></returns>
        private void PinSetzen(byte pinID, PinStatus pinStatus, string prompt)
        {           
            PinDialog pinbox;
            string altePIN = "";
            string neuePIN = "";
            DialogResult dr = DialogResult.Cancel;

            // Pin-Eingabe (getrennt nach Anzahl der benoetigten PINs)
            if (pinStatus == PinStatus.RetryCounter)
            {
                pinbox = new PinDialog("PIN-Eingabe", "Alte " + prompt, "Neue " + prompt);
                dr = pinbox.ShowDialog();
                if (dr == DialogResult.OK)
                {
                    altePIN = pinbox.Value;
                    neuePIN = pinbox.Value2;
                }
            }
            else
            {
                pinbox = new PinDialog("PIN-Eingabe", prompt);
                dr = pinbox.ShowDialog();
                if (dr == DialogResult.OK)
                    neuePIN = pinbox.Value2;
            }

            // Wurde Dialog mit OK beendet?
            if (dr == DialogResult.OK)
            {
                // Beide PINs ueberpruefen
                Regex pattern = new Regex(@"^[0-9]{6,8}$");
                if (((pinStatus == PinStatus.RetryCounter) && !pattern.IsMatch(altePIN)) || (!pattern.IsMatch(neuePIN)))
                {
                    MessageBox.Show("PINs müssen aus 6 bis 8 Ziffern bestehen!", "Eingabefehler");
                    return;
                }

                // Aus dem String ASCII-Zeichen machen
                System.Text.ASCIIEncoding enc = new System.Text.ASCIIEncoding();
                byte[] oldPIN = enc.GetBytes(altePIN);
                byte[] newPIN = enc.GetBytes(neuePIN);

                // PIN setzen
                karte.setPIN(pinID, pinStatus, newPIN, oldPIN);
            }
        }
    }


}
