//
//	tsp.cpp
//	=======
//
//	Software License Agreement (BSD License)
//	----------------------------------------
//	Copyright (c) 2013 Thorsten Radde (thorstenr@idealsoftware.com). All rights reserved.
//	Source code: www.IdealSoftware.com
//
//	Redistribution and use in source and binary forms, with or without modification,
//	are permitted provided that the following conditions are met:
//
//	* Redistributions of source code must retain the above copyright notice, this 
//	  list of conditions and the following disclaimer.
//
//	* Redistributions in binary form must reproduce the above copyright notice, this
//	  list of conditions and the following disclaimer in the documentation and/or
//	  other materials provided with the distribution.
//
//	* Neither the names Thorsten Radde or IDEAL Software GmbH, nor the names of contributors
//	  may be used to endorse or promote products derived from this software without
//	  specific prior written permission of Thorsten Radde.
//
//	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
//	ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//	DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
//	ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
//	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
//	LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
//	ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
//	(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
//	SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


#include "stdafx.h"
#include "../resource.h"

#include <vector>
#include <list>
#include <algorithm>
using namespace std;

#include <limits.h>
#include <float.h>
#include <time.h>
#include <math.h>
#include "../MersenneTwister/SFMT.h"	// "good" random numbers

#ifndef _DEBUG
    #include <omp.h>		// OpenMP support
#endif

#include "../assert.h"
#include "../Splash.h"
#include "../galib.h"
#include "../ga_app.h"
#include "tsp.h"



// ================================================================================================================
//												TSP Application Class
// ================================================================================================================
template<class TGenome, class TVisualization, class TSelection>
class CTspApp : public CGaApplication<TGenome, TVisualization, TSelection>
{
protected:
	TGenome		m_BruteForceGenome;			// best solution found by Brute-Force algorithm

public:
	virtual void Init(HINSTANCE hInstance);
	virtual void CalcGenetic();
	virtual void DoCalcBruteForce();
	virtual void MainWndPaint(HWND hWnd, HDC hdc);
	virtual void PopWndPaint(HWND hWnd, HDC hdc);
};


// ================================================================================================================
//												Global Variables
// ================================================================================================================
CTspApp<CTspGenome, CGaAppVisualize<CTspGenome>, CNaturalSelection<CTspGenome>> gGeneticApp;


// ================================================================================================================
//											Application Interface Class
//
// This is just glue-code.
// ================================================================================================================
class CAppInterface : public CGaAppInterface
{
public:
	virtual void MainWndCreate() { gGeneticApp.MainWndCreate(); }
	virtual bool MainWndCommand(int wmId, int wmEvent) { return gGeneticApp.MainWndCommand(wmId, wmEvent); }
	virtual void MainWndPaint(HWND hWnd, HDC hdc) { gGeneticApp.MainWndPaint(hWnd, hdc); }
	virtual void MainWndDestroy() { gGeneticApp.MainWndDestroy(); }

	virtual void MeanWndPaint(HWND hWnd, HDC hdc) { gGeneticApp.MeanWndPaint(hWnd, hdc); }
	virtual void PopWndPaint(HWND hWnd, HDC hdc) { gGeneticApp.PopWndPaint(hWnd, hdc); }
};


// ================================================================================================================
//											InitApplication()
// ================================================================================================================
void InitApplication(HINSTANCE hInstance)
{
	gGeneticApp.Init(hInstance);
}


// ================================================================================================================
//											Init()
// ================================================================================================================
template<class TGenome, class TVisualization, class TSelection>
void CTspApp<TGenome, TVisualization, TSelection>::Init(HINSTANCE hInstance)
{
	// initialize Application Interface Object
	gpGaAppInterface = new CAppInterface();

	// perform application specific initializations
	ComputeDistances();

	// call base class code
	CGaApplication<TGenome, TVisualization, TSelection>::Init(hInstance);
}


// ================================================================================================================
//												CalcGenetic()
// ================================================================================================================
template<class TGenome, class TVisualization, class TSelection>
void CTspApp<TGenome, TVisualization, TSelection>::CalcGenetic()
{
	m_GeneticAlgo.SetPopulationSize(1024);

	m_GeneticAlgo.SetCrossoverProbability(60);
	m_GeneticAlgo.SetMutationProbability(40);
	m_GeneticAlgo.SetElitismPercentage(5);
	m_GeneticAlgo.SetStableGenerations(64);
	m_GeneticAlgo.SetMaxGenerations(256);

	CGaApplication<TGenome, TVisualization, TSelection>::CalcGenetic();
}


// ================================================================================================================
//												DoCalcBruteForce()
// ================================================================================================================
template<class TGenome, class TVisualization, class TSelection>
void CTspApp<TGenome, TVisualization, TSelection>::DoCalcBruteForce()
{
	MessageBox(NULL, _T("Brute Force is not implemented for the TSP problem."), _T("Note:"), MB_OK);
}


// ================================================================================================================
//												MainWndPaint()
// ================================================================================================================
template<class TGenome, class TVisualization, class TSelection>
void CTspApp<TGenome, TVisualization, TSelection>::MainWndPaint(HWND hWnd, HDC hdc)
{
	TCHAR s[256];
	const int nLineHeight = 14;
	int y = 10;
	size_t i;
	int min_x = INT_MAX, min_y = INT_MAX;
	int max_x = 0, max_y = 0;

	for (i = 0; i < CityCount; i++)
	{
		// find bounding rect
		if (Cities[i].xPos < min_x)
			min_x = Cities[i].xPos;

		if (Cities[i].xPos > max_x)
			max_x = Cities[i].xPos;

		if (Cities[i].yPos < min_y)
			min_y = Cities[i].yPos;

		if (Cities[i].yPos > max_y)
			max_y = Cities[i].yPos;
	}

	double dx = max_x - min_x;
	double dy = max_y - min_y;

	RECT rcClient;
	GetClientRect(hWnd, &rcClient);
	double client_dx = rcClient.right - rcClient.left;
	int org_client_dy = rcClient.bottom - rcClient.top;
	double client_dy = org_client_dy;

	const int margin = 15;
	client_dx -= margin * 2;
	client_dy -= margin * 2;

	double scale_x = client_dx / dx;
	double scale_y = client_dy / dy;

	double scale = min(scale_x, scale_y);

	TGenome solution = m_GeneticAlgo.GetSolution();

	// we select the ANSI_VAR_FONT, which is some smaller than the normal system font.
	HFONT hOldFont = (HFONT)SelectObject(hdc, GetStockObject(ANSI_VAR_FONT));

	for (i = 0; i < CityCount; i++)
	{
		int x = (int)((Cities[i].xPos - min_x) * scale) + margin;
		int y = (int)((Cities[i].yPos - min_y) * scale) + margin;
	#ifdef MAP_UPSIDE_DOWN				// to be compatible to Concorde, the map is painted upside down
		y = org_client_dy - y;
	#endif

		Ellipse(hdc, x - 3, y - 3, x + 3, y + 3);

		if (solution.GetFitness() > 0)
		{
			for (size_t n = 0; n < solution.m_nNumChromosomes; n++)
			{
				if (solution.GetChromosome(n) == i)
				{
					wsprintf(s, _T("%s"), Cities[i].Name);
					break;
				}
			}
		}
		else
			wcscpy_s(s, Cities[i].Name);

		TextOut(hdc, x + 5, y - 5, s, (int)_tcslen(s));
	}

	if (solution.GetFitness() > 0)
	{
		// There is a solution available
		_stprintf_s(s, _T("Route Length %g (Optimum %d)"), solution.GetRouteLength(), nOptimum);
		TextOut(hdc, 10, y, s, (int)_tcslen(s));
		y += nLineHeight;

		_stprintf_s(s, _T("Generations %d (%d computations)"), m_GeneticAlgo.GetGenerations(), m_GeneticAlgo.GetGenerations() * m_GeneticAlgo.GetPopulationSize());
		TextOut(hdc, 10, y, s, (int)_tcslen(s));
		y += nLineHeight;

		_stprintf_s(s, _T("Solution found in generation %d"), m_GeneticAlgo.GetBSGeneration());
		TextOut(hdc, 10, y, s, (int)_tcslen(s));
		y += nLineHeight;

		if (!m_GeneticAlgo.GetIsComputing())
		{
			_stprintf_s(s, _T("done."));
			TextOut(hdc, 10, y, s, (int)_tcslen(s));
			y += nLineHeight;

			m_Timer.GetElapsedTime(s);
			TextOut(hdc, 10, y, s, (int)_tcslen(s));
			y += nLineHeight;
		}

		TChromType n = solution.GetChromosome(0);
		int x = (int)((Cities[n].xPos - min_x) * scale) + margin;
		int y = (int)((Cities[n].yPos - min_y) * scale) + margin;
	#ifdef MAP_UPSIDE_DOWN				// to be compatible to Concorde, the map is painted upside down
		y = org_client_dy - y;
	#endif
		MoveToEx(hdc, x, y, NULL);

		for (i = 1; i < solution.m_nNumChromosomes; i++)
		{
			n = solution.GetChromosome(i);
			x = (int)((Cities[n].xPos - min_x) * scale) + margin;
			y = (int)((Cities[n].yPos - min_y) * scale) + margin;
		#ifdef MAP_UPSIDE_DOWN				// to be compatible to Concorde, the map is painted upside down
			y = org_client_dy - y;
		#endif
			LineTo(hdc, x, y);
		}

		n = solution.GetChromosome(0);
		x = (int)((Cities[n].xPos - min_x) * scale) + margin;
		y = (int)((Cities[n].yPos - min_y) * scale) + margin;
	#ifdef MAP_UPSIDE_DOWN				// to be compatible to Concorde, the map is painted upside down
		y = org_client_dy - y;
	#endif
		LineTo(hdc, x, y);
	}

	SelectObject(hdc, hOldFont);
}


//---------------------------------------------------------------------------
//                        COLOR RAMP FUNCTION
//
// The color ramp function describes a path from black to white along
// seven edges of the color cube.
// Since each edge of the color cube is 256 pixels long, the length of
// the entire path is 7 x 256 or 1792 pixels.
//
// Source: http://www.duke.edu/web/isis/gessler/borland/color-ramp.txt
//---------------------------------------------------------------------------
DWORD colorRamp(int value, int maximum)
{
    int pixelDistanceAlongPath = (value * 1792) / maximum;
    int red, green, blue;

    // Which edge of the color cube are we on?
    if (pixelDistanceAlongPath < 256) {
        // Edge 1 from BLACK to BLUE
        red=0; green=0; blue=pixelDistanceAlongPath;
    }
    else if (pixelDistanceAlongPath < 512) {
        // Edge 2 from BLUE to CYAN
        red =0; green=pixelDistanceAlongPath-256; blue=255;
    }
    else if (pixelDistanceAlongPath < 768) {
        // Edge 3 from CYAN to GREEN
        red =0; green =255; blue= 255-(pixelDistanceAlongPath-512);
    }
    else if (pixelDistanceAlongPath < 1024) {
        // Edge 4 from GREEN to YELLOW
        red= (pixelDistanceAlongPath-768); green =255; blue =0;
    }
    else if (pixelDistanceAlongPath <1280) {
        // Edge 5 from YELLOW to RED
        red =255; green=255-(pixelDistanceAlongPath-1024); blue =0;
    }
    else if (pixelDistanceAlongPath < 1536) {
        // Edge 6 from RED to MAGENTA
        red =255; green=0; blue=pixelDistanceAlongPath -1280;
    }
    else {
        // Edge 7 from MAGENTA to WHITE
        red =255; green=pixelDistanceAlongPath-1536; blue =255;
    }

    return RGB(red, green, blue);
}


// ================================================================================================================
//													PopWndPaint()
// ================================================================================================================
template<class TGenome, class TVisualization, class TSelection>
void CTspApp<TGenome, TVisualization, TSelection>::PopWndPaint(HWND hWnd, HDC hdc)
{
	RECT rcClient;
	GetClientRect(hWnd, &rcClient);
	LONG client_dx = rcClient.right - rcClient.left;
	LONG client_dy = rcClient.bottom - rcClient.top;

	if (m_enVisualization == enVisualizeOff)
	{
		PatBlt(hdc, 0, 0, client_dx, client_dy, WHITENESS);
		return;
	}

	CPopulation<TGenome> *pPopulation = m_GeneticAlgo.GetVisualization().GetPopulation();
	if (!pPopulation || client_dx == 0 || client_dy == 0)
	{
		PatBlt(hdc, 0, 0, client_dx, client_dy, WHITENESS);
		return;
	}

	LONG height = min(client_dy, (LONG)pPopulation->size());
	LONG width = (LONG)TGenome::m_nNumChromosomes;

	if (height == 0)
	{
		PatBlt(hdc, 0, 0, client_dx, client_dy, WHITENESS);
		return;
	}

	// create RGB bitmap
	BITMAPINFO *pbmi = (BITMAPINFO *)malloc(sizeof(BITMAPINFOHEADER));
	pbmi->bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
	pbmi->bmiHeader.biWidth = width;
	pbmi->bmiHeader.biHeight = height;
	pbmi->bmiHeader.biPlanes = 1;
	pbmi->bmiHeader.biBitCount = 24;
	pbmi->bmiHeader.biCompression = BI_RGB;
	pbmi->bmiHeader.biSizeImage = 0;
	pbmi->bmiHeader.biXPelsPerMeter = 0;
	pbmi->bmiHeader.biYPelsPerMeter = 0;
	pbmi->bmiHeader.biClrUsed = 0;
	pbmi->bmiHeader.biClrImportant = 0;

	// Macro to determine the number of bytes in a DWORD aligned DIB scanline
	#define BYTESPERLINE(Width, BPP) ((WORD)((((DWORD)(Width) * (DWORD)(BPP) + 31) >> 5)) << 2)

	DWORD bytes_per_line = BYTESPERLINE(width, pbmi->bmiHeader.biBitCount);
	const size_t bmp_size = bytes_per_line * height;

	BYTE *pBitmap = (BYTE *)malloc(bmp_size);
	memset(pBitmap, 0, bmp_size);
	BYTE *p = pBitmap;
	BYTE *start = pBitmap;

	LONG line = 0;
	for (vector<TGenome>::iterator it = pPopulation->begin(); it != pPopulation->end(); it++)
	{
		for (size_t i = 0; i < it->m_nNumChromosomes; i++)
		{
			int val = it->GetChromosome(i);
			DWORD color = colorRamp(val, (int)TGenome::m_nChromosomeScale);
			*p++ = (BYTE)((color >> 16) & 0xff);
			*p++ = (BYTE)((color >>  8) & 0xff);
			*p++ = (BYTE)((color      ) & 0xff);
		}

		start += bytes_per_line;
		p = start;

		line++;
		if (line >= height)
			break;
	}

	// Blit
	StretchDIBits(
	  hdc,				// handle to DC
	  0,				// x-coord of destination upper-left corner
	  0,				// y-coord of destination upper-left corner
	  client_dx,		// width of destination rectangle
	  client_dy,		// height of destination rectangle
	  0,				// x-coord of source upper-left corner
	  0,				// y-coord of source upper-left corner
	  width,			// width of source rectangle
	  height,			// height of source rectangle
	  pBitmap,			// bitmap bits
	  pbmi,				// bitmap data
	  DIB_RGB_COLORS,	// usage options
	  SRCCOPY			// raster operation code
	);

	free(pbmi);
	free(pBitmap);
}
