package com.linkesoft.ctlongitude;

import android.app.Activity;
import android.graphics.Bitmap;
import android.graphics.drawable.BitmapDrawable;
import android.hardware.Sensor;
import android.hardware.SensorEvent;
import android.hardware.SensorEventListener;
import android.hardware.SensorManager;
import android.location.Location;
import android.net.Uri;
import android.os.Bundle;
import android.os.Handler;
import android.support.wearable.view.GridPagerAdapter;
import android.support.wearable.view.GridViewPager;
import android.support.wearable.view.ImageReference;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.TextView;

import com.google.android.gms.common.ConnectionResult;
import com.google.android.gms.common.GooglePlayServicesClient;
import com.google.android.gms.common.api.GoogleApiClient;
import com.google.android.gms.common.api.ResultCallback;
import com.google.android.gms.location.LocationClient;
import com.google.android.gms.location.LocationListener;
import com.google.android.gms.location.LocationRequest;
import com.google.android.gms.wearable.DataApi;
import com.google.android.gms.wearable.DataEvent;
import com.google.android.gms.wearable.DataEventBuffer;
import com.google.android.gms.wearable.DataItem;
import com.google.android.gms.wearable.DataItemBuffer;
import com.google.android.gms.wearable.DataMapItem;
import com.google.android.gms.wearable.Wearable;
import com.linkesoft.ctlongitude.common.Friend;

import org.json.JSONObject;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.text.Collator;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

public class MainActivity extends Activity implements SensorEventListener,DataApi.DataListener,GoogleApiClient.ConnectionCallbacks,GoogleApiClient.OnConnectionFailedListener,LocationListener,
        GooglePlayServicesClient.ConnectionCallbacks,
        GooglePlayServicesClient.OnConnectionFailedListener {

    private GoogleApiClient googleApiClient;
    private SensorManager sensorManager;
    private LocationClient locationClient;
    private float[] gravity = new float[3]; // for compass calculation
    private float compass = 0; // in degrees

    private Map<Uri, Friend> dataitems = new ConcurrentHashMap<Uri, Friend>(); // used for the sync from the device
    private List<Friend> sortedfriends; // list of friends, sorted by name, used for the view pager

    private GridViewPager friendsPager;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_main);
        // GridView einrichten
        friendsPager = (GridViewPager) findViewById(R.id.friendsPager);
        friendsPager.setAdapter(friendsPagerAdapter);
        friendsPager.setOnPageChangeListener(friendsPageChangedListener);
        // Google client für Kommunikation mit Smartphone
        googleApiClient = new GoogleApiClient.Builder(this, this, this).addApi(Wearable.API).build();
    }

    @Override
    protected void onStart() {
        super.onStart();
        googleApiClient.connect();
    }

    @Override
    public void onConnected(Bundle bundle) {
        Log.v(getClass().getSimpleName(), "Google API connected, listening for data items");
        loadDataItems();
        Wearable.DataApi.addListener(googleApiClient, this); // listener für Änderungen an geladenen DataItems
        startSensors();
        startLocationUpdates();
    }

    @Override
    public void onDisconnected() {
        Log.v(getClass().getSimpleName(), "Google API disconnected");
    }

    @Override
    public void onConnectionFailed(ConnectionResult connectionResult) {
        Log.v(getClass().getSimpleName(), "onConnectionFailed");

    }

    @Override
    public void onConnectionSuspended(int i) {
        Log.v(getClass().getSimpleName(), "onConnectionSuspended");

    }

    @Override
    protected void onStop() {
        if (sensorManager != null)
            sensorManager.unregisterListener(this);
        if (googleApiClient != null && googleApiClient.isConnected()) {
            Wearable.DataApi.removeListener(googleApiClient, this);
            locationClient.removeLocationUpdates(this);
            googleApiClient.disconnect();
        }
        super.onStop();
    }

    private void loadDataItems() {
        Wearable.DataApi.getDataItems(googleApiClient).setResultCallback(new ResultCallback<DataItemBuffer>() {
            @Override
            public void onResult(DataItemBuffer dataItemBuffer) {
                for(DataItem dataitem: dataItemBuffer)
                {
                    DataMapItem item = DataMapItem.fromDataItem(dataitem);
                    Friend friend = new Friend(item.getDataMap().toBundle());
                    dataitems.put(item.getUri(), friend);
                }
                dataItemBuffer.release(); // wichtig um Resourcen freizugeben
                updateSortedFriends();
            }
        });
    }
    @Override
    public synchronized void onDataChanged(DataEventBuffer dataEvents) {
        Log.v(getClass().getSimpleName(),"Data received from device "+dataEvents);
        // wird in einem background thread aufgerufen, dataitems sind nach Ende der Methode nicht mehr verfügbar
        for(DataEvent dataEvent:dataEvents) {
            if (dataEvent.getType() == DataEvent.TYPE_DELETED)
                dataitems.remove(dataEvent.getDataItem().getUri());
            else {
                if (dataEvent.getType() == DataEvent.TYPE_CHANGED) {
                    DataMapItem item = DataMapItem.fromDataItem(dataEvent.getDataItem());
                    Friend friend = new Friend(item.getDataMap().toBundle());
                    dataitems.put(dataEvent.getDataItem().getUri(), friend);
                }
            }
        }
        updateSortedFriends();
    }
    private void updateSortedFriends() {
        sortedfriends=new ArrayList<Friend>(dataitems.values());
        final Collator collator = Collator.getInstance();
        collator.setStrength(Collator.PRIMARY);
        // sortiere nach Name (oder wahlweise nach Abstand zum aktuellen Standort)
        Collections.sort(sortedfriends, new Comparator<Friend>() {
            @Override
            public int compare(Friend lhs, Friend rhs) {
                return collator.compare(lhs.name,rhs.name);
            }
        });

        // update UI
        runOnUiThread(new Runnable() {
            @Override
            public void run() {
                // GridView hat eine Bug beim dynamischen Aktualisieren
                // http://stackoverflow.com/questions/24742427/dynamically-adding-items-to-fragmentgridpageradapter
                // https://code.google.com/p/android/issues/detail?id=75309
                friendsPager.setAdapter(friendsPagerAdapter);
                Runnable dirtyHack = new Runnable() {
                    @Override
                    public void run() {
                        friendsPager.setCurrentItem(0, 0);
                        friendsPageChangedListener.onPageSelected(0,0); // um Hintegrundbild zu aktualisieren
                        friendsPagerAdapter.notifyDataSetChanged();
                    }
                };
                Handler handler = new Handler();
                handler.postDelayed(dirtyHack, 1000);
            }
        });
    }


    private void updateFriendViews() {
        if(sortedfriends==null)
            return;
        for(Friend friend:sortedfriends)
        {
            if(friend.view!=null && locationClient!=null && locationClient.isConnected())
            {
                Location currentLocation=locationClient.getLastLocation();
                float distance =  currentLocation.distanceTo(friend.location); // Meter
                float bearing=currentLocation.bearingTo(friend.location); // Grad
                //Log.v(getClass().getSimpleName(), "Updating distance "+distance+" m /bearing "+bearing+" deg for friend "+friend.name);
                // Pfeil-Richtung
                TextView arrow = (TextView) friend.view.findViewById(R.id.arrow);
                arrow.setText("⬆");
                arrow.setRotation(bearing-compass);
                // Abstand in m oder km
                String str;
                if(distance<10000)
                    str=String.format("%d m",(int)distance);
                else
                    str=String.format("%.1f km",distance/1000);
                TextView distanceTextView = (TextView) friend.view.findViewById(R.id.distanceTextView);
                distanceTextView.setText(str);
            }
        }
    }
    // sensors
    private void startSensors() {
        sensorManager = (SensorManager)getSystemService(SENSOR_SERVICE);
        sensorManager.registerListener(this,
                sensorManager.getDefaultSensor(Sensor.TYPE_ACCELEROMETER),
                SensorManager.SENSOR_DELAY_NORMAL);
        Sensor magneticfieldsensor = sensorManager.getDefaultSensor(Sensor.TYPE_MAGNETIC_FIELD);
        if(magneticfieldsensor!=null)
            sensorManager.registerListener(this, magneticfieldsensor, SensorManager.SENSOR_DELAY_UI);
        else
            Log.v(getClass().getSimpleName(),"no magnetic sensor");
    }
    private void onMagneticFieldChanged(float[] magvalues) {
        float rotationMatrix[] = new float[3 * 3];
        if (SensorManager.getRotationMatrix(rotationMatrix, null, gravity, magvalues)) {

            float orientation[] = new float[3];
            SensorManager.getOrientation(rotationMatrix, orientation);
            float phi = orientation[0]; // -PI..PI
            compass=(float)(phi / Math.PI * 180);
            //Log.v(getClass().getSimpleName(), "Current compass reading " + compass);
            updateFriendViews();
        }

    }

    // location
    private void startLocationUpdates()
    {
        locationClient = new LocationClient(this, new GooglePlayServicesClient.ConnectionCallbacks() {
            @Override
            public void onConnected(Bundle bundle) {
                LocationRequest locationRequest = LocationRequest.create();
                // Use high accuracy
                locationRequest.setPriority(
                        LocationRequest.PRIORITY_HIGH_ACCURACY);
                // Set the update interval to 5 seconds
                locationRequest.setInterval(5*1000);
                // Set the fastest update interval to 1 second
                locationRequest.setFastestInterval(1*1000);
                locationClient.requestLocationUpdates(locationRequest, MainActivity.this);
            }

            @Override
            public void onDisconnected() {

            }
        }, this);
        locationClient.connect();
    }

    @Override
    public void onLocationChanged(Location location) {
        updateFriendViews();
    }

    // sensors
    @Override
    public void onSensorChanged(SensorEvent event) {
        if (event.sensor.getType() == Sensor.TYPE_MAGNETIC_FIELD) {
            onMagneticFieldChanged(event.values);
        }
        else if (event.sensor.getType() == Sensor.TYPE_ACCELEROMETER) {
            gravity = event.values.clone();
        }

    }

    @Override
    public void onAccuracyChanged(Sensor sensor, int accuracy) {

    }

    private Friend getFriend(int index)
    {
        if(sortedfriends==null || sortedfriends.size()<=index)
            return null;
        else
            return sortedfriends.get(index);
    }
    // grid pages
    final GridPagerAdapter friendsPagerAdapter=new GridPagerAdapter() {

        @Override
        public ImageReference getBackground(int row, int column) {
            return ImageReference.NONE;
        }

        @Override
        public int getRowCount() {
            // must not return 0
            if(sortedfriends==null||sortedfriends.size()==0)
                return 1;
            else
                return sortedfriends.size();
        }

        @Override
        public int getColumnCount(int row) {
            return 1;
        }

        @Override
        protected Object instantiateItem(ViewGroup container, int row, int col) {
            Log.v(getClass().getSimpleName(),"instantiateItem "+row+":"+col);
            final Friend friend = getFriend(row);
            if(col==0) {
                final View view = LayoutInflater.from(getApplicationContext()).inflate(R.layout.frienddirection, container, false);
                container.addView(view);
                TextView nameTextView = (TextView) view.findViewById(R.id.nameTextView);

                if(friend!=null) {
                    nameTextView.setText(friend.name);
                    friend.view = view;
                    updateFriendViews(); // aktualisiere Richtung und Entfernung
                }
                return view;
            }
            else
                return null;
        }

        @Override
        protected void destroyItem(ViewGroup container, int row, int col, Object o) {
            Log.v(getClass().getSimpleName(),"destroyItem "+row+":"+col);
            if(col==0) {
                Friend friend = getFriend(row);
                if(friend!=null)
                    friend.view = null;
                container.removeView((View) o);
            } else {

            }
        }

        @Override
        public boolean isViewFromObject(View view, Object o) {
            return view==o;
        }
    };

    final GridViewPager.OnPageChangeListener friendsPageChangedListener=new GridViewPager.OnPageChangeListener() {
        @Override
        public void onPageScrolled(int i, int i2, float v, float v2, int i3, int i4) {

        }

        @Override
        public void onPageSelected(int row, int col) {
        // setze Hintergrundbild falls verfügbar
            BitmapDrawable drawable=null; // default kein Hintergrund
            Friend friend =getFriend(row);
            if(friend!=null) {
                Bitmap bmp = friend.getBitmap();
                if (bmp != null)
                    drawable = new BitmapDrawable(getResources(), bmp);
            }
            View mainView=findViewById(R.id.mainView);
            mainView.setBackground(drawable);
        }

        @Override
        public void onPageScrollStateChanged(int i) {

        }
    };





}
