package com.linkesoft.midiintervals;

import android.annotation.SuppressLint;
import android.app.ListActivity;
import android.os.Bundle;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ArrayAdapter;
import android.widget.ListView;
import android.widget.TextView;
import android.widget.Toast;

public class MainActivity extends ListActivity implements Midi.MidiNoteListener{
    // die Notenintervalle (Index = Anzahl Halbtonschritte)
    private static final String intervals[]= {
        "Prime",
        "Kleine Sekunde",
        "Große Sekunde",
        "Kleine Terz",
        "Große Terz",
        "Quarte",
        "Tritonus", // auch übermäßige Quarte genannt
        "Quinte",
        "Kleine Sexte",
        "Große Sexte",
        "Kleine Septime",
        "Große Septime",
        "Oktave",
    };

    private Midi midi; // zentrale MIDI-Klasse
    private byte lastNote; // letzte gespielte Note

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        // nutzt Standard-List-Layout

        ArrayAdapter<String> adapter=new ArrayAdapter<String>(this,android.R.layout.simple_list_item_2,android.R.id.text1,intervals) {
            @SuppressLint("SetTextI18n")
            @Override
            public View getView(int position, View convertView, ViewGroup parent) {
                View view= super.getView(position, convertView, parent);
                // text1 automatisch von super.getView gesetzt, text2 gibt die Noten des Intervalls an, basierend auf lastNote
                TextView smallTextView=(TextView)view.findViewById(android.R.id.text2);
                byte startNote=lastNote != 0 ? lastNote : Midi.MidiConstants.NOTE_MIDDLE_C;
                smallTextView.setText(Midi.nameForNote(startNote)+" - "+Midi.nameForNote((byte)(startNote+(byte)position)));
                if(getListView().isItemChecked(position)) {
                    view.setBackgroundColor(0x7FFFFFFF & getColor(android.R.color.holo_orange_dark)); // halb-transparentes Orange
                }
                else
                    view.setBackgroundColor(0); // transparent
                return view;
            }
        };
        setListAdapter(adapter);
        getListView().setChoiceMode(ListView.CHOICE_MODE_SINGLE);
        getListView().setItemChecked(4,true);
    }

    @Override
    protected void onResume() {
        super.onResume();
        if(Midi.isMidiSupported(this)) {
            midi = new Midi(this,this);
        }
        else {
            Toast.makeText(this,"Keine MIDI-Unterstützung", Toast.LENGTH_LONG).show();
        }
    }

    @Override
    protected void onPause() {
        if(midi!=null)
            midi.close();
        super.onPause();
    }

    @Override
    protected void onListItemClick(ListView l, View v, int position, long id) {
        super.onListItemClick(l, v, position, id);
        getListView().setItemChecked(position, true);
        if(midi!=null)
            midi.sendNotes(new byte[]{Midi.MidiConstants.NOTE_MIDDLE_C, (byte) (Midi.MidiConstants.NOTE_MIDDLE_C + position)});
        resetList(1000);
    }

    @Override
    public void onMidiNoteReceived(byte note) {

        if(lastNote==0) {
            lastNote=note;
        } else {
            // zwei Noten = Interval kann bestimmt werden
            int interval=note-lastNote;
            // normalisieren
            if(interval<0)
                interval=-interval;
            while(interval>intervals.length)
                interval-=12;
            // in Liste markieren
            getListView().setItemChecked(interval, true);
            // Listeintrag in sichtbaren Bereich scrollen
            getListView().setSelection(interval);
            // und nach einiger Zeit Markierung entfernen
            resetList(1000);
        }
    }

    // setzte Listmarkierung zurück mit Verzögerung
    private void resetList(int delay_ms) {
        new android.os.Handler().postDelayed(new Runnable() {
            @Override
            public void run() {
                getListView().clearChoices();
                ((ArrayAdapter)getListAdapter()).notifyDataSetChanged();
                lastNote=0;
            }
        },delay_ms);
    }
}
