﻿<#
.SYNOPSIS
Bindet die virtuellen Laufwerke aus zwei Kopien derselben VHD-Datei ins 
System ein. Dazu ändert es bei der Kopie die Datenträgersignatur.

.NOTES
Copyright © 2017 Heise Medien GmbH & Co. KG / c't
Geschrieben von Hajo Schulz hos@ct.de

.LINK
https://ct.de/ypwx
#>

#---------- Hier an eigene Bedürfnisse anpassen ----------
$VhdPath = (Split-Path $MyInvocation.MyCommand.Path) # oder z. B.: "C:\Users\$env:USERNAME\Dropbox"
$VhdName = "Dropbox.vhdx"
$AutoUnlock = $true # $false, um die Passwort-Abfrage und das automatische Entsperren zu unterdrücken
#----------------- Ende der Anpassungen ------------------

# Auf Admin-Rechte prüfen; wenn nicht vorhanden, mit RunAs neu starten
$identity = [System.Security.Principal.WindowsIdentity]::GetCurrent()
$princ = New-Object System.Security.Principal.WindowsPrincipal($identity)
if(-not $princ.IsInRole([System.Security.Principal.WindowsBuiltInRole]::Administrator)) {
    Start-Process "$psHome\powershell.exe" -ArgumentList $MyInvocation.MyCommand.Path -Verb RunAs
    exit
}

# Liefert eine Liste der Laufwerksbuchstaben, die zu einem Datenträger gehören
# $disk: Ein MSFT_Disk-Objekt, wie es beispielsweise Get-Disk liefert.
function VolumesOnDisk($disk) {
    $wmidisk = Get-WmiObject -Class Win32_DiskDrive -Filter ("Index = " + $disk.Number)
    $partitions = Get-WmiObject -Query "Associators of {Win32_DiskDrive.DeviceID='$($wmidisk.DeviceID)'} WHERE ResultRole=Dependent"
    $logicalDisks = $partitions | ForEach-Object {
        Get-WmiObject -Query "Associators of {Win32_DiskPartition.DeviceID='$($_.DeviceID)'} WHERE AssocClass = Win32_LogicalDiskToPartition"
    }
    $volumes = @()
    foreach($ld in $logicalDisks) {
        $volumes += $ld.DeviceID
    }
    return $volumes
}

# Setzt den Status eines Laufwerks auf Online; ändert dazu wenn nötig dessen Signatur
# $disk: Ein MSFT_Disk-Objekt, wie es beispielsweise Get-Disk liefert.
function ForceOnline($disk) {
    $selectCmd = "select disk {0}" -f $disk.Number
    if($disk.PartitionStyle -eq "MBR") {
        $newID = Get-Random
        $idCmd = "uniqueid disk id={0:x8}" -f $newID
    } else {
        $newID = (New-Guid).Guid
        $idCmd = "uniqueid disk id={0}" -f $newID
    }
    $output = $selectCmd, $idCmd, "online disk" | diskpart.exe
}

# Initialisierung
$VhdFile = $VhdPath + "\" + $VhdName
$password = $null
if($AutoUnlock) {
    $password = Read-Host "Bitte Ihr Bitlocker-Passwort für $VhdName eingeben" -AsSecureString
}

# Original-VHD einbinden ...
$vhd = Get-DiskImage $VhdFile
if(-not $vhd.Attached) {
    $vhd | Mount-DiskImage
}
# ... und sicherstellen, dass sie online ist
$vhd = Get-DiskImage $VhdFile
$disk = Get-Disk -Number $vhd.Number
if($disk) {
    if($disk.OperationalStatus -eq 'Offline') {
        ForceOnline($disk)
    }
    $origVolumes = @(VolumesOnDisk($disk))
    "{0} als Laufwerk(e) {1} eingebunden." -f $VhdName, ($origVolumes -join ", ")
    if($AutoUnlock -and $password) {
        Unlock-BitLocker -MountPoint $origVolumes[0] -Password $password | Out-Null
    }
} else {
    "Fehler: {0} konnte nicht eingebunden werden." -f $VhdName
    exit
}

# Konflikt-VHD suchen
$name = $VhdName.Substring(0, $VhdName.LastIndexOf('.'))
$ext = $VhdName.Substring($VhdName.LastIndexOf('.'))
$mask = $name + "*" + $ext
$conflict = @(Get-ChildItem -Path $VhdPath -Filter $mask | where {$_.Name -match "$name (.* in Konflikt stehende Kopie .*)"})
if(-not $conflict) {
    "Keine in Konflikt stehende Kopie gefunden."
    exit
}
$conflict = $conflict[0]

# Konflikt-VHD einbinden ...
$vhd = Get-DiskImage $conflict.FullName
if(-not $vhd.Attached) {
    # Konflikt-VHD einbinden
    $vhd | Mount-DiskImage
}
# ... und sicherstellen, dass sie online ist
$vhd = Get-DiskImage $conflict.FullName
$disk = Get-Disk -Number $vhd.Number
if($disk) {
    if($disk.OperationalStatus -eq 'Offline') {
        ForceOnline($disk)
    }
    $conflictVolumes = @(VolumesOnDisk($disk))
    "{0} als Laufwerk(e) {1} eingebunden." -f $conflict.Name, ($conflictVolumes -join ", ")
    if($AutoUnlock -and $password) {
        Unlock-BitLocker -MountPoint $conflictVolumes[0] -Password $password | Out-Null
    }
} else {
    "Fehler: {0} konnte nicht eingebunden werden." -f $conflict.Name
    exit
}

if($AutoUnlock) {
    # Hier könnte man noch ein Diff-Programm zum Beheben der Konflikte aufrufen
}
Read-Host -Prompt "Zum Beenden Enter drücken"
