/**
 * @overview Sammlung kleiner Hilfsfunktionen und -werte, die von sowohl von
 * deploy.js als auch von play.js genutzt werden. Konkret in der Datei enhalten
 * sind
 * {@link stake},
 * {@link domReady|domReady()},
 * {@link hideAll|hideAll()},
 * {@link show|show()},
 * {@link connectToMetaMask|connectToMetaMask()},
 * {@link getAccount|getAccount()},
 * {@link fetchABI|fetchABI()},
 * {@link fetchContract|fetchContract()} und
 * {@link sha256|sha256()}.
 * @author Lars Hupel
 * @author Sylvester Tremmel
 */

/**
 * Betrag der von beiden Spielern jeweils gesetzt werden muss.
 * 10^15 Wei = 1 Finney = 0,001 Ether
 * @constant
 * @type {wei}
 * @default
 */
const stake = "1000000000000000";

/**
 * String, der nur eine Ganzzahl enthält, die als Ether-Betrag in der Einheit
 * "Wei" aufgefasst werden soll (10^18 Wei = 1 Ether).
 * @typedef {string} wei
 */

/**
 * Ruft die übergebene callback-Funktion sofort auf, falls die Website bereits
 * vollständig geladen ist. Andersfalls wird auf das Event "DOMContentLoaded"
 * gewartet bevor die übergebene Funktion aufgerufen wird.
 * @function
 * @param {function} callback - Funktion, die ausgeführt werden soll,
 * sobald die Website geladen ist.
 */
var domReady = (callback) => {
  if (document.readyState != "loading") callback();
  else document.addEventListener("DOMContentLoaded", callback);
}

/**
 * Versteckt alle DOM-Elements, die die Klasse "hidden" haben, indem die
 * Eigenschaft
 * {@link https://developer.mozilla.org/en-US/docs/Web/API/HTMLElement/hidden|"hidden"}
 * auf true gesetzt wird.
 */
function hideAll() {
  for (const elem of document.querySelectorAll(".hidden"))
    elem.hidden = true;
}

/**
 * Zeigt das übergebene DOM-Element an, indem die Eigenschaft
 * {@link https://developer.mozilla.org/en-US/docs/Web/API/HTMLElement/hidden|"hidden"}
 * auf false gesetzt wird.
 * @param {string} id - ID des anzuzeigenden DOM-Elements.
 */
function show(id) {
  document.getElementById(id).hidden = false;
}

/**
 * Baut eine Verbindung mit
 * {@link https://docs.metamask.io/guide/ethereum-provider.html#ethereum-request-args|MetaMask}
 * auf und nutzt sie um
 * {@link https://web3js.readthedocs.io/en/v1.4.0/web3.html#web3-instance|Web3js zu initialisieren}.
 * Im Erfolgsfall wird das Element mit der ID "conn-succ" angezeigt, im
 * Fehlerfall "conn-fail" (siehe {@link show}).
 * @throws Wirft jeglichen Fehler, der von MetaMask oder Web3js produziert wird.
 */
async function connectToMetaMask() {
  try {
    await window.ethereum.request({ method: 'eth_requestAccounts' });
    window.web3 = new Web3(window.ethereum);
    show("conn-succ");
  }
  catch (err) {
    show("conn-fail");
    throw err;
  }
}

/**
 * Gibt den aktuell in MetaMask eingestellten Account zurück (siehe
 * {@link https://web3js.readthedocs.io/en/v1.4.0/web3-eth.html#getaccounts|web3.eth.getAccounts()}
 * ).
 * @returns {string} Ethereum-Adresse des Accounts.
 * @throws {Error} Wirft einen Fehler, falls unerwarteterweise nicht genau ein
 * Account zur Verfügung steht.
 */
async function getAccount() {
  const accounts = await window.web3.eth.getAccounts();
  if (accounts.length != 1)
    throw new Error("Expected one account");
  return accounts[0];
}

/**
 * Lädt per
 * {@link https://developer.mozilla.org/en-US/docs/Web/API/WindowOrWorkerGlobalScope/fetch|fetch()}
 * die Datei "game_sol_RockPaperScissors.abi" vom Server und gibt ihren Inhalt
 * zurück.
 * @returns {Promise} JSON-ABI-Beschreibung des Vertrags; ist ein
 * {@link https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Promise|Promise},
 * das zu einem String erfüllt wird.
 */
async function fetchABI() {
  const response = await fetch(`game_sol_RockPaperScissors.abi`);
  return response.text();
}

/**
 * Lädt per
 * {@link https://developer.mozilla.org/en-US/docs/Web/API/WindowOrWorkerGlobalScope/fetch|fetch}
 * die Datei "game_sol_RockPaperScissors.bin" vom Server und gibt ihren Inhalt
 * zurück.
 * @returns {Promise} Code des Vertrags; ist ein
 * {@link https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Promise|Promise},
 * das zu einem String erfüllt wird.
 */
async function fetchContract() {
  const response = await fetch(`game_sol_RockPaperScissors.bin`);
  return response.text();
}

/**
 * Nutzt {@link https://github.com/Caligatio/jsSHA|jsSHA}, um den SHA-256-Hash
 * des übergebenen Parameters zu berechnen. jsSHA muss zur Verfügung stehen,
 * etwa indem {@link https://unpkg.com/jssha} geladen wird.
 * @param {string} string - Zeichenkette, deren SHA-256-Hash berechnet werden
 * soll.
 * @returns {string} SHA-256-Hash
 */
function sha256(string) {
  const sha = new jsSHA("SHA-256", "TEXT");
  sha.update(string);
  return sha.getHash("HEX");
}

/*
 * Verstecke alle zu versteckenden Element, sobald die Seite fertig geladen ist.
 */
domReady(hideAll);
