// SPDX-License-Identifier: CC0-1.0
pragma solidity >=0.7 <0.9;

/**
 * @title Ein einfaches Schere-Stein-Papier-Spiel
 * @author Lars Hupel
 * @author Sylvester Tremmel
 * @notice Dieser Vertrag ist ein Beispielprojekt, um Solidity zu lernen. Er
 *         nicht zur produktiven Nutzung gedacht und ENTHÄLT BEKANNTE PROBLEME!
 *         Für weitere Informationen siehe
 *         https://www.heise.de/select/ct/2021/12/seite-150 und
 *         https://www.heise.de/select/ct/2021/19/seite-136 .
 */
contract RockPaperScissors {
    // Adressen der beiden Spieler
    address payable player1;
    address payable player2;

    // Zeitpunkt des Deployments
    uint start;

    // Einsatz von Spieler 2 verbucht
    bool paymentComplete;

    // Spieler-Commitments (SHA-256-Hashes von [choice, secret_string])
    bytes32 comm1;
    bytes32 comm2;

    // Wahlen der Spieler (1: Schere, 2: Stein, 3: Papier)
    uint8 reveal1;
    uint8 reveal2;

    /**
     * @dev Erstellt ein neues Schere-Stein-Papier-Spiel. Spieler 1 ist der
     *      Ersteller des Vertrages, die Adresse von Spieler 2 wird übergeben.
     * @param _player2 Adresse des zweiten Spielers
     */
    constructor(address payable _player2) payable {
        require(_player2 != msg.sender, "Spieler muessen verschieden sein");
        player1 = payable(msg.sender);
        player2 = _player2;
        start = block.timestamp;
    }

    /**
     * @dev Verbucht den Einsatz von Spieler 2. Der mit diesem Funktionsaufruf
     *      überwiesene Betrag muss exakt dem Betrag entsprechen, den Spieler 1
     *      beim Erstellen des Vertrags überwiesen hat.
     */
    function pay() public payable {
        require(msg.sender == player2, "Ungueltiger Spieler");
        require(!paymentComplete, "Bereits gezahlt.");
        require((msg.value * 2) == address(this).balance, "Falscher Betrag");
        paymentComplete = true;
    }

    /**
     * @dev Ein Commitment festlegen; kann nur von Spieler 1 oder 2 aufgerufen
     *      werden, kann nur jeweils einmal aufgerufen werden und kann erst
     *      aufgerufen werden, wenn auch Spieler 2 Geld gesetzt hat (siehe
     *      pay()).
     * @param comm 32 Bytes Commitment, die den SHA-256-Hash aus Spielerwahl
     *             (einer der Strings "1", "2" oder "3") konkateniert mit einem
     *             geheim gewählten String darstellen sollen.
     */
    function commit(bytes32 comm) public {
        require(
            msg.sender == player1 || msg.sender == player2,
            "Ungueltiger Spieler"
        );
        require(comm != bytes32(0), "Null-Commits sind ungueltig"); // Leere
            // Hashes werden genutzt, um anzuzeigen, dass noch kein Commitment
            // vorliegt.
        require(paymentComplete, "Zweiter Einsatz ausstehend");

        if (msg.sender == player1) {
            require(comm1 == bytes32(0), "Commit schon vorhanden");
            comm1 = comm;
        } else {
            require(comm2 == bytes32(0), "Commit schon vorhanden");
            comm2 = comm;
        }
    }

    /**
     * @dev Ein Commitment aufdecken; kann nur von Spieler 1 oder 2 aufgerufen
     *      werden, kann nur jeweils einmal aufgerufen werden und kann erst
     *      aufgerufen werden, wenn beide Spieler bereits ein Commitment
     *      abgegeben haben (siehe commit()).
     * @param choice Die beim Commitment getroffene Wahl (1, 2 oder 3). Eine
     *               abweichende Wahl wird nicht akzeptiert. Wenn beim
     *               Commitment eine Zahl <1 oder >3 gewählt wurde, kann nicht
     *               aufgedeckt werden.
     * @param secret Der beim Commitment gewählte geheime String. Ein
     *               abweichender String wird nicht akzeptiert. Wenn der String
     *               nicht mehr bekannt ist, kann nicht aufgedeckt werden.
     */
    function reveal(uint8 choice, string calldata secret) public {
        require(
            msg.sender == player1 || msg.sender == player2,
            "Ungueltiger Spieler"
        );
        require(
            comm1 != bytes32(0) && comm2 != bytes32(0),
            "Commits noch ausstehend"
        );
        require(
            choice == 1 || choice == 2 || choice == 3,
            "Ungueltige Wahl"
        );

        // encodePacked() und die Konvertierung von Choice nach String sind
        // nötig, damit die gleichen SHA-256-Hashes entstehen, wie bei einer
        // einfachen Berechnung in einer Kommandozeile.
        bytes32 expected = sha256(
            abi.encodePacked(
                choice2str(choice),
                secret
            )
        );

        if (msg.sender == player1) {
            require(expected == comm1, "Reveal ungueltig.");
            require(reveal1 == 0, "Reveal bereits vorhanden.");
            reveal1 = choice;
        } else {
            require(expected == comm2, "Reveal ungueltig.");
            require(reveal2 == 0, "Reveal bereits vorhanden.");
            reveal2 = choice;
        }
    }

    /**
     * @dev Den Spielgewinner berechnen und den Spieleinsatz an ihn oder sie
     *      auszahlen. Beide Spieler müssen ihr Commitment aufgedeckt haben
     *      (siehe reveal()) damit finish() ausgeführt wird. Bei einem Patt
     *      (siehe getWinner()) wird der Vertrag zurückgesetzt (siehe
     *      reset()), sodass beide Spieler erneut committen können.
     */
    function finish() public {
        require(reveal1 > 0 && reveal2 > 0, "Reveals noch ausstehend");

        address payable winner = getWinner();
        if (winner != payable(0))
            selfdestruct(winner);
        else {
            reset();
        }
    }

    /**
     * @dev Zahlt an den jeweils anderen Spieler aus, falls einer der beiden
     *      Spieler säumig ist. Falls beide Spieler mit dem Commiten (siehe
     *      commit()) oder mit dem Aufdecken ihres Commitments (siehe reveal())
     *      säumig sind, wird der Vertrag zurückgesetzt (siehe reset()), sodass
     *      erneut gespielt werden kann. Im ersten Fall ist das fast ein NOOP,
     *      nur der Startzeitpunkt ändert sich. Wichtig ist der zweite Fall, der
     *      erlaubt, den Vertrag zurückzusetzen, wenn beide Spieler Fehler
     *      gemacht haben und nicht revealen können. Spieler werden als säumig
     *      betrachtet, wenn Startzeitpunkt des Vertrages muss mindestens eine
     *      Woche in der Vergangenheit liegt. Falls abort() zu früh aufgerufen
     *      wird bricht die Funktion ab. Falls das Spiel bereits beendet ist,
     *      hat die Funktion keinen Effekt.
     */
    function abort() public {
        require(
            msg.sender == player1 || msg.sender == player2,
            "Ungueltiger Spieler"
        );
        require(
            block.timestamp > start + 1 weeks,
            "Frist noch nicht vorbei"
        );

        if (
            (!paymentComplete) ||
            ((comm1 != bytes32(0)) && (comm2 == bytes32(0))) ||
            ((reveal1 > 0) && (reveal2 == 0))
        ) {
            // Spieler 2 hat kein Geld gesetzt, oder trödelt mit dem Commitment 
            // oder dem Reveal, obwohl Spieler 1 bereits committet bzw. revealt
            // hat.
            selfdestruct(player1);
        }

        if (
            ((comm1 == bytes32(0)) && (comm2 != bytes32(0))) ||
            ((reveal1 == 0) && (reveal2 > 0))
        ) {
            // Spieler 1 trödelt mit dem Commitment oder dem Reveal, obwohl
            // Spieler 1 bereits committet bzw. revealt hat.
            selfdestruct(player2);
        }
        
        if ((reveal1 == 0) && (reveal2 == 0)) {
            // Beide spieler trödeln mit dem reveal (oder können nicht revealen,
            // weil sie beide ungültige Commitments abgegeben oder ihren
            // geheimen String vergessen haben).
            reset();
        }
    }

    /**
     * @dev Konvertiert die übergebene Ganzzahl in einen String. Die Zahlen 1
     *      und 2 werden in "1" und "2" konvertiert, alle anderen Zahlen in den
     *      String "3".
     * @param choice Die zu konvertierende Zahl
     * @return Der konvertierte String ("1", "2" oder "3")
     */
    function choice2str(uint8 choice) private pure returns (string memory) {
        if (choice == 1) return "1";
        if (choice == 2) return "2";
        return "3";
    }

    /**
     * @dev Ermittelt den Spielgewinner. Erfordert, dass beide Spieler eine
     *      Wahl abgegeben haben. (Dies wird in finish() geprüft.)
     * @return Die Adresse des Gewinners (der Wert von player1 oder player2)
     *         oder 0, wenn kein Gewinner ermittelt werden kann
     */
    function getWinner() private view returns (address payable) {
        // Gewinner nach den Regeln ermittlen
        if (
            (reveal1 == 1 && reveal2 == 2) || // Schere vs. Stein
            (reveal1 == 2 && reveal2 == 3) || // Stein vs. Papier
            (reveal1 == 3 && reveal2 == 1)    // Papier vs. Schere
        )
            return player2;
        else if (reveal1 == reveal2) // Patt
            return payable(0);
        else
            return player1; // Alle anderen Fälle
    }
    
    /**
     * @dev Setzt Reveals und Commitments zurück und den Startzeitpunkt
     *      auf den aktuellen Block, sodass das Spiel erneut gespielt
     *      werden kann.
     */
    function reset() private {
        start = block.timestamp;
        comm1 = bytes32(0);
        comm2 = bytes32(0);
        reveal1 = 0;
        reveal2 = 0;
    }
}

