/**
 * @overview JavaScript-Code für play.html. Implementiert alle Funktionen, die
 * zum Spielen mit dem Vertrag nötig sind:
 * {@link selectContract|selectContract()},
 * {@link payStake|payStake()},
 * {@link commit|commit()},
 * {@link reveal|reveal()},
 * {@link finish|finish()} und
 * eine anonyme Hilfsfunktion, die hier als
 * {@link getContractAdressFromURL|getContractAdressFromURL()} dokumentiert ist.
 * @author Lars Hupel
 * @author Sylvester Tremmel
 */

domReady(
  /**
   * Anonyme Funktion (hier mit einem Alias dokumentiert), die ausgeführt wird,
   * sobald das Dokument geladen ist (siehe {@link domReady}). Falls ein
   * URL-Parameter "addr" vorhanden ist, wird dessen Wert in das Feld für die
   * Vertragsauswahl (siehe {@link selectContract}) übernommen.
   * @alias getContractAdressFromURL
   */
  function() {
    const params = (new URL(document.location)).searchParams;
    const addr = params.get('addr');
    document.getElementById("contract").value = addr;
  }
);

/**
 * Wählt den Vertrag auf der Blockchain aus, mit dem gespielt werden soll. Der
 * Vertrag wird über die Adresse in dem Eingabefeld mit der ID "contract"
 * identifiziert. Per {@link fetchABI} wird die JSON-Repräsentation des
 * zugehörigen ABI geladen. Im Erfolgsfall wird das Element mit der ID
 * "slct-succ" angezeigt, im Fehlerfall "slct-fail" (siehe {@link show}).
 * @throws Wirft jeglichen Fehler, der von Web3js produziert wird.
 */
async function selectContract() {
  const address = document.getElementById("contract").value;

  try {
    // JSON-Repräsentation der ABI laden, interpretieren und zusammen mit der
    // Adresse eines passenden Vertrags ein Contract-Object erstellen.
    const abi = await fetchABI();
    /**
     * In Play.js genutztes {@link https://web3js.readthedocs.io/en/v1.4.0/web3-eth-contract.html|Vertragsobjekt}.
     * Wird von {@link selectContract} erstellt.
     * @global
     * @type object
     */
    window.contract = new window.web3.eth.Contract(JSON.parse(abi), address);
    
    // Erfolgsmeldung anzeigen
    show("slct-succ");
  }
  catch (err) {
    // Fehlermeldung anzeigen (und Fehler weiter eskalieren).
    show("slct-fail");
    throw err;
  }
}

/**
 * Erstellt und versendet eine Ethereum-Transaktion, mit der Spieler 2 seinen
 * Einsatz setzt. Absender der Transaktion (also Spieler 2) ist der aktuell in
 * MetaMask eingestellte Account, der Transaktionsbetrag (also Spieleinsatz) ist
 * {@link stake} aus commons.js. Vor Versand der Transaktion wird das Element
 * mit der ID "pay-pend" als Wartemeldung angezeigt (siehe {@link show}). Im
 * Erfolgsfall wird das Element mit der ID "pay-succ" angezeigt, sonst
 * "pay-fail".
 * @throws Wirft jeglichen Fehler, der von MetaMask oder Web3js produziert wird.
 */
async function payStake() {
  try {
    // Wartemeldung anzeigen.
    show("pay-pend");
    
    // Adresse des Spielers beziehen und Transaktion versenden. pay() ist
    // letzlich eine Methode des Vertrags (siehe game.sol), die Web3js hier als
    // JavaScript-Methode zur Verfügung stellt.
    const player = await getAccount();
    await contract.methods.pay().send({
      from: player,
      value: stake
    });

    // Erfolgsmeldung anzeigen
    show("pay-succ");
  }
  catch (err) {
    // Fehlermeldung anzeigen (und Fehler weiter eskalieren).
    show("pay-fail");
    throw err;
  }
}

/**
 * Erstellt und versendet eine Ethereum-Transaktion, mit der beide Spieler ihren
 * jeweiligen Spielzug festlegen (ohne ihn zu verraten) – das sogenannte
 * Commitment. Dazu berechnet diese Funktion aus dem Spielzug (der Wert des
 * Dropdown-Feldes mit der ID "choice") und einem nur dem jeweiligen Spieler
 * bekannten Kennwort (der Wert des Eingabefeldes mit der ID "secret") einen
 * SHA-256-Hash (siehe {@link sha256}) und übermittelt diesen an den Vertrag.
 * Absender und Informationsmeldungen werden wie bei {@link payStake}
 * gehandhabt.
 * @throws Wirft jeglichen Fehler, der von MetaMask oder Web3js produziert wird.
 */
async function commit() {
  // Spielzug des Spielers (1, 2 oder 3 für Schere, Stein oder Papier)
  const choice = document.getElementById("choice").value;
  // Kennwort des Spielers
  const secret = document.getElementById("secret").value;

  try {
    // Wartemeldung anzeigen.
    show("comm-pend");

    // Adresse des Spielers beziehen ...
    const player = await getAccount();
    // ... Hash berechnen ...
    const hash = sha256(choice + secret);
    // ... und Transaktion versenden. commit() ist letzlich eine Methode des
    // Vertrags (siehe game.sol), die Web3js hier als JavaScript-Methode zur
    // Verfügung stellt.
    await contract.methods.commit(`0x${hash}`).send({
      from: player,
      value: 0
    });

    // Erfolgsmeldung anzeigen
    show("comm-succ");
  }
  catch (err) {
    // Fehlermeldung anzeigen (und Fehler weiter eskalieren).
    show("comm-fail");
    throw err;
  }
}

/**
 * Erstellt und versendet eine Ethereum-Transaktion, mit der beide Spieler ihren
 * jeweiligen Spielzug offenbaren und validieren. Dazu übermittelt diese
 * Funktion den Spielzug (der Wert des Dropdown-Feldes mit der ID "choice") und
 * das nur dem jeweiligen Spieler bekannte Kennwort (der Wert des Eingabefeldes
 * mit der ID "secret") an den Vertrag. (Der Vertrag akzeptiert nur Werte, aus
 * denen sich ein zum {@link commit} passender Wert ergibt.) Absender und
 * Informationsmeldungen werden wie bei {@link payStake} gehandhabt.
 * @throws Wirft jeglichen Fehler, der von MetaMask oder Web3js produziert wird.
 */
async function reveal() {
  const choice = document.getElementById("choice").value;
  const secret = document.getElementById("secret").value;

  try {
    // Wartemeldung anzeigen.
    show("revl-pend");
    
    // Adresse des Spielers beziehen und Transaktion versenden. reveal() ist
    // letzlich eine Methode des Vertrags (siehe game.sol), die Web3js hier als
    // JavaScript-Methode zur Verfügung stellt.
    const player = await getAccount();
    await contract.methods.reveal(+choice, secret).send({
      from: player,
      value: 0
    });

    // Erfolgsmeldung anzeigen
    show("revl-succ");
  }
  catch (err) {
    // Fehlermeldung anzeigen (und Fehler weiter eskalieren).
    show("revl-fail");
    throw err;
  }
}

/**
 * Erstellt und versendet eine Ethereum-Transaktion, mit der jeder der beiden
 * Spieler das Spiel beenden können. (Der Vertrag zerstört sich daraufhin selbst
 * und zahlt die Einsätze an den Gewinner aus.) Absender und
 * Informationsmeldungen werden wie bei {@link payStake} gehandhabt.
 * @throws Wirft jeglichen Fehler, der von MetaMask oder Web3js produziert wird.
 */
async function finish() {
  try {
    // Wartemeldung anzeigen.
    show("fnsh-pend");

    // Adresse des Spielers beziehen und Transaktion versenden. finish() ist
    // letzlich eine Methode des Vertrags (siehe game.sol), die Web3js hier als
    // JavaScript-Methode zur Verfügung stellt.
    const player = await getAccount();
    await contract.methods.finish().send({
      from: player,
      value: 0
    });

    // Erfolgsmeldung anzeigen
    show("fnsh-succ");
  }
  catch (err) {
    // Fehlermeldung anzeigen (und Fehler weiter eskalieren).
    show("fnsh-fail");
    throw err;
  }
}
