﻿<#PSScriptInfo

.VERSION 1.0.0

.GUID 022030bf-5917-426d-b98c-baced5933949

.AUTHOR Hajo Schulz <hos@ct.de>

.COMPANYNAME c't Magazin für Computertechnik

.COPYRIGHT Copyright © 2023 Heise Medien GmbH & Co. KG / c't

.PROJECTURI https://ct.de/yesq

#>

<#
.SYNOPSIS
   Kopiert die lokale AppLocker-Policy in den MDM Bridge WMI Provider
.DESCRIPTION
   Das Skript kopiert die lokale AppLocker-Policy in den MDM Bridge WMI 
   Provider. Für einen sinnvollen Betrieb ist unbedingt die Lektüre des 
   Artikels "An der kurzen Leine" in c't 20/2023, S. 70, erforderlich!
.EXAMPLE
   C:\PS> AppLockerPolicy2MDM
.INPUTS
   None
.OUTPUTS
   None
#>

[CmdletBinding()]
param (
  [Parameter(Mandatory=$false)]
  [ValidateNotNullOrEmpty()]
  [string] $FromFile,
  [Parameter()]
  [switch] $KillAll
)

function IsWindowsHome() {
  $prodName = (Get-ItemProperty "HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion").ProductName
  return $prodName -match "Home"
}

function IsSystem() {
  if($ExecutionContext.SessionState.LanguageMode -ne [System.Management.Automation.PSLanguageMode]::FullLanguage) {
    return $false
  }
  $identity = [System.Security.Principal.WindowsIdentity]::GetCurrent()
  return $identity.IsSystem
}

if(-not (IsWindowsHome)) {
  Write-Warning 'Dieses Skript ist nur für Windows Home vorgesehen.'
  Write-Host 'Wenn Sie sicher sind, was Sie tun, geben Sie bitte ' -NoNewline
  Write-Host 'yes' -ForegroundColor DarkGreen -BackgroundColor White -NoNewline
  Write-Host ' ein: ' -NoNewline
  $answer = Read-Host
  if($answer -ne 'yes') {
    return;
  }
}

if(IsSystem) {
  Push-Location (Split-Path $PSCommandPath)
}
else {
  Write-Warning "Dieses Skript muss mit Systemrechten laufen. Versuche Neustart ..."
  $pse = Get-Command .\psexec -ErrorAction SilentlyContinue
  if(!$pse) {
    $pse = Get-Command psexec -ErrorAction SilentlyContinue
    if(!$pse) {
      Write-Warning "Der Start mit Systemrechten ist fehlgeschlagen, weil das dafür notwendige Sysinternals-Programm psexec nicht gefunden wurde."
      return
    }
  }
  # Reconstruct my arguments
  $scriptArgs = ''
  if($FromFile) {
    $scriptArgs += "-FromFile `"$FromFile`" "
  }
  if($KillAll) {
    $scriptArgs += "-KillAll "
  }
  # Run myself as admin
  Start-Process $pse.Source "-accepteula -d -i -s `"$((Get-Process -PID $PID).Path)`" -NoProfile -EP Bypass -NoExit -File `"$PSCommandPath`" $scriptArgs" -Verb RunAs
  return
}

$namespaceName = "root\cimv2\mdm\dmmap" #Do not change this
$GroupName = "ctAppLocker" #You can use your own Groupname, don't use special charaters or wite space
$parentID = "./Vendor/MSFT/AppLocker/ApplicationLaunchRestrictions/$GroupName"

if($FromFile) {
  $xml = [Xml](Get-Content $FromFile)
}
elseif ($KillAll) {
  $xml = [xml]@"
<AppLockerPolicy Version="1">
  <RuleCollection Type="Appx" EnforcementMode="NotConfigured" />
  <RuleCollection Type="Dll" EnforcementMode="NotConfigured" />
  <RuleCollection Type="Exe" EnforcementMode="NotConfigured" />
  <RuleCollection Type="Msi" EnforcementMode="NotConfigured" />
  <RuleCollection Type="Script" EnforcementMode="NotConfigured" />
</AppLockerPolicy>
"@
}
else {
  $xml = [Xml](Get-AppLockerPolicy -Local -Xml)
}
foreach($rc in $xml.SelectNodes('//RuleCollection')) {
  Write-Host "Verarbeite $($rc.Type)-Regeln ..."
  switch($rc.Type) {
    'Appx'   { $className = 'MDM_AppLocker_ApplicationLaunchRestrictions01_StoreApps03'; $instanceID = 'StoreApps'; break }
    'Dll'    { $className = 'MDM_AppLocker_DLL03'; $instanceID = 'Dll'; break }
    'Exe'    { $className = 'MDM_AppLocker_ApplicationLaunchRestrictions01_EXE03'; $instanceID = 'Exe'; break }
    'Msi'    { $className = 'MDM_AppLocker_MSI03'; $instanceID = 'Msi'; break }
    'Script' { $className = 'MDM_AppLocker_Script03'; $instanceID = 'Script'; break }
    Default  { Write-Error "$($rc.Type): Unbekannte Regelkategorie"; return }
  }
  $existing = Get-CimInstance -Namespace $namespaceName -ClassName $className -Filter "ParentID='$parentID'"
  if($rc.ChildNodes.Count -gt 0) {
    $ruleXml = [System.Net.WebUtility]::HtmlEncode($rc.OuterXml)
    if($existing) {
      $existing | Set-CimInstance -Property @{InstanceID=$instanceID; Policy=$ruleXml}
    }
    else {
      New-CimInstance -Namespace $namespaceName -ClassName $className -Property @{ParentID=$parentID; InstanceID=$instanceID; Policy=$ruleXml} | Out-Null
    }
  }
  else {
    if($existing) {
      $existing | Remove-CimInstance
    }
  }
}
Write-Host "Fertig. Sie können dieses Fenster jetzt schließen."

