﻿<#PSScriptInfo

.VERSION 1.0.0

.GUID e6479933-108f-46c1-a466-efc0fe66ba7e

.AUTHOR Hajo Schulz <hos@ct.de>

.COMPANYNAME c't Magazin für Computertechnik

.COPYRIGHT Copyright © 2023 Heise Medien GmbH & Co. KG / c't

.PROJECTURI https://ct.de/yesq

#>

<#
.SYNOPSIS
   Prüft AppLocker-Regeln auf Lücken durch beschreibbare Ordner
.DESCRIPTION
   Das Skript analysiert die lokale AppLocker-Konfiguration. Es prüft, ob in 
   Pfadregeln angesprochene Ordner Unterverzeichnisse enthalten, in die man 
   ohne Administratorrechte schreiben. Fundstellen trägt es auf Wunsch als
   Ausnahmen in die betroffenen Regeln ein.
.EXAMPLE
   C:\PS> Check-AppLockerRules
.INPUTS
   None
.OUTPUTS
   None
#>

[CmdletBinding()]
param (
  [switch] $FullAuto
)

function IsAdmin() {
  $identity = [System.Security.Principal.WindowsIdentity]::GetCurrent()
  $princ = New-Object System.Security.Principal.WindowsPrincipal($identity)
  return ($princ.IsInRole([System.Security.Principal.WindowsBuiltInRole]::Administrator))
}

if($ExecutionContext.SessionState.LanguageMode -ne [System.Management.Automation.PSLanguageMode]::FullLanguage) {
  Write-Warning "Skript-Ausführung fehlgeschlagen."
  Write-Host "Bitte prüfen Sie, ob dieses Skript durch Ihre AppLocker-Regeln erfasst ist"
  Write-Host "und/oder starten Sie es mit Administratorrechten neu."
  return
}

if(IsAdmin) {
  Push-Location (Split-Path $PSCommandPath)
}
else {
  # Reconstruct my arguments
  $scriptArgs = ''
  if($FullAuto) {
    $scriptArgs += "-FullAuto "
  }
  # Run myself as admin
  Start-Process $((Get-Process -PID $PID).Path) "-NoProfile -EP Bypass -NoExit -File `"$PSCommandPath`" $scriptArgs" -Verb RunAs
  exit
}

$source = @"
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.IO;
using System.Linq;
using System.Security.AccessControl;
using System.Security.Principal;

public class AppLockerHelper
{
    private int screenWidth;

    private HashSet<SecurityIdentifier> _nonAdminSids = null;
    private HashSet<SecurityIdentifier> NonAdminSids
    {
        get
        {
            if (_nonAdminSids == null)
            {
                _nonAdminSids = new HashSet<SecurityIdentifier>();
                _nonAdminSids.Add(WindowsIdentity.GetCurrent().User);
                foreach (var group in WindowsIdentity.GetCurrent().Groups)
                {
                    if (group.Value != "S-1-5-114" // NT-AUTORITÄT\Lokales Konto und Mitglied der Gruppe "Administratoren"
                        && group.Value != "S-1-5-32-544" // VORDEFINIERT\Administratoren
                    )
                    {
                        _nonAdminSids.Add((SecurityIdentifier)group.Translate(typeof(SecurityIdentifier)));
                    }
                }
            }
            return _nonAdminSids;
        }
    }

    private bool IsFolderNonAdminWriteable(DirectoryInfo dir)
    {
        DirectorySecurity acl;
        try
        {
            try
            {
                acl = dir.GetAccessControl(AccessControlSections.Access);
            }
            catch (Exception ex)
            {
                if (ex is ArgumentException || ex is PathTooLongException)
                {
                    if (dir.FullName.Length > 248)
                    {
                        dir = new DirectoryInfo(@"\\?\" + dir.FullName);
                        acl = dir.GetAccessControl(AccessControlSections.Access);
                    }
                    else
                    {
                        throw;
                    }
                }
                else
                {
                    throw;
                }
            }
        }
        catch (UnauthorizedAccessException)
        {
            // HACK: Wenn wir noch nicht mal als Admin die Rechte lesen dürfen, wird wohl auch kein Benutzer schreiben können.
            return false;
        }
        catch (Exception/* ex*/)
        {
            // Console.WriteLine("\r\n\r\n" + dir.FullName);
            // Console.WriteLine(ex.ToString() + "\r\n\r\n");
            // throw;
            return false; // Wir haben unser Bestes versucht.
        }
        AuthorizationRuleCollection rules = acl.GetAccessRules(true, true, typeof(SecurityIdentifier));
        foreach (AuthorizationRule rule in rules)
        {
            if (NonAdminSids.Contains(rule.IdentityReference as SecurityIdentifier))
            {
                if ((((FileSystemAccessRule)rule).FileSystemRights & FileSystemRights.WriteData) != 0)
                {
                    return true;
                }
            }
        }
        return false;
    }

    private IEnumerable<DirectoryInfo> WriteableSubDirs(DirectoryInfo dir, int depth, bool reportProgress)
    {
        if (IsFolderNonAdminWriteable(dir))
        {
            yield return dir;
        }
        else
        {
            DirectoryInfo[] subDirs;
            try
            {
                subDirs = dir.GetDirectories();
            }
            catch (UnauthorizedAccessException)
            {
                // HACK: Wenn wir noch nicht mal als Admin die Unterordner lesen dürfen, wird wohl auch kein Benutzer schreiben können.
                yield break;
            }
            catch (IOException)
            {
                // dir existiert nicht, ungültiger Name, etc.
                yield break;
            }
            foreach (DirectoryInfo subFolder in subDirs)
            {
                if (reportProgress && depth < 2)
                {
                    LogProgress(subFolder.FullName);
                }
                foreach (DirectoryInfo writeable in WriteableSubDirs(subFolder, depth + 1, reportProgress))
                {
                    yield return writeable;
                }
            }
        }
    }

    private void LogProgress(String currentFolder)
    {
        String line = String.Format("Prüfe {0} ...", currentFolder);
        if (line.Length > screenWidth)
            line = line.Substring(0, screenWidth);
        Console.Write(line + "\x1b[0K\x1b[1G");
    }

    public AppLockerHelper()
    {
        this.screenWidth = 80;
    }

    public AppLockerHelper(int screenWidth)
    {
        this.screenWidth = screenWidth;
    }

    public List<DirectoryInfo> FindWriteableFolders(DirectoryInfo root, bool reportProgress = true)
    {
        var result = WriteableSubDirs(root, 0, reportProgress).ToList();
        if(reportProgress)
            Console.WriteLine("\x1b[0K");
        return result;
    }
}
"@

Add-Type -TypeDefinition $source
$helper = New-Object AppLockerHelper($Host.UI.RawUI.WindowSize.Width)

$PATH_VARS = @{
  '%WINDIR%' = @($env:SystemRoot);
  '%SYSTEM32%' = @($($env:SystemRoot + '\System32'), $($env:SystemRoot + '\SysWOW64'));
  '%OSDRIVE%' = @($env:SystemDrive);
  '%PROGRAMFILES%' = @($env:ProgramFiles, ${env:ProgramFiles(x86)});
  '%REMOVABLE%' = @();
  '%HOT%' = @()
}

$KNOWN_PATHS = @{
  $($env:SystemRoot + '\System32') = '%SYSTEM32%';
  $($env:SystemRoot + '\SysWOW64') = '%SYSTEM32%';
  $env:SystemRoot = '%WINDIR%';
  ${env:ProgramFiles(x86)} = '%PROGRAMFILES%';
  $env:ProgramFiles = '%PROGRAMFILES%';
  $env:SystemDrive = '%OSDRIVE%';
}

function ExpandPathVar($str) {
  foreach($var in $PATH_VARS.Keys) {
    if($str.StartsWith($var)) {
      $result = @()
      foreach($path in $PATH_VARS[$var]) {
        $result += "$path$($str.Substring($var.Length))"
      }
      return $result
    }
  }
  return @($str)
}

function ApplyPathVar($str) {
  foreach($path in $KNOWN_PATHS.Keys|Sort-Object -Property Length -Descending) {
    if($str.StartsWith($path)) {
      return "$($KNOWN_PATHS[$path])$($str.Substring($path.Length))"
    }
  }
  return $str
}

function Make-Choice($options, $message = 'Bitte wählen Sie {0}: ') {
  $optStr = ''
  $options = $options.ToUpper()
  for($i = 0; $i -lt $options.Length - 2; $i++) {
    $optStr += ('{0}, ' -f $options[$i])
  }
  $optStr += ('{0} oder ' -f $options[$options.Length - 2])
  $optStr += ('{0}' -f $options[$options.Length - 1])
  Write-Host ($message -f $optStr) -NoNewline
  if($psISE) {
    $choice = [Char]::ToUpper((Read-Host)[0])
    while(-not $options.Contains($choice)) {
      [Console]::Beep(1000,250)
      Write-Host ($message -f $optStr) -NoNewline
      $choice = Read-Host
      if($choice) {
        $choice = [Char]::ToUpper($choice[0])
      } else {
        $choice = [Char]1
      }
    }
  } else {
    $choice = [Char]::ToUpper($host.ui.RawUI.ReadKey("NoEcho,IncludeKeyDown").Character)
    while(-not $options.Contains($choice)) {
      [Console]::Beep(1000,250)
      $choice = [Char]::ToUpper($host.ui.RawUI.ReadKey("NoEcho,IncludeKeyDown").Character)
    }
    Write-Host $choice
  }
  return $choice
}

$pathCache = @{}
$changes = 0
$policy = Get-AppLockerPolicy -Local
foreach($coll in $policy.RuleCollections) {
  Write-Host ("`nVerarbeite {0}-Regeln ..." -f $coll.RuleCollectionType)
  :rules foreach($rule in $coll) {
    if($rule.GetType() -eq [Microsoft.Security.ApplicationId.PolicyManagement.PolicyModel.FilePathRule]) {
      if($rule.Action -eq 'Allow') {
        $oldExceptions = @()
        foreach($ex in $rule.PathExceptions) {
          $oldExceptions += $ex.Path.Path
        }
        $newExceptions = @()
        foreach($cond in $rule.PathConditions) {
          $path = $cond.Path.Path
          if($path -ne '*') {
            Write-Host ('- Überprüfe {0} in "{1}" ...' -f $path, $rule.Name)
            if($path.EndsWith('\*')) {
              $path = $path.SubString(0, $path.Length - 1)
            }
            foreach($osPath in (ExpandPathVar $path)) {
              Write-Host ('  - Scanne {0} ...' -f $osPath)
              $holes = $null
              if($pathCache.Contains($osPath)) {
                $holes = $pathCache[$osPath]
              }
              else {
                $dirInfo = Get-Item $osPath -ErrorAction SilentlyContinue
                if($dirInfo -and ($dirInfo -is [System.IO.DirectoryInfo])) {
                  $holes = $helper.FindWriteableFolders($dirInfo, !$psISE)
                  $pathCache[$osPath] = $holes
                }
              }
              foreach($hole in $holes) {
                if($hole -eq $dirInfo) {
                  Write-Warning ('Der Ordner {0} ist beschreibbar!' -f $hole.FullName)
                }
                else {
                  $holePath = (ApplyPathVar $hole.FullName) + '\*'
                  if(-not $newExceptions.Contains($holePath)) {
                    $newExceptions += $holePath
                  }
                }
              }
            }
          }
        }
        if($newExceptions) {
          Write-Host ("  Der Pfad für die Regel `"{0}`" enthält folgende beschreibbaren Unterordner:" -f $rule.Name)
          foreach($ex in $newExceptions) {
            if($oldExceptions.Contains($ex)) {
              $mark = '(Ausnahme schon vorhanden)'
            } else {
              $mark = ''
            }
            Write-Host ("  {0} {1}" -f $ex, $mark)
          }
          $confirm = $false
          if(-not $FullAuto) {
            Write-Host "`n  Wie wollen Sie vorgehen?"
            Write-Host "  [A]lle als Ausnahme übernehmen"
            Write-Host "  [N]ichts ändern"
            Write-Host "  [E]inzeln bestätigen"
            Write-Host "  [V]ollautomatik für diese und weitere Regeln"
            $answer = Make-Choice "ANEV" '  Bitte wählen Sie {0}: '
            switch($answer) {
              'A' { $confirm = $false }
              'N' { continue rules }
              'E' { $confirm = $true }
              'V' { $FullAuto = $true }
            }
          }
          foreach($ex in $newExceptions) {
            if($oldExceptions.Contains($ex)) {
              if($confirm) {
                $question = '  Ausnahme für {0} beibehalten (J/N)? ' -f $ex
                $answer = Make-Choice "JN" $question
                if($answer -eq 'N') {
                  foreach($r in $rule.PathExceptions) {
                    if($r.Path.Path -eq $ex) {
                      $success = $rule.PathExceptions.Remove($r)
                      if($success) {
                        $changes += 1
                      }
                      break
                    }
                  }
                }
              }
            } else {
              $answer = 'J'
              if($confirm) {
                $question = '  Ausnahme für {0} eintragen (J/N)? ' -f $ex
                $answer = Make-Choice "JN" $question
              }
              if($answer -eq 'J') {
                $c = New-Object -TypeName Microsoft.Security.ApplicationId.PolicyManagement.PolicyModel.FilePathCondition -ArgumentList $ex
                $rule.PathExceptions.Add($c)
                $changes += 1
              }
            }
          }
          if($confirm) {
            foreach($ex in $oldExceptions) {
              if(!$newExceptions.Contains($ex)) {
                $question = '  Ordner {0} nicht gefunden. Ausnahme trotzdem beibehalten (J/N)? ' -f $ex
                $answer = Make-Choice "JN" $question
                if($answer -eq 'N') {
                  foreach($r in $rule.PathExceptions) {
                    if($r.Path.Path -eq $ex) {
                      $success = $rule.PathExceptions.Remove($r)
                      if($success) {
                        $changes += 1
                      }
                      break
                    }
                  }
                }
              }
            }
          }
        }
      }
    }
  }
}
if($changes -gt 0) {
  Set-AppLockerPolicy -PolicyObject $policy
  Write-Host "Fertig."
  if( (Get-Process mmc) ) {
    Write-Host "(Bitte denken Sie daran, GpEdit.msc und/oder SecPol.msc neu zu starten, falls sie geöffnet sind.)"
  }
}
else {
  Write-Host "Nichts zu tun."
}
