#Requires -RunAsAdministrator
<#
.SYNOPSIS
  c't-RE-Info
.DESCRIPTION
  Lesen Sie unbedingt die Anleitung zum Skript in c't 18/2024!
.NOTES
  Version:        23
  Author:         Axel Vahldiek <axv@ct.de>
  Creation Date:  2024-07-15
#>

[Console]::OutputEncoding = [System.Text.Encoding]::UTF8

$Hinweis = @"

####################################################################
### Lesen Sie unbedingt die Anleitung zum Skript in c't 18/2024! ###
####################################################################
"@
$outputPath = Join-Path -Path $PSScriptRoot -ChildPath ($MyInvocation.MyCommand.Name.Replace('.ps1', '.txt'))

function Ausgabe {
	param([String]$Text = "")
	$Text | Out-File -FilePath $outputPath -Append -Encoding UTF8
	$Text | Write-Output
}


function ImageInfo {
    param (
        [string]$ImagePath
    )
    $ImageCount = (Get-WindowsImage -ImagePath $ImagePath).Count
    foreach ($ImageIndex in 1..$ImageCount) {
		    $ImageInfo = Get-WindowsImage -ImagePath $ImagePath -index $ImageIndex
        $ImageName = $ImageInfo.ImageName
        $ImageLanguages = ($ImageInfo.Languages -join ", ")
        $ImageCreated = $ImageInfo.CreatedTime.ToString('g')
        $ImageModified = $ImageInfo.ModifiedTime.ToString('g')
		    $ImageVersion = $ImageInfo.Version
        return @"
Image ${ImageIndex}: $ImageName $ImageLanguages
Erstellt: $ImageCreated
Zuletzt geaendert: $ImageModified
RE-Build: $ImageVersion
"@
    }
}

Clear-Host
Ausgabe $Hinweis

# Computername ermitteln
$PCname = "Analyse von $env:COMPUTERNAME (" + (Get-Date -Format "yyyy-MM-dd HH:mm:ss") +")"
Ausgabe
Ausgabe $PCname
Ausgabe

# Windows-Partition analysieren
$WinDriveLetter = $env:SystemDrive
$WinPart = Get-Partition -DriveLetter $WinDriveLetter.Trim(':')
$WinDisk = Get-Disk -Number $WinPart.DiskNumber
$WinVolume = Get-Volume -DriveLetter $WinDriveLetter.Trim(':')
$WinSize = [Math]::Round($WinVolume.Size / 1MB).ToString("N0")
$WinRemain = [math]::Round($WinVolume.SizeRemaining / 1MB).ToString("N0")
$WinLabel = $WinVolume.FileSystemLabel; if(-not $WinLabel) {$WinLabel = "<Kein Label>"}
$WinMajor = (Get-ItemProperty "HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion").CurrentMajorVersionNumber
$WinMinor = (Get-ItemProperty "HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion").CurrentMinorVersionNumber
$WinBuild = (Get-ItemProperty "HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion").CurrentBuildNumber
$WinPatch = (Get-ItemProperty "HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion").UBR
$WinStatus = @"
Windows liegt auf: Disk $($WinPart.DiskNumber) ($($WinDisk.PartitionStyle)), Partition $($WinPart.PartitionNumber) ($WinDriveletter).
Label: $Winlabel
Platz: $WinSize MByte, davon $WinRemain MByte frei.
Windows-Build $WinMajor.$WinMinor.$WinBuild.$WinPatch
"@


# RE analysieren
$REStatus = reagentc /info
if ($REStatus -match 'disable') {
	$REStatus = "RE ist ausgeschaltet"
}
elseif ($REStatus -match 'enable') {
	$RELocation = $REStatus -match 'GLOBALROOT'
	if($RELocation) {
		if($RELocation[0] -match 'harddisk(\d+)\\partition(\d+)') {
			$REDiskNo = $Matches[1]
			$REPartNo = $Matches[2]
		}
	}
	$REDisk = Get-Disk -Number $REDiskNo
	$REPart = Get-Partition -DiskNumber $REDiskNo -PartitionNumber $REPartNo
	$REVolume = $REPart | Get-Volume
	$RESize = [math]::Round($REPart.Size / 1MB).ToString("N0")
	$RERemain = [math]::Round($REVolume.SizeRemaining / 1MB).ToString("N0")
	$RELabel = $REVolume.FileSystemLabel; if(-not $RELabel) {$RELabel = "<Kein Label>"}
	$REID = if ($REPart.GptType -eq '{de94bba4-06d1-4d40-a16a-bfd50179d6ac}' -or $REPart.MbrType -eq 39) {"Ja"} else {"Nein"}
	$REFollowsWin = if ($WinDisk.DiskNumber -eq $REDiskNo -and $WinPart.PartitionNumber -eq $REPartNo - 1) {"Ja"} else {"Nein"}
	$REIsWin = if ($WinDisk.DiskNumber -eq $REDiskNo -and $WinPart.PartitionNumber -eq $REPartNo) {"Ja"} else {"Nein"}
	$WinREReg = (Get-ItemProperty -Path 'HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion' -Name 'WinREVersion' -ErrorAction SilentlyContinue).WinREVersion; if(-not $WinREReg) {$WinREReg = "Keine"}
	$REStatus = @"
RE ist eingeschaltet (enabled)

Verwendete Partition: Disk $REDiskNo ($($REDisk.PartitionStyle)), Partition $REPartNo
Label: $RELabel
Platz: $RESize MByte, davon $RERemain MByte frei
Hat Recovery-ID: $REID
Liegt direkt hinter Windows-Partition: $REFollowsWin
Ist Windows-Partition: $REIsWin
"@
	# WIM analysieren
	$WimFile = ($RELocation[0] -replace '.*:', '').Trim() + "\Winre.wim"
	$WimFileShort = $WimFile -replace '^.*?\\partition\d+(.*)', '$1'
	$WimFI = New-Object System.IO.FileInfo $WimFile
	if ($WimFI.Exists) {
		$WimFISize = [math]::Round($WimFI.Length / 1MB, 2)
		$WimStatus = @"
Windows verweist auf $WimFileShort
Datei ist vorhanden ($WimFISize MByte)
$(ImageInfo -Imagepath $WimFile)
"@
	} Else {
		$WimStatus = @"
Windows verweist auf $WimFileShort
Datei ist aber nicht vorhanden
"@
	}	
}

Ausgabe @"
#############################
### Status von Windows RE ###
#############################

$REStatus

$WimStatus

In der Registry hinterlegte RE-Version: $WinREReg

"@

Ausgabe @"
##########################
### Status von Windows ###
##########################

$WinStatus

"@


# Vorhandene Recovery-Partitionen suchen
Ausgabe "###################################"
Ausgabe "### Partitionen mit Recovery-ID ###"
Ausgabe "###################################"
$HitCount = 0
Get-Disk | ForEach-Object {
    $HitDisk = $_.Number
	  $HitLayout = $_.PartitionStyle
    Get-Partition -DiskNumber $HitDisk -ErrorAction SilentlyContinue | Where-Object {
		  $_.GptType -eq '{de94bba4-06d1-4d40-a16a-bfd50179d6ac}' -or $_.MbrType -eq 39
		} | ForEach-Object {
			$HitCount++
			$HitPart = $_.PartitionNumber
			$HitSize = [math]::Round($_.Size / 1MB).ToString("N0")
			$HitLabel = ($_ | Get-Volume | ForEach-Object {if ($_.FileSystemLabel) {$_.FileSystemLabel} else {"<Kein Label>"}})
			$HitFollowsWin = if ($WinDisk.DiskNumber -eq $HitDisk -and $WinPart.PartitionNumber -eq $HitPart - 1) {"Ja"} else {"Nein"}
			Ausgabe @"

Partition mit RE-ID: Disk $HitDisk ($HitLayout), Partition $HitPart
Label: $HitLabel
Platz: $HitSize MByte
Partition liegt direkt hinter Windows: $HitFollowsWin.

"@
		}
}
Ausgabe "Anzahl Partitionen mit Recovery-ID: $HitCount"
Ausgabe


Ausgabe "############################"
Ausgabe "### Vorhandene Winre.wim ###"
Ausgabe "############################"

$WimCount = 0
foreach ($Part in (Get-Partition)) {
	if (-not ($Part.gpttype -eq '{de94bba4-06d1-4d40-a16a-bfd50179d6ac}' -or $Part.mbrtype -eq 39))	{
		if ($Part.driveletter) {
			$LWB = $Part.driveletter		
			$Ordners = @(
				"${LWB}:\Recovery\",
				"${LWB}:\Windows\Recovery\",
				"${LWB}:\Windows\System32\Recovery\"
				)
			Foreach ($Ordner in $Ordners) {
				$REFiles = get-childitem -path $Ordner -force -Recurse -filter "Winre.wim" -ErrorAction SilentlyContinue | Select-Object -ExpandProperty FullName
				foreach ($REFile in $REFiles) {
					$WimCount++
					$Funddisk = $Part.DiskNumber
					$FundLayout = (Get-Disk -Number $FundDisk).PartitionStyle
					$Fundpart = $Part.PartitionNumber
					$Fundsize = [Math]::Round((Get-Item $REFile -force).Length / 1MB, 2)
					$FundFollowsWin = if ($WinDisk.DiskNumber -eq $FundDisk -and $WinPart.PartitionNumber -eq $FundPart - 1) {"Ja"} else {"Nein"}
					$FundIsWin = if ($WinDisk.DiskNumber -eq $FundDisk -and $WinPart.PartitionNumber -eq $FundPart) {"Ja"} else {"Nein"}
					Ausgabe @"

Disk $FundDisk ($FundLayout), Partition ${FundPart}: $REFile ($FundSize MByte)
Hat RE-ID: Nein
Liegt direkt hinter Windows-Partition: $FundFollowsWin
Ist Windows-Partition: $FundIsWin
Dateinformationen:
$(ImageInfo -Imagepath $REFile)
"@
				}
			}
		}
	}
	else
	{
		$MountBase = Join-Path -Path $env:TEMP -ChildPath ([System.IO.Path]::GetFileNameWithoutExtension($MyInvocation.MyCommand.Name))
		if (-not (Test-Path $MountBase)) {New-Item -ItemType Directory -Path $MountBase | Out-Null}
		$MountPoint = Join-Path -Path $MountBase -ChildPath ("Mount_" + $Part.DiskNumber + "_" + $Part.PartitionNumber)
		New-Item -ItemType Directory -Path $MountPoint | Out-Null
		$SearchPath = (Get-Volume -Partition $Part).Path
		mountvol $MountPoint $Searchpath
		$REFiles = Get-ChildItem -force -Path $MountPoint -Filter "Winre.wim" -Recurse -ErrorAction SilentlyContinue -File
		foreach ($REFile in $REFiles) {
			$WimCount++
			$Fundname = $REFile.Fullname.Replace(${MountPoint}, '')
			$Funddisk = $Part.DiskNumber
			$FundLayout = (Get-Disk -Number $FundDisk).PartitionStyle
			$Fundpart = $Part.PartitionNumber
			$Fundsize = [Math]::Round($REFile.Length / 1MB, 2)
			$FundFollowsWin = if ($WinDisk.DiskNumber -eq $FundDisk -and $WinPart.PartitionNumber -eq $FundPart - 1) {"Ja"} else {"Nein"}
			$FundIsWin = if ($WinDisk.DiskNumber -eq $FundDisk -and $WinPart.PartitionNumber -eq $FundPart) {"Ja"} else {"Nein"}		
			Ausgabe @"

Disk $FundDisk ($FundLayout), Partition ${FundPart}: $Fundname ($FundSize MByte)
Hat RE-ID: Ja
Liegt direkt hinter Windows-Partition: $FundFollowsWin
Ist Windows-Partition: $FundIsWin
Dateinformationen:
$(ImageInfo -Imagepath $REFile.Fullname)
"@
		}
		mountvol $MountPoint /D
		Remove-Item -Path $MountPoint -Force -Recurse
	}
}

Ausgabe "`r`nAnzahl Winre.wim: $WimCount`r`n"

Write-Host "`nAlle Infos wurden in $outputPath gespeichert."
Write-Host "`nEnter zum Beenden... " -NoNewline
$host.UI.RawUI.ReadKey('NoEcho,IncludeKeyDown') | Out-Null
