/**************************************************************************/
/**************************************************************************/
/**                                                                      **/
/**               TU Muenchen - Institut fuer Informatik                 **/
/**                                                                      **/
/** Program for the game of Solitair  V1.1                               **/
/**                                                                      **/
/**            Joern Eichler                                             **/
/**            Jochen Jaeger                                             **/
/**            Thomas Ludwig                                             **/
/**                                                                      **/
/** File:      solitair.c                                                **/
/**                                                                      **/
/**************************************************************************/
/**************************************************************************/

/* Include some header files */

#include <stdio.h>
#include <time.h>
#include <stdlib.h>
#include <sys/file.h>
#include "solitair.h"

#ifdef UseDiskTab
FILE *disktable;
#endif

#ifdef UseIndexTab
FILE *disktable;
#endif

/* Variables for the rules R1-R6 */
int r1 = 0;
int r2o = 0, r2j = 0, r3 = 0;
int r4 = 0, r5 = 0, r6 = 0;

/**************************************************************************/
/* tables for rules                                                       */
/**************************************************************************/

unsigned char table46[256];
unsigned char table13[256];
unsigned char table02[256];
unsigned char table0[256];
unsigned char table4[256];
unsigned char table5[256];

/**************************************************************************/
/* global variables                                                       */
/**************************************************************************/

ULONG HASHSIZE = 218;

STYPE solutions = (STYPE) 0;
int level = 0;
int act_move_number;

time_t starttime;
time_t endtime;

unsigned char board[5];

struct HashEntry *hashtable;
ULONG hashsize;
ULONG hashfkt;			/* Biggest prime number < hashsize */

int valid;

STYPE positions = 0;

/* some statistical data */
ULONG hash_entries = 0;
ULONG hash_colls = 0;
int hash_full = 0;

/**************************************************************************/
/* functions for machines using little endian byte order                  */
/**************************************************************************/

#ifndef BIG_ENDIAN
#define BIG_ENDIAN 4321
#endif
#ifndef LITTLE_ENDIAN
#define LITTLE_ENDIAN 1234
#endif

unsigned short byte_order = BIG_ENDIAN;		/* hope for the best... */

union {
    ULONG ll;
    unsigned char bytes[4];
} endian;

void check_byte_order(void)
{
    endian.ll = 0x01020304;
    if (endian.bytes[0] == 1)
	/* Big Endian 
	 * so nothing to change... */
	;
    else
	/* Hmm... Little Endian
	 * we have to change all values... */
	byte_order = LITTLE_ENDIAN;
}

STYPE little_to_big(STYPE x)
{
    STYPE y;

    ((char *) &y)[0] = ((char *) &x)[7];
    ((char *) &y)[1] = ((char *) &x)[6];
    ((char *) &y)[2] = ((char *) &x)[5];
    ((char *) &y)[3] = ((char *) &x)[4];
    ((char *) &y)[4] = ((char *) &x)[3];
    ((char *) &y)[5] = ((char *) &x)[2];
    ((char *) &y)[6] = ((char *) &x)[1];
    ((char *) &y)[7] = ((char *) &x)[0];

    return y;
}

ULONG ULONG_little_to_big(ULONG x)
{
    /* same thing for 32-bit */

    ULONG y;

    ((char *) &y)[0] = ((char *) &x)[3];
    ((char *) &y)[1] = ((char *) &x)[2];
    ((char *) &y)[2] = ((char *) &x)[1];
    ((char *) &y)[3] = ((char *) &x)[0];
    return y;
}

/**************************************************************************/
/* hash functions                                                         */
/**************************************************************************/

struct HashEntry *get_hash_node(unsigned char *board)
{
    struct HashEntry *h;
    STYPE hashindex;
    ULONG i, j;
    ULONG step;

    /* calculate hash function for given board */
    hashindex = (((((((((STYPE) board[4] << (STYPE) 8) + (STYPE) board[3]) << (STYPE) 8) +
		     (STYPE) board[2]) << (STYPE) 8) + (STYPE) board[1]) << (STYPE) 8) + (STYPE) board[0]);

    /* calculate step size for double hashing */
    step = (ULONG) ((STYPE) hashindex % (hashfkt / 3) + 1);

    i = hashindex % (STYPE) hashfkt;

    h = hashtable + i;
    j = i;

    do {
	if (h->hashindex == 0) {
	    /* key was not in the table
	     * generate new entry */
	    valid = 0;
	    hash_entries++;
	    if (j != i)
		hash_colls++;
	    h->hashindex = hashindex;
	    return h;
	}
	if (h->hashindex == hashindex) {
	    /* key was in the table */
	    valid = 1;
	    return (h);
	}
	/* try next place in the table (add step to index) */
	i = (i + step) % hashfkt;
	h = hashtable + i;
    }
    while (i != j);

    /* we coudln't find the right position neither a free slot ->
     * hashtable full!
     * thats very bad... we have to give up! */

    printf("Hashtable full!\n");
    exit(1);
}

/**************************************************************************/
/* This function gets the value of a position using the precalculated     */
/* table from disk. It therefore calls the function index_search in       */
/* the file scan_table.c                                                  */
/**************************************************************************/

#ifdef UseIndexTab
#define UseDiskTab

struct HashEntry *get_disk_hash_node(unsigned char *board)
{
    struct HashEntry *h;
    STYPE hashindex;

    h = malloc(sizeof(struct HashEntry));

    /* calculate hash index */
    hashindex = (((((((((STYPE) board[4] << (STYPE) 8) + (STYPE) board[3]) << (STYPE) 8) +
		     (STYPE) board[2]) << (STYPE) 8) + (STYPE) board[1]) << (STYPE) 8) + (STYPE) board[0]);

    h->hashindex = hashindex;
    h->solutions = index_search(hashindex);
    valid = 1;

    return h;
}

#endif

/**************************************************************************/
/* init memory for hash table                                             */
/**************************************************************************/

void InitHash()
{
    int hsize = HASHSIZE * 1024 * 1024;
    ULONG z, y;
    int prim;

    if (hsize == 0)
	hsize = 1024;

    hashsize = (hsize / sizeof(struct HashEntry));
    hashtable = (struct HashEntry *) calloc(hashsize, sizeof(struct HashEntry));
    if ((!hashtable)) {
	fprintf(stderr, "Couldn't get memory for index array\n");
	exit(1);
    }
    hashfkt = 0;
    /* hashsize-(1-(hashsize%2)) is always the greatest odd number <= hashsize */
    for (z = hashsize - (1 - (hashsize % 2)); z > 0; z -= 2) {
	/* find first prime number smaller than hashsize */
	prim = 1;
	for (y = 3; y * y < z && prim; y += 2)
	    if (z % y == 0)
		prim = 0;
	if (prim) {
	    hashfkt = z;
	    break;
	}
    }
    /* Hmm. couldn't find one... hashsize is too small (shouldn't happen!) */
    if (hashfkt == 0) {
	printf("Error!\n");
	exit(0);
    }
}

/**************************************************************************/
/* initialization of tables for rules                                     */
/**************************************************************************/

int bittest(int i, int bit)
{
    /* return 1 if bit 'bit' is set in 'i' */
    return ((i & (1 << bit)) >> bit);
}

void InitR()
{
    int i;
    for (i = 0; i < 256; i++) {
	table46[i] = bittest(i, 4) + bittest(i, 6);
	table13[i] = bittest(i, 1) + bittest(i, 3);
	table02[i] = bittest(i, 0) + bittest(i, 2);
	table0[i] = bittest(i, 0);
	table4[i] = bittest(i, 4);
	table5[i] = bittest(i, 5);
    }
}

/**************************************************************************/
/* initialization of the board                                            */
/**************************************************************************/

void InitSearch()
{
    int i;
    int maparray[] =
    {0, 4, 2, 1, 5, 3, 10, 11, 7, 6, 31, 25, 24, 12, 13, 14, 32, 30, 29, 28,
     8, 9, 15, 22, 23, 27, 26, 19, 21, 17, 18, 20, 16, 0};

    /* initial position */

    int board0[33] =
    {
	      1, 1, 1,
	      1, 1, 1,
	1, 1, 1, 1, 1, 1, 1,
	1, 1, 1, 0, 1, 1, 1,
	1, 1, 1, 1, 1, 1, 1,
	      1, 1, 1,
	      1, 1, 1
    };

    for (i = 0; i < 33; i++)
	putsquare(board, maparray[i], board0[i]);
}

/**************************************************************************/
/* Read position if calles by WWW                                         */
/**************************************************************************/

void ReadInit(void)
{
    /* maparray contains the index of each square in the internal
     * representation, which differs from the one used in the TCL-
     * Application due to easier calculation of symmetries */

    int i = 0, j = 0;
    int maparray[] =
    {0, 4, 2, 1, 5, 3, 10, 11, 7, 6, 31, 25, 24, 12, 13, 14, 32, 30, 29, 28,
     8, 9, 15, 22, 23, 27, 26, 19, 21, 17, 18, 20, 16, 0};
    int ende = 0;
    char *s = getenv("QUERY_STRING");
    char *c = s;
    long prim, z, y;
    ULONG hashsize;

    while (!ende) {
	if (*c == 'x') {
	    putsquare(board, maparray[j], 1);
	    i++;
	} else if (*c == 'o')
	    putsquare(board, maparray[j], 0);
	else
	    ende = 1;
	j++;
	c++;
    }

    level = 32 - i;
    act_move_number = 32 - i;

    /* Init hash table */

    hashsize = ((HASHSIZE * 1024 * 1024) / sizeof(struct HashEntry));
    hashfkt = 0;
    for (z = hashsize - 1; z > 0; z -= 2) {
	prim = 1;
	for (y = 3; y * y < z && prim; y += 2)
	    if (z % y == 0)
		prim = 0;
	if (prim) {
	    hashfkt = z;
	    break;
	}
    }
    if (hashfkt == 0) {
	printf("Error!\n");
	exit(0);
    }
}

/**************************************************************************/
/* search routine                                                         */
/**************************************************************************/

void search(int move_number)
{
    /* variables for rules */
    int No, Nj, Nvo, Nvj, Nb, Noj, Njo, Nbo, Nbj;

#ifdef WWW
    STYPE altsol = 0;
#endif

    int i, j, t;
    STYPE psol;			/* solutions from this pos. */
    struct HashEntry *hash;
    unsigned char old_board[5];

    memcpy(old_board, board, 5);

    if (move_number == MAX_MOVE) {
//      if(getsquare(board,32))  //use faster code for this
	if (board[4])
	    solutions++;
    } else {			/* look deeper */
	positions++;
	psol = solutions;

	Nb = table5[board[0]] + table5[board[1]] +
	    table0[board[4]] + table5[board[3]] + table5[board[2]];

	/* rule one: */
	/* there must be at least one blue peg left */
	if (Nb < 1) {
	    r1++;
	    return;
	}
	/* rules two: */
	No = table13[board[0]] + table46[board[1]] +
	    table46[board[3]] + table13[board[2]];
	Nj = table46[board[0]] + table13[board[1]] +
	    table13[board[3]] + table46[board[2]];
	Nvo = table02[board[0]] + table02[board[2]];
	Nvj = table02[board[1]] + table02[board[3]];

	if (No < Nvo) {
	    r2o++;
	    return;		/* rule 2o */
	}
	if (Nj < Nvj) {
	    r2j++;
	    return;		/* rule 2j */
	}
	/* rule three: */
	if ((No + Nj) < (Nvo + Nvj + 1)) {
	    r3++;
	    return;
	}
	Noj = table4[board[1]] + table4[board[3]];
	Njo = table4[board[0]] + table4[board[2]];

	if ((No + Nj + Nb) < (Nvo + Nvj + Noj + Njo + 2)) {
	    r4++;
	    return;
	}
	Nbo = table5[board[0]] + table5[board[2]];

	if ((No + Nj + Nbo) < (Nvo + Nvj + Njo)) {
	    r5++;
	    return;
	}
	Nbj = table5[board[1]] + table5[board[3]];

	if ((No + Nj + Nbj) < (Nvo + Nvj + Noj)) {
	    r6++;
	    return;
	}
	/* find hash entry */
#ifdef UseDiskTab
	if (act_move_number == move_number) {
	    hash = malloc(sizeof(struct HashEntry));
	    valid = 0;
	} else {
	    hash = get_disk_hash_node(board);
	}
#else
	hash = get_hash_node(board);
#endif
	if (valid)
	    solutions += hash->solutions;
	else {
	    for (i = 0; i < 33; i++) {
		if (!board[BoardIndex(i)]) {
		    i += 7;
		    continue;
		}
		if (getsquare(board, i)) {	/* square occupied ? */
		    t = i * 9;
		    for (j = 0; j < move_pos[t]; j += 2) {
			if ((getsquare(board, move_pos[t + j + 1])) &&
			/* move possible? */
			    (!getsquare(board, move_pos[t + j + 2])))
			    /* okay, make move */
			{
			    clearsquare(board, i);
			    clearsquare(board, move_pos[t + j + 1]);
			    setsquare(board, move_pos[t + j + 2]);
			    /* hash index */
			    normalize_board(board);
			    search(move_number + 1);
			    memcpy(board, old_board, 5);
#ifdef WWW
			    if (act_move_number == move_number) {
				STYPE sols = solutions - altsol;
				if (sols > 0) {
				    printf("Zug: %s-%s (%.0f Loesungen)\n",
					   sqnames[i], sqnames[move_pos[t + j + 2]], (double) sols);
				    altsol = solutions;
				}
			    }
#endif
			}
		    }
		}
	    }
	    hash->solutions = solutions - psol;
#ifdef UseDiskTab
	    free(hash);
#endif
	}
    }
}

/**************************************************************************/
/* main                                                                   */
/**************************************************************************/

int main(int argc, char *argv[])
{
#ifdef WriteTable
    ULONG z;
    FILE *disk;
#endif
#if defined(PUTENV) && defined(WWW)
    char s[256];
#endif

    check_byte_order();

    /* user can specify the size of the hash table */
    if (argc == 2) {
	/* we only use this creating the table file */
	/* otherwise this value will be overwritten
	 * later to value 1 */
	HASHSIZE = atoi(argv[1]);
	if (HASHSIZE < 0)
	    HASHSIZE = 8;
    }
#ifdef UseDiskTab
    /* use the tabel file */
    disktable = fopen("disktable", "rb");
    if (!disktable) {
	printf("Disktable file not found!\n");
	exit(0);
    }
#endif

#ifdef UseIndexTab
    /* when using the table we don't need more than one MByte */
    HASHSIZE = 1;
#endif

    printf("\n");
    printf("-========================-\n");
    printf("  Solitair - Sequentiell  \n");
    printf("  V1.1 by  Joern Eichler  \n");
    printf("      and  Jochen Jaeger  \n");
    printf("-========================-\n\n");

#ifndef UseDiskTab
    printf("Using %ld MByte memory for my table\n\n", HASHSIZE);
#endif

#ifdef WWW

#ifndef UseDiskTab
    InitHash();
#endif
#ifndef PUTENV
    if (argc == 2)
	setenv("QUERY_STRING", argv[1], 0);
#else
    sprintf(s, "QUERY_STRING=%s", argv[1]);
    putenv(s);
#endif
    ReadInit();

#else

#ifndef UseDiskTab
    InitHash();
#endif

    InitSearch();

#endif

    InitR();

#ifndef WWW
    printf("searching...\n");
#endif

    starttime = time(NULL);
    search(act_move_number);
    endtime = time(NULL);

    printf("Calculation time: %lu seconds\n", (endtime - starttime));
    printf("Solutions: %.0f\n\n", (double) solutions);

#ifndef WWW
    printf("Hash_Entries: %ld / %ld \t(%.1f%%)\n", hash_entries, hashsize,
	   (double) hash_entries / hashsize * 100.0);
    printf("Hash_Colls:   %ld \t(%.1f%%)\n\n", hash_colls,
	   (double) hash_colls / hash_entries * 100.0);
    printf("Positions: %llu\n\n", positions);

    printf("R1=%d\n", r1);
    printf("R2o=%d, R2j=%d, R3=%d\n", r2o, r2j, r3);
    printf("R4=%d, R5=%d, R6=%d\n", r4, r5, r6);
    printf("\n");
#endif

#ifdef WriteTable
    /* write the hash table to disk */

    if (!(disk = fopen("soltable", "w"))) {
	fprintf(stderr, "Couldn't open table file!\n");
	exit(1);
    }
    /* if we run under LITTLE_ENDIAN machine we have to change
     * byte order when writing the table because later we expect
     * the table to have BIG_ENDIAN */

    if (byte_order == LITTLE_ENDIAN) {
	printf("Converting table to big endian...");
	fflush(stdout);
	for (z = 0; z < hashsize; z++) {
	    hashtable[z].hashindex = little_to_big(hashtable[z].hashindex);
	    hashtable[z].solutions = little_to_big(hashtable[z].solutions);
	}
	HASHSIZE = ULONG_little_to_big(HASHSIZE);
	printf("\n");
    }
    printf("Writing table...");
    fflush(stdout);

    fwrite((const void *) &HASHSIZE, sizeof(HASHSIZE), 1, disk);
    fwrite((const void *) hashtable, sizeof(struct HashEntry), hashsize, disk);

    fclose(disk);
    printf("\n\n");
#endif

#ifdef UseDiskTab
    fclose(disktable);
#endif

    /* clean up */
    free(hashtable);

    exit(0);
}
