unit MainUnit;

interface

uses
	Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
	Menus, ExtCtrls, StdCtrls;

const
	InfStrings: array [1..23] of string = (
		'; Erstellt mit dem Bildschirmtreiber-Editor von SeReSoft, Standardversion 1.3',
		'',
		'[Version]',
		'Class=Monitor',
		'Provider=%Prov%',
		'',
		'[Manufacturer]',
		'%Mfg%=Mfg',
		'',
		'[Mfg]',
		'%Monitor%=Monitor.Install,',
		'',
		'[Monitor.Install]',
		'DelReg=DelCurrentReg',
		'AddReg=AddNewReg',
		'',
		'[DelCurrentReg]',
		'HKR,MODES',
		'HKR,,MaxResolution',
		'HKR,,DPMS',
		'HKR,,ICMProfile',
		'',
		'[AddNewReg]');

type
	TDisplayMode = class(TObject)
	private
		function PolToStr (Pol: Boolean): string;
	public
		ResX,
		ResY,
		FreqHorz,
		FreqVert: string;
		PolHorz,
		PolVert:  Boolean;
		function GetDisplayStr: string;
		function GetResStr: string;
		function SaveToStr: string;
	end;
	TMainWindow = class(TForm)
		MainMenu: TMainMenu;
		MainMenuFile: TMenuItem;
		MainMenuFileExit: TMenuItem;
		MainMenuFileLine1: TMenuItem;
		MainMenuFileSaveAs: TMenuItem;
		MainMenuFileSave: TMenuItem;
		MainMenuFileOpen: TMenuItem;
		MainMenuFileNew: TMenuItem;
		GroupBoxNameSettings: TGroupBox;
		LabelMonitorName: TLabel;
		LabelManufacturer: TLabel;
		LabelProvider: TLabel;
		EditProvider: TEdit;
		EditManufacturer: TEdit;
		EditMonitorName: TEdit;
		GroupBoxMonitorSettings: TGroupBox;
		ListBoxModes: TListBox;
		ButtonModeAdd: TButton;
		ButtonModeEdit: TButton;
		ButtonModeDelete: TButton;
		GroupBoxExtendedSettings: TGroupBox;
		LabelHardwareID: TLabel;
		EditHardwareID: TEdit;
		CheckBoxDPMS: TCheckBox;
		CheckBoxICMProfile: TCheckBox;
		EditICMProfile: TEdit;
		BevelMenuSeparator: TBevel;
		LabelRes: TLabel;
		LabelResUnity: TLabel;
    MainMenuDriverCompileAs: TMenuItem;
		OpenDialogProject: TOpenDialog;
		SaveDialogProject: TSaveDialog;
		SaveDialogDriver: TSaveDialog;
    MainMenuDriverCompile: TMenuItem;
		MainMenuHelp: TMenuItem;
		MainMenuHelpInfo: TMenuItem;
		MainMenuHelpMainHelp: TMenuItem;
		MainMenuHelpLine1: TMenuItem;
    MainMenuDriver: TMenuItem;
    ComboBoxRes: TComboBox;
    LabelCurrentResDescription: TLabel;
    LabelCurrentRes: TLabel;
    LabelCurrentResUnity: TLabel;
		procedure FileExit(Sender: TObject);
		procedure FileNew(Sender: TObject);
		procedure CheckBoxICMProfileClick(Sender: TObject);
		procedure AddMode(Sender: TObject);
		procedure ListBoxModesClick(Sender: TObject);
		procedure DeleteMode(Sender: TObject);
		procedure EditMode(Sender: TObject);
		procedure FileCompileAs(Sender: TObject);
		procedure FormCreate(Sender: TObject);
		procedure EditNumberChange(Sender: TObject);
		procedure FileCompile(Sender: TObject);
		procedure FileSaveAs(Sender: TObject);
		procedure FileSave(Sender: TObject);
		procedure FileOpenDlg(Sender: TObject);
		procedure FileOpen(Sender: TObject);
		procedure NameChange(Sender: TObject);
		procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
		procedure HelpAbout(Sender: TObject);
		procedure MainHelp(Sender: TObject);
    procedure ListBoxModesDblClick(Sender: TObject);
		procedure ListBoxModesKeyDown(Sender: TObject; var Key: Word;
			Shift: TShiftState);
		procedure ComboBoxResChange(Sender: TObject);
	private
	public
		NewHardwareID,
		NewManufacturer,
		NewProvider,
		ProjectFile,
		DriverFile:      string;
		WasAborted,
		Modified:        Boolean;
		procedure UpdateListBoxButtons;
		procedure UpdateICMProfileEdit;
		procedure UpdateMainMenu;
		procedure AssignMode (Index: Integer);
		procedure ReadMode (Index: Integer);
		procedure UpdateMode (Index: Integer);
		procedure Modify;
		function CancelChanges: Boolean;
		function GetResX (const Box: TComboBox): string;
		function GetResY (const Box: TComboBox): string;
	end;

var
	MainWindow: TMainWindow;

implementation

uses
	Registry, IniFiles,
	ModeUnit, InfoUnit, HelpUnit, WzrdUnit;

{$R *.DFM}

{ TDisplayMode }

function TDisplayMode.GetDisplayStr: string;
begin
	Result := 'H: ' + FreqHorz + ' kHz; V: ' + FreqVert + ' Hz; ' + PolToStr (PolHorz) + '/' + PolToStr (PolVert);
	if (ResX <> '') and (ResY <> '') then
		Result := Result + '; ' + ResX + 'x' + ResY;
end;

function TDisplayMode.GetResStr: string;
begin
	if (ResX = '') or (ResY = '') then
		with MainWindow do
			Result := GetResX (ComboBoxRes) + ',' + GetResY (ComboBoxRes)
	else
		Result := ResX + ',' + ResY;
end;

function TDisplayMode.PolToStr(Pol: Boolean): string;
begin
	if Pol then
		Result := '+'
	else
		Result := '-';
end;

function TDisplayMode.SaveToStr: string;
begin
	Result := FreqHorz + ',' + FreqVert + ',' + PolToStr (PolHorz) + ',' + PolToStr (PolVert);
end;

{ TMainWindow }

procedure TMainWindow.FileExit(Sender: TObject);
begin
	Close;
end;

procedure TMainWindow.FileNew(Sender: TObject);
begin
	if CancelChanges then begin
		ProjectFile                := '';
		DriverFile                 := '';
		if NewHardwareID = 'Default_Monitor' then
			EditMonitorName.Text     := 'Neuer Bildschirm'
		else
			EditMonitorName.Text     := NewHardwareID;
		EditManufacturer.Text      := NewManufacturer;
		EditProvider.Text          := NewProvider;
		with ComboBoxRes do
			if Screen.Width > 1024 then
				Text                   := LabelCurrentRes.Caption
			else
				ItemIndex              := 2;
		ListBoxModes.Clear;
		UpdateListBoxButtons;
		EditHardwareID.Text        := 'Monitor\' + NewHardwareID;
		CheckBoxDPMS.Checked       := True;
		CheckBoxICMProfile.Checked := False;
		UpdateICMProfileEdit;
		UpdateMainMenu;
		Modified                   := False;
	end;
end;

procedure TMainWindow.UpdateICMProfileEdit;
begin
	if EditICMProfile.Enabled <> CheckBoxICMProfile.Checked then begin
		EditICMProfile.Enabled := CheckBoxICMProfile.Checked;
		with EditICMProfile do begin
			if Enabled then
				Text := '9'
			else
				Text := '';
		end;
	end;
end;

procedure TMainWindow.UpdateListBoxButtons;
begin
	ButtonModeEdit.Enabled := (ListBoxModes.ItemIndex >= 0) and (ListBoxModes.Items.Count > 0);
	ButtonModeDelete.Enabled := ButtonModeEdit.Enabled;
end;

procedure TMainWindow.CheckBoxICMProfileClick(Sender: TObject);
begin
	UpdateICMProfileEdit;
	Modify;
end;

procedure TMainWindow.AddMode(Sender: TObject);
var
	MsgBoxResult,
	InchBoxResult: Integer;
	ModelStr,
	ModelNoStr:    string;
	I,
	MonFreq,
	MonSize:       Integer;
	IsValid:       Boolean;
	NewMode:       TDisplayMode;
begin
	if ListBoxModes.Items.Count = 0 then
		MsgBoxResult := Application.MessageBox
			('Falls Sie kein Handbuch zum Monitor besitzen, kann der Modell-Assistent Ihnen unter Umstnden behilflich sein. Mchten Sie jetzt versuchen, den Bildschirmtreiber mit Hilfe der Modell-Nummer des Bildschirms zu erstellen?',
			'Modell-Assistent', mb_YesNoCancel +
			mb_IconQuestion + mb_DefButton2)
	else
		MsgBoxResult := idNo;
	if MsgBoxResult <> idCancel then begin
		if MsgBoxResult = idYes then
			with ModelNoWizardWindow do begin
				if ShowModal = mrOK then begin
					IsValid := False;
					ModelStr := EditModelNo.Text;
					ModelNoStr := '';
					MonFreq := 0;
					MonSize := 0;
					I := Length (ModelStr);
					while (I > 0) and (ModelStr [I] in ['0'..'9']) do begin
						ModelNoStr := ModelStr [I] + ModelNoStr;
						Dec (I);
					end;
					if ModelNoStr <> '' then begin
						MonFreq := StrToInt (ModelNoStr);
						MonSize := MonFreq div 100;
						MonFreq := MonFreq mod 100;
					end;
					if (MonSize in [14..25]) and (MonFreq in [40..65]) then begin
						InchBoxResult := Application.MessageBox
							(PChar ('Ist der Bildschirm ein ' + IntToStr (MonSize) + '"-Monitor?'),
							'Bildschirmgre', mb_YesNoCancel +
							mb_IconQuestion);
						case InchBoxResult of
							idCancel: Exit;
							idYes:    IsValid := True;
						end;
					end;
					if IsValid then begin
						NewMode := TDisplayMode.Create;
						with NewMode do begin
							ResX := '';
							ResY := '';
							FreqHorz := '30.0-' + IntToStr (MonFreq) + '.0';
							FreqVert := '50.0-85.0';
							PolHorz := False;
							PolVert := False;
							ListBoxModes.Items.AddObject (GetDisplayStr, NewMode);
						end;
						UpdateListBoxButtons;
						Modify;
					end else
						Application.MessageBox
							('Die Modell-Nummer enthlt leider nicht die richtigen Informationen zur Erstellung eines Bildschirmtreibers.',
							'Ungltige Modell-Nummer', mb_OK +
							mb_IconError);
				end;
		end else begin
			if DisplayModeWindow.ShowModal = mrOK then begin
				with ListBoxModes.Items do begin
					BeginUpdate;
					AssignMode (AddObject ('', TDisplayMode.Create));
					EndUpdate;
				end;
				UpdateListBoxButtons;
				Modify;
			end;
		end;
	end;
end;

procedure TMainWindow.AssignMode(Index: Integer);
begin
	with ListBoxModes.Items.Objects [Index] as TDisplayMode, DisplayModeWindow do begin
		if CheckBoxSpecialResolution.Checked then begin
			ResX   := GetResX (ComboBoxSpecialRes);
			ResY   := GetResY (ComboBoxSpecialRes);
		end else begin
			ResX   := '';
			ResY   := '';
		end;
		FreqHorz := EditFreqHorz.Text;
		FreqVert := EditFreqVert.Text;
		PolHorz  := RadioButtonPolHorzPos.Checked;
		PolVert  := RadioButtonPolVertPos.Checked;
	end;
	UpdateMode (Index);
end;

procedure TMainWindow.UpdateMode(Index: Integer);
begin
	with ListBoxModes.Items do
		Strings [Index] := (Objects[Index] as TDisplayMode).GetDisplayStr;
end;

procedure TMainWindow.ListBoxModesClick(Sender: TObject);
begin
	UpdateListBoxButtons;
end;

procedure TMainWindow.DeleteMode(Sender: TObject);
begin
	with ListBoxModes do
		with Items do begin
			(Objects[ItemIndex] as TDisplayMode).Free;
			Delete (ItemIndex);
		end;
	UpdateListBoxButtons;
	Modify;
end;

procedure TMainWindow.EditMode(Sender: TObject);
begin
	with ListBoxModes do begin
		ReadMode (ItemIndex);
		if DisplayModeWindow.ShowModal = mrOK then begin
			AssignMode (ItemIndex);
			UpdateListBoxButtons;
			Modify;
		end;
	end;
end;

procedure TMainWindow.ReadMode(Index: Integer);
begin
	with ListBoxModes.Items.Objects [Index] as TDisplayMode, DisplayModeWindow do begin
		CheckBoxSpecialResolution.Checked := (ResX <> '') and (ResY <> '');
		if CheckBoxSpecialResolution.Checked then
			ComboBoxSpecialRes.Text := ResX + 'x' + ResY
		else
			ComboBoxSpecialRes.Text := '';
		EditFreqHorz.Text := FreqHorz;
		EditFreqVert.Text := FreqVert;
		if PolHorz then
			RadioButtonPolHorzPos.Checked := True
		else
			RadioButtonPolHorzNeg.Checked := True;
		if PolVert then
			RadioButtonPolVertPos.Checked := True
		else
			RadioButtonPolVertNeg.Checked := True;
	end;
end;

procedure TMainWindow.FileCompileAs(Sender: TObject);
begin
	WasAborted := False;
	with SaveDialogDriver do begin
		FileName := DriverFile;
		if Execute then begin
			DriverFile := FileName;
			FileCompile (Sender);
			UpdateMainMenu;
			Modify;
		end else
			WasAborted := True;
	end;
end;

procedure TMainWindow.FormCreate(Sender: TObject);
const
	ExtDesc = 'MonitorDriverProject';
var
	I,
	J:       Integer;
	DirBuf:  array [0..Max_Path] of Char;
	Reg:     TRegistry;
	SubKeys: TStringList;
begin
	if Copy (ShortDateFormat, Length (ShortDateFormat) - 3, 4) <> 'yyyy' then
		ShortDateFormat := ShortDateFormat + 'yy';
	GetWindowsDirectory (DirBuf, Max_Path);
	SaveDialogDriver.InitialDir := StrPas (DirBuf) + '\Inf\Other';
	SubKeys         := TStringList.Create;
	Reg             := TRegistry.Create;
	NewHardwareID   := 'Default_Monitor';
	NewManufacturer := '';
	with Reg do begin
		RootKey := HKey_Local_Machine;
		OpenKeyReadOnly ('\Enum\Monitor');
		GetKeyNames (SubKeys);
		with SubKeys do
			for I := Count - 1 downto 0 do
				if UpperCase (Strings [I]) <> 'DEFAULT_MONITOR' then begin
					NewHardwareID := Strings [I];
					J := 1;
					while (J < Length (NewHardwareID)) and (NewHardwareID [J] in ['A'..'Z', 'a'..'z']) do begin
						if J = 1 then
							NewManufacturer := NewManufacturer + UpperCase (NewHardwareID [J])
						else
							NewManufacturer := NewManufacturer + LowerCase (NewHardwareID [J]);
						Inc (J);
					end;
					Break;
				end;
		NewProvider := '';
		try
			RootKey := HKey_Current_User;
			OpenKeyReadOnly ('\Software\Microsoft\MS Setup (ACME)\User Info');
			NewProvider := ReadString ('DefName');
		except end;
		if NewProvider = '' then try
			RootKey := HKey_Local_Machine;
			OpenKeyReadOnly ('\Software\Microsoft\Windows\CurrentVersion');
			NewProvider := ReadString ('RegisteredOwner');
		except end;
		RootKey := HKey_Classes_Root;
		OpenKey ('\.mon', True);
		WriteString ('', ExtDesc);
		OpenKey ('\' + ExtDesc, True);
		WriteString ('', 'Bildschirmtreiber-Projekt');
		OpenKey ('DefaultIcon', True);
		GetSystemDirectory (DirBuf, Max_Path);
		WriteString ('', StrPas (DirBuf) + '\Deskcp16.dll,0');
		OpenKey ('\' + ExtDesc + '\Shell\open\command', True);
		WriteString ('', '"' + ParamStr (0) + '" "%1"');
		Free;
	end;
	SubKeys.Free;
	if NewManufacturer = '' then
		NewManufacturer := 'Neuer Hersteller';
	with Screen do
		LabelCurrentRes.Caption := IntToStr (Width) + 'x' + IntToStr (Height);
	ProjectFile := ParamStr (1);
	FileOpen (Sender);
end;

procedure TMainWindow.EditNumberChange(Sender: TObject);
var
	S:    string;
	I,
	SelS,
	SelL: Integer;
begin
	with Sender as TEdit do begin
		S    := Text;
		SelS := SelStart;
		SelL := SelLength;
		I    := 1;
		while I <= Length (S) do begin
			if not (S [I] in ['0'..'9']) then begin
				Delete (S, I, 1);
				Dec (I);
			end;
			Inc (I);
		end;
		Text      := S;
		SelStart  := SelS;
		SelLength := SelL;
	end;
	Modify;
end;

procedure TMainWindow.UpdateMainMenu;
var
	F: string;
begin
	MainMenuDriverCompile.Enabled := (DriverFile <> '');
	if ProjectFile = '' then
		Caption := 'Bildschirmtreiber-Editor'
	else begin
		F := ExtractFileName (ProjectFile);
		Caption := 'Bildschirmtreiber-Editor - ' + Copy (F, 1, Pos (ExtractFileExt (F), F) - 1);
	end;
end;

procedure TMainWindow.FileCompile(Sender: TObject);
var
	ModeList: TStringList;
function GetModeNum (Res: string): Integer;
begin
	with ModeList do
		if Values [Res] = '' then begin
			Result := 1;
			Add (Res + '=1');
		end else begin
			Result := StrToInt (Values [Res]);
			Inc (Result);
			Values [Res] := IntToStr (Result);
		end;
end;
var
	InfFile: System.Text;
	I:       Integer;
begin
	WasAborted := False;
	ModeList := TStringList.Create;
	AssignFile (InfFile, DriverFile);
	ReWrite (InfFile);
	WriteLn (InfFile, '; Bildschirmtreiber fr den Monitor "' + EditMonitorName.Text + '" (Hersteller: ' + EditManufacturer.Text + ')');
	WriteLn (InfFile, '; Letzte Aktualisierung: ' + DateToStr (Date) + ' von ' + EditProvider.Text);
	for I := 1 to High (InfStrings) do begin
		if (InfStrings [I] <> '') and (InfStrings [I] [Length (InfStrings [I])] = ',') then
			WriteLn (InfFile, InfStrings [I] + EditHardwareID.Text)
		else
			WriteLn (InfFile, InfStrings [I]);
	end;
	if CheckBoxDPMS.Checked then
		WriteLn (InfFile, 'HKR,,DPMS,,1');
	if CheckBoxICMProfile.Checked then
		WriteLn (InfFile, 'HKR,,ICMProfile,1,' + EditICMProfile.Text);
	WriteLn (InfFile, 'HKR,,MaxResolution,,"' + GetResX (ComboBoxRes) + ',' + GetResY (ComboBoxRes) + '"');
	with ListBoxModes.Items do
		for I := 0 to Count - 1 do
			with Objects [I] as TDisplayMode do
				WriteLn (InfFile, 'HKR,"MODES\' + GetResStr + '",Mode' + IntToStr (GetModeNum (GetResStr)) + ',,"' + SaveToStr + '"');
	WriteLn (InfFile);
	WriteLn (InfFile, '[Strings]');
	WriteLn (InfFile, 'Prov="' + EditProvider.Text + '"');
	WriteLn (InfFile, 'Mfg="' + EditManufacturer.Text + '"');
	WriteLn (InfFile, 'Monitor="' + EditMonitorName.Text + '"');
	CloseFile (InfFile);
	ModeList.Free;
end;

procedure TMainWindow.FileSaveAs(Sender: TObject);
begin
	WasAborted := False;
	with SaveDialogProject do begin
		FileName := ProjectFile;
		if Execute then begin
			ProjectFile := FileName;
			FileSave (Sender);
			UpdateMainMenu;
		end else
			WasAborted := True;
	end;
end;

procedure TMainWindow.FileSave(Sender: TObject);
var
	ProjFile: TIniFile;
	I:        Integer;
begin
	WasAborted := False;
	if ProjectFile = '' then
		FileSaveAs (Sender)
	else begin
		if FileExists (ProjectFile) then
			DeleteFile (ProjectFile);
		ProjFile := TIniFile.Create (ProjectFile);
		with ProjFile do begin
			WriteString ('Names', 'Monitor',      EditMonitorName.Text);
			WriteString ('Names', 'Manufacturer', EditManufacturer.Text);
			WriteString ('Names', 'Provider',     EditProvider.Text);
			WriteString ('Names', 'Driver File',  DriverFile);
			WriteString ('Resolution', 'X', GetResX (ComboBoxRes));
			WriteString ('Resolution', 'Y', GetResY (ComboBoxRes));
			with ListBoxModes.Items do
				for I := 1 to Count do
					with Objects [I-1] as TDisplayMode do begin
						WriteString ('Mode ' + IntToStr (I), 'Resolution X',            ResX);
						WriteString ('Mode ' + IntToStr (I), 'Resolution Y',            ResY);
						WriteString ('Mode ' + IntToStr (I), 'Horizontal Frequency',    FreqHorz);
						WriteString ('Mode ' + IntToStr (I), 'Vertical Frequency',      FreqVert);
						WriteBool   ('Mode ' + IntToStr (I), 'Horizontal Polarization', PolHorz);
						WriteBool   ('Mode ' + IntToStr (I), 'Vertical Polarization',   PolVert);
					end;
			WriteString ('Enhanced Settings', 'Hardware ID', EditHardwareID.Text);
			WriteBool   ('Enhanced Settings', 'DPMS',        CheckBoxDPMS.Checked);
			WriteString ('Enhanced Settings', 'ICM Profile', EditICMProfile.Text);
			Free;
		end;
		Modified := False;
	end;
end;

procedure TMainWindow.FileOpenDlg(Sender: TObject);
begin
	if CancelChanges then begin
		WasAborted := False;
		if OpenDialogProject.Execute then begin
			ProjectFile := OpenDialogProject.FileName;
			FileOpen (Sender);
		end else
			WasAborted := True;
	end;
end;

procedure TMainWindow.FileOpen(Sender: TObject);
var
	ProjFile: TIniFile;
	I,
	Idx:      Integer;
begin
	WasAborted := False;
	if ProjectFile = '' then
		FileNew (Sender)
	else begin
		ProjFile := TIniFile.Create (ProjectFile);
		with ProjFile do begin
			EditMonitorName.Text  := ReadString ('Names', 'Monitor',      'Neuer Bildschirm');
			EditManufacturer.Text := ReadString ('Names', 'Manufacturer', 'Neuer Hersteller');
			EditProvider.Text     := ReadString ('Names', 'Provider',     'Neuer Anbieter');
			DriverFile            := ReadString ('Names', 'Driver File',  '');
			ComboBoxRes.Text      := ReadString ('Resolution', 'X', '1024') + 'x' + ReadString ('Resolution', 'Y', '768');
			I := 1;
			with ListBoxModes.Items do begin
				BeginUpdate;
				ListBoxModes.Clear;
				while SectionExists ('Mode ' + IntToStr (I)) do begin
					Idx := AddObject ('', TDisplayMode.Create);
					with Objects [Idx] as TDisplayMode do begin
						ResX     := ReadString ('Mode ' + IntToStr (I), 'Resolution X',            '');
						ResY     := ReadString ('Mode ' + IntToStr (I), 'Resolution Y',            '');
						FreqHorz := ReadString ('Mode ' + IntToStr (I), 'Horizontal Frequency',    '31.5');
						FreqVert := ReadString ('Mode ' + IntToStr (I), 'Vertical Frequency',      '50.0-70.0');
						PolHorz  := ReadBool   ('Mode ' + IntToStr (I), 'Horizontal Polarization', False);
						PolVert  := ReadBool   ('Mode ' + IntToStr (I), 'Vertical Polarization',   False);
					end;
					UpdateMode (Idx);
					Inc (I);
				end;
				EndUpdate;
			end;
			UpdateListBoxButtons;
			EditHardwareID.Text  := ReadString ('Enhanced Settings', 'Hardware ID', 'Monitor\Default_Monitor');
			CheckBoxDPMS.Checked := ReadBool   ('Enhanced Settings', 'DPMS',        True);
			EditICMProfile.Text  := ReadString ('Enhanced Settings', 'ICM Profile', '');
			with EditICMProfile do begin
				Enabled := (Text <> '');
				CheckBoxICMProfile.Checked := Enabled;
			end;
			Free;
			UpdateMainMenu;
		end;
		Modified := False;
	end;
end;

procedure TMainWindow.Modify;
begin
	Modified := True;
end;

procedure TMainWindow.NameChange(Sender: TObject);
begin
	Modify;
end;

function TMainWindow.CancelChanges: Boolean;
begin
	if Modified then begin
		case Application.MessageBox
			('Die aktuelle Datei wurde gendert. Mchten Sie die nderungen speichern?',
			'Treiber gendert',
			mb_YesNoCancel + mb_IconQuestion) of
			idYes: begin
				FileSave (Self);
				Result := not WasAborted;
			end;
			idNo:
				Result := True;
			else
				Result := False;
		end;
	end else
		Result := True;
end;

procedure TMainWindow.FormCloseQuery(Sender: TObject;
	var CanClose: Boolean);
begin
	CanClose := CancelChanges;
end;

procedure TMainWindow.HelpAbout(Sender: TObject);
begin
	AboutWindow.ShowModal;
end;

procedure TMainWindow.MainHelp(Sender: TObject);
begin
	HelpWindow.ShowModal;
end;

procedure TMainWindow.ListBoxModesDblClick(Sender: TObject);
begin
	if ButtonModeEdit.Enabled then
		ButtonModeEdit.Click
	else
		ButtonModeAdd.Click;
end;

procedure TMainWindow.ListBoxModesKeyDown(Sender: TObject; var Key: Word;
	Shift: TShiftState);
begin
	case Key of
		vk_Delete:
			if ButtonModeDelete.Enabled then
				ButtonModeDelete.Click;
		vk_Return:
			ListBoxModesDblClick (Sender);
	end;
end;

function TMainWindow.GetResX(const Box: TComboBox): string;
begin
	if Pos ('X', UpperCase (Box.Text)) <= 0 then
		Result := '640'
	else
		Result := Copy (Box.Text, 1, Pos ('X', UpperCase (Box.Text)) - 1);
end;

function TMainWindow.GetResY(const Box: TComboBox): string;
begin
	if Pos ('X', UpperCase (Box.Text)) <= 0 then
		Result := '480'
	else
		Result := Copy (Box.Text, Pos ('X', UpperCase (Box.Text)) + 1, Length (Box.Text));
end;

procedure TMainWindow.ComboBoxResChange(Sender: TObject);
var
	S:    string;
	I,
	SelS,
	SelL: Integer;
	Sep:  Boolean;
begin
	with Sender as TComboBox do begin
		S    := Text;
		SelS := SelStart;
		SelL := SelLength;
		I    := 1;
		Sep  := False;
		while I <= Length (S) do begin
			if not (S [I] in ['0'..'9', 'x', 'X']) then begin
				Delete (S, I, 1);
				Dec (I);
			end else
				if S [I] in ['x', 'X'] then begin
					if Sep then begin
						Delete (S, I, 1);
						Dec (I);
					end else
						Sep := True;
				end;
			Inc (I);
		end;
		Text      := S;
		SelStart  := SelS;
		SelLength := SelL;
	end;
	Modify;
end;

end.
