unit D3DRMiniU;  // Minimalistische 3DRM-Demo, 09-NOV-99 as
interface
uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls,
  Forms, Dialogs, ExtCtrls, StdCtrls, DXTimer, // schneller Timer
{$IFDEF DIRECTX7}  // DX7-Header (Zukunftsmusik)
  DirectDraw, Direct3D, Direct3DRM;
{$ELSE}  // DX6-Header
  DDraw,D3D,D3DTypes,D3DCaps,D3DRM,D3DRmDef,D3DRMObj,D3DRMWin;
{$ENDIF}

type
  TD3DRMiniForm = class(TForm)
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormPaint(Sender: TObject);
  private  // 50 Fortschreibungen pro Sekunde
    DXTimer: TDXTimer;
    procedure OnDXTimer(Sender: TObject);
  private  // Direct3DRM-Schnittstellen
    D3dInitialized: Boolean;
    D3DDevGUID: TGUID;  // GUID des (HAL-)Treibers
    Device: IDirect3DRMDevice3;
    D3DRMInterface: IDirect3DRM3;
    View: IDirect3DRMViewPort2;
    Scene, Camera: IDirect3DRMFrame3;
  protected
    function InitD3DInterfaces: Boolean;
    function BuildScene:Boolean;
    function RenderScene:Boolean;
  end;

var D3DRMiniForm: TD3DRMiniForm;

{$DEFINE DIRECTLIGHT}  // dritte Lichtquelle
{$DEFINE TEXTURE}      // Textur-Bitmap
{$DEFINE CUBE}         // zweites Objekt

implementation
{$R *.DFM}

function CheckRes(Res: HResult; ErrMsg: String): Boolean;
begin  // Makro, sozusagen
  Result := SUCCEEDED(Res);
  if not Result then ShowMessage('Fehler bei: '+ErrMsg);
end;

// D3D-Treiber abzhlen, Clipper und D3DRM-Schnittstellen anlegen
function TD3DRMiniForm.InitD3DInterfaces: Boolean;
var D3DRMTemp: IDirect3DRM;  // Urversion
    DDrawClipper: IDirectDrawClipper;

  // Rckruf beim Abzhlen der D3D-Gerte bzw. Treiber; bricht ab,
  // sobald der erste Hardware-Treiber gemeldet wird
  function EnumDevicesCallback(const lpGuid: TGUID;
     lpDeviceDescription, lpDeviceName: LPSTR;
     const lpD3DHWDeviceDesc, lpD3DHELDeviceDesc: TD3DDeviceDesc;
        lpUserArg: Pointer) : HResult; stdcall;
  begin
    if Assigned(@lpGUID) then PGUID(lpUserArg)^ := lpGuid;
    if lpD3DHWDeviceDesc.dcmColorModel = D3DCOLOR_RGB
      then Result := DDENUMRET_CANCEL  // OK, ende
      else Result := DDENUMRET_OK;  // weitermachen
  end;

  // DirectDraw-Schnittstelle anlegen, dort nach IDirect3D fragen,
  // IDirect3D zum Abzhlen der Gerte verwenden, danach beide
  // Schnittstellen (automatisch) wieder freigeben
  function Enum3DDevices(var D3DDevGuid: TGuid): Boolean;
  var DDrawObject: IDirectDraw; D3DObject: IDirect3D;
  begin
    Result := SUCCEEDED(DirectDrawCreate(nil, DDrawObject, nil))
     and SUCCEEDED(DDrawObject.QueryInterface(IID_IDirect3D,
       D3DObject)) and SUCCEEDED(D3DObject.EnumDevices
        (@EnumDevicesCallback, @D3DDevGUID));
  end;

begin
  Result := False;

  // DirectD3DRM-Schnittstelle (Urversion), Abfrage nach RM3
  if not CheckRes(Direct3DRMCreate(D3DRMTemp),
    'Direct3DRM-Schnittstelle anlegen')
   or not CheckRes(D3DRMTemp.QueryInterface(IID_IDirect3DRM3,
     D3DRMInterface),' QueryInterface nach RM3') then Exit;

  // DirectDrawClipper-Objekt anlegen, mit dem Fenster verbinden
  if not CheckRes(DirectDrawCreateClipper(0, DDrawClipper, nil),
    'DirectDraw-Clipper anlegen')
   or not CheckRes(DDrawClipper.SetHWnd(0, Handle),
    'DirectDraw-Clipper mit Fenster verbinden') then Exit;

  // GUID des ersten HAL-Treibers abfragen. Das ginge auch
  // ohne die vorherige Initialisierung von D3DRM
  if not Enum3DDevices(D3DDevGUID) then Exit;

  // D3DRM-Gert ber den DirectDrawClipper anlegen. NIL anstelle
  // des GUIDs wre "HEL ramp" anstelle von "HAL"
  if not CheckRes(D3DRMInterface.CreateDeviceFromClipper(
    DDrawClipper, @D3DDevGUID, ClientWidth, ClientHeight,
    Device), 'CreateDeviceFromClipper') then Exit;

  // Master- (= Hintergrund) und Kamera-Frame anlegen
  if not CheckRes(D3DRMInterface.CreateFrame(nil, Scene),
    'CreateFrame fr den Hintergrund')
   or not CheckRes(D3DRMInterface.CreateFrame(Scene, Camera),
    'CreateFrame fr die Kamera') then Exit;

   // Hintergrundfarbe (hier: 128er-Grau), keine Fehlerprfung
   Scene.SetSceneBackgroundRGB(0.5, 0.5, 0.5);

   // Viewport anlegen, hier fr das gesamte Fenster
   // (GetWidth/GetHeight = ClientWidth/ClientHeight)
   Result := CheckRes(D3DRMInterface.CreateViewport(Device,
    Camera, 0, 0, Device.GetWidth, Device.GetHeight, View),
     'CreateViewport');
end;

function TD3DRMiniForm.BuildScene: Boolean;
var
  MeshBuilder: IDirect3DRMMeshBuilder3;
  Lights, TeaFrame: IDirect3DRMFrame3;
{$IFDEF CUBE}
  CubeFrame: IDirect3DRMFrame3;
{$ENDIF}
  Light1, Light2: IDirect3DRMLight;
  Material: IDirect3DRMMaterial2;
  ProgPath: String;
{$IFDEF TEXTURE}
  // Textur einsetzen. Ursprung und Skalierung sind hier
  // der Einfachheit halber fix, d.h. nicht ber GetBox
  // des Meshes berechnet
  procedure DoTexture(BMPName: String);
  var MeshTex: IDirect3DRMTexture3;
      pWrap: IDirect3DRmWrap;
  begin
    if not CheckRes(D3DRMInterface.LoadTexture(
      PChar(BMPName),MeshTex),'Load Texture '+BmpName) then Exit;
    MeshBuilder.SetTexture(MeshTex);
    D3DRMInterface.CreateWrap(
      D3DRMWRAP_Sphere, nil,
       0,0,0,  // Wrap Org im Frame
       0,1,0,  // Z-Achse (Dir)
       0,1,0,  // Y-Achse (Up)
       0,0,  // Org in der Textur
       1,1, // Skalierung der Textur
       pWrap);
    CheckRes(pWrap.Apply(Meshbuilder),'Apply Wrap');
  end;
{$ENDIF}
begin
  Result := False; ProgPath := ExtractFilePath(ParamStr(0));
  // Render-Qualitt (in der Praxis wird da Gouraud draus)
  if not CheckRes(Device.SetQuality(D3DRMRENDER_PHONG),
    'SetQuality') then Exit;
  // Meshbuilder anlegen und Utah-Teetopf aus einer .X-Datei laden
  if not CheckRes(D3DRMInterface.CreateMeshBuilder(MeshBuilder),
    'CreateMeshBuilder')
   or not CheckRes(MeshBuilder.Load(PChar(ProgPath+'teapot.x'),nil,
     D3DRMLOAD_FROMFILE, nil, nil), 'Load TEAPOT.X') then Exit;
{$IFDEF TEXTURE}
  DoTexture(ProgPath+'env2.bmp');
{$ENDIF}

  // Das Gitternetz mit einem Material versehen (5.0 = metallisch,
  // je hher, desto plastikartiger wird es), Farbe auf hellgrn
  if not CheckRes(D3DRMInterface.CreateMaterial(35.0,Material),
    'CreateMaterial (fr den Teetopf)')
   or not CheckRes(MeshBuilder.SetMaterial(Material),
     'SetMaterial (Teetopf)')
   or not CheckRes(MeshBuilder.SetColorRGB(0.0, 0.7, 0.0),
     'SetColorRGB (Teetop-Material)')  then Exit;

  // Child-Frame in der Szene anlegen, den Teetopf dort einsetzen
  if not CheckRes(D3DRMInterface.CreateFrame(Scene, TeaFrame),
    'CreateFrame (fr den Teetopf)')
  or not CheckRes(TeaFrame.AddVisual(MeshBuilder),
    'AddVisual (Teetopf -> TeaFrame)') then Exit;

{$IFDEF CUBE}
  // Reinitalisierung. (Unbedingt mal ausklammern und angucken,
  // was dann herauskommt)
  D3DRMInterface.CreateMeshBuilder(MeshBuilder);
  if CheckRes(MeshBuilder.Load(PChar(ProgPath+'cube.x'),
     nil, D3DRMLOAD_FROMFILE, nil, nil), 'Load CUBE.X') then
  begin
    MeshBuilder.SetMaterial(Material);
    MeshBuilder.SetColorRGB(0.8,0.3,0.3);
    if SUCCEEDED(D3DRMInterface.CreateFrame(Scene, CubeFrame)) then
    begin
      CubeFrame.AddVisual(MeshBuilder);
      CubeFrame.SetPosition(Scene, 3,1,4);
      CubeFrame.SetOrientation(Scene,1,1,1,0,1,0);
{$IFDEF TEXTURE}
      DoTexture(ExtractFilePath(ParamStr(0))+'env2.bmp');
{$ENDIF}
    end;
  end;
{$ENDIF}

  // Kameraposition setzen. Objekte (= Frames) mit denselben X/Y-
  // Koordinaten wie die Kamera liegen sozusagen vor der Nase
  // des Benutzers, der Z-Wert bestimmt die Entfernung.
  Camera.SetPosition(Scene, 0, 0, -7);

  // Rotation an allen drei Achsen mit 0.01 Rad pro Frame.
  TeaFrame.SetRotation(Scene, 1,1,1, 0.01); // Pfannkucheneffekt

  // Beleuchtung. Positionierung geht wieder ber Frames
  // (und ist hier rechts neben der Kamera)
  if not CheckRes(D3DRMInterface.CreateFrame(Scene, Lights),
    'CreateFrame (fr die Beleuchtung)')
   or not CheckRes(Lights.SetPosition(Scene, 5, 0, -7),
    'SetPosition (fr die Beleuchtung)') then Exit;

  // Die erste Lichtquelle ist ein Punktlicht
  if CheckRes(D3DRMInterface.CreateLightRGB(
    D3DRMLIGHT_PARALLELPOINT, 1.0, 0.8, 0.9, Light1),
    'CreateLight fr Parallellicht') then Lights.AddLight(Light1);

  // 2. Lichtquelle ist eine gleichmige Umgebungshelligkeit
  if CheckRes(D3DRMInterface.CreateLightRGB(
    D3DRMLIGHT_AMBIENT, 0.1, 0.1, 0.1, Light2),
    'CreateLight fr Umgebung') then Lights.AddLight(Light2);
{$IFDEF DIRECTLIGHT}
  // Eine dritte, gerichtete und bewegte Lichtquelle
  if CheckRes(D3DRMInterface.CreateLightRGB(
    D3DRMLIGHT_DIRECTIONAL, 1.0, 0.8, 0.9, Light1),
    'CreateLight fr gerichtetes Licht') then
  begin
    Lights.AddLight(Light1);    // Dir   Up
    Lights.SetOrientation(Scene, 0,0,1, 0,1,0);
     // Drehachse: 45 Grad in XYZ
    Lights.SetRotation(Scene, 1, 1, 1, 0.05);
  end;
{$ENDIF}
  Result:=True;  // wunnebar...
end;

function TD3DRMiniForm.RenderScene: Boolean;
begin
  // Objekt(e) um 1 Frame bewegen. Scene.Move(1.0) ginge auch
  Result := CheckRes(D3DRMInterface.Tick(1.0),'Tick')
   // Viewport lschen
  and CheckRes(View.Clear(D3DRMCLEAR_ALL),'View.Clear')
  // Szene in den Viewport zeichnen
  and CheckRes(View.Render(Scene),'View.Render')
  // und Fenster aktualisieren
  and CheckRes(Device.Update,'Device.Update (RenderScene)');
end;

procedure TD3DRMiniForm.FormCreate(Sender: TObject);
begin
  Caption := 'D3DRM Minidemo';
  ClientWidth := 400; ClientHeight := 300;
  D3DInitialized := InitD3DInterfaces;
  if D3DInitialized and BuildScene then
  begin
    DXTimer := TDXTimer.Create(Self);
    DXTimer.Interval := 20; DXTimer.OnTimer := OnDXTimer;
  end
    else PostQuitMessage(0);
end;

procedure TD3DRMiniForm.FormDestroy(Sender: TObject);
begin  // Die Reihenfolge spielt mal wieder eine Rolle...
  D3DInitialized:=False;
  View := nil; Camera := nil;
  Scene :=nil; Device := nil;
  D3DRMInterface := nil;
end;

procedure TD3DRMiniForm.OnDXTimer(Sender: TObject);
begin
  if D3DInitialized then
    if not RenderScene then
    begin
      PostQuitMessage(0);
      DXTimer.Enabled := False;
    end;
end;

// Sorgt dafr, dass D3DRM bei WM_PAINT den gesamten Zeichen-
// puffer neu malt - und nicht nur die Bereiche, die bei
// normaler Fortschreibung aktualisiert werden mten
procedure TD3DRMiniForm.FormPaint(Sender: TObject);
var WinDev: IDirect3DRMWinDevice;
begin
  // WinDevice-Schnittstelle von RMDevice3
  // bernimmt die Reaktion auf WM_PAINT
  if SUCCEEDED(Device.QueryInterface(
    IID_IDirect3DRMWinDevice,WinDev))
   then WinDev.HandlePaint(Canvas.Handle);
end;


end.


