
// -------------------------------------------------
// Raycast.c++
// Experimentelle Raycast-Engine 1995 by Stefan Becker
// Geschrieben fr Macintosh und MacOS ab 7.5
// Code ist grtenteils Mac-spezifisch
// -------------------------------------------------

//
// ANSI-Includes
//

#include	<math.h>
#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>

//
// System-Includes:
//

#include	<Packages.h>
#include	<GestaltEqu.h>
#include	<QDOffscreen.h>
#include	<Controls.h>
#include	<ToolUtils.h>
#include	<Script.h>
#include	<Sound.h>

//
// Eigene Includes:
//

#include	"Globals.h"				// Allg. Definitionen
#include	"Raycast.pro"			// Prototypen dieses Moduls
#include	"Player.h"				// Definitionen des Spielers+
#include	"Render.pro"			// Prototypen des Render-Moduls
#include	"Bresenham.h"			// Bresenham-Modul

//
// Globale Variablen:
//

#ifdef	_NO_BRESENHAM_		// Nur beim berspringen von Bresenham ntig

SKIP_TAB	hsc_dx_tab,		// dx/dy-Tabellen fr effizienteres Raycasting
			hsc_dy_tab;
SKIP_TAB	vsc_dx_tab,
			vsc_dy_tab;

#endif						// _NO_BRESENHAM_

int			my3d_flag;		// Umschaltvariable zwischen 3D- und Kartensicht
Player		pl;				// Spielerdaten
GWorldPtr	texture_gworld;	// Geladene Textur als Offscreen-Bitmap
GWorldPtr	frame_buffer;	// Neuer Frame wird hier erst aufgebaut und dann gezeigt
short		quit_flag;		// Flagge, die bestimmt, ob das Programm verlassen wird
WINKEL_TAB	x_winkel_tab,	// Tabelle fr Richtungsvektoren aller Winkel
			y_winkel_tab;	// (Ntig, um schnell einen Einheitsvektor dieser Richtung zu ermitteln)
MAZE		maze;			// Wird spter aus maze_user generiert.
MAZE        maze_user =    	// Definition des begehbaren Areals: '*':Wand, ' ':freie Flche
                    {       // Achtung, wird noch vertikal gespiegelt, damit (0,0) links unten ist
                        "****************", // 15
                        "*          *   *", // 14
                        "*    *         *", // 13
                        "*        *******", // 12
                        "*   ******     *", // 11
                        "*        *     *", // 10
                        "****     *     *", // 9
                        "*  *        *  *", // 8
                        "*  *           *", // 7
                        "*     ****  ****", // 6
                        "*  *  *        *", // 5
                        "*******   ******", // 4
                        "*  ** *        *", // 3
                        "*         *    *", // 2
                        "*  *          **", // 1 Feld (1,1) mu frei sein (Spieler wird dahin
                        "****************"  // 0       mit Blickrichtung nach rechts positioniert!)
                    };  // (0,0) links unten, erster Index ist X

//
// Toolbox-Initialisierung (komplett Mac-spezifisch):
//

void ToolBoxInit(void)
{
	InitGraf(&qd.thePort);	// Setzt globale Variablen fr Quickdraw
	InitFonts();			// Fonts initialisieren
	InitWindows();			// Window Manager initialisieren
	InitMenus();			// MenuManager initialisieren
	TEInit();				// TextScrap leeren
	InitDialogs(0L);		// Dialogmanager initialisieren
	InitCursor();			// Maus-Init (sichtbar, Pfeil)
}

//
// Einen Sound aus dem Resource abspielen (komplett Mac-spezifisch):
//

void play_sound(short id)
{
	Handle	mySndHandle;	// Handle auf "snd "-Resource

	// Handle auf Sound aus dem Resource besorgen:
	if((mySndHandle=GetResource('snd ',id)) == nil)
		return;		// Das wars! Ohne Handle kein Sound. Seufz!

	// Handle auf Sound ist da. Sperren, abspielen, freigeben:
	HLock(mySndHandle);				// Handle sperren
	SndPlay(nil,(SndListHandle)mySndHandle,false);		// Sound abspielen
	HUnlock(mySndHandle);			// Handle entsperren
	ReleaseResource(mySndHandle);	// Handle wieder freigeben.
}

//
// Das Display-Window ffnen (komplett Mac-spezifisch):
//

WindowPtr make_display(void)
{
	WindowPtr	window;		// Verweis auf eine Mac-Fensterstruktur

	// Fenster aus dem Resource auslesen:
	if((window = GetNewWindow(WINDOW_ID,nil,(WindowPtr)-1)) == nil)
	{
		fatal_error("Das Raycast-Fenster konnte nicht geffnet werden!");
	}

	// Viewport auf dieses Fenster setzen:
	SetPort(window);

	// Fenster mit Titel versehen:
	SetWTitle(window,"\p Raycast-Fenster ");

	// Fenster auch auf dem Schirm zeigen:
	ShowWindow(window);

	// WindowPtr zurckgeben:
	return window;
}

//
// Fenster wieder dicht machen (komplett Mac-spezifisch):
//

void kill_display(WindowPtr window)
{
	// Weg damit!
	DisposeWindow(window);
}

//
// Fenster aktivieren/deaktivieren (komplett Mac-spezifisch):
//

void HandleActivate(WindowPtr theWindow, short becomingActive)
{
	// Tut nix, da wir keine Slider und keinen Sizer im Fenster haben
	theWindow = theWindow;
	becomingActive = becomingActive;
}

//
// Mausklicks abarbeiten (komplett Mac-spezifisch):
//

void HandleMouseDown(EventRecord *eventPtr)
{
	short			thePart;		// Was wurde angeklickt?
	WindowPtr		whichWindow;	// Welches Fenster. OK, wir haben nur eins.
	
	// Wohin wurde geklickt?
	thePart = FindWindow(eventPtr->where,&whichWindow);
	switch (thePart)	// Fallunterscheidung
	{
		case inMenuBar	:	// Menue-Klick (wir haben keins!):
							break;
		case inSysWindow: 	// Klick frs System (weiterleiten):
							SystemClick(eventPtr,whichWindow);
							break;
		case inDrag		: 	// Fenster verschieben:
							DragWindow(whichWindow,eventPtr->where,&qd.screenBits.bounds);
							break;
		case inGoAway	:	// Fenster schlieen. Verlt das Programm!
							if(TrackGoAway(whichWindow,eventPtr->where))
								quit_flag = true;	// Raus aus dem Programm 
							break;
	}	// switch
}

//
// Fenster komplett neumalen (komplett Mac-spezifisch):
//

void DrawWindow(WindowPtr window)
{
	int		plx,ply,plw;					// Spielerdaten

	pl.get(&plx,&ply,&plw);					// Spieler-Infos auslesen
	do_raycasting(window,plx,ply,plw);		// Einmal Raycasting machen
}

//
// Redraw abarbeiten (komplett Mac-spezifisch):
//

void DoUpdate(EventRecord *eventPtr)
{
	WindowPtr	theWindow = (WindowPtr) eventPtr->message;

	// Bescheid sagen, da Redraw nun beginnt:
	BeginUpdate(theWindow);

	// Mu man berhaupt etwas tun?
	if(!EmptyRgn(theWindow->visRgn))
	{
		// Ja: Fenster komplett neu malen:
		do_complete_redraw(theWindow);
	}	// if

	// Bescheid sagen, da Redraw nun beendet ist:
	EndUpdate(theWindow);
}

//
// Events abarbeiten (komplett Mac-spezifisch):
//

void DoEvent(WindowPtr window, EventRecord *eventPtr)
{
	int		plx,ply,plw;		// Spielerdaten
	char	theChar;			// gedrckte Taste
	long    myMessage;			
	
	switch (eventPtr->what)	// Welcher Event ist aufgelaufen?
	{
		case mouseDown	: 	// Mausklick:
							HandleMouseDown(eventPtr);
							break;
		case keyDown	:
		case autoKey	:	// Taste
							theChar = eventPtr->message & charCodeMask;
							if(theChar == ' ')	// Umschalten 3D<->Karte?
							{
								// Ja: Umschalten:
								my3d_flag = !my3d_flag;				// Karte <-> 3D
								pl.get(&plx,&ply,&plw);				// Spieler-Infos auslesen
								do_raycasting(window,plx,ply,plw);	// Einmal Raycasting machen
							}
							else
							{
								// Nein: Spielerbewegung?
								if(pl.move(theChar))					// Diese Bewegung mglich?
								{
									// Erfolgreiche Bewegung:
									pl.get(&plx,&ply,&plw);				// Spieler-Infos auslesen
									do_raycasting(window,plx,ply,plw);	// Einmal Raycasting machen
								}	// if
								else
									play_sound(UHH_SOUND_ID);			// Uhh... fr Bewegungen ohne Effekt.
							}	// else
							break;
		case updateEvt	:	// Update-Event (Fenster komplett neu zeichnen):
							DoUpdate(eventPtr);
							break;
		case activateEvt:	// Fenster hochklicken
							HandleActivate((WindowPtr)eventPtr->message,
							(eventPtr->modifiers & activeFlag) != 0 );
							break;
		case app4Evt: 		//Suspend- und Resume-Events
							myMessage = (eventPtr->message >> 24) & 0x0FF;
							if(myMessage == 1)
							{
								if((eventPtr->message&1)!=0)	// Falls mich einer hochklickt,
									InitCursor();				// Cursor wieder auf Pfeil stellen
								HandleActivate(FrontWindow(),((eventPtr->message&1)!=0));
						   	}
						   	break;
	}
}

//
// Haupt-Event-Schleife (komplett Mac-spezifisch):
//

void do_event_loop(WindowPtr window)
{
	int				plx,ply,plw;	// Spielerstandpunkt
	EventRecord		event;			// Puffervariable fr ankommende Events

	// Zunchst einmal die Start-Weltsicht generieren:
	my3d_flag = true;		// 3D-Darstellung ist default
	pl.set(150,320,0);		// Da soll der Spieler hin!
	pl.get(&plx,&ply,&plw);					// Spieler-Infos auslesen
	do_raycasting(window,plx,ply,plw);		// Einmal Raycasting machen

	// Haupt-Event-Schleife hier:
	quit_flag = false;		// Noch soll das Programm nicht verlassen werden.
	do
	{
		// Auf Events warten:
		if(WaitNextEvent(everyEvent,&event,0xFFFFFFFF,nil ))
			DoEvent(window,&event);	// Auf Events reagieren

	} while(!quit_flag);		// So lange bis quit gesetzt wird.
}

//
// Unscharfer Null-Test (fr alle Rechnersysteme):
//

int fast_null(float x)
{
	return (ABS(x) < EPSILON);	// Unscharfer Null-Test
}

//
// Tabellen initialisieren etc. (fr alle Rechnersysteme):
//

// Spielfeld vertikal spiegeln. Ntig, damit vorinitialisiertes
// Feld oben "richtig herum" im Quellcode angezeigt wird.

void flip_maze(void)
{						
	int		zeile,spalte;					// Indizes

	for(spalte=0; spalte<MAZE_X; spalte++)
		for(zeile=0; zeile<MAZE_Y; zeile++)
			maze[spalte][zeile] = maze_user[MAZE_Y-1-zeile][spalte];		// Daten bernehmen
}

void Init_Tabellen(void)		// dx/dy-Tabelle und Winkeltabellen errechnen:
{
	Bresenham	bres;
	float		alpha,d_alpha;	// Streckfaktor, Winkel und Winkel-Inkrement
	int			i;				// Laufvariable

	//
	// Tabelle fr Richtungsvektoren initialisieren:
	//

	// Gre eines Winkelschrittes berechnen (2/Winkelanzahl):
	d_alpha = (2.0 * M_PI) / (float)ANZ_WINKEL;

	// Fr alle Winkel den zugehrigen Richtungsvektor errechnen.
	// Dabei ist der 0. Winkel bei 0 Grad (entspricht rechts).
	for(alpha=0.0, i=0; i<ANZ_WINKEL; i++, alpha+=d_alpha)	// Alle Winkel...
	{
		// Punkte auf dem Einheitskreis als Richtungsvektoren:
		x_winkel_tab[i] = cos(alpha);	// Cos gibt die X-Koordinate an
		y_winkel_tab[i] = sin(alpha);	// Sin gibt die Y-Koordinate an
	}	// for

#ifdef	_NO_BRESENHAM_	// Falls Bresenham so weit wie mglich vermieden werden soll.

	{	// Einschub fr _NO_BRESENHAM_
		float		d;				// Ergebnis der trig. Berechnung
		//
		// Tabelle fr schnelleres dx/dy-Raycasting initialisieren:
		//
		for(i=0; i<ANZ_WINKEL; i++)	// Alle Winkel...
		{
			//
			// Tabelle fr horizontalen Scan generieren:
			//
			if(fast_null(y_winkel_tab[i]))				// Degenerierte Richtung fr hor. Scan?
				hsc_dx_tab[i] = hsc_dy_tab[i] = 0.0;	// "Degenerierte" Richtung fr hor. Scan!
			else
			{
				hsc_dy_tab[i] = ((y_winkel_tab[i] > 0.0) ? TEX_SIZE : -TEX_SIZE);
				d = hsc_dy_tab[i] * (x_winkel_tab[i] / y_winkel_tab[i]);
				if(ABS(d) > MAX_SICHT)
					hsc_dx_tab[i] = hsc_dy_tab[i] = 0.0;// Dieser Winkel ist auch "degeneriert"!
				else
					hsc_dx_tab[i] = d;	// Dieser Wert kann verwendet werden.
			}
			//
			// Tabelle fr vertikalen Scan erstellen:
			//
			if(fast_null(x_winkel_tab[i]))				// Degenerierte Richtung fr vert. Scan?
				vsc_dx_tab[i] = vsc_dy_tab[i] = 0.0;	// "Degenerierte" Richtung fr vert. Scan!
			else
			{
				vsc_dx_tab[i] = ((x_winkel_tab[i] > 0.0) ? TEX_SIZE : -TEX_SIZE);
				d = vsc_dx_tab[i] * (y_winkel_tab[i] / x_winkel_tab[i]);
				if(ABS(d) > MAX_SICHT)
					vsc_dx_tab[i] = vsc_dy_tab[i] = 0.0;// Dieser Winkel ist auch "degeneriert"!
				else
					vsc_dy_tab[i] = d;	// Dieser Wert kann verwendet werden.
			}
		}	// for
	}	// _NO_BRESENHAM_ - Einschub
#endif	// _NO_BRESENHAM_

	//
	// Spielfeld vertikal spiegeln:
	//
	flip_maze();
}

//
// Handle auf eine Offscreen-Gworld des Textur-PICTs im Resource besorgen:
// (komplett Mac-spezifisch)
//

GWorldPtr get_texture_pic(void)
{
	GWorldPtr	oldGW, newWorld;	// Offscreen-Bitmaps
	GDHandle	oldGD;				// altes Device
	Rect		pictRect;			// Rechteck mit Grenangabe des PICTs
	PicHandle	texture_handle;		// Handle auf das geladene Textur-PICT

	// Einen Handle auf das Texturbild besorgen:
	texture_handle = GetPicture(TEXTURE_ID);	// Handle auf das Resource-PICT holen
	if(texture_handle == nil)					// Fehler!!!
		return nil;		// Fehler!

	// Gre des Bildes auslesen:
	pictRect = (**(texture_handle)).picFrame;

	// Schauen, da top und left jeweils 0 sind, damit bottom und right 
	// die Auflsung des Bildes angeben (eventuell berflssig, sicherheitshalber).
	if((pictRect.top!=0) || (pictRect.left!=0))
	{
		// Kann eigentlich nie passieren. Falls doch, dann Rechteck nach (0,0) setzen:
		pictRect.bottom -= pictRect.top;
		pictRect.right -= pictRect.left;
		pictRect.top = pictRect.left = 0;
	}

	// Stimmt die Gre des Bildes mit der Vorgabe fr die Texturgre berein?
	if((pictRect.right!=TEX_SIZE) || (pictRect.bottom!=TEX_SIZE))
	{
		// Die Textur hat die falsche Gre! Aargh!
		KillPicture(texture_handle);	// Bild wieder freigeben
		fatal_error("Das im Resource abgelegte Textur-PICT hat die falsche Gre!");
	}	// if

	// Nun eine Offscreen-Bitmap in passender Gre erzeugen:
	GetGWorld(&oldGW,&oldGD);			// Alter Port und altes Device merken
	if(NewGWorld(&newWorld,TEXTURE_DEPTH,&pictRect,nil,nil,0) != 0)
	{
		// Kein Speicher fr die Offscreen-Bitmap da!
		ReleaseResource((Handle)texture_handle);// Bild wieder freigeben
		fatal_error("Die Offscreen-Bitmap kann fr die Textur kann nicht angelegt werden. Kein Speicher?");
	}

	// Offscreen-Bitmap setzen und PICT reinkopieren:
	LockPixels(GetGWorldPixMap(newWorld));		// Bild ist jetzt nicht verschiebbar, da gezeichnet wird!
	SetGWorld(newWorld,nil);					// Es soll nun in die Offscreen-Bitmap gezeichnet werden.
	DrawPicture(texture_handle,&pictRect);		// Bild nun in die Offscreen-Bitmap malen.

	// Alten Port und Device wieder setzen:
	SetGWorld(oldGW,oldGD);						// Alten Port und Device wieder setzen.
	UnlockPixels(GetGWorldPixMap(newWorld));	// Offscreen-Bitmap darf wieder verschoben werden.

	// PICT-Handle wieder freigeben:
	ReleaseResource((Handle)texture_handle);	// Bild wieder freigeben

	// Schlielich einen Verweis auf die Offscreen-Bitmap liefern:
	return newWorld;	// Bittesehr: Hier ist die neue Offscreen-Bitmap.
}

//
// Framebuffer allokieren (komplett Mac-spezifisch):
//

GWorldPtr get_frame_buffer(void)
{
	GWorldPtr	oldGW, newWorld;	// Offscreen-Bitmaps
	GDHandle	oldGD;				// altes Device
	Rect		frameRect;			// Rechteck mit Grenangabe des Zielbildes

	// Gre des Framebuffers ist XAUFxYAUF:
	SetRect(&frameRect,0,0,XAUF-1,YAUF-1);

	// Nun eine Offscreen-Bitmap in passender Gre erzeugen:
	GetGWorld(&oldGW,&oldGD);			// Alter Port und altes Device merken
	if(NewGWorld(&newWorld,TEXTURE_DEPTH,&frameRect,nil,nil,0) != 0)
	{
		// Kein Speicher fr die Offscreen-Bitmap da!
		fatal_error("Die Framebuffer-Offscreen-Bitmap kann nicht erzeugt werden. Kein Speicher?");
	}

	// Offscreen-Bitmap setzen und prophylaktisch lschen:
	LockPixels(GetGWorldPixMap(newWorld));		// Bild ist jetzt nicht verschiebbar, da gezeichnet wird!
	SetGWorld(newWorld,nil);					// Es soll nun in die Offscreen-Bitmap gezeichnet werden.
	EraseRect(&frameRect);						// Offscreen-Bitmap lschen.

	// Alten Port und Device wieder setzen:
	SetGWorld(oldGW,oldGD);						// Alten Port und Device wieder setzen.
	UnlockPixels(GetGWorldPixMap(newWorld));	// Offscreen-Bitmap darf wieder verschoben werden.

	// Schlielich einen Verweis auf die Offscreen-Bitmap liefern:
	return newWorld;	// Bittesehr!
}

//
// Copyright-Dialog zeigen (komplett Mac-spezifisch):
//

void show_copyright(void)
{
	Alert(COPYR_ALERT_ID,nil);			// Copyright-Dialog zeigen
}

//
// Das Hauptprogramm:
// (grtenteils Mac-spezifisch):
//

void main(void)
{
	WindowPtr	window;					// Verweis auf Fenster

	MaxApplZone();						// Applikations-Speicher verfgbar machen (Mac)
	ToolBoxInit();						// ToolBox-Funktionen initialisieren (Mac)

	texture_gworld=get_texture_pic();	// Textur als Grafik lesen (Mac)
	frame_buffer=get_frame_buffer();	// Framebuffer allokieren (Mac)

	Init_Tabellen();					// Diverse Tabellen vorbelegen (alle Rechnertypen)

	show_copyright();					// Copyright-Dialog zeigen (Mac)

	window=make_display();				// Fenster aufmachen (Mac)
	do_event_loop(window);				// Dann ab in die Event-Schleife (Mac)
	kill_display(window);				// Fenster wieder weg! (Mac)
}	// main

//
// Ende von Raycast.c++
//
